#!/bin/bash
# create archive from current source using git
set -e

NAME="symas-buildtools"
UPSTREAM_TAG_PREFIX="v"

if test -n "$CI_COMMIT_TAG"; then
    # Undo tag mangling (https://dep-team.pages.debian.net/deps/dep14/) and
    # retrieve upstream version name only
    VERSION="$(echo "${CI_COMMIT_TAG}" | sed -e 'y/%_/:~/' -e 's/#//')"
    VERSION="${VERSION#*/"$NAME"-}"
    COMMIT="$CI_COMMIT_TAG"
else
    UPSTREAM_VERSION="$(git describe --abbrev=0 --tags --match "${UPSTREAM_TAG_PREFIX}*")"
    LAST_PACKAGE="$(git describe --first-parent --abbrev=0 --tags | sed -e 'y/%_/:~/' -e 's/#//')"
    CURRENT="$(git describe --tags --long)"

    UPSTREAM_VERSION="${UPSTREAM_VERSION#"${UPSTREAM_TAG_PREFIX}"}"
    LAST_PACKAGE="${LAST_PACKAGE#*/"$NAME"-}"

    if [[ "$CURRENT" =~ / ]]; then
        CURRENT="${CURRENT#*/"$NAME"-}"
        CURRENT="$(echo "$CURRENT" | sed -e 'y/%_/:~/' -e 's/#//')"
    else
        CURRENT="${CURRENT#"${UPSTREAM_TAG_PREFIX}"}"
    fi

    if test "${CURRENT::${#LAST_PACKAGE}}" = "$LAST_PACKAGE"; then
        # Appears to be new packaging in progress based on last released
        # version - make sure we sort accordingly (after the last package
        # *iff* we have local commits)
        VERSION="$(echo "$CURRENT" | sed \
            -e 's/-\([0-9]*\)-\(g[0-9a-f]*\)$/+\1~\2/' -e 's/+0~/~/')"
    elif test "${CURRENT::${#UPSTREAM_VERSION}}" = "${UPSTREAM_VERSION}"; then
        # Appears to be a new upstream version
        VERSION="$UPSTREAM_VERSION~0"
    else
        # What does it mean?
        exit 1
    fi

    COMMIT=HEAD
fi

COMMIT_SHA="$(git rev-parse "$COMMIT")"
IFS=- read -r VERSION RELEASE <<<"$VERSION"

OUTPATH=pkg/archives/dev
NAMEVER="$NAME-v$VERSION"
ARCHIVE="$NAMEVER.tar.xz"
ARPATH="$OUTPATH/$ARCHIVE"

mkdir -p "$OUTPATH"

# We need to tell apkg what local release to encode into the source package
# but due to how our CI is structured, the environment we use to actually
# prepare each source package doesn't necessarily have the information that we
# have now. Keep it near the generated artifacts.
RELEASE_INFO="${OUTPATH}/.buildtools.cache.${COMMIT_SHA}"
rm -f "$RELEASE_INFO"
echo "VERSION=${VERSION@Q}" >>"$RELEASE_INFO"
echo "RELEASE=${RELEASE@Q}" >>"$RELEASE_INFO"

# FIXME for reproducible builds: git archive appears to be deterministic,
# whether its output is stable across versions is another question:
# - If we call git archive with a commit id, it will encode that in the
#   archive, so we wouldn't be able to reproduce the archive across local
#   versions, resolve it to the actual tree-id
faketime -f '1970-01-01 00:00:00' \
git -c tar.tar.xz.command="xz -c" archive --worktree-attributes --format=tar.xz --output "$ARPATH" --prefix "$NAMEVER/" "$COMMIT_SHA^{tree}"

echo "archive: '$ARPATH'"
echo "version: '$VERSION'"
