#!/bin/bash
# Copyright (c) 2022 Symas Corporation
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# * Redistributions of source code must retain the above copyright
# notice, this list of conditions and the following disclaimer.
# * Redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in
# the documentation and/or other materials provided with the
# distribution.
# * Neither the name of the Symas Corporation nor the names of its
# contributors may be used to endorse or promote products derived
# from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# bs.sh
# -----
# Build the Symas OpenLDAP source RPMs in the order in which they need to
# be built and put them in the correct places. The result is a set of
# binary RPMs that comprise Symas OpenLDAP. No root privileges are needed
# to run this script.
#
# WARNING
# This script overwrites the contents of the directory pointed to by
# $syPrefix, $builddir, and $syWork.
#
#set -x
#
# Set file globbing to return the empty string instead of the literal.
shopt -s nullglob

# We *do not* want LD_LIBRARY_PATH to be set
unset LD_LIBRARY_PATH

echo "Creating Symas OpenLDAP binary RPMs from source RPMs"

# Pick up common variables and functions
source ./common.sh

# Check the environment to make sure it's sane
checkenv || exit 1

# Do a little cleanup
rm -rf $buildDir $syWork

deleteInstalledServer || exit 2

# Make our necessary directories
mkdir -p $syWork $rpmSrcDir || exit 3

#
# Loop, installing each of the source RPMs needed for the tarball. If the
# SRPM isn't present, fetch it first and then install it.
#
echo "Installing the necessary SRPMs:"
while read projectName srpmName rpmList; do
    # Fetch the SRPM if it's not here
    if [ ! -f $rpmSrcDir/$srpmName -a -n "$soldRepoUrl" ]; then
        echo "Fetching $srpmName from the Symas repository..."
        wget --progress=dot:mega $soldRepoUrl/$srpmName -P $rpmSrcDir || exit 4
    elif [ ! -f $rpmSrcDir/$srpmName ]; then
        ls -l  $rpmSrcDir/$srpmName && echo "soldRepoUrl=$soldRepoUrl"
        echo "$srpmName not present and no source repo URL defined"
        exit 5
    else
        echo "$srpmName already present"
    fi

    # Install the SRPM
    echo "Installing source RPM $srpmName..."
    rpm -ivh $rpmSrcDir/$srpmName || exit 6
done <<< $projectInfoList

#
# Loop, adding each of the specified patches to the SPEC file for each project.
#
echo "Adding post-release patches to the SPEC files:"
while read projectName srpmName rpmList; do
	projectSpecFile=$buildDir/SPECS/${projectName}.spec
	projectSourcesDir=$buildDir/SOURCES
	projectVersion=$(eval "echo \$$(echo ${projectName/-/_}Ver)")
	projectPatchDir=$patchDir/${projectName}-$projectVersion
	patchNum=500
	unset patchFileList patchDirectiveList
	echo "Looking for post-release patches in $projectPatchDir"
	if [ -d "$projectPatchDir" ]; then
		echo "Adding post-release patches"
		# Copy each patch file into the project's rpmbuild SOURCE directory,
		# then update the spec file.
		for p in $projectPatchDir/*; do
			echo "Found post-release patch $p and adding as patch $patchNum"
			cp $p $projectSourcesDir/
			patchFileList="${patchFileList}Patch${patchNum}: $(basename $p)\\n"
			patchDirectiveList="${patchDirectiveList}%patch$((patchNum++)) -p1\\n"
		done
	fi
	
	# Update the spec file for the project with the patches
	if [ -n "$patchFileList" ]; then
		echo "Updating spec file with post-release patches"
		# If autosetup or autopatch are present then we don't need to
		# do anything.
		if grep "%autosetup" $projectSpecFile; then
			echo "Spec file contains %autosetup, nothing to do."
		elif grep "%autopatch" $projectSpecFile; then
			echo "Spec file contains %autopatch, unable to patch."
			exit 7
		else
			# Add the Patch directives in the main section of the file.
			# We simply add it to the beginning of the file because there's
			# no good anchor to place it after. This is not pretty, but
			# should work 100% of the time.
			sed -i -e "1i # Define post-release patches:\\n$patchFileList"\
				 $projectSpecFile || exit 8

			# Next add the %patch{N} directive after the %setup directive
			# We won't worry about patch ordering, so we can start
			# inserting patch directives immediately after the %setup.
			# The strategy here is to read lines from the spec work file
			# until we come across a line that starts with %setup.
			# Then we insert the contents of the patch directive
			# and write the rest of the file out.
			sed -i -e "/^%setup/a # Apply post-release patches:\\n$patchDirectiveList"\
				$projectSpecFile || exit 9
		
			# Update the changelog section so that we know this RPM
			# has been patched 
			sed -i -e "/%changelog/a * $(date "+%a %b %d %Y") $patchUserInfo($0) - ${projectVersion}X\n* Apply post-release patches"\
				$projectSpecFile || exit 10
		fi
	else
		echo "No post-release patches for $projectName $projectVersion"
	fi
	# Disable symbol table stripping:
	 sed -i -e "1i %define debug_package %{nil}\n%define __strip /bin/true"\
		$projectSpecFile || exit 11

done <<< $projectInfoList
#
# Build each of the SRPMs, installing each of the RPMs that an SRPM produced.
while read projectName srpmName rpmList; do
    # Build the RPMs
    cd $buildDir/SPECS || exit 12
    echo Building $srpmName...
	projectSpecFile=$buildDir/SPECS/${projectName/_/-}.spec
    rpmbuild -bb -D "_syprefix $syPrefix" -D "_varfix $varFix" \
	 --nodeps $(basename $projectSpecFile) || exit 13

    # Install the RPM(s) it produced that we'll need to build the next
    # ones
    for r in $rpmList; do
        echo Fake-installing built rpm $r
        rpm2cpio $buildDir/RPMS/x86_64/$r | cpio -id -D $syWork || exit 14
        cp -r $syWork/$syPrefixRoot/* $syPrefixRoot || exit 15
        rm -r $syWork || exit 16
    done
done <<< $projectInfoList

# Clean up the working directories
deleteInstalledServer

echo "Symas OpenLDAP binary RPM creation is complete."

