#!/bin/sh
#########################################################################
#
# Copyright (c) 2007-2021 Symas Corporation
# All rights reserved.
# 
# Redistribution and use in source and binary forms, with or without 
# modification, are permitted provided that the following conditions are 
# met:
# 
# * Redistributions of source code must retain the above copyright 
#   notice, this list of conditions and the following disclaimer.
# * Redistributions in binary form must reproduce the above 
#   copyright notice, this list of conditions and the following disclaimer 
#   in the documentation and/or other materials provided with the 
#   distribution.
# * Neither the name of the Symas Corporation nor the names of its 
#   contributors may be used to endorse or promote products derived from 
#   this software without specific prior written permission.
# 
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# "portably" echo without a newline
pecho () {
	if [ "x$(echo -n foo\cbar)x" = "x-n foox" ]; then
		# ancient bourne shells only have \c
		echo $1\\c
	elif [ "x$(echo -n foo\\cbar)x" = "x-n foox" ]; then
		# some shells unquote differently
		echo $1\\c
	elif [ "x$(echo -n foo\cbar)x" = "xfoocbarx" ]; then
		# bash likes it csh style
		echo -n $1
	elif [ "x$(echo -n foo\\cbar)x" = "xfoocbarx" ]; then
		# some shells unquote differently
		echo -n $1
	else
		# ouch
		echo $1
	fi
}

# Must be root to run this script
if [ "$(id -ru)" -ne "0" ]; then
	echo "Because this script writes to directories that are owned by root,"
	echo "you must be root to run this script."
	exit 1
fi

#
# Set sane values for these variables. If you downloaded this script from 
# the Symas Forum or if you've moved either of these directories around
# you may need to set these to match your distribution.
BIN=/opt/symas/bin
SBIN=/opt/symas/sbin
LIB=/opt/symas/lib

if [ ! -x $LIB/slapd ]; then
	echo "Can't locate slapd executable. This script will exit."
	exit 1
fi

if [ ! -x $BIN/ldapsearch ]; then
	echo "Can't locate ldapsearch executable. This script will exit."
	exit 1
fi

STARTSERVER="systemctl start slapd"
STOPSERVER="systemctl stop slapd"

EXAMPLEDIR=/var/symas/openldap-data/example
SLAPD_CONF=/opt/symas/etc/openldap/slapd.conf
CONFIGDIR=/opt/symas/etc/openldap/slapd.d

SLAPD_GROUP=root

TESTSEARCH="$BIN/ldapsearch -x -H ldap://localhost:389/ -b dc=example,dc=com -D dc=example,dc=com -w secret"

cat <<EOF

                     *****  CONFIGURATION SELECTION *****

This script can either use the deprecated slapd.conf configuration file or
the cn=config configuration database.  Please choose:
1) slapd.conf
2) cn=config

EOF

pecho "Select 1 or 2, anything else to cancel: "
read config
if [ "$config" != "1" -a "$config" != "2" ]; then
  echo "Exiting.  No changes have been made to the system."
  exit 1
fi

if [ "$config" = "1" -a -d $CONFIGDIR ]; then
  echo "                     *****  WARNING *****"
  echo "Existing cn=config configuration directory present"
  echo "Enter YES to permanently ERASE this configuration."
  echo ""
  pecho "Type YES to continue, anything else to cancel: "
  read yn
  if [ "$yn" != "YES" ]; then
    exit 1
  else
    echo "Deleting contents of $CONFIGDIR..."
    rm -rf $CONFIGDIR
  fi
fi

if [ "$config" = "1" ]; then
  cat <<EOF


                     *****  WARNING *****

This script will set up an example database called dc=example,dc=com
and configure Symas OpenLDAP to use this database.

In the process it will delete the contents of $EXAMPLEDIR
and replace the following file with an example version:
    $SLAPD_CONF

Any previously existing versions of these files will be lost!

EOF
fi

if [ "$config" = "2" ]; then
  cat <<EOF


                     *****  WARNING *****

This script will set up an example database called dc=example,dc=com
and configure Symas OpenLDAP to use this database.

In the process it will delete the contents of $EXAMPLEDIR and $CONFIGDIR.

Any previously existing version of this file will be lost!

EOF
fi

pecho "Type YES to continue, anything else to cancel: "
read yn
if [ "$yn" != "YES" ]; then
	echo "Sample database creation has NOT taken place. No changes"
	echo "have been made to your system."
	exit 1
fi

$STOPSERVER

echo "Deleting contents of $EXAMPLEDIR..."
mkdir -p $EXAMPLEDIR 2>/dev/null
rm $EXAMPLEDIR/* 2>/dev/null

if [ "$config" = "2" ]; then
  echo "Deleting contents of $CONFIGDIR..."
  rm -rf $CONFIGDIR
  mkdir -p $CONFIGDIR
fi

if [ "$config" = "1" ]; then
    echo "Creating $SLAPD_CONF..."
    cat > $SLAPD_CONF <<EOF
#
# See slapd.conf(5) for details on configuration options.
# This file should NOT be world readable.
#
# Schema files. Note that not all of these schemas co-exist peacefully.
# Use only those you need and leave the rest commented out.
include     /opt/symas/etc/openldap/schema/core.schema
include     /opt/symas/etc/openldap/schema/cosine.schema
include     /opt/symas/etc/openldap/schema/inetorgperson.schema

# Files in which to store the process id and startup arguments.
# These files are needed by the init scripts, so only change
# these if you are prepared to edit those scripts as well.
pidfile         /var/symas/run/slapd.pid
argsfile        /var/symas/run/slapd.args

# Set the log level
loglevel stats sync

# Choose the directory for loadable modules.
modulepath  /opt/symas/lib/openldap

# Uncomment the moduleloads as needed to enable backend
# functionality.
# Load dynamic backend modules:
moduleload  back_mdb.la
moduleload  back_monitor.la

# Example access control policy:
#   Allow read access of root DSE
#   Allow self write access
#   Allow authenticated users read access
#   Allow anonymous users to authenticate
# Directives needed to implement policy:
access to dn="" by * read
access to *
    by self write
    by sockurl="^ldapi:///$" write
    by users read
    by anonymous auth
#
# if no access controls are present, the default policy is:
#   Allow read by all
#
# rootdn can always write!

#######################################################################
# Example mdb database definitions
#######################################################################
database    mdb
suffix      "dc=example,dc=com"
rootdn      "dc=example,dc=com"
# Cleartext passwords, especially for the rootdn, should
# be avoided. See slappasswd(8) and slapd.conf(5) for details describing
# the creation of encrypted passwords.
rootpw      secret

# Indices to maintain

# index default sets the basic type of indexing to perform if there isn't
# any indexing specified for a given attribute
index   default     eq
index   objectClass
index   cn

# The database directory MUST exist prior to running slapd AND
# should only be accessible by the slapd/tools. Mode 700 recommended.
# One directory will be needed for each backend, so you should
# create a subdirectory beneath /var/symas/openldap-data for each
# new backend. This is also where the DB_CONFIG file needs to be
# placed.
directory   /var/symas/openldap-data/example

# Here we specify the maximum on-disk size of the database. It is
# Recommended to set this near the expected free-space availability
# for the machine. This paramiter is not pre-allocated and simply
# represents the upward limit to which the database will be allowed
# to grow. Note: Specified in *bytes*. Here, we set it to 1gb.
maxsize 1073741824

#######################################################################
# Monitor database
#######################################################################
database    monitor
EOF
fi

if [ "$config" = "2" ]; then
  echo "Creating server configuration..."
  $SBIN/slapadd -q -n 0 -F $CONFIGDIR <<EOF
dn: cn=config
objectClass: olcGlobal
cn: config
olcLogLevel: Sync
olcLogLevel: Stats
olcPidFile: /var/symas/run/slapd.pid
olcArgsFile: /var/symas/run/slapd.args

dn: cn=schema,cn=config
objectClass: olcSchemaConfig
cn: schema

include: file:///opt/symas/etc/openldap/schema/core.ldif
include: file:///opt/symas/etc/openldap/schema/cosine.ldif
include: file:///opt/symas/etc/openldap/schema/inetorgperson.ldif

dn: cn=module{0},cn=config
objectClass: olcModuleList
cn: module{0}
olcModulePath: /opt/symas/lib/openldap
olcModuleLoad: {0}back_mdb.la
olcModuleLoad: {1}back_monitor.la

dn: olcDatabase={-1}frontend,cn=config
objectClass: olcDatabaseConfig
objectClass: olcFrontendConfig
olcDatabase: {-1}frontend
olcAccess: {0}to dn=""  by * read
olcAccess: {1}to *  by self write  by sockurl.exact="ldapi:///" write  by users read  by anonymous auth

dn: olcDatabase={0}config,cn=config
objectClass: olcDatabaseConfig
olcDatabase: {0}config
olcRootPW: secret
olcAccess: {0}to *  by * none

dn: olcDatabase={1}mdb,cn=config
objectClass: olcDatabaseConfig
objectClass: olcMdbConfig
olcDatabase: {1}mdb
olcSuffix: dc=example,dc=com
olcRootDN: dc=example,dc=com
olcRootPw: secret
olcDbDirectory: /var/symas/openldap-data/example
olcDbIndex: default eq
olcDbIndex: objectClass
olcDbIndex: cn
olcDbMaxSize: 1073741824

dn: olcDatabase={2}monitor,cn=config
objectClass: olcDatabaseConfig
olcDatabase: {2}monitor
olcRootDn: cn=config
olcMonitoring: FALSE
EOF

fi

echo "Creating the example database..."
$SBIN/slapadd -q <<EOF
dn: dc=example,dc=com
objectClass: top
objectClass: organization
objectClass: dcObject
o: example
dc: example

dn: ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: organizationalunit
ou: Accounting

dn: ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: organizationalunit
ou: Product Development

dn: ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: organizationalunit
ou: Product Testing

dn: ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: organizationalunit
ou: Human Resources

dn: ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: organizationalunit
ou: Payroll

dn: ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: organizationalunit
ou: Janitorial

dn: ou=Management, dc=example,dc=com
objectClass: top
objectClass: organizationalunit
ou: Management

dn: ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: organizationalunit
ou: Administrative

dn: ou=Peons, dc=example,dc=com
objectClass: top
objectClass: organizationalunit
ou: Peons

dn: ou=Planning, dc=example,dc=com
objectClass: top
objectClass: organizationalunit
ou: Planning

dn: ou=KerberosPrincipals, dc=example,dc=com
objectClass: top
objectClass: organizationalunit
ou: KerberosPrincipals

dn: cn=Katha Petree, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Katha Petree
sn: Petree
description: This is Katha Petree's description
facsimileTelephoneNumber: +1 804 572-2449
l: Milpitas
ou: Peons
postalAddress: example$Peons$Dept # 533
telephoneNumber: +1 408 136-9364
title: Supreme Peons President
userPassword: eertePahta
uid: Katha_Petree
givenName: Katha
mail: Katha_Petree@example.com
carLicense: YIN9D3G
departmentNumber: 2518
employeeType: Employee
homePhone: +1 303 811-5175
initials: K. P.
mobile: +1 510 729-3926
pager: +1 804 321-1156
manager: cn=Crissie Wayler
secretary: cn=Mer Percy
roomNumber: 9527

dn: cn=Te-Wei Menashian, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Te-Wei Menashian
sn: Menashian
description: This is Te-Wei Menashian's description
facsimileTelephoneNumber: +1 510 733-9612
l: Emeryville
ou: Peons
postalAddress: example$Peons$Dept # 710
telephoneNumber: +1 818 936-6205
title: Senior Peons Sales Rep
userPassword: naihsaneMi
uid: Te-Wei_Menashian
givenName: Te-Wei
mail: Te-Wei_Menashian@example.com
carLicense: E9DAV2R
departmentNumber: 4375
employeeType: Contract
homePhone: +1 303 330-7311
initials: T. M.
mobile: +1 415 242-6860
pager: +1 818 979-7582
manager: cn=Deryck Gramiak
secretary: cn=Emelyne Settels
roomNumber: 1704

dn: cn=Hung Nehring, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Hung Nehring
sn: Nehring
description: This is Hung Nehring's description
facsimileTelephoneNumber: +1 804 930-6996
l: San Jose
ou: Product Development
postalAddress: example$Product Development$Dept # 291
telephoneNumber: +1 804 594-3030
title: Associate Product Development Accountant
userPassword: gnirheNgnu
uid: Hung_Nehring
givenName: Hung
mail: Hung_Nehring@example.com
carLicense: ZB0UL1J
departmentNumber: 1687
employeeType: Temp
homePhone: +1 71 278-6297
initials: H. N.
mobile: +1 71 631-8088
pager: +1 213 531-8152
manager: cn=Mkt Silgardo
secretary: cn=Kien-Nghiep McKeage
roomNumber: 1611

dn: cn=Grant Dransfield, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Grant Dransfield
sn: Dransfield
description: This is Grant Dransfield's description
facsimileTelephoneNumber: +1 71 211-2977
l: Fremont
ou: Planning
postalAddress: example$Planning$Dept # 384
telephoneNumber: +1 206 134-2715
title: Senior Planning Mascot
userPassword: dleifsnarD
uid: Grant_Dransfield
givenName: Grant
mail: Grant_Dransfield@example.com
carLicense: S3E00VH
departmentNumber: 6184
employeeType: Employee
homePhone: +1 510 360-6966
initials: G. D.
mobile: +1 71 658-3718
pager: +1 804 923-4914
manager: cn=Paulie Saisho
secretary: cn=Sarette Valia
roomNumber: 8490

dn: cn=Greta Ifill, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Greta Ifill
sn: Ifill
description: This is Greta Ifill's description
facsimileTelephoneNumber: +1 71 387-9510
l: San Mateo
ou: Product Development
postalAddress: example$Product Development$Dept # 25
telephoneNumber: +1 71 864-3915
title: Chief Product Development Figurehead
userPassword: llifIaterG
uid: Greta_Ifill
givenName: Greta
mail: Greta_Ifill@example.com
carLicense: T2TYGBB
departmentNumber: 8443
employeeType: Manager
homePhone: +1 818 117-6606
initials: G. I.
mobile: +1 804 857-3067
pager: +1 818 180-9782
manager: cn=Pulak Locicero
secretary: cn=Venkataraman Hurd
roomNumber: 2397

dn: cn=Ursa Kitzmiller, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ursa Kitzmiller
sn: Kitzmiller
description: This is Ursa Kitzmiller's description
facsimileTelephoneNumber: +1 71 793-7201
l: Redmond
ou: Janitorial
postalAddress: example$Janitorial$Dept # 293
telephoneNumber: +1 818 107-3194
title: Junior Janitorial Madonna
userPassword: rellimztiK
uid: Ursa_Kitzmiller
givenName: Ursa
mail: Ursa_Kitzmiller@example.com
carLicense: 3WCAXAW
departmentNumber: 4704
employeeType: Contract
homePhone: +1 415 399-6169
initials: U. K.
mobile: +1 804 365-7746
pager: +1 408 927-5149
manager: cn=Nedi Ashraf
secretary: cn=Hedda Curley
roomNumber: 1939

dn: cn=Pammi Valente, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Pammi Valente
sn: Valente
description: This is Pammi Valente's description
facsimileTelephoneNumber: +1 415 463-6393
l: Milpitas
ou: Planning
postalAddress: example$Planning$Dept # 780
telephoneNumber: +1 303 977-5996
title: Elite Planning Consultant
userPassword: etnelaVimm
uid: Pammi_Valente
givenName: Pammi
mail: Pammi_Valente@example.com
carLicense: YBQ9A2W
departmentNumber: 5643
employeeType: Employee
homePhone: +1 804 224-3675
initials: P. V.
mobile: +1 415 431-6328
pager: +1 213 684-9761
manager: cn=Melisse Nelon
secretary: cn=Gaby Ligon
roomNumber: 3619

dn: cn=Tineke Metler, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tineke Metler
sn: Metler
description: This is Tineke Metler's description
facsimileTelephoneNumber: +1 510 510-1291
l: San Mateo
ou: Peons
postalAddress: example$Peons$Dept # 36
telephoneNumber: +1 415 934-5428
title: Master Peons Admin
userPassword: relteMeken
uid: Tineke_Metler
givenName: Tineke
mail: Tineke_Metler@example.com
carLicense: BBPPUFQ
departmentNumber: 5007
employeeType: Temp
homePhone: +1 415 387-4578
initials: T. M.
mobile: +1 71 214-7101
pager: +1 818 907-4547
manager: cn=Andrei Hopley
secretary: cn=Lian Wrigglesworth
roomNumber: 6303

dn: cn=Gleda Klamner, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gleda Klamner
sn: Klamner
description: This is Gleda Klamner's description
facsimileTelephoneNumber: +1 804 506-9247
l: Armonk
ou: Product Testing
postalAddress: example$Product Testing$Dept # 214
telephoneNumber: +1 213 136-1420
title: Chief Product Testing Figurehead
userPassword: renmalKade
uid: Gleda_Klamner
givenName: Gleda
mail: Gleda_Klamner@example.com
carLicense: SMM8BG0
departmentNumber: 6925
employeeType: Temp
homePhone: +1 213 179-6240
initials: G. K.
mobile: +1 303 485-9637
pager: +1 510 470-8588
manager: cn=Ginnie Todd
secretary: cn=Nessie Buley
roomNumber: 9564

dn: cn=Starr Kasumovich, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Starr Kasumovich
sn: Kasumovich
description: This is Starr Kasumovich's description
facsimileTelephoneNumber: +1 408 275-9420
l: Alameda
ou: Janitorial
postalAddress: example$Janitorial$Dept # 113
telephoneNumber: +1 804 885-9038
title: Chief Janitorial Punk
userPassword: hcivomusaK
uid: Starr_Kasumovich
givenName: Starr
mail: Starr_Kasumovich@example.com
carLicense: ED7JA85
departmentNumber: 2291
employeeType: Contract
homePhone: +1 818 747-5196
initials: S. K.
mobile: +1 206 937-1390
pager: +1 510 397-4977
manager: cn=Marline Saul
secretary: cn=Mildred Eggleton
roomNumber: 6557

dn: cn=Zhanna Briere, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zhanna Briere
sn: Briere
description: This is Zhanna Briere's description
facsimileTelephoneNumber: +1 510 732-4159
l: San Francisco
ou: Payroll
postalAddress: example$Payroll$Dept # 612
telephoneNumber: +1 206 509-2872
title: Elite Payroll Consultant
userPassword: ereirBanna
uid: Zhanna_Briere
givenName: Zhanna
mail: Zhanna_Briere@example.com
carLicense: JD7AX71
departmentNumber: 4210
employeeType: Temp
homePhone: +1 804 227-1032
initials: Z. B.
mobile: +1 71 146-3040
pager: +1 206 635-5658
manager: cn=Katusha Brunton
secretary: cn=Essie Hocutt
roomNumber: 9432

dn: cn=Dimitri Overby, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dimitri Overby
sn: Overby
description: This is Dimitri Overby's description
facsimileTelephoneNumber: +1 71 767-6661
l: Cupertino
ou: Peons
postalAddress: example$Peons$Dept # 572
telephoneNumber: +1 303 149-9698
title: Senior Peons Janitor
userPassword: ybrevOirti
uid: Dimitri_Overby
givenName: Dimitri
mail: Dimitri_Overby@example.com
carLicense: LU1FF8A
departmentNumber: 8962
employeeType: Contract
homePhone: +1 71 984-7586
initials: D. O.
mobile: +1 408 143-8879
pager: +1 804 763-8868
manager: cn=Reed Research
secretary: cn=Dorice Hollandsworth
roomNumber: 6199

dn: cn=Farrand Bhandari, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Farrand Bhandari
sn: Bhandari
description: This is Farrand Bhandari's description
facsimileTelephoneNumber: +1 804 494-2129
l: Armonk
ou: Product Development
postalAddress: example$Product Development$Dept # 703
telephoneNumber: +1 415 115-9048
title: Chief Product Development Fellow
userPassword: iradnahBdn
uid: Farrand_Bhandari
givenName: Farrand
mail: Farrand_Bhandari@example.com
carLicense: 37D7RP9
departmentNumber: 8802
employeeType: Contract
homePhone: +1 408 325-2076
initials: F. B.
mobile: +1 303 869-5982
pager: +1 206 701-5301
manager: cn=Herronald Humphrey
secretary: cn=Christel Arnon
roomNumber: 9817

dn: cn=Loralee Homayoun, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Loralee Homayoun
sn: Homayoun
description: This is Loralee Homayoun's description
facsimileTelephoneNumber: +1 415 120-4563
l: Orem
ou: Product Development
postalAddress: example$Product Development$Dept # 869
telephoneNumber: +1 408 390-6261
title: Elite Product Development Grunt
userPassword: nuoyamoHee
uid: Loralee_Homayoun
givenName: Loralee
mail: Loralee_Homayoun@example.com
carLicense: B7UAB5A
departmentNumber: 1016
employeeType: Employee
homePhone: +1 804 572-3652
initials: L. H.
mobile: +1 818 511-6642
pager: +1 213 946-8035
manager: cn=Michaella Frierson
secretary: cn=Bernelle Hannah
roomNumber: 761

dn: cn=Ophelie Benedek, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ophelie Benedek
sn: Benedek
description: This is Ophelie Benedek's description
facsimileTelephoneNumber: +1 510 372-2586
l: Orem
ou: Management
postalAddress: example$Management$Dept # 307
telephoneNumber: +1 804 720-3031
title: Associate Management Manager
userPassword: kedeneBeil
uid: Ophelie_Benedek
givenName: Ophelie
mail: Ophelie_Benedek@example.com
carLicense: XXB309G
departmentNumber: 3029
employeeType: Contract
homePhone: +1 510 234-2942
initials: O. B.
mobile: +1 71 233-3279
pager: +1 206 689-9631
manager: cn=Noelyn Ramlogan
secretary: cn=Alida Hanna
roomNumber: 5444

dn: cn=Justina Steeves, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Justina Steeves
sn: Steeves
description: This is Justina Steeves's description
facsimileTelephoneNumber: +1 818 599-1461
l: Cambridge
ou: Human Resources
postalAddress: example$Human Resources$Dept # 786
telephoneNumber: +1 213 825-8065
title: Senior Human Resources Consultant
userPassword: seveetSani
uid: Justina_Steeves
givenName: Justina
mail: Justina_Steeves@example.com
carLicense: M5YX6UO
departmentNumber: 1162
employeeType: Temp
homePhone: +1 510 209-9450
initials: J. S.
mobile: +1 71 203-4372
pager: +1 71 508-6628
manager: cn=Cathal Soumis
secretary: cn=Dulce Modl
roomNumber: 9123

dn: cn=Marina Lemaire, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marina Lemaire
sn: Lemaire
description: This is Marina Lemaire's description
facsimileTelephoneNumber: +1 71 485-6032
l: Palo Alto
ou: Human Resources
postalAddress: example$Human Resources$Dept # 549
telephoneNumber: +1 213 522-7876
title: Chief Human Resources Czar
userPassword: eriameLani
uid: Marina_Lemaire
givenName: Marina
mail: Marina_Lemaire@example.com
carLicense: RU6R1SJ
departmentNumber: 5315
employeeType: Manager
homePhone: +1 818 245-4519
initials: M. L.
mobile: +1 213 388-6377
pager: +1 71 509-9816
manager: cn=Coriss Rynties
secretary: cn=Pegeen Postlethwaite
roomNumber: 2719

dn: cn=Patches Derbyshire, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Patches Derbyshire
sn: Derbyshire
description: This is Patches Derbyshire's description
facsimileTelephoneNumber: +1 415 501-8333
l: San Jose
ou: Janitorial
postalAddress: example$Janitorial$Dept # 190
telephoneNumber: +1 206 126-2236
title: Junior Janitorial Mascot
userPassword: erihsybreD
uid: Patches_Derbyshire
givenName: Patches
mail: Patches_Derbyshire@example.com
carLicense: TQD663V
departmentNumber: 4594
employeeType: Contract
homePhone: +1 408 841-2501
initials: P. D.
mobile: +1 510 287-3008
pager: +1 510 891-6326
manager: cn=Ernestine Dias
secretary: cn=Leanne Bivens
roomNumber: 6527

dn: cn=Tildy Hoskin, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tildy Hoskin
sn: Hoskin
description: This is Tildy Hoskin's description
facsimileTelephoneNumber: +1 415 649-7690
l: Santa Clara
ou: Peons
postalAddress: example$Peons$Dept # 312
telephoneNumber: +1 408 344-5686
title: Master Peons Yahoo
userPassword: niksoHydli
uid: Tildy_Hoskin
givenName: Tildy
mail: Tildy_Hoskin@example.com
carLicense: QUK1330
departmentNumber: 7288
employeeType: Normal
homePhone: +1 213 701-5449
initials: T. H.
mobile: +1 415 692-1013
pager: +1 71 787-5332
manager: cn=Wan Kramer
secretary: cn=Cocos Hinchey
roomNumber: 1357

dn: cn=Lrc Melanson, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lrc Melanson
sn: Melanson
description: This is Lrc Melanson's description
facsimileTelephoneNumber: +1 206 315-2540
l: Orem
ou: Administrative
postalAddress: example$Administrative$Dept # 765
telephoneNumber: +1 804 281-1079
title: Elite Administrative President
userPassword: nosnaleMcr
uid: Lrc_Melanson
givenName: Lrc
mail: Lrc_Melanson@example.com
carLicense: D5XAYYD
departmentNumber: 2311
employeeType: Contract
homePhone: +1 804 633-5469
initials: L. M.
mobile: +1 303 663-9879
pager: +1 804 308-9318
manager: cn=Jordanna Thornton
secretary: cn=Samuel Bawek
roomNumber: 2439

dn: cn=Bengt Quigley, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bengt Quigley
sn: Quigley
description: This is Bengt Quigley's description
facsimileTelephoneNumber: +1 213 914-4098
l: Fremont
ou: Planning
postalAddress: example$Planning$Dept # 46
telephoneNumber: +1 510 718-5158
title: Master Planning Janitor
userPassword: yelgiuQtgn
uid: Bengt_Quigley
givenName: Bengt
mail: Bengt_Quigley@example.com
carLicense: 9RKXD9Z
departmentNumber: 5930
employeeType: Employee
homePhone: +1 303 989-2850
initials: B. Q.
mobile: +1 71 380-9979
pager: +1 71 980-1170
manager: cn=Rosina Verification
secretary: cn=Antonie Barksdale
roomNumber: 8018

dn: cn=Jordanna FASTONE, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jordanna FASTONE
sn: FASTONE
description: This is Jordanna FASTONE's description
facsimileTelephoneNumber: +1 71 158-1818
l: Milpitas
ou: Human Resources
postalAddress: example$Human Resources$Dept # 637
telephoneNumber: +1 804 928-6588
title: Associate Human Resources Evangelist
userPassword: ENOTSAFann
uid: Jordanna_FASTONE
givenName: Jordanna
mail: Jordanna_FASTONE@example.com
carLicense: SEPIWM6
departmentNumber: 59
employeeType: Normal
homePhone: +1 71 396-4665
initials: J. F.
mobile: +1 213 466-1002
pager: +1 303 261-1330
manager: cn=Kiri Register
secretary: cn=Wanda Urbick
roomNumber: 1657

dn: cn=Lilia Lalu, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lilia Lalu
sn: Lalu
description: This is Lilia Lalu's description
facsimileTelephoneNumber: +1 804 550-2153
l: Redwood Shores
ou: Product Testing
postalAddress: example$Product Testing$Dept # 436
telephoneNumber: +1 206 165-2522
title: Associate Product Testing Artist
userPassword: ulaLailiL
uid: Lilia_Lalu
givenName: Lilia
mail: Lilia_Lalu@example.com
carLicense: D17UMBF
departmentNumber: 5441
employeeType: Temp
homePhone: +1 71 298-3798
initials: L. L.
mobile: +1 510 529-2970
pager: +1 71 655-2394
manager: cn=Laverna Kearney
secretary: cn=Ethelda Gaffney
roomNumber: 6209

dn: cn=Manami Varano, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Manami Varano
sn: Varano
description: This is Manami Varano's description
facsimileTelephoneNumber: +1 818 724-3231
l: Fremont
ou: Administrative
postalAddress: example$Administrative$Dept # 759
telephoneNumber: +1 818 663-9466
title: Chief Administrative Artist
userPassword: onaraViman
uid: Manami_Varano
givenName: Manami
mail: Manami_Varano@example.com
carLicense: VDYGUJ1
departmentNumber: 2127
employeeType: Temp
homePhone: +1 818 115-2547
initials: M. V.
mobile: +1 415 950-1894
pager: +1 213 491-2978
manager: cn=Ulla Sirevicius
secretary: cn=Norma Dwyer
roomNumber: 7220

dn: cn=Nike Dorn, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Nike Dorn
sn: Dorn
description: This is Nike Dorn's description
facsimileTelephoneNumber: +1 818 538-9631
l: Emeryville
ou: Planning
postalAddress: example$Planning$Dept # 109
telephoneNumber: +1 71 308-5243
title: Associate Planning Pinhead
userPassword: nroDekiN
uid: Nike_Dorn
givenName: Nike
mail: Nike_Dorn@example.com
carLicense: 1UAP19L
departmentNumber: 1302
employeeType: Normal
homePhone: +1 510 512-3464
initials: N. D.
mobile: +1 71 484-2404
pager: +1 303 101-4105
manager: cn=Kirstie Bartholomew
secretary: cn=Nady Shirai
roomNumber: 2603

dn: cn=Denys Cooper, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Denys Cooper
sn: Cooper
description: This is Denys Cooper's description
facsimileTelephoneNumber: +1 71 560-5439
l: Palo Alto
ou: Payroll
postalAddress: example$Payroll$Dept # 36
telephoneNumber: +1 818 205-1817
title: Elite Payroll Grunt
userPassword: repooCsyne
uid: Denys_Cooper
givenName: Denys
mail: Denys_Cooper@example.com
carLicense: 7MYC39T
departmentNumber: 2073
employeeType: Contract
homePhone: +1 408 908-8099
initials: D. C.
mobile: +1 71 315-8213
pager: +1 213 589-9770
manager: cn=Kishore Denison
secretary: cn=Daphene Dragan
roomNumber: 7412

dn: cn=Marjan Lukassen, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marjan Lukassen
sn: Lukassen
description: This is Marjan Lukassen's description
facsimileTelephoneNumber: +1 213 952-1994
l: San Mateo
ou: Janitorial
postalAddress: example$Janitorial$Dept # 919
telephoneNumber: +1 408 188-6244
title: Elite Janitorial Developer
userPassword: nessakuLna
uid: Marjan_Lukassen
givenName: Marjan
mail: Marjan_Lukassen@example.com
carLicense: QO7KGQI
departmentNumber: 1650
employeeType: Manager
homePhone: +1 206 500-9364
initials: M. L.
mobile: +1 415 161-6485
pager: +1 303 227-1193
manager: cn=Truus Hoelscher
secretary: cn=Jacquelyn Roseland
roomNumber: 4938

dn: cn=Salis Lundhild, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Salis Lundhild
sn: Lundhild
description: This is Salis Lundhild's description
facsimileTelephoneNumber: +1 213 604-7319
l: Orem
ou: Janitorial
postalAddress: example$Janitorial$Dept # 246
telephoneNumber: +1 415 804-9623
title: Supreme Janitorial Vice President
userPassword: dlihdnuLsi
uid: Salis_Lundhild
givenName: Salis
mail: Salis_Lundhild@example.com
carLicense: YI55DU4
departmentNumber: 844
employeeType: Manager
homePhone: +1 510 223-2526
initials: S. L.
mobile: +1 408 422-8947
pager: +1 206 384-4202
manager: cn=Shirin Cech
secretary: cn=Cherilyn Croxford
roomNumber: 5201

dn: cn=Penang Nava, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Penang Nava
sn: Nava
description: This is Penang Nava's description
facsimileTelephoneNumber: +1 818 691-4786
l: Sunnyvale
ou: Human Resources
postalAddress: example$Human Resources$Dept # 718
telephoneNumber: +1 213 146-9070
title: Senior Human Resources Developer
userPassword: avaNgnaneP
uid: Penang_Nava
givenName: Penang
mail: Penang_Nava@example.com
carLicense: U434FGT
departmentNumber: 7310
employeeType: Manager
homePhone: +1 303 258-9102
initials: P. N.
mobile: +1 206 778-4551
pager: +1 213 829-2714
manager: cn=Julius Satkunaseelan
secretary: cn=Kynthia Dziamba
roomNumber: 4716

dn: cn=Pieter Nilakantan, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Pieter Nilakantan
sn: Nilakantan
description: This is Pieter Nilakantan's description
facsimileTelephoneNumber: +1 510 891-1705
l: Menlo Park
ou: Administrative
postalAddress: example$Administrative$Dept # 730
telephoneNumber: +1 213 227-9259
title: Master Administrative Evangelist
userPassword: natnakaliN
uid: Pieter_Nilakantan
givenName: Pieter
mail: Pieter_Nilakantan@example.com
carLicense: FH3QN9L
departmentNumber: 268
employeeType: Employee
homePhone: +1 804 822-6008
initials: P. N.
mobile: +1 206 453-3297
pager: +1 818 647-1304
manager: cn=Claus Myroon
secretary: cn=Rec Mickens
roomNumber: 5653

dn: cn=Nooshin Kramar, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Nooshin Kramar
sn: Kramar
description: This is Nooshin Kramar's description
facsimileTelephoneNumber: +1 408 392-5438
l: Milpitas
ou: Planning
postalAddress: example$Planning$Dept # 380
telephoneNumber: +1 510 679-5782
title: Chief Planning Architect
userPassword: ramarKnihs
uid: Nooshin_Kramar
givenName: Nooshin
mail: Nooshin_Kramar@example.com
carLicense: TBHGVGS
departmentNumber: 402
employeeType: Contract
homePhone: +1 415 977-5735
initials: N. K.
mobile: +1 510 410-8688
pager: +1 804 667-5477
manager: cn=Sam Schaller
secretary: cn=Shandeigh Massoudian
roomNumber: 2658

dn: cn=Sky Szypulski, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sky Szypulski
sn: Szypulski
description: This is Sky Szypulski's description
facsimileTelephoneNumber: +1 206 123-9549
l: Menlo Park
ou: Planning
postalAddress: example$Planning$Dept # 521
telephoneNumber: +1 303 401-9309
title: Master Planning Vice President
userPassword: ikslupyzSy
uid: Sky_Szypulski
givenName: Sky
mail: Sky_Szypulski@example.com
carLicense: H43RZD7
departmentNumber: 446
employeeType: Contract
homePhone: +1 804 570-7137
initials: S. S.
mobile: +1 206 442-7052
pager: +1 510 446-9661
manager: cn=Quyen Hubley
secretary: cn=Clement Unxlb
roomNumber: 3846

dn: cn=Eoin Dilen, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Eoin Dilen
sn: Dilen
description: This is Eoin Dilen's description
facsimileTelephoneNumber: +1 818 800-9230
l: Fremont
ou: Human Resources
postalAddress: example$Human Resources$Dept # 415
telephoneNumber: +1 818 689-3752
title: Supreme Human Resources Admin
userPassword: neliDnioE
uid: Eoin_Dilen
givenName: Eoin
mail: Eoin_Dilen@example.com
carLicense: BPWEE2T
departmentNumber: 1111
employeeType: Employee
homePhone: +1 71 172-1790
initials: E. D.
mobile: +1 510 558-9742
pager: +1 415 406-8107
manager: cn=Isaac Kurylyk
secretary: cn=Nanette Guin
roomNumber: 88

dn: cn=Angie Quattrucci, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Angie Quattrucci
sn: Quattrucci
description: This is Angie Quattrucci's description
facsimileTelephoneNumber: +1 818 951-9124
l: Menlo Park
ou: Peons
postalAddress: example$Peons$Dept # 840
telephoneNumber: +1 71 658-5962
title: Elite Peons Visionary
userPassword: iccurttauQ
uid: Angie_Quattrucci
givenName: Angie
mail: Angie_Quattrucci@example.com
carLicense: P63ZFCZ
departmentNumber: 6809
employeeType: Manager
homePhone: +1 510 354-4073
initials: A. Q.
mobile: +1 303 915-7613
pager: +1 510 473-5228
manager: cn=Gwynith Corbeil
secretary: cn=Cyb SYS
roomNumber: 1434

dn: cn=Carree Colton, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Carree Colton
sn: Colton
description: This is Carree Colton's description
facsimileTelephoneNumber: +1 206 518-6934
l: Cambridge
ou: Administrative
postalAddress: example$Administrative$Dept # 341
telephoneNumber: +1 408 205-5357
title: Elite Administrative Vice President
userPassword: notloCeerr
uid: Carree_Colton
givenName: Carree
mail: Carree_Colton@example.com
carLicense: CHOQMJS
departmentNumber: 1724
employeeType: Normal
homePhone: +1 303 706-3439
initials: C. C.
mobile: +1 510 333-2875
pager: +1 408 431-1579
manager: cn=Ernestine Wadasinghe
secretary: cn=Zonda Logarajah
roomNumber: 8840

dn: cn=Stacey Cuffling, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Stacey Cuffling
sn: Cuffling
description: This is Stacey Cuffling's description
facsimileTelephoneNumber: +1 303 959-6773
l: Milpitas
ou: Payroll
postalAddress: example$Payroll$Dept # 658
telephoneNumber: +1 206 294-7855
title: Supreme Payroll Janitor
userPassword: gnilffuCye
uid: Stacey_Cuffling
givenName: Stacey
mail: Stacey_Cuffling@example.com
carLicense: TUXZ3MU
departmentNumber: 340
employeeType: Contract
homePhone: +1 206 830-7198
initials: S. C.
mobile: +1 213 199-3217
pager: +1 206 919-2006
manager: cn=Connie Mulqueen
secretary: cn=Shobana Goold
roomNumber: 2950

dn: cn=Lenka Recycling, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lenka Recycling
sn: Recycling
description: This is Lenka Recycling's description
facsimileTelephoneNumber: +1 415 302-8519
l: Redwood Shores
ou: Peons
postalAddress: example$Peons$Dept # 285
telephoneNumber: +1 408 465-7183
title: Senior Peons Accountant
userPassword: gnilcyceRa
uid: Lenka_Recycling
givenName: Lenka
mail: Lenka_Recycling@example.com
carLicense: E4K4WQS
departmentNumber: 5130
employeeType: Contract
homePhone: +1 213 417-5706
initials: L. R.
mobile: +1 71 263-5791
pager: +1 206 115-7598
manager: cn=Wanids Decker
secretary: cn=Wits ETAS
roomNumber: 9709

dn: cn=Audry Tsunoda, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Audry Tsunoda
sn: Tsunoda
description: This is Audry Tsunoda's description
facsimileTelephoneNumber: +1 804 684-2111
l: San Jose
ou: Human Resources
postalAddress: example$Human Resources$Dept # 145
telephoneNumber: +1 303 267-1575
title: Supreme Human Resources Developer
userPassword: adonusTyrd
uid: Audry_Tsunoda
givenName: Audry
mail: Audry_Tsunoda@example.com
carLicense: AS9127F
departmentNumber: 6349
employeeType: Temp
homePhone: +1 71 373-6012
initials: A. T.
mobile: +1 71 708-6730
pager: +1 408 374-6781
manager: cn=Nadim Kuhlkamp
secretary: cn=Miguela Nyce
roomNumber: 1979

dn: cn=Marena Mastellar, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marena Mastellar
sn: Mastellar
description: This is Marena Mastellar's description
facsimileTelephoneNumber: +1 804 172-1820
l: Milpitas
ou: Payroll
postalAddress: example$Payroll$Dept # 550
telephoneNumber: +1 303 414-3131
title: Senior Payroll Architect
userPassword: ralletsaMa
uid: Marena_Mastellar
givenName: Marena
mail: Marena_Mastellar@example.com
carLicense: PM1DW3J
departmentNumber: 446
employeeType: Employee
homePhone: +1 804 379-6259
initials: M. M.
mobile: +1 804 138-3415
pager: +1 510 190-7433
manager: cn=Gino Shyu
secretary: cn=Van Katcher
roomNumber: 8573

dn: cn=Sephira Beaudet, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sephira Beaudet
sn: Beaudet
description: This is Sephira Beaudet's description
facsimileTelephoneNumber: +1 804 618-2331
l: San Mateo
ou: Planning
postalAddress: example$Planning$Dept # 192
telephoneNumber: +1 408 162-3755
title: Junior Planning President
userPassword: teduaeBari
uid: Sephira_Beaudet
givenName: Sephira
mail: Sephira_Beaudet@example.com
carLicense: 9ETUUM5
departmentNumber: 7153
employeeType: Manager
homePhone: +1 804 313-4133
initials: S. B.
mobile: +1 510 749-2995
pager: +1 206 570-8482
manager: cn=Josey Plastina
secretary: cn=Parkinson Cutrufello
roomNumber: 1227

dn: cn=Toyoji Zottola, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Toyoji Zottola
sn: Zottola
description: This is Toyoji Zottola's description
facsimileTelephoneNumber: +1 206 989-7405
l: Menlo Park
ou: Product Development
postalAddress: example$Product Development$Dept # 435
telephoneNumber: +1 818 610-3160
title: Supreme Product Development Sales Rep
userPassword: alottoZijo
uid: Toyoji_Zottola
givenName: Toyoji
mail: Toyoji_Zottola@example.com
carLicense: D6D5MT9
departmentNumber: 9953
employeeType: Contract
homePhone: +1 213 678-4694
initials: T. Z.
mobile: +1 510 885-4076
pager: +1 71 493-4359
manager: cn=Isl Homa
secretary: cn=Stateson Raynard
roomNumber: 3891

dn: cn=Coors Moree, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Coors Moree
sn: Moree
description: This is Coors Moree's description
facsimileTelephoneNumber: +1 408 813-6240
l: San Mateo
ou: Accounting
postalAddress: example$Accounting$Dept # 646
telephoneNumber: +1 303 836-3845
title: Supreme Accounting Assistant
userPassword: eeroMsrooC
uid: Coors_Moree
givenName: Coors
mail: Coors_Moree@example.com
carLicense: O89NVOS
departmentNumber: 5578
employeeType: Manager
homePhone: +1 415 113-4325
initials: C. M.
mobile: +1 818 512-9511
pager: +1 415 974-5164
manager: cn=Marvell Zeidler
secretary: cn=Ilise Dubeau
roomNumber: 7984

dn: cn=Cornelius Hazelton, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Cornelius Hazelton
sn: Hazelton
description: This is Cornelius Hazelton's description
facsimileTelephoneNumber: +1 818 667-1576
l: Cupertino
ou: Peons
postalAddress: example$Peons$Dept # 837
telephoneNumber: +1 415 777-3658
title: Chief Peons Technician
userPassword: notlezaHsu
uid: Cornelius_Hazelton
givenName: Cornelius
mail: Cornelius_Hazelton@example.com
carLicense: 7DZ9RCQ
departmentNumber: 7943
employeeType: Employee
homePhone: +1 206 366-3619
initials: C. H.
mobile: +1 206 627-1023
pager: +1 213 268-2112
manager: cn=Leia Chasse
secretary: cn=Junia Eleftheriou
roomNumber: 3119

dn: cn=Byron Evers, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Byron Evers
sn: Evers
description: This is Byron Evers's description
facsimileTelephoneNumber: +1 804 497-6758
l: San Jose
ou: Accounting
postalAddress: example$Accounting$Dept # 357
telephoneNumber: +1 206 803-9974
title: Elite Accounting Developer
userPassword: srevEnoryB
uid: Byron_Evers
givenName: Byron
mail: Byron_Evers@example.com
carLicense: PA9UYBD
departmentNumber: 3329
employeeType: Normal
homePhone: +1 818 676-8405
initials: B. E.
mobile: +1 818 716-3538
pager: +1 206 437-6113
manager: cn=Janenna Bourque
secretary: cn=Staci De Boer
roomNumber: 6310

dn: cn=Happy Gates, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Happy Gates
sn: Gates
description: This is Happy Gates's description
facsimileTelephoneNumber: +1 303 627-9787
l: Cupertino
ou: Human Resources
postalAddress: example$Human Resources$Dept # 398
telephoneNumber: +1 510 242-2971
title: Master Human Resources Consultant
userPassword: setaGyppaH
uid: Happy_Gates
givenName: Happy
mail: Happy_Gates@example.com
carLicense: 35K5S1O
departmentNumber: 2027
employeeType: Temp
homePhone: +1 303 955-6641
initials: H. G.
mobile: +1 408 167-2225
pager: +1 213 514-1012
manager: cn=Moyra Cascarini
secretary: cn=Bellina Skillen
roomNumber: 2341

dn: cn=Uta McMannen, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Uta McMannen
sn: McMannen
description: This is Uta McMannen's description
facsimileTelephoneNumber: +1 71 495-9676
l: Redmond
ou: Management
postalAddress: example$Management$Dept # 443
telephoneNumber: +1 213 393-1818
title: Elite Management Writer
userPassword: nennaMcMat
uid: Uta_McMannen
givenName: Uta
mail: Uta_McMannen@example.com
carLicense: R8MU4OO
departmentNumber: 4555
employeeType: Manager
homePhone: +1 408 883-8577
initials: U. M.
mobile: +1 818 736-6007
pager: +1 206 874-6837
manager: cn=Alissa Tello
secretary: cn=Mahesh Zenkevicius
roomNumber: 8182

dn: cn=Ty Stevens, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ty Stevens
sn: Stevens
description: This is Ty Stevens's description
facsimileTelephoneNumber: +1 213 318-3148
l: Fremont
ou: Human Resources
postalAddress: example$Human Resources$Dept # 699
telephoneNumber: +1 408 250-9862
title: Chief Human Resources Sales Rep
userPassword: snevetSyT
uid: Ty_Stevens
givenName: Ty
mail: Ty_Stevens@example.com
carLicense: AUKD2Z3
departmentNumber: 9795
employeeType: Normal
homePhone: +1 206 586-9013
initials: T. S.
mobile: +1 510 919-6750
pager: +1 71 807-1102
manager: cn=Stafani Hayes
secretary: cn=Cassandra Applications
roomNumber: 6916

dn: cn=Kapsch Panter, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kapsch Panter
sn: Panter
description: This is Kapsch Panter's description
facsimileTelephoneNumber: +1 804 892-6408
l: Menlo Park
ou: Product Testing
postalAddress: example$Product Testing$Dept # 234
telephoneNumber: +1 303 633-5806
title: Master Product Testing President
userPassword: retnaPhcsp
uid: Kapsch_Panter
givenName: Kapsch
mail: Kapsch_Panter@example.com
carLicense: T4QCGI8
departmentNumber: 4234
employeeType: Manager
homePhone: +1 206 715-8979
initials: K. P.
mobile: +1 206 838-5130
pager: +1 415 874-2777
manager: cn=Estella Beauvais
secretary: cn=Melisandra Gomm
roomNumber: 6615

dn: cn=Allie Linegar, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Allie Linegar
sn: Linegar
description: This is Allie Linegar's description
facsimileTelephoneNumber: +1 303 669-1524
l: Fremont
ou: Product Development
postalAddress: example$Product Development$Dept # 457
telephoneNumber: +1 206 317-5935
title: Master Product Development Artist
userPassword: rageniLeil
uid: Allie_Linegar
givenName: Allie
mail: Allie_Linegar@example.com
carLicense: 366L4CY
departmentNumber: 7139
employeeType: Normal
homePhone: +1 415 999-2443
initials: A. L.
mobile: +1 213 628-9505
pager: +1 415 836-1152
manager: cn=Viva Obermyer
secretary: cn=Iteke Ripa
roomNumber: 4909

dn: cn=Elisa Jasmin, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Elisa Jasmin
sn: Jasmin
description: This is Elisa Jasmin's description
facsimileTelephoneNumber: +1 408 968-2146
l: Emeryville
ou: Product Testing
postalAddress: example$Product Testing$Dept # 988
telephoneNumber: +1 206 270-1329
title: Junior Product Testing Janitor
userPassword: nimsaJasil
uid: Elisa_Jasmin
givenName: Elisa
mail: Elisa_Jasmin@example.com
carLicense: BI5VLJ8
departmentNumber: 6992
employeeType: Temp
homePhone: +1 71 892-4047
initials: E. J.
mobile: +1 415 203-3869
pager: +1 303 703-2221
manager: cn=Katha Bilsborough
secretary: cn=Beatrice Mobley
roomNumber: 3108

dn: cn=Mauro Guimond, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mauro Guimond
sn: Guimond
description: This is Mauro Guimond's description
facsimileTelephoneNumber: +1 804 786-4314
l: Emeryville
ou: Administrative
postalAddress: example$Administrative$Dept # 353
telephoneNumber: +1 415 750-7749
title: Chief Administrative Punk
userPassword: dnomiuGoru
uid: Mauro_Guimond
givenName: Mauro
mail: Mauro_Guimond@example.com
carLicense: CSOB28C
departmentNumber: 7802
employeeType: Manager
homePhone: +1 303 155-6935
initials: M. G.
mobile: +1 213 293-5543
pager: +1 818 921-7041
manager: cn=Pete Broberg
secretary: cn=Jagjeet Rasmus
roomNumber: 3246

dn: cn=Ermengarde Schick, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ermengarde Schick
sn: Schick
description: This is Ermengarde Schick's description
facsimileTelephoneNumber: +1 510 685-2099
l: Sunnyvale
ou: Planning
postalAddress: example$Planning$Dept # 642
telephoneNumber: +1 510 765-1652
title: Master Planning Artist
userPassword: kcihcSedra
uid: Ermengarde_Schick
givenName: Ermengarde
mail: Ermengarde_Schick@example.com
carLicense: 1EMVOQB
departmentNumber: 8429
employeeType: Normal
homePhone: +1 408 556-2369
initials: E. S.
mobile: +1 71 789-2658
pager: +1 206 381-7041
manager: cn=Marie-Josee McDowell
secretary: cn=Tine Beaudry
roomNumber: 8490

dn: cn=Goldia Korpela, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Goldia Korpela
sn: Korpela
description: This is Goldia Korpela's description
facsimileTelephoneNumber: +1 510 910-7839
l: Cambridge
ou: Janitorial
postalAddress: example$Janitorial$Dept # 351
telephoneNumber: +1 408 745-4848
title: Associate Janitorial Admin
userPassword: aleproKaid
uid: Goldia_Korpela
givenName: Goldia
mail: Goldia_Korpela@example.com
carLicense: US4BNF8
departmentNumber: 5578
employeeType: Manager
homePhone: +1 213 513-3767
initials: G. K.
mobile: +1 510 821-7500
pager: +1 71 850-8225
manager: cn=Bertrand Davalo
secretary: cn=Dante Dalrymple
roomNumber: 2711

dn: cn=Takis Bour, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Takis Bour
sn: Bour
description: This is Takis Bour's description
facsimileTelephoneNumber: +1 206 328-7272
l: Fremont
ou: Product Testing
postalAddress: example$Product Testing$Dept # 325
telephoneNumber: +1 804 550-7887
title: Senior Product Testing Janitor
userPassword: ruoBsikaT
uid: Takis_Bour
givenName: Takis
mail: Takis_Bour@example.com
carLicense: JHOHQNO
departmentNumber: 1708
employeeType: Manager
homePhone: +1 213 966-8094
initials: T. B.
mobile: +1 510 244-3496
pager: +1 71 234-5371
manager: cn=Shiela Puckett
secretary: cn=Silvestro Ritter
roomNumber: 8997

dn: cn=Dixie Bourdin, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dixie Bourdin
sn: Bourdin
description: This is Dixie Bourdin's description
facsimileTelephoneNumber: +1 510 585-8546
l: Mountain View
ou: Human Resources
postalAddress: example$Human Resources$Dept # 457
telephoneNumber: +1 206 634-6581
title: Junior Human Resources Sales Rep
userPassword: nidruoBeix
uid: Dixie_Bourdin
givenName: Dixie
mail: Dixie_Bourdin@example.com
carLicense: YK0J5T7
departmentNumber: 9790
employeeType: Temp
homePhone: +1 71 462-8278
initials: D. B.
mobile: +1 408 510-4697
pager: +1 213 741-2133
manager: cn=Charyl Nava
secretary: cn=Dolley Puglia
roomNumber: 5734

dn: cn=Keven Rychlicki, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Keven Rychlicki
sn: Rychlicki
description: This is Keven Rychlicki's description
facsimileTelephoneNumber: +1 303 124-8353
l: San Jose
ou: Human Resources
postalAddress: example$Human Resources$Dept # 124
telephoneNumber: +1 804 191-1531
title: Elite Human Resources Grunt
userPassword: ikcilhcyRn
uid: Keven_Rychlicki
givenName: Keven
mail: Keven_Rychlicki@example.com
carLicense: ZBLG4RV
departmentNumber: 834
employeeType: Contract
homePhone: +1 71 567-3412
initials: K. R.
mobile: +1 71 968-3996
pager: +1 303 731-9082
manager: cn=Mindy Goodman
secretary: cn=Mignonne Shayanpour
roomNumber: 7098

dn: cn=Donall Rantala, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Donall Rantala
sn: Rantala
description: This is Donall Rantala's description
facsimileTelephoneNumber: +1 408 303-4522
l: Milpitas
ou: Payroll
postalAddress: example$Payroll$Dept # 328
telephoneNumber: +1 71 301-6813
title: Chief Payroll Visionary
userPassword: alatnaRlla
uid: Donall_Rantala
givenName: Donall
mail: Donall_Rantala@example.com
carLicense: MX4TUQ6
departmentNumber: 9947
employeeType: Normal
homePhone: +1 303 864-6341
initials: D. R.
mobile: +1 818 165-2183
pager: +1 71 401-5780
manager: cn=Devonne Polashock
secretary: cn=Anabel Campeau
roomNumber: 4308

dn: cn=Marline Klotz, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marline Klotz
sn: Klotz
description: This is Marline Klotz's description
facsimileTelephoneNumber: +1 510 983-5629
l: Orem
ou: Payroll
postalAddress: example$Payroll$Dept # 407
telephoneNumber: +1 818 268-9539
title: Supreme Payroll Czar
userPassword: ztolKenilr
uid: Marline_Klotz
givenName: Marline
mail: Marline_Klotz@example.com
carLicense: 32KPAL0
departmentNumber: 2403
employeeType: Temp
homePhone: +1 71 872-9086
initials: M. K.
mobile: +1 415 594-1651
pager: +1 415 917-6733
manager: cn=Elleke Register
secretary: cn=Milka Lesniak
roomNumber: 7758

dn: cn=Fekri Visockis, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Fekri Visockis
sn: Visockis
description: This is Fekri Visockis's description
facsimileTelephoneNumber: +1 71 137-3361
l: Orem
ou: Product Testing
postalAddress: example$Product Testing$Dept # 237
telephoneNumber: +1 71 956-3730
title: Supreme Product Testing Grunt
userPassword: sikcosiVir
uid: Fekri_Visockis
givenName: Fekri
mail: Fekri_Visockis@example.com
carLicense: 7TMONCZ
departmentNumber: 5655
employeeType: Contract
homePhone: +1 408 533-4693
initials: F. V.
mobile: +1 408 940-7060
pager: +1 71 816-8956
manager: cn=Shina Benjamin
secretary: cn=Anallese Kupe
roomNumber: 7880

dn: cn=Candace PATCOR, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Candace PATCOR
sn: PATCOR
description: This is Candace PATCOR's description
facsimileTelephoneNumber: +1 206 131-7351
l: Redmond
ou: Administrative
postalAddress: example$Administrative$Dept # 864
telephoneNumber: +1 206 845-3696
title: Elite Administrative Warrior
userPassword: ROCTAPecad
uid: Candace_PATCOR
givenName: Candace
mail: Candace_PATCOR@example.com
carLicense: 5KEJAI8
departmentNumber: 3432
employeeType: Contract
homePhone: +1 206 601-9313
initials: C. P.
mobile: +1 213 775-9778
pager: +1 206 781-9544
manager: cn=Amir Michalos
secretary: cn=Marieann Hearnden
roomNumber: 4467

dn: cn=Truus McKinlay, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Truus McKinlay
sn: McKinlay
description: This is Truus McKinlay's description
facsimileTelephoneNumber: +1 408 195-6650
l: Sunnyvale
ou: Product Development
postalAddress: example$Product Development$Dept # 17
telephoneNumber: +1 818 325-8417
title: Associate Product Development Accountant
userPassword: yalniKcMsu
uid: Truus_McKinlay
givenName: Truus
mail: Truus_McKinlay@example.com
carLicense: QIEZFB2
departmentNumber: 1541
employeeType: Employee
homePhone: +1 303 526-8978
initials: T. M.
mobile: +1 206 450-3967
pager: +1 71 431-6521
manager: cn=Federica Nigam
secretary: cn=Trudey Berryhill
roomNumber: 6635

dn: cn=Marie-Josee Sanders, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marie-Josee Sanders
sn: Sanders
description: This is Marie-Josee Sanders's description
facsimileTelephoneNumber: +1 71 525-1361
l: Mountain View
ou: Planning
postalAddress: example$Planning$Dept # 79
telephoneNumber: +1 510 349-7810
title: Master Planning President
userPassword: srednaSees
uid: Marie-Josee_Sanders
givenName: Marie-Josee
mail: Marie-Josee_Sanders@example.com
carLicense: FYWUH72
departmentNumber: 7976
employeeType: Temp
homePhone: +1 213 628-9235
initials: M. S.
mobile: +1 408 871-9747
pager: +1 206 539-8562
manager: cn=Ardelle Russett
secretary: cn=Fabienne Fetting
roomNumber: 3962

dn: cn=Alpine Mullaney, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Alpine Mullaney
sn: Mullaney
description: This is Alpine Mullaney's description
facsimileTelephoneNumber: +1 510 900-8598
l: Redwood Shores
ou: Product Testing
postalAddress: example$Product Testing$Dept # 226
telephoneNumber: +1 206 993-8923
title: Junior Product Testing Sales Rep
userPassword: yenalluMen
uid: Alpine_Mullaney
givenName: Alpine
mail: Alpine_Mullaney@example.com
carLicense: WGBX355
departmentNumber: 3067
employeeType: Employee
homePhone: +1 213 952-3838
initials: A. M.
mobile: +1 510 868-4334
pager: +1 818 976-7889
manager: cn=Mala Pepin
secretary: cn=Mauricio Serbin
roomNumber: 7803

dn: cn=Jirina Ketterer, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jirina Ketterer
sn: Ketterer
description: This is Jirina Ketterer's description
facsimileTelephoneNumber: +1 71 141-9174
l: Palo Alto
ou: Administrative
postalAddress: example$Administrative$Dept # 893
telephoneNumber: +1 510 466-4173
title: Supreme Administrative Director
userPassword: reretteKan
uid: Jirina_Ketterer
givenName: Jirina
mail: Jirina_Ketterer@example.com
carLicense: W3ATUY9
departmentNumber: 3226
employeeType: Employee
homePhone: +1 818 218-1098
initials: J. K.
mobile: +1 213 679-1507
pager: +1 303 561-2677
manager: cn=Sher Spohn
secretary: cn=Brande Dube
roomNumber: 3148

dn: cn=Desire Shames, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Desire Shames
sn: Shames
description: This is Desire Shames's description
facsimileTelephoneNumber: +1 408 385-6057
l: Cupertino
ou: Product Testing
postalAddress: example$Product Testing$Dept # 86
telephoneNumber: +1 804 126-3732
title: Supreme Product Testing Grunt
userPassword: semahSeris
uid: Desire_Shames
givenName: Desire
mail: Desire_Shames@example.com
carLicense: 27NFE5O
departmentNumber: 6240
employeeType: Temp
homePhone: +1 408 300-7065
initials: D. S.
mobile: +1 71 159-1833
pager: +1 818 820-1159
manager: cn=Kim-Tram Grimshaw
secretary: cn=Randie Burgi
roomNumber: 5357

dn: cn=Farra Boles, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Farra Boles
sn: Boles
description: This is Farra Boles's description
facsimileTelephoneNumber: +1 303 186-3696
l: Cupertino
ou: Product Development
postalAddress: example$Product Development$Dept # 832
telephoneNumber: +1 415 451-5315
title: Junior Product Development Yahoo
userPassword: seloBarraF
uid: Farra_Boles
givenName: Farra
mail: Farra_Boles@example.com
carLicense: VGCU9QH
departmentNumber: 548
employeeType: Normal
homePhone: +1 818 924-1171
initials: F. B.
mobile: +1 213 317-5535
pager: +1 206 429-2114
manager: cn=Sofeya Attenborough
secretary: cn=Kelwin Strider
roomNumber: 4776

dn: cn=Edmx Beaty, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Edmx Beaty
sn: Beaty
description: This is Edmx Beaty's description
facsimileTelephoneNumber: +1 408 460-7625
l: San Mateo
ou: Janitorial
postalAddress: example$Janitorial$Dept # 331
telephoneNumber: +1 303 427-6104
title: Chief Janitorial Vice President
userPassword: ytaeBxmdE
uid: Edmx_Beaty
givenName: Edmx
mail: Edmx_Beaty@example.com
carLicense: 4XOW1HI
departmentNumber: 1279
employeeType: Contract
homePhone: +1 206 683-8029
initials: E. B.
mobile: +1 303 400-9306
pager: +1 408 324-7874
manager: cn=Jilleen Funston
secretary: cn=Chick Bulifant
roomNumber: 389

dn: cn=Arshad Pridgen, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Arshad Pridgen
sn: Pridgen
description: This is Arshad Pridgen's description
facsimileTelephoneNumber: +1 303 515-5437
l: Alameda
ou: Accounting
postalAddress: example$Accounting$Dept # 403
telephoneNumber: +1 408 223-8594
title: Master Accounting Architect
userPassword: negdirPdah
uid: Arshad_Pridgen
givenName: Arshad
mail: Arshad_Pridgen@example.com
carLicense: D28KMCL
departmentNumber: 2137
employeeType: Temp
homePhone: +1 303 973-8037
initials: A. P.
mobile: +1 213 495-7493
pager: +1 408 755-4594
manager: cn=Henri Moebes
secretary: cn=Madelaine Abdo
roomNumber: 8913

dn: cn=Elfreda Lavers, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Elfreda Lavers
sn: Lavers
description: This is Elfreda Lavers's description
facsimileTelephoneNumber: +1 510 469-9254
l: San Francisco
ou: Administrative
postalAddress: example$Administrative$Dept # 156
telephoneNumber: +1 206 395-9094
title: Master Administrative Consultant
userPassword: srevaLader
uid: Elfreda_Lavers
givenName: Elfreda
mail: Elfreda_Lavers@example.com
carLicense: ZB0NSRM
departmentNumber: 5851
employeeType: Contract
homePhone: +1 415 646-9727
initials: E. L.
mobile: +1 415 713-2127
pager: +1 71 459-2508
manager: cn=Viv Beveridge
secretary: cn=Candace Norby
roomNumber: 4356

dn: cn=Shirley-Ann Killam, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shirley-Ann Killam
sn: Killam
description: This is Shirley-Ann Killam's description
facsimileTelephoneNumber: +1 818 694-5222
l: Santa Clara
ou: Payroll
postalAddress: example$Payroll$Dept # 905
telephoneNumber: +1 818 273-7131
title: Chief Payroll Manager
userPassword: malliKnnA-
uid: Shirley-Ann_Killam
givenName: Shirley-Ann
mail: Shirley-Ann_Killam@example.com
carLicense: 3JDUXZN
departmentNumber: 3391
employeeType: Employee
homePhone: +1 213 809-9622
initials: S. K.
mobile: +1 415 600-2402
pager: +1 804 244-2185
manager: cn=Danica Newell
secretary: cn=Kristine Hazell
roomNumber: 3785

dn: cn=Bertrand Gause, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bertrand Gause
sn: Gause
description: This is Bertrand Gause's description
facsimileTelephoneNumber: +1 415 315-1909
l: Alameda
ou: Product Development
postalAddress: example$Product Development$Dept # 464
telephoneNumber: +1 510 847-5638
title: Supreme Product Development Fellow
userPassword: esuaGdnart
uid: Bertrand_Gause
givenName: Bertrand
mail: Bertrand_Gause@example.com
carLicense: 8RL8VQ7
departmentNumber: 6491
employeeType: Employee
homePhone: +1 303 369-1227
initials: B. G.
mobile: +1 71 834-6359
pager: +1 303 452-4725
manager: cn=Ashoka Shamblin
secretary: cn=Bel Network-Ops
roomNumber: 4026

dn: cn=Ronnie Naem, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ronnie Naem
sn: Naem
description: This is Ronnie Naem's description
facsimileTelephoneNumber: +1 303 725-3898
l: Cambridge
ou: Administrative
postalAddress: example$Administrative$Dept # 546
telephoneNumber: +1 206 546-8134
title: Associate Administrative Manager
userPassword: meaNeinnoR
uid: Ronnie_Naem
givenName: Ronnie
mail: Ronnie_Naem@example.com
carLicense: 6IN2K0T
departmentNumber: 8511
employeeType: Manager
homePhone: +1 408 635-4147
initials: R. N.
mobile: +1 206 261-5574
pager: +1 510 247-6002
manager: cn=Ladell Kudrewatych
secretary: cn=Wilkin Millette
roomNumber: 932

dn: cn=Pardip Hendrickse, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Pardip Hendrickse
sn: Hendrickse
description: This is Pardip Hendrickse's description
facsimileTelephoneNumber: +1 804 478-9034
l: Palo Alto
ou: Product Testing
postalAddress: example$Product Testing$Dept # 944
telephoneNumber: +1 206 746-3144
title: Chief Product Testing Figurehead
userPassword: eskcirdneH
uid: Pardip_Hendrickse
givenName: Pardip
mail: Pardip_Hendrickse@example.com
carLicense: J3YMHFM
departmentNumber: 5834
employeeType: Normal
homePhone: +1 510 846-8385
initials: P. H.
mobile: +1 415 967-9127
pager: +1 71 547-6822
manager: cn=Nerty Hyndman
secretary: cn=Duane Vasile
roomNumber: 9014

dn: cn=Allen Forecasting, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Allen Forecasting
sn: Forecasting
description: This is Allen Forecasting's description
facsimileTelephoneNumber: +1 818 812-7241
l: Mountain View
ou: Janitorial
postalAddress: example$Janitorial$Dept # 846
telephoneNumber: +1 510 130-7532
title: Associate Janitorial Architect
userPassword: gnitsacero
uid: Allen_Forecasting
givenName: Allen
mail: Allen_Forecasting@example.com
carLicense: CRR07CE
departmentNumber: 5949
employeeType: Manager
homePhone: +1 408 285-1016
initials: A. F.
mobile: +1 206 738-2006
pager: +1 510 746-7114
manager: cn=Catriona Sanderson
secretary: cn=Konstance Slotnick
roomNumber: 4639

dn: cn=Fred Whitlock, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Fred Whitlock
sn: Whitlock
description: This is Fred Whitlock's description
facsimileTelephoneNumber: +1 206 352-1952
l: Redwood Shores
ou: Accounting
postalAddress: example$Accounting$Dept # 433
telephoneNumber: +1 213 597-2051
title: Junior Accounting Stooge
userPassword: kcoltihWde
uid: Fred_Whitlock
givenName: Fred
mail: Fred_Whitlock@example.com
carLicense: V1DEMN9
departmentNumber: 7174
employeeType: Temp
homePhone: +1 71 946-4846
initials: F. W.
mobile: +1 213 432-1110
pager: +1 510 957-6711
manager: cn=Shinichiro Wingar
secretary: cn=Saeed Moorer
roomNumber: 4124

dn: cn=Mervyn Applications, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mervyn Applications
sn: Applications
description: This is Mervyn Applications's description
facsimileTelephoneNumber: +1 804 261-3157
l: San Francisco
ou: Administrative
postalAddress: example$Administrative$Dept # 524
telephoneNumber: +1 71 413-5934
title: Associate Administrative Punk
userPassword: snoitacilp
uid: Mervyn_Applications
givenName: Mervyn
mail: Mervyn_Applications@example.com
carLicense: 36BYATU
departmentNumber: 4942
employeeType: Employee
homePhone: +1 818 207-4366
initials: M. A.
mobile: +1 415 655-2093
pager: +1 213 251-3236
manager: cn=Rafaelita Tamasi
secretary: cn=Ineke Marcellus
roomNumber: 2876

dn: cn=Carlis Poulos, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Carlis Poulos
sn: Poulos
description: This is Carlis Poulos's description
facsimileTelephoneNumber: +1 415 675-6465
l: Orem
ou: Product Development
postalAddress: example$Product Development$Dept # 94
telephoneNumber: +1 415 483-9097
title: Supreme Product Development Developer
userPassword: soluoPsilr
uid: Carlis_Poulos
givenName: Carlis
mail: Carlis_Poulos@example.com
carLicense: Y1KXSLV
departmentNumber: 5357
employeeType: Contract
homePhone: +1 71 357-8932
initials: C. P.
mobile: +1 408 878-8956
pager: +1 408 836-4803
manager: cn=Ketti Bahoric
secretary: cn=Hulda Ludwick
roomNumber: 3005

dn: cn=Doralyn Cracknell, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Doralyn Cracknell
sn: Cracknell
description: This is Doralyn Cracknell's description
facsimileTelephoneNumber: +1 408 771-3063
l: Armonk
ou: Janitorial
postalAddress: example$Janitorial$Dept # 114
telephoneNumber: +1 213 521-1169
title: Chief Janitorial Technician
userPassword: llenkcarCn
uid: Doralyn_Cracknell
givenName: Doralyn
mail: Doralyn_Cracknell@example.com
carLicense: NNBPX46
departmentNumber: 8129
employeeType: Temp
homePhone: +1 415 570-8371
initials: D. C.
mobile: +1 804 660-5466
pager: +1 71 388-8815
manager: cn=Sue Ching
secretary: cn=Thomasina Zolmer
roomNumber: 5462

dn: cn=Gray Daniels, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gray Daniels
sn: Daniels
description: This is Gray Daniels's description
facsimileTelephoneNumber: +1 71 326-6034
l: San Jose
ou: Accounting
postalAddress: example$Accounting$Dept # 334
telephoneNumber: +1 415 800-8060
title: Elite Accounting Dictator
userPassword: sleinaDyar
uid: Gray_Daniels
givenName: Gray
mail: Gray_Daniels@example.com
carLicense: DKL6Z8V
departmentNumber: 2834
employeeType: Employee
homePhone: +1 804 364-6354
initials: G. D.
mobile: +1 804 851-3507
pager: +1 71 155-5635
manager: cn=Danielle Potter
secretary: cn=Sheilah Vilmansen
roomNumber: 586

dn: cn=Shari Bhasin, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shari Bhasin
sn: Bhasin
description: This is Shari Bhasin's description
facsimileTelephoneNumber: +1 303 648-5011
l: Cambridge
ou: Administrative
postalAddress: example$Administrative$Dept # 354
telephoneNumber: +1 510 520-4823
title: Elite Administrative Punk
userPassword: nisahBirah
uid: Shari_Bhasin
givenName: Shari
mail: Shari_Bhasin@example.com
carLicense: T31G80F
departmentNumber: 2088
employeeType: Normal
homePhone: +1 408 119-3492
initials: S. B.
mobile: +1 415 321-1103
pager: +1 510 601-4952
manager: cn=Danit Evans
secretary: cn=Monique Wessels
roomNumber: 3363

dn: cn=Bess McNamara, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bess McNamara
sn: McNamara
description: This is Bess McNamara's description
facsimileTelephoneNumber: +1 804 814-6509
l: Sunnyvale
ou: Accounting
postalAddress: example$Accounting$Dept # 671
telephoneNumber: +1 510 411-1693
title: Elite Accounting Janitor
userPassword: aramaNcMss
uid: Bess_McNamara
givenName: Bess
mail: Bess_McNamara@example.com
carLicense: G12WSLR
departmentNumber: 611
employeeType: Manager
homePhone: +1 71 363-1248
initials: B. M.
mobile: +1 206 575-4956
pager: +1 510 613-8294
manager: cn=Gelais Ikeda
secretary: cn=Esam Adimari
roomNumber: 3261

dn: cn=Shaylyn Sharratt, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shaylyn Sharratt
sn: Sharratt
description: This is Shaylyn Sharratt's description
facsimileTelephoneNumber: +1 818 680-5587
l: Orem
ou: Administrative
postalAddress: example$Administrative$Dept # 819
telephoneNumber: +1 408 918-8301
title: Associate Administrative President
userPassword: ttarrahSny
uid: Shaylyn_Sharratt
givenName: Shaylyn
mail: Shaylyn_Sharratt@example.com
carLicense: SI9N0ZQ
departmentNumber: 1064
employeeType: Normal
homePhone: +1 213 653-4304
initials: S. S.
mobile: +1 213 652-1817
pager: +1 804 754-1963
manager: cn=Doug Dyna
secretary: cn=Marquita Engelhart
roomNumber: 9955

dn: cn=Belen Kindel, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Belen Kindel
sn: Kindel
description: This is Belen Kindel's description
facsimileTelephoneNumber: +1 804 347-4747
l: San Francisco
ou: Peons
postalAddress: example$Peons$Dept # 140
telephoneNumber: +1 408 393-7079
title: Associate Peons Grunt
userPassword: ledniKnele
uid: Belen_Kindel
givenName: Belen
mail: Belen_Kindel@example.com
carLicense: YXBFXG6
departmentNumber: 4615
employeeType: Normal
homePhone: +1 71 632-1119
initials: B. K.
mobile: +1 415 362-8247
pager: +1 804 853-1114
manager: cn=Kipp Piche
secretary: cn=Tine Lapchak
roomNumber: 1447

dn: cn=Vivia Orders, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Vivia Orders
sn: Orders
description: This is Vivia Orders's description
facsimileTelephoneNumber: +1 818 238-4580
l: San Mateo
ou: Janitorial
postalAddress: example$Janitorial$Dept # 110
telephoneNumber: +1 71 423-6695
title: Junior Janitorial Director
userPassword: sredrOaivi
uid: Vivia_Orders
givenName: Vivia
mail: Vivia_Orders@example.com
carLicense: N0GIZG1
departmentNumber: 3173
employeeType: Manager
homePhone: +1 415 181-8474
initials: V. O.
mobile: +1 818 553-5255
pager: +1 510 323-6318
manager: cn=Jerrilyn Ragland
secretary: cn=Jacques Gould
roomNumber: 9210

dn: cn=Ajit Haas, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ajit Haas
sn: Haas
description: This is Ajit Haas's description
facsimileTelephoneNumber: +1 71 520-4965
l: Sunnyvale
ou: Product Testing
postalAddress: example$Product Testing$Dept # 789
telephoneNumber: +1 510 389-9823
title: Elite Product Testing Vice President
userPassword: saaHtijA
uid: Ajit_Haas
givenName: Ajit
mail: Ajit_Haas@example.com
carLicense: ZPG5SB0
departmentNumber: 7319
employeeType: Manager
homePhone: +1 303 478-5253
initials: A. H.
mobile: +1 415 432-4102
pager: +1 804 459-9086
manager: cn=Adriena Nemec
secretary: cn=Sallee Heinzman
roomNumber: 2814

dn: cn=Anastassia Kurylyk, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Anastassia Kurylyk
sn: Kurylyk
description: This is Anastassia Kurylyk's description
facsimileTelephoneNumber: +1 71 624-2992
l: Cambridge
ou: Administrative
postalAddress: example$Administrative$Dept # 201
telephoneNumber: +1 71 188-9159
title: Junior Administrative Admin
userPassword: kylyruKais
uid: Anastassia_Kurylyk
givenName: Anastassia
mail: Anastassia_Kurylyk@example.com
carLicense: 8FP9RJ0
departmentNumber: 6526
employeeType: Employee
homePhone: +1 818 915-4509
initials: A. K.
mobile: +1 71 614-9963
pager: +1 206 340-8900
manager: cn=Sherline Zubans
secretary: cn=Lilith Gouhara
roomNumber: 6144

dn: cn=Zainab Cholette, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zainab Cholette
sn: Cholette
description: This is Zainab Cholette's description
facsimileTelephoneNumber: +1 510 897-8790
l: Cupertino
ou: Peons
postalAddress: example$Peons$Dept # 731
telephoneNumber: +1 71 723-3944
title: Master Peons Czar
userPassword: ettelohCba
uid: Zainab_Cholette
givenName: Zainab
mail: Zainab_Cholette@example.com
carLicense: SVQX1AW
departmentNumber: 1204
employeeType: Contract
homePhone: +1 415 990-5494
initials: Z. C.
mobile: +1 303 827-4839
pager: +1 408 796-1288
manager: cn=Bethanne Palmer
secretary: cn=Camilla Schenck
roomNumber: 9840

dn: cn=Kevina Sayer, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kevina Sayer
sn: Sayer
description: This is Kevina Sayer's description
facsimileTelephoneNumber: +1 510 845-1787
l: Redmond
ou: Administrative
postalAddress: example$Administrative$Dept # 24
telephoneNumber: +1 408 848-6391
title: Junior Administrative Director
userPassword: reyaSanive
uid: Kevina_Sayer
givenName: Kevina
mail: Kevina_Sayer@example.com
carLicense: ZQYCZJV
departmentNumber: 8861
employeeType: Contract
homePhone: +1 71 338-5161
initials: K. S.
mobile: +1 510 321-5364
pager: +1 213 655-8043
manager: cn=Karisa Geary
secretary: cn=Vicuong Hurtado
roomNumber: 9905

dn: cn=Charissa Bourguignon, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Charissa Bourguignon
sn: Bourguignon
description: This is Charissa Bourguignon's description
facsimileTelephoneNumber: +1 818 206-6968
l: Milpitas
ou: Product Development
postalAddress: example$Product Development$Dept # 334
telephoneNumber: +1 818 356-7277
title: Master Product Development Manager
userPassword: nongiugruo
uid: Charissa_Bourguignon
givenName: Charissa
mail: Charissa_Bourguignon@example.com
carLicense: IJIQ4FU
departmentNumber: 3392
employeeType: Normal
homePhone: +1 818 329-4249
initials: C. B.
mobile: +1 818 990-2558
pager: +1 415 356-6834
manager: cn=Alfonzo Vigeant
secretary: cn=Ruthann Costadimas
roomNumber: 6193

dn: cn=Stesha Kolk, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Stesha Kolk
sn: Kolk
description: This is Stesha Kolk's description
facsimileTelephoneNumber: +1 71 625-9907
l: Sunnyvale
ou: Janitorial
postalAddress: example$Janitorial$Dept # 924
telephoneNumber: +1 71 678-2046
title: Associate Janitorial Madonna
userPassword: kloKahsetS
uid: Stesha_Kolk
givenName: Stesha
mail: Stesha_Kolk@example.com
carLicense: 8MM8997
departmentNumber: 2179
employeeType: Manager
homePhone: +1 206 732-6542
initials: S. K.
mobile: +1 206 767-2832
pager: +1 510 735-6377
manager: cn=Naser Vucinich
secretary: cn=Bill Hasen
roomNumber: 8698

dn: cn=Den Van Vrouwerff, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Den Van Vrouwerff
sn: Van Vrouwerff
description: This is Den Van Vrouwerff's description
facsimileTelephoneNumber: +1 213 670-2974
l: Menlo Park
ou: Janitorial
postalAddress: example$Janitorial$Dept # 58
telephoneNumber: +1 213 206-7058
title: Associate Janitorial Sales Rep
userPassword: ffrewuorVn
uid: Den_Van Vrouwerff
givenName: Den
mail: Den_Van Vrouwerff@example.com
carLicense: 15HPUA7
departmentNumber: 8462
employeeType: Contract
homePhone: +1 804 229-2581
initials: D. V.
mobile: +1 408 959-9170
pager: +1 818 941-8681
manager: cn=Usa Decker
secretary: cn=Kippy Rutter
roomNumber: 6499

dn: cn=Phylys Tripps, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Phylys Tripps
sn: Tripps
description: This is Phylys Tripps's description
facsimileTelephoneNumber: +1 213 806-1298
l: San Francisco
ou: Peons
postalAddress: example$Peons$Dept # 263
telephoneNumber: +1 510 355-3414
title: Elite Peons President
userPassword: sppirTsyly
uid: Phylys_Tripps
givenName: Phylys
mail: Phylys_Tripps@example.com
carLicense: 3CZN75N
departmentNumber: 7178
employeeType: Temp
homePhone: +1 303 633-2086
initials: P. T.
mobile: +1 804 161-1091
pager: +1 510 148-9098
manager: cn=Onette Dalton
secretary: cn=Jeannine Kennaday
roomNumber: 6754

dn: cn=Nana Bleile, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Nana Bleile
sn: Bleile
description: This is Nana Bleile's description
facsimileTelephoneNumber: +1 415 575-4766
l: Sunnyvale
ou: Product Development
postalAddress: example$Product Development$Dept # 260
telephoneNumber: +1 206 410-7593
title: Chief Product Development Evangelist
userPassword: elielBanaN
uid: Nana_Bleile
givenName: Nana
mail: Nana_Bleile@example.com
carLicense: 5OD820R
departmentNumber: 7648
employeeType: Temp
homePhone: +1 303 948-3739
initials: N. B.
mobile: +1 71 232-4963
pager: +1 804 995-4537
manager: cn=Ashley Sufcak
secretary: cn=Antonia Leiker
roomNumber: 7074

dn: cn=Maryam Zegray, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Maryam Zegray
sn: Zegray
description: This is Maryam Zegray's description
facsimileTelephoneNumber: +1 303 896-7882
l: Cambridge
ou: Administrative
postalAddress: example$Administrative$Dept # 800
telephoneNumber: +1 71 384-8261
title: Associate Administrative Warrior
userPassword: yargeZmayr
uid: Maryam_Zegray
givenName: Maryam
mail: Maryam_Zegray@example.com
carLicense: 7PE0BSL
departmentNumber: 4858
employeeType: Temp
homePhone: +1 804 847-3265
initials: M. Z.
mobile: +1 804 537-5963
pager: +1 408 189-4710
manager: cn=Cindelyn Tabor
secretary: cn=Ashley Duguay
roomNumber: 5756

dn: cn=Thomas Grafton, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Thomas Grafton
sn: Grafton
description: This is Thomas Grafton's description
facsimileTelephoneNumber: +1 303 860-3939
l: Redmond
ou: Planning
postalAddress: example$Planning$Dept # 685
telephoneNumber: +1 818 525-4686
title: Supreme Planning Artist
userPassword: notfarGsam
uid: Thomas_Grafton
givenName: Thomas
mail: Thomas_Grafton@example.com
carLicense: CM7B2YA
departmentNumber: 6226
employeeType: Manager
homePhone: +1 818 148-2665
initials: T. G.
mobile: +1 303 129-4914
pager: +1 804 372-4422
manager: cn=Duquette Langenberg
secretary: cn=Alanah Panke
roomNumber: 4969

dn: cn=Alyse Vastine, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Alyse Vastine
sn: Vastine
description: This is Alyse Vastine's description
facsimileTelephoneNumber: +1 408 728-8488
l: Milpitas
ou: Accounting
postalAddress: example$Accounting$Dept # 738
telephoneNumber: +1 415 283-5336
title: Master Accounting Writer
userPassword: enitsaVesy
uid: Alyse_Vastine
givenName: Alyse
mail: Alyse_Vastine@example.com
carLicense: SVFFSBQ
departmentNumber: 8200
employeeType: Temp
homePhone: +1 818 115-1404
initials: A. V.
mobile: +1 408 624-9737
pager: +1 213 701-5457
manager: cn=Doyle Brunelle
secretary: cn=Hqs Marrone
roomNumber: 5128

dn: cn=Jessika Shirley, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jessika Shirley
sn: Shirley
description: This is Jessika Shirley's description
facsimileTelephoneNumber: +1 303 132-8975
l: Menlo Park
ou: Administrative
postalAddress: example$Administrative$Dept # 418
telephoneNumber: +1 415 482-9480
title: Elite Administrative President
userPassword: yelrihSaki
uid: Jessika_Shirley
givenName: Jessika
mail: Jessika_Shirley@example.com
carLicense: NN9B3MS
departmentNumber: 5551
employeeType: Manager
homePhone: +1 510 420-8465
initials: J. S.
mobile: +1 206 549-8539
pager: +1 71 246-9558
manager: cn=Austin Wray
secretary: cn=Dael Rao
roomNumber: 2852

dn: cn=Purnam Wilsey, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Purnam Wilsey
sn: Wilsey
description: This is Purnam Wilsey's description
facsimileTelephoneNumber: +1 213 548-5517
l: Milpitas
ou: Planning
postalAddress: example$Planning$Dept # 20
telephoneNumber: +1 206 413-4448
title: Elite Planning Assistant
userPassword: yesliWmanr
uid: Purnam_Wilsey
givenName: Purnam
mail: Purnam_Wilsey@example.com
carLicense: 4THLTBM
departmentNumber: 9063
employeeType: Contract
homePhone: +1 818 107-8788
initials: P. W.
mobile: +1 206 869-1238
pager: +1 818 423-3448
manager: cn=Goldina Van Eyk
secretary: cn=Sheila-kathryn Teague
roomNumber: 6208

dn: cn=Diamond Dejongh, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Diamond Dejongh
sn: Dejongh
description: This is Diamond Dejongh's description
facsimileTelephoneNumber: +1 415 690-7520
l: Redmond
ou: Peons
postalAddress: example$Peons$Dept # 399
telephoneNumber: +1 71 359-2616
title: Supreme Peons Technician
userPassword: hgnojeDdno
uid: Diamond_Dejongh
givenName: Diamond
mail: Diamond_Dejongh@example.com
carLicense: T4M7LKD
departmentNumber: 713
employeeType: Manager
homePhone: +1 213 789-8779
initials: D. D.
mobile: +1 818 552-2619
pager: +1 510 205-2550
manager: cn=Seema Ong
secretary: cn=Gabie Sturrock
roomNumber: 1724

dn: cn=Meridian Milotte, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Meridian Milotte
sn: Milotte
description: This is Meridian Milotte's description
facsimileTelephoneNumber: +1 818 998-6569
l: Milpitas
ou: Accounting
postalAddress: example$Accounting$Dept # 788
telephoneNumber: +1 71 833-1231
title: Elite Accounting Artist
userPassword: ettoliMnai
uid: Meridian_Milotte
givenName: Meridian
mail: Meridian_Milotte@example.com
carLicense: S6PQA01
departmentNumber: 8153
employeeType: Temp
homePhone: +1 206 306-3546
initials: M. M.
mobile: +1 510 315-6592
pager: +1 415 677-4723
manager: cn=Holli Wojcik
secretary: cn=Florida Labarge
roomNumber: 3260

dn: cn=Clark Soto, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Clark Soto
sn: Soto
description: This is Clark Soto's description
facsimileTelephoneNumber: +1 510 450-7054
l: San Francisco
ou: Janitorial
postalAddress: example$Janitorial$Dept # 395
telephoneNumber: +1 415 574-1770
title: Junior Janitorial Sales Rep
userPassword: otoSkralC
uid: Clark_Soto
givenName: Clark
mail: Clark_Soto@example.com
carLicense: PPTYC3Z
departmentNumber: 7897
employeeType: Employee
homePhone: +1 804 304-2863
initials: C. S.
mobile: +1 206 393-4632
pager: +1 510 651-1646
manager: cn=Gerianne Skrebels
secretary: cn=Vrouwerff Komenda
roomNumber: 5847

dn: cn=Anurag Gores, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Anurag Gores
sn: Gores
description: This is Anurag Gores's description
facsimileTelephoneNumber: +1 415 579-3094
l: San Mateo
ou: Product Testing
postalAddress: example$Product Testing$Dept # 499
telephoneNumber: +1 415 489-1448
title: Elite Product Testing Fellow
userPassword: seroGgarun
uid: Anurag_Gores
givenName: Anurag
mail: Anurag_Gores@example.com
carLicense: JXH0SUV
departmentNumber: 2859
employeeType: Temp
homePhone: +1 804 967-3243
initials: A. G.
mobile: +1 408 376-8981
pager: +1 818 769-2160
manager: cn=Carry Wennerstrom
secretary: cn=Crawford Ayukawa
roomNumber: 8359

dn: cn=Priti Mathus, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Priti Mathus
sn: Mathus
description: This is Priti Mathus's description
facsimileTelephoneNumber: +1 303 909-3247
l: Fremont
ou: Payroll
postalAddress: example$Payroll$Dept # 139
telephoneNumber: +1 213 812-5903
title: Supreme Payroll Figurehead
userPassword: suhtaMitir
uid: Priti_Mathus
givenName: Priti
mail: Priti_Mathus@example.com
carLicense: QMTZUJ6
departmentNumber: 5971
employeeType: Temp
homePhone: +1 818 121-6367
initials: P. M.
mobile: +1 415 618-5833
pager: +1 213 617-7627
manager: cn=Aruna Harris
secretary: cn=Minnie Dolan
roomNumber: 8375

dn: cn=Joann Tencer, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Joann Tencer
sn: Tencer
description: This is Joann Tencer's description
facsimileTelephoneNumber: +1 818 235-3432
l: Mountain View
ou: Janitorial
postalAddress: example$Janitorial$Dept # 76
telephoneNumber: +1 206 423-4983
title: Junior Janitorial Vice President
userPassword: recneTnnao
uid: Joann_Tencer
givenName: Joann
mail: Joann_Tencer@example.com
carLicense: S2AWHAJ
departmentNumber: 2653
employeeType: Manager
homePhone: +1 206 345-3712
initials: J. T.
mobile: +1 510 810-5835
pager: +1 804 253-7958
manager: cn=Nolie Deslandes
secretary: cn=Aubrey Feldberg
roomNumber: 6014

dn: cn=Stephani Prevost, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Stephani Prevost
sn: Prevost
description: This is Stephani Prevost's description
facsimileTelephoneNumber: +1 510 611-6202
l: Cambridge
ou: Payroll
postalAddress: example$Payroll$Dept # 324
telephoneNumber: +1 415 322-7383
title: Chief Payroll Janitor
userPassword: tsoverPina
uid: Stephani_Prevost
givenName: Stephani
mail: Stephani_Prevost@example.com
carLicense: NL8S1ZS
departmentNumber: 3383
employeeType: Contract
homePhone: +1 213 181-6675
initials: S. P.
mobile: +1 213 978-9750
pager: +1 818 996-4936
manager: cn=Pcta Ceponis
secretary: cn=Antoine Groleau
roomNumber: 5810

dn: cn=Anneliese Baldridge, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Anneliese Baldridge
sn: Baldridge
description: This is Anneliese Baldridge's description
facsimileTelephoneNumber: +1 408 317-2040
l: Milpitas
ou: Product Development
postalAddress: example$Product Development$Dept # 415
telephoneNumber: +1 206 685-7272
title: Associate Product Development Writer
userPassword: egdirdlaBe
uid: Anneliese_Baldridge
givenName: Anneliese
mail: Anneliese_Baldridge@example.com
carLicense: YI622JD
departmentNumber: 8362
employeeType: Manager
homePhone: +1 818 759-4260
initials: A. B.
mobile: +1 818 488-6138
pager: +1 71 997-6404
manager: cn=Jirina Knorr
secretary: cn=Jayesh Malhotra
roomNumber: 5925

dn: cn=Katya Kimler, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Katya Kimler
sn: Kimler
description: This is Katya Kimler's description
facsimileTelephoneNumber: +1 303 952-3948
l: Menlo Park
ou: Human Resources
postalAddress: example$Human Resources$Dept # 937
telephoneNumber: +1 818 356-2917
title: Associate Human Resources Consultant
userPassword: relmiKayta
uid: Katya_Kimler
givenName: Katya
mail: Katya_Kimler@example.com
carLicense: 9S3RWOV
departmentNumber: 2807
employeeType: Normal
homePhone: +1 408 516-1513
initials: K. K.
mobile: +1 213 484-2095
pager: +1 206 723-1364
manager: cn=Perle Antinucci
secretary: cn=Nicol Hartleb
roomNumber: 8148

dn: cn=Balaji Chilton, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Balaji Chilton
sn: Chilton
description: This is Balaji Chilton's description
facsimileTelephoneNumber: +1 510 735-7327
l: Sunnyvale
ou: Payroll
postalAddress: example$Payroll$Dept # 549
telephoneNumber: +1 510 980-2807
title: Master Payroll Developer
userPassword: notlihCija
uid: Balaji_Chilton
givenName: Balaji
mail: Balaji_Chilton@example.com
carLicense: YGOOWNV
departmentNumber: 3737
employeeType: Manager
homePhone: +1 408 239-5774
initials: B. C.
mobile: +1 415 976-9429
pager: +1 510 528-4585
manager: cn=Grant Selic
secretary: cn=Ianthe Jonkheer
roomNumber: 7077

dn: cn=Arabella Luetchford, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Arabella Luetchford
sn: Luetchford
description: This is Arabella Luetchford's description
facsimileTelephoneNumber: +1 213 471-7402
l: Redmond
ou: Product Development
postalAddress: example$Product Development$Dept # 138
telephoneNumber: +1 510 641-6260
title: Elite Product Development Czar
userPassword: drofhcteuL
uid: Arabella_Luetchford
givenName: Arabella
mail: Arabella_Luetchford@example.com
carLicense: SUBK5US
departmentNumber: 1005
employeeType: Contract
homePhone: +1 415 445-2918
initials: A. L.
mobile: +1 408 308-7725
pager: +1 804 427-9393
manager: cn=Gaylene Wieland
secretary: cn=Jade Fredette
roomNumber: 3428

dn: cn=Elsa Lytle, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Elsa Lytle
sn: Lytle
description: This is Elsa Lytle's description
facsimileTelephoneNumber: +1 818 332-6087
l: Santa Clara
ou: Payroll
postalAddress: example$Payroll$Dept # 547
telephoneNumber: +1 415 147-3266
title: Chief Payroll Dictator
userPassword: eltyLaslE
uid: Elsa_Lytle
givenName: Elsa
mail: Elsa_Lytle@example.com
carLicense: SSAVPSH
departmentNumber: 9001
employeeType: Employee
homePhone: +1 818 209-9459
initials: E. L.
mobile: +1 213 106-8761
pager: +1 818 418-9513
manager: cn=Benthem Patchsqa
secretary: cn=Jaman Nomura
roomNumber: 2275

dn: cn=Blinnie MacLennan, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Blinnie MacLennan
sn: MacLennan
description: This is Blinnie MacLennan's description
facsimileTelephoneNumber: +1 206 180-6916
l: Armonk
ou: Accounting
postalAddress: example$Accounting$Dept # 101
telephoneNumber: +1 804 729-5408
title: Junior Accounting Yahoo
userPassword: nanneLcaMe
uid: Blinnie_MacLennan
givenName: Blinnie
mail: Blinnie_MacLennan@example.com
carLicense: FBU7VSE
departmentNumber: 8636
employeeType: Normal
homePhone: +1 408 502-1710
initials: B. M.
mobile: +1 510 120-2151
pager: +1 804 744-7495
manager: cn=Krzysztof McDuffie
secretary: cn=Wiele Winlow
roomNumber: 3835

dn: cn=Bqb Testing, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bqb Testing
sn: Testing
description: This is Bqb Testing's description
facsimileTelephoneNumber: +1 206 415-1952
l: Redmond
ou: Administrative
postalAddress: example$Administrative$Dept # 804
telephoneNumber: +1 213 697-3839
title: Senior Administrative Vice President
userPassword: gnitseTbqB
uid: Bqb_Testing
givenName: Bqb
mail: Bqb_Testing@example.com
carLicense: JYFN7VI
departmentNumber: 406
employeeType: Employee
homePhone: +1 415 246-2701
initials: B. T.
mobile: +1 71 591-5998
pager: +1 818 868-3686
manager: cn=Evangelina Goodridge
secretary: cn=Ramez Sherk
roomNumber: 8009

dn: cn=Melosa Garcia-Molina, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Melosa Garcia-Molina
sn: Garcia-Molina
description: This is Melosa Garcia-Molina's description
facsimileTelephoneNumber: +1 804 468-4833
l: Armonk
ou: Payroll
postalAddress: example$Payroll$Dept # 239
telephoneNumber: +1 206 799-4746
title: Supreme Payroll Madonna
userPassword: aniloM-aic
uid: Melosa_Garcia-Molina
givenName: Melosa
mail: Melosa_Garcia-Molina@example.com
carLicense: 2ZWRAH4
departmentNumber: 3218
employeeType: Temp
homePhone: +1 415 586-7945
initials: M. G.
mobile: +1 213 388-4058
pager: +1 206 760-4662
manager: cn=Marrilee Prestia
secretary: cn=Dani Schreier
roomNumber: 9512

dn: cn=Karry Youngblood, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Karry Youngblood
sn: Youngblood
description: This is Karry Youngblood's description
facsimileTelephoneNumber: +1 818 100-1452
l: Fremont
ou: Janitorial
postalAddress: example$Janitorial$Dept # 503
telephoneNumber: +1 213 184-4069
title: Master Janitorial Manager
userPassword: doolbgnuoY
uid: Karry_Youngblood
givenName: Karry
mail: Karry_Youngblood@example.com
carLicense: YY1HBNJ
departmentNumber: 751
employeeType: Temp
homePhone: +1 303 821-8818
initials: K. Y.
mobile: +1 303 332-3514
pager: +1 206 212-7247
manager: cn=Dael Bogert
secretary: cn=Chu-Chay Ahmadi
roomNumber: 8889

dn: cn=Rebekah Roob, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rebekah Roob
sn: Roob
description: This is Rebekah Roob's description
facsimileTelephoneNumber: +1 510 604-5615
l: Alameda
ou: Payroll
postalAddress: example$Payroll$Dept # 357
telephoneNumber: +1 510 721-4818
title: Master Payroll Madonna
userPassword: booRhakebe
uid: Rebekah_Roob
givenName: Rebekah
mail: Rebekah_Roob@example.com
carLicense: 9I1AFWY
departmentNumber: 6681
employeeType: Temp
homePhone: +1 804 224-8637
initials: R. R.
mobile: +1 510 999-2221
pager: +1 206 902-9608
manager: cn=Abigail Chacko
secretary: cn=Sonny Pacheco
roomNumber: 9394

dn: cn=Car Oates, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Car Oates
sn: Oates
description: This is Car Oates's description
facsimileTelephoneNumber: +1 818 684-5067
l: San Francisco
ou: Planning
postalAddress: example$Planning$Dept # 696
telephoneNumber: +1 71 322-1117
title: Chief Planning Warrior
userPassword: setaOraC
uid: Car_Oates
givenName: Car
mail: Car_Oates@example.com
carLicense: 06NVVNE
departmentNumber: 7341
employeeType: Employee
homePhone: +1 510 190-3084
initials: C. O.
mobile: +1 206 285-8142
pager: +1 303 309-3552
manager: cn=Sylvia Ehrenfried
secretary: cn=Stone Apostolopoulos
roomNumber: 7577

dn: cn=Blinni Kuzemka, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Blinni Kuzemka
sn: Kuzemka
description: This is Blinni Kuzemka's description
facsimileTelephoneNumber: +1 303 531-1360
l: Cambridge
ou: Product Development
postalAddress: example$Product Development$Dept # 43
telephoneNumber: +1 415 908-1379
title: Senior Product Development Accountant
userPassword: akmezuKinn
uid: Blinni_Kuzemka
givenName: Blinni
mail: Blinni_Kuzemka@example.com
carLicense: G8NU9ZO
departmentNumber: 5790
employeeType: Employee
homePhone: +1 408 485-8799
initials: B. K.
mobile: +1 818 816-2078
pager: +1 206 356-6808
manager: cn=Lilllie Woodall
secretary: cn=Nomi Scarffe
roomNumber: 9555

dn: cn=Ingaborg Querengesser, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ingaborg Querengesser
sn: Querengesser
description: This is Ingaborg Querengesser's description
facsimileTelephoneNumber: +1 206 732-6897
l: Redwood Shores
ou: Human Resources
postalAddress: example$Human Resources$Dept # 424
telephoneNumber: +1 213 698-6472
title: Elite Human Resources Artist
userPassword: ressegnere
uid: Ingaborg_Querengesser
givenName: Ingaborg
mail: Ingaborg_Querengesser@example.com
carLicense: 72A9EG0
departmentNumber: 9511
employeeType: Normal
homePhone: +1 408 107-9263
initials: I. Q.
mobile: +1 213 663-2420
pager: +1 415 180-7950
manager: cn=Brena Ballard
secretary: cn=Hall Falkenstrom
roomNumber: 8945

dn: cn=Chau Wargnier, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Chau Wargnier
sn: Wargnier
description: This is Chau Wargnier's description
facsimileTelephoneNumber: +1 206 814-4393
l: Santa Clara
ou: Administrative
postalAddress: example$Administrative$Dept # 171
telephoneNumber: +1 213 191-6545
title: Senior Administrative Technician
userPassword: reingraWua
uid: Chau_Wargnier
givenName: Chau
mail: Chau_Wargnier@example.com
carLicense: 0BK0MIA
departmentNumber: 7972
employeeType: Employee
homePhone: +1 213 898-6905
initials: C. W.
mobile: +1 408 635-9678
pager: +1 213 966-5147
manager: cn=Janeta Corson
secretary: cn=Kikelia Stenson
roomNumber: 2391

dn: cn=Shorwan Velasquez, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shorwan Velasquez
sn: Velasquez
description: This is Shorwan Velasquez's description
facsimileTelephoneNumber: +1 408 758-2210
l: Santa Clara
ou: Product Development
postalAddress: example$Product Development$Dept # 545
telephoneNumber: +1 213 899-5824
title: Supreme Product Development Grunt
userPassword: zeuqsaleVn
uid: Shorwan_Velasquez
givenName: Shorwan
mail: Shorwan_Velasquez@example.com
carLicense: SMQ1XWB
departmentNumber: 1496
employeeType: Contract
homePhone: +1 408 261-7871
initials: S. V.
mobile: +1 804 604-4166
pager: +1 804 282-3825
manager: cn=Ken Piecaitis
secretary: cn=Gwenni Hertler
roomNumber: 9830

dn: cn=Nicolina Carmona, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Nicolina Carmona
sn: Carmona
description: This is Nicolina Carmona's description
facsimileTelephoneNumber: +1 510 185-8337
l: San Jose
ou: Product Development
postalAddress: example$Product Development$Dept # 383
telephoneNumber: +1 804 459-8566
title: Associate Product Development Writer
userPassword: anomraCani
uid: Nicolina_Carmona
givenName: Nicolina
mail: Nicolina_Carmona@example.com
carLicense: F8HC165
departmentNumber: 5296
employeeType: Contract
homePhone: +1 206 976-1598
initials: N. C.
mobile: +1 71 102-2974
pager: +1 804 315-7898
manager: cn=Manmohan Nguyen
secretary: cn=Edouard Rashidi
roomNumber: 8354

dn: cn=Celesta Marketing, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Celesta Marketing
sn: Marketing
description: This is Celesta Marketing's description
facsimileTelephoneNumber: +1 303 138-1000
l: Mountain View
ou: Payroll
postalAddress: example$Payroll$Dept # 275
telephoneNumber: +1 415 923-5744
title: Junior Payroll Technician
userPassword: gnitekraMa
uid: Celesta_Marketing
givenName: Celesta
mail: Celesta_Marketing@example.com
carLicense: DQ24O7Z
departmentNumber: 1828
employeeType: Manager
homePhone: +1 818 365-9314
initials: C. M.
mobile: +1 303 929-4566
pager: +1 510 467-5264
manager: cn=Kimmi Toplis
secretary: cn=Sidonia Yuill
roomNumber: 1977

dn: cn=Alex Pinchen, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Alex Pinchen
sn: Pinchen
description: This is Alex Pinchen's description
facsimileTelephoneNumber: +1 510 103-2167
l: Menlo Park
ou: Product Development
postalAddress: example$Product Development$Dept # 964
telephoneNumber: +1 415 225-2391
title: Supreme Product Development Architect
userPassword: nehcniPxel
uid: Alex_Pinchen
givenName: Alex
mail: Alex_Pinchen@example.com
carLicense: ME29T4Q
departmentNumber: 4571
employeeType: Employee
homePhone: +1 71 596-9249
initials: A. P.
mobile: +1 415 171-6971
pager: +1 818 297-6326
manager: cn=Annemarie Tester
secretary: cn=Myrta Latour
roomNumber: 9463

dn: cn=Ernesto Nerby, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ernesto Nerby
sn: Nerby
description: This is Ernesto Nerby's description
facsimileTelephoneNumber: +1 206 652-1611
l: Redwood Shores
ou: Accounting
postalAddress: example$Accounting$Dept # 773
telephoneNumber: +1 408 139-7115
title: Senior Accounting Vice President
userPassword: ybreNotsen
uid: Ernesto_Nerby
givenName: Ernesto
mail: Ernesto_Nerby@example.com
carLicense: Y1PMBOP
departmentNumber: 1241
employeeType: Employee
homePhone: +1 303 795-7886
initials: E. N.
mobile: +1 804 706-1694
pager: +1 408 988-7072
manager: cn=Subramaniam Sassine
secretary: cn=Tiffanie Keane
roomNumber: 4335

dn: cn=Ibrahim Sproule, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ibrahim Sproule
sn: Sproule
description: This is Ibrahim Sproule's description
facsimileTelephoneNumber: +1 408 640-2419
l: Redwood Shores
ou: Janitorial
postalAddress: example$Janitorial$Dept # 214
telephoneNumber: +1 510 190-6101
title: Master Janitorial Madonna
userPassword: eluorpSmih
uid: Ibrahim_Sproule
givenName: Ibrahim
mail: Ibrahim_Sproule@example.com
carLicense: 8SWCJYI
departmentNumber: 188
employeeType: Contract
homePhone: +1 206 386-9152
initials: I. S.
mobile: +1 213 401-1321
pager: +1 408 139-7904
manager: cn=Blinni Rudiak
secretary: cn=Albrecht Monfre
roomNumber: 4370

dn: cn=Ronda Cristescu, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ronda Cristescu
sn: Cristescu
description: This is Ronda Cristescu's description
facsimileTelephoneNumber: +1 510 864-6492
l: Orem
ou: Accounting
postalAddress: example$Accounting$Dept # 293
telephoneNumber: +1 415 645-1547
title: Chief Accounting Artist
userPassword: ucsetsirCa
uid: Ronda_Cristescu
givenName: Ronda
mail: Ronda_Cristescu@example.com
carLicense: FS3QOOW
departmentNumber: 1146
employeeType: Temp
homePhone: +1 510 408-1132
initials: R. C.
mobile: +1 213 275-3983
pager: +1 415 344-2023
manager: cn=Marv Bnrinfo
secretary: cn=Arly Damena
roomNumber: 554

dn: cn=Philippine Jeronimo, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Philippine Jeronimo
sn: Jeronimo
description: This is Philippine Jeronimo's description
facsimileTelephoneNumber: +1 415 716-5555
l: Redwood Shores
ou: Product Testing
postalAddress: example$Product Testing$Dept # 444
telephoneNumber: +1 510 208-6902
title: Master Product Testing Punk
userPassword: ominoreJen
uid: Philippine_Jeronimo
givenName: Philippine
mail: Philippine_Jeronimo@example.com
carLicense: U2WN2EY
departmentNumber: 7466
employeeType: Temp
homePhone: +1 818 835-7136
initials: P. J.
mobile: +1 213 686-3140
pager: +1 804 513-1137
manager: cn=Tao Haverty
secretary: cn=Yueh Rosson
roomNumber: 2005

dn: cn=Eadie Borodajluk, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Eadie Borodajluk
sn: Borodajluk
description: This is Eadie Borodajluk's description
facsimileTelephoneNumber: +1 213 452-1906
l: Cambridge
ou: Product Development
postalAddress: example$Product Development$Dept # 323
telephoneNumber: +1 303 445-2378
title: Elite Product Development Figurehead
userPassword: kuljadoroB
uid: Eadie_Borodajluk
givenName: Eadie
mail: Eadie_Borodajluk@example.com
carLicense: RC1TZQP
departmentNumber: 7767
employeeType: Manager
homePhone: +1 510 780-5248
initials: E. B.
mobile: +1 303 403-3123
pager: +1 804 399-6145
manager: cn=Kayle Weiser
secretary: cn=Werner Alford
roomNumber: 7021

dn: cn=Jaclin Boehms, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jaclin Boehms
sn: Boehms
description: This is Jaclin Boehms's description
facsimileTelephoneNumber: +1 510 138-5625
l: Mountain View
ou: Peons
postalAddress: example$Peons$Dept # 629
telephoneNumber: +1 71 612-4011
title: Chief Peons Engineer
userPassword: smheoBnilc
uid: Jaclin_Boehms
givenName: Jaclin
mail: Jaclin_Boehms@example.com
carLicense: 0HGDB1Q
departmentNumber: 6375
employeeType: Employee
homePhone: +1 303 400-5548
initials: J. B.
mobile: +1 206 126-6173
pager: +1 804 942-4898
manager: cn=Annecorinne Galvin
secretary: cn=Hideo Colquhoun
roomNumber: 8526

dn: cn=Isabelle Felder, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Isabelle Felder
sn: Felder
description: This is Isabelle Felder's description
facsimileTelephoneNumber: +1 415 851-3994
l: Armonk
ou: Human Resources
postalAddress: example$Human Resources$Dept # 708
telephoneNumber: +1 510 277-1010
title: Chief Human Resources Consultant
userPassword: redleFelle
uid: Isabelle_Felder
givenName: Isabelle
mail: Isabelle_Felder@example.com
carLicense: SUSHAJ9
departmentNumber: 6767
employeeType: Temp
homePhone: +1 206 330-9826
initials: I. F.
mobile: +1 408 243-7429
pager: +1 510 113-7411
manager: cn=Kaman Krautle
secretary: cn=Maggee Zhelka
roomNumber: 4898

dn: cn=Mal Salkilld, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mal Salkilld
sn: Salkilld
description: This is Mal Salkilld's description
facsimileTelephoneNumber: +1 818 216-1414
l: Redwood Shores
ou: Administrative
postalAddress: example$Administrative$Dept # 773
telephoneNumber: +1 213 200-8828
title: Chief Administrative Developer
userPassword: dlliklaSla
uid: Mal_Salkilld
givenName: Mal
mail: Mal_Salkilld@example.com
carLicense: C25Q0HN
departmentNumber: 7571
employeeType: Employee
homePhone: +1 510 225-4247
initials: M. S.
mobile: +1 408 370-2772
pager: +1 804 721-9213
manager: cn=Connie Alvarez
secretary: cn=Mer Evans
roomNumber: 4251

dn: cn=Api-Ecm Willmore, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Api-Ecm Willmore
sn: Willmore
description: This is Api-Ecm Willmore's description
facsimileTelephoneNumber: +1 408 114-1738
l: San Mateo
ou: Accounting
postalAddress: example$Accounting$Dept # 700
telephoneNumber: +1 213 647-4281
title: Elite Accounting Dictator
userPassword: eromlliWmc
uid: Api-Ecm_Willmore
givenName: Api-Ecm
mail: Api-Ecm_Willmore@example.com
carLicense: C6GVAPO
departmentNumber: 3407
employeeType: Contract
homePhone: +1 408 676-3261
initials: A. W.
mobile: +1 818 655-9407
pager: +1 408 794-3155
manager: cn=Yu-Kai Fischer
secretary: cn=Romina Kriegler
roomNumber: 5062

dn: cn=Prab Stover, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Prab Stover
sn: Stover
description: This is Prab Stover's description
facsimileTelephoneNumber: +1 303 454-9353
l: Redwood Shores
ou: Administrative
postalAddress: example$Administrative$Dept # 607
telephoneNumber: +1 213 896-9678
title: Supreme Administrative Warrior
userPassword: revotSbarP
uid: Prab_Stover
givenName: Prab
mail: Prab_Stover@example.com
carLicense: WOGGB4G
departmentNumber: 3248
employeeType: Normal
homePhone: +1 818 889-3903
initials: P. S.
mobile: +1 303 722-5331
pager: +1 71 553-3990
manager: cn=Maggi Deere
secretary: cn=Lulu Feyen
roomNumber: 7129

dn: cn=Tom Spohn, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tom Spohn
sn: Spohn
description: This is Tom Spohn's description
facsimileTelephoneNumber: +1 818 852-9003
l: Redwood Shores
ou: Human Resources
postalAddress: example$Human Resources$Dept # 320
telephoneNumber: +1 71 459-4079
title: Chief Human Resources Director
userPassword: nhopSmoT
uid: Tom_Spohn
givenName: Tom
mail: Tom_Spohn@example.com
carLicense: EXW05UD
departmentNumber: 7720
employeeType: Contract
homePhone: +1 408 417-4336
initials: T. S.
mobile: +1 818 574-3281
pager: +1 415 914-5974
manager: cn=Saraann Waigh
secretary: cn=Blanche Coucopoulos
roomNumber: 6395

dn: cn=Perla Klavkalns, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Perla Klavkalns
sn: Klavkalns
description: This is Perla Klavkalns's description
facsimileTelephoneNumber: +1 408 593-3266
l: Palo Alto
ou: Accounting
postalAddress: example$Accounting$Dept # 785
telephoneNumber: +1 818 391-8407
title: Associate Accounting Director
userPassword: snlakvalKa
uid: Perla_Klavkalns
givenName: Perla
mail: Perla_Klavkalns@example.com
carLicense: 73RCSKR
departmentNumber: 3270
employeeType: Normal
homePhone: +1 818 437-8811
initials: P. K.
mobile: +1 303 598-5727
pager: +1 206 934-2727
manager: cn=Debera Weibust
secretary: cn=Taffy Sastry
roomNumber: 953

dn: cn=Modestia Coviensky, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Modestia Coviensky
sn: Coviensky
description: This is Modestia Coviensky's description
facsimileTelephoneNumber: +1 415 883-4842
l: San Mateo
ou: Janitorial
postalAddress: example$Janitorial$Dept # 978
telephoneNumber: +1 303 686-7438
title: Elite Janitorial Accountant
userPassword: yksneivoCa
uid: Modestia_Coviensky
givenName: Modestia
mail: Modestia_Coviensky@example.com
carLicense: 7MZP5P0
departmentNumber: 6588
employeeType: Contract
homePhone: +1 213 316-3891
initials: M. C.
mobile: +1 206 826-9119
pager: +1 415 313-2422
manager: cn=Ninno Engle
secretary: cn=Sarajane Stanke
roomNumber: 4252

dn: cn=Rini Meier, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rini Meier
sn: Meier
description: This is Rini Meier's description
facsimileTelephoneNumber: +1 213 526-7575
l: Redmond
ou: Peons
postalAddress: example$Peons$Dept # 502
telephoneNumber: +1 71 438-4729
title: Master Peons Artist
userPassword: reieMiniR
uid: Rini_Meier
givenName: Rini
mail: Rini_Meier@example.com
carLicense: QMB8GH5
departmentNumber: 1103
employeeType: Contract
homePhone: +1 213 627-1021
initials: R. M.
mobile: +1 804 353-1875
pager: +1 408 377-2005
manager: cn=Jeroen Dunlay
secretary: cn=Linette Surray
roomNumber: 3593

dn: cn=Ting Audet, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ting Audet
sn: Audet
description: This is Ting Audet's description
facsimileTelephoneNumber: +1 408 585-2475
l: Cambridge
ou: Management
postalAddress: example$Management$Dept # 23
telephoneNumber: +1 415 456-5711
title: Chief Management Accountant
userPassword: teduAgniT
uid: Ting_Audet
givenName: Ting
mail: Ting_Audet@example.com
carLicense: DWXTYWJ
departmentNumber: 8714
employeeType: Employee
homePhone: +1 303 890-4765
initials: T. A.
mobile: +1 71 180-7262
pager: +1 206 393-9581
manager: cn=Rocio Vezina
secretary: cn=Gilemette Erguven
roomNumber: 5931

dn: cn=Teymour Patry, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Teymour Patry
sn: Patry
description: This is Teymour Patry's description
facsimileTelephoneNumber: +1 71 906-2570
l: San Francisco
ou: Product Development
postalAddress: example$Product Development$Dept # 976
telephoneNumber: +1 206 637-9717
title: Associate Product Development Writer
userPassword: yrtaPruomy
uid: Teymour_Patry
givenName: Teymour
mail: Teymour_Patry@example.com
carLicense: INYJU0D
departmentNumber: 2937
employeeType: Temp
homePhone: +1 415 744-9261
initials: T. P.
mobile: +1 510 665-2491
pager: +1 818 935-7075
manager: cn=Mattie Weaver
secretary: cn=Gwendolin Hadden
roomNumber: 6829

dn: cn=Helsa Dressler, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Helsa Dressler
sn: Dressler
description: This is Helsa Dressler's description
facsimileTelephoneNumber: +1 818 884-5702
l: Redwood Shores
ou: Peons
postalAddress: example$Peons$Dept # 622
telephoneNumber: +1 415 317-1038
title: Supreme Peons Architect
userPassword: relsserDas
uid: Helsa_Dressler
givenName: Helsa
mail: Helsa_Dressler@example.com
carLicense: FLGCRQV
departmentNumber: 8092
employeeType: Employee
homePhone: +1 818 167-2407
initials: H. D.
mobile: +1 408 415-4273
pager: +1 818 554-5228
manager: cn=Linnea Van Meter
secretary: cn=Shelbi Thomas
roomNumber: 2280

dn: cn=Jobyna Ackaouy, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jobyna Ackaouy
sn: Ackaouy
description: This is Jobyna Ackaouy's description
facsimileTelephoneNumber: +1 818 499-3930
l: Orem
ou: Janitorial
postalAddress: example$Janitorial$Dept # 718
telephoneNumber: +1 510 489-5638
title: Senior Janitorial Artist
userPassword: yuoakcAany
uid: Jobyna_Ackaouy
givenName: Jobyna
mail: Jobyna_Ackaouy@example.com
carLicense: PWM4Z1O
departmentNumber: 5874
employeeType: Normal
homePhone: +1 213 629-8161
initials: J. A.
mobile: +1 206 798-4549
pager: +1 804 265-8346
manager: cn=Joachim Rahal
secretary: cn=Athene Zonner
roomNumber: 8666

dn: cn=Edeline Cegelski, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Edeline Cegelski
sn: Cegelski
description: This is Edeline Cegelski's description
facsimileTelephoneNumber: +1 71 220-3831
l: Cambridge
ou: Product Development
postalAddress: example$Product Development$Dept # 879
telephoneNumber: +1 213 524-9029
title: Supreme Product Development Assistant
userPassword: ikslegeCen
uid: Edeline_Cegelski
givenName: Edeline
mail: Edeline_Cegelski@example.com
carLicense: SGAENU8
departmentNumber: 373
employeeType: Manager
homePhone: +1 303 682-4481
initials: E. C.
mobile: +1 213 897-9227
pager: +1 206 624-1940
manager: cn=Zena Cottingham
secretary: cn=Rania Brann
roomNumber: 7985

dn: cn=Sharon Bolding, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sharon Bolding
sn: Bolding
description: This is Sharon Bolding's description
facsimileTelephoneNumber: +1 408 987-2312
l: Emeryville
ou: Accounting
postalAddress: example$Accounting$Dept # 485
telephoneNumber: +1 408 842-3786
title: Junior Accounting Writer
userPassword: gnidloBnor
uid: Sharon_Bolding
givenName: Sharon
mail: Sharon_Bolding@example.com
carLicense: KCDS8YF
departmentNumber: 5881
employeeType: Normal
homePhone: +1 206 105-7175
initials: S. B.
mobile: +1 213 923-6505
pager: +1 303 283-7542
manager: cn=Deanna Lambregts
secretary: cn=Naser Kahhale
roomNumber: 4980

dn: cn=Benoit Stampley, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Benoit Stampley
sn: Stampley
description: This is Benoit Stampley's description
facsimileTelephoneNumber: +1 510 376-8381
l: Palo Alto
ou: Planning
postalAddress: example$Planning$Dept # 458
telephoneNumber: +1 206 338-1295
title: Associate Planning Architect
userPassword: yelpmatSti
uid: Benoit_Stampley
givenName: Benoit
mail: Benoit_Stampley@example.com
carLicense: CLK3X3D
departmentNumber: 8307
employeeType: Temp
homePhone: +1 804 305-9283
initials: B. S.
mobile: +1 415 859-2869
pager: +1 206 465-2730
manager: cn=Cynde Vahary
secretary: cn=Franc Smoot
roomNumber: 6379

dn: cn=Said Relations, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Said Relations
sn: Relations
description: This is Said Relations's description
facsimileTelephoneNumber: +1 213 281-5932
l: San Francisco
ou: Product Development
postalAddress: example$Product Development$Dept # 546
telephoneNumber: +1 804 997-7533
title: Master Product Development Figurehead
userPassword: snoitaleRd
uid: Said_Relations
givenName: Said
mail: Said_Relations@example.com
carLicense: NA0SZ53
departmentNumber: 2137
employeeType: Manager
homePhone: +1 415 960-5640
initials: S. R.
mobile: +1 415 307-2107
pager: +1 510 368-5019
manager: cn=Issy Rau
secretary: cn=Brandais Poindexter
roomNumber: 3833

dn: cn=Deva Cakarevic, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Deva Cakarevic
sn: Cakarevic
description: This is Deva Cakarevic's description
facsimileTelephoneNumber: +1 804 899-3758
l: Fremont
ou: Accounting
postalAddress: example$Accounting$Dept # 825
telephoneNumber: +1 206 319-3019
title: Elite Accounting Punk
userPassword: civerakaCa
uid: Deva_Cakarevic
givenName: Deva
mail: Deva_Cakarevic@example.com
carLicense: URSX167
departmentNumber: 935
employeeType: Manager
homePhone: +1 818 153-4850
initials: D. C.
mobile: +1 408 385-9712
pager: +1 303 126-3737
manager: cn=Wallie Kouhi
secretary: cn=Jobi Coggins
roomNumber: 6423

dn: cn=Ermina Lannan, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ermina Lannan
sn: Lannan
description: This is Ermina Lannan's description
facsimileTelephoneNumber: +1 415 101-8882
l: Sunnyvale
ou: Management
postalAddress: example$Management$Dept # 782
telephoneNumber: +1 213 680-9698
title: Associate Management Sales Rep
userPassword: nannaLanim
uid: Ermina_Lannan
givenName: Ermina
mail: Ermina_Lannan@example.com
carLicense: D8L35SS
departmentNumber: 139
employeeType: Manager
homePhone: +1 71 140-5966
initials: E. L.
mobile: +1 415 363-7285
pager: +1 510 237-3675
manager: cn=Ahmed Lassig
secretary: cn=Giuseppe Downey
roomNumber: 6461

dn: cn=Bqb Marette, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bqb Marette
sn: Marette
description: This is Bqb Marette's description
facsimileTelephoneNumber: +1 804 222-9616
l: Cupertino
ou: Planning
postalAddress: example$Planning$Dept # 647
telephoneNumber: +1 804 939-1042
title: Supreme Planning Madonna
userPassword: etteraMbqB
uid: Bqb_Marette
givenName: Bqb
mail: Bqb_Marette@example.com
carLicense: WQ70RXI
departmentNumber: 169
employeeType: Contract
homePhone: +1 303 898-7967
initials: B. M.
mobile: +1 510 742-2632
pager: +1 213 378-3866
manager: cn=Levent Hummerston
secretary: cn=Doralin Georges
roomNumber: 8753

dn: cn=Liese Wolczanski, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Liese Wolczanski
sn: Wolczanski
description: This is Liese Wolczanski's description
facsimileTelephoneNumber: +1 510 573-6007
l: Mountain View
ou: Planning
postalAddress: example$Planning$Dept # 638
telephoneNumber: +1 206 831-2154
title: Master Planning Architect
userPassword: iksnazcloW
uid: Liese_Wolczanski
givenName: Liese
mail: Liese_Wolczanski@example.com
carLicense: 0CLR48Z
departmentNumber: 314
employeeType: Temp
homePhone: +1 804 330-5492
initials: L. W.
mobile: +1 804 511-1187
pager: +1 415 616-5291
manager: cn=Meade Talmont
secretary: cn=Fereidoon Herscovici
roomNumber: 7599

dn: cn=Ahmet Covey, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ahmet Covey
sn: Covey
description: This is Ahmet Covey's description
facsimileTelephoneNumber: +1 206 196-9623
l: Armonk
ou: Administrative
postalAddress: example$Administrative$Dept # 621
telephoneNumber: +1 213 318-6865
title: Senior Administrative Pinhead
userPassword: yevoCtemhA
uid: Ahmet_Covey
givenName: Ahmet
mail: Ahmet_Covey@example.com
carLicense: LLJX9BE
departmentNumber: 1598
employeeType: Employee
homePhone: +1 510 515-5928
initials: A. C.
mobile: +1 303 574-6475
pager: +1 71 787-5024
manager: cn=Oren Kannel
secretary: cn=Shiva Baer
roomNumber: 2934

dn: cn=Nona Knorr, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Nona Knorr
sn: Knorr
description: This is Nona Knorr's description
facsimileTelephoneNumber: +1 510 986-6709
l: Alameda
ou: Payroll
postalAddress: example$Payroll$Dept # 287
telephoneNumber: +1 804 342-2409
title: Junior Payroll Accountant
userPassword: rronKanoN
uid: Nona_Knorr
givenName: Nona
mail: Nona_Knorr@example.com
carLicense: IBBEKR8
departmentNumber: 569
employeeType: Contract
homePhone: +1 415 972-2312
initials: N. K.
mobile: +1 71 299-2716
pager: +1 408 697-8594
manager: cn=Sharron Kishi
secretary: cn=Emmey Raaflaub
roomNumber: 8189

dn: cn=Alane deMontluzin, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Alane deMontluzin
sn: deMontluzin
description: This is Alane deMontluzin's description
facsimileTelephoneNumber: +1 206 966-7764
l: Redmond
ou: Management
postalAddress: example$Management$Dept # 994
telephoneNumber: +1 804 268-8328
title: Junior Management Architect
userPassword: nizultnoMe
uid: Alane_deMontluzin
givenName: Alane
mail: Alane_deMontluzin@example.com
carLicense: Q2SBGF5
departmentNumber: 7468
employeeType: Manager
homePhone: +1 408 352-4589
initials: A. d.
mobile: +1 71 684-3917
pager: +1 303 840-4607
manager: cn=Aloise Engelhart
secretary: cn=Nicol Pancholy
roomNumber: 8247

dn: cn=Karlene Combellack, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Karlene Combellack
sn: Combellack
description: This is Karlene Combellack's description
facsimileTelephoneNumber: +1 303 375-7590
l: Cupertino
ou: Management
postalAddress: example$Management$Dept # 72
telephoneNumber: +1 510 123-4455
title: Junior Management Vice President
userPassword: kcallebmoC
uid: Karlene_Combellack
givenName: Karlene
mail: Karlene_Combellack@example.com
carLicense: OCDZU2F
departmentNumber: 9006
employeeType: Employee
homePhone: +1 804 125-5891
initials: K. C.
mobile: +1 408 126-4843
pager: +1 804 602-5945
manager: cn=Astra Durant
secretary: cn=Petronilla Shabatura
roomNumber: 9322

dn: cn=Lowell Piotto, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lowell Piotto
sn: Piotto
description: This is Lowell Piotto's description
facsimileTelephoneNumber: +1 408 485-1853
l: Alameda
ou: Accounting
postalAddress: example$Accounting$Dept # 114
telephoneNumber: +1 206 123-8513
title: Junior Accounting Fellow
userPassword: ottoiPllew
uid: Lowell_Piotto
givenName: Lowell
mail: Lowell_Piotto@example.com
carLicense: DRS8XSG
departmentNumber: 2716
employeeType: Temp
homePhone: +1 213 404-7970
initials: L. P.
mobile: +1 408 924-2754
pager: +1 71 622-1922
manager: cn=Julietta Galluzzi
secretary: cn=Wannell Towsley
roomNumber: 4029

dn: cn=JR Pezzullo, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: JR Pezzullo
sn: Pezzullo
description: This is JR Pezzullo's description
facsimileTelephoneNumber: +1 71 460-7618
l: San Mateo
ou: Administrative
postalAddress: example$Administrative$Dept # 630
telephoneNumber: +1 818 796-1485
title: Associate Administrative Artist
userPassword: olluzzePRJ
uid: JR_Pezzullo
givenName: JR
mail: JR_Pezzullo@example.com
carLicense: PRQ6ADM
departmentNumber: 5668
employeeType: Employee
homePhone: +1 804 363-4761
initials: J. P.
mobile: +1 415 187-3333
pager: +1 510 125-1940
manager: cn=Chastity Hamori
secretary: cn=Conway Levin
roomNumber: 1379

dn: cn=Binni Munsey, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Binni Munsey
sn: Munsey
description: This is Binni Munsey's description
facsimileTelephoneNumber: +1 510 472-4712
l: Cambridge
ou: Planning
postalAddress: example$Planning$Dept # 449
telephoneNumber: +1 303 415-8355
title: Master Planning Consultant
userPassword: yesnuMinni
uid: Binni_Munsey
givenName: Binni
mail: Binni_Munsey@example.com
carLicense: VH2YC4W
departmentNumber: 3217
employeeType: Temp
homePhone: +1 804 302-6225
initials: B. M.
mobile: +1 206 459-4235
pager: +1 213 267-7393
manager: cn=Corissa McNerlan
secretary: cn=Pic Kohn
roomNumber: 1672

dn: cn=Gernot Mirek, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gernot Mirek
sn: Mirek
description: This is Gernot Mirek's description
facsimileTelephoneNumber: +1 71 744-5990
l: Sunnyvale
ou: Human Resources
postalAddress: example$Human Resources$Dept # 491
telephoneNumber: +1 408 323-8792
title: Senior Human Resources Engineer
userPassword: keriMtonre
uid: Gernot_Mirek
givenName: Gernot
mail: Gernot_Mirek@example.com
carLicense: 56PVP59
departmentNumber: 8407
employeeType: Normal
homePhone: +1 818 598-3576
initials: G. M.
mobile: +1 206 863-2296
pager: +1 818 403-1609
manager: cn=Meris Omura
secretary: cn=Kirsten Bullion
roomNumber: 297

dn: cn=Madalene Hesk, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Madalene Hesk
sn: Hesk
description: This is Madalene Hesk's description
facsimileTelephoneNumber: +1 408 954-9133
l: Sunnyvale
ou: Payroll
postalAddress: example$Payroll$Dept # 427
telephoneNumber: +1 206 629-7573
title: Senior Payroll Stooge
userPassword: kseHenelad
uid: Madalene_Hesk
givenName: Madalene
mail: Madalene_Hesk@example.com
carLicense: JL1NDFK
departmentNumber: 3011
employeeType: Manager
homePhone: +1 303 349-2087
initials: M. H.
mobile: +1 804 205-5440
pager: +1 408 482-1246
manager: cn=Yuan Mitsui
secretary: cn=Khosro Paulin
roomNumber: 7563

dn: cn=Ketti Kehr, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ketti Kehr
sn: Kehr
description: This is Ketti Kehr's description
facsimileTelephoneNumber: +1 213 436-1385
l: Alameda
ou: Human Resources
postalAddress: example$Human Resources$Dept # 785
telephoneNumber: +1 408 721-5498
title: Junior Human Resources Architect
userPassword: rheKitteK
uid: Ketti_Kehr
givenName: Ketti
mail: Ketti_Kehr@example.com
carLicense: Y0QLRIU
departmentNumber: 6558
employeeType: Employee
homePhone: +1 303 755-9515
initials: K. K.
mobile: +1 71 844-4943
pager: +1 415 558-2691
manager: cn=Dalip Danko
secretary: cn=Tessi Campara
roomNumber: 7697

dn: cn=Antonia Marcom, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Antonia Marcom
sn: Marcom
description: This is Antonia Marcom's description
facsimileTelephoneNumber: +1 213 211-8314
l: San Mateo
ou: Payroll
postalAddress: example$Payroll$Dept # 134
telephoneNumber: +1 71 504-2602
title: Senior Payroll Warrior
userPassword: mocraMaino
uid: Antonia_Marcom
givenName: Antonia
mail: Antonia_Marcom@example.com
carLicense: WIZ5G1X
departmentNumber: 7766
employeeType: Manager
homePhone: +1 408 112-2793
initials: A. M.
mobile: +1 408 113-3390
pager: +1 818 400-6100
manager: cn=Fuzal Yerigan
secretary: cn=Mouna Forghani
roomNumber: 3247

dn: cn=Xuan-Lien Thoms, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Xuan-Lien Thoms
sn: Thoms
description: This is Xuan-Lien Thoms's description
facsimileTelephoneNumber: +1 818 700-9243
l: Redmond
ou: Planning
postalAddress: example$Planning$Dept # 279
telephoneNumber: +1 818 695-7186
title: Elite Planning Developer
userPassword: smohTneiL-
uid: Xuan-Lien_Thoms
givenName: Xuan-Lien
mail: Xuan-Lien_Thoms@example.com
carLicense: RR8HW30
departmentNumber: 5729
employeeType: Contract
homePhone: +1 804 759-9967
initials: X. T.
mobile: +1 804 976-9015
pager: +1 206 634-9113
manager: cn=Oralia Rajwani
secretary: cn=Shandie Maxin
roomNumber: 6773

dn: cn=Estelle Leenher, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Estelle Leenher
sn: Leenher
description: This is Estelle Leenher's description
facsimileTelephoneNumber: +1 818 222-1752
l: San Francisco
ou: Accounting
postalAddress: example$Accounting$Dept # 168
telephoneNumber: +1 415 310-8124
title: Junior Accounting Engineer
userPassword: rehneeLell
uid: Estelle_Leenher
givenName: Estelle
mail: Estelle_Leenher@example.com
carLicense: HZMBN5S
departmentNumber: 9010
employeeType: Contract
homePhone: +1 818 281-4909
initials: E. L.
mobile: +1 804 354-3908
pager: +1 510 142-5148
manager: cn=Sayeeda Traut
secretary: cn=Shayne Pardi
roomNumber: 4701

dn: cn=Adriana Brummitt, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Adriana Brummitt
sn: Brummitt
description: This is Adriana Brummitt's description
facsimileTelephoneNumber: +1 804 719-7444
l: Sunnyvale
ou: Payroll
postalAddress: example$Payroll$Dept # 779
telephoneNumber: +1 510 344-1272
title: Junior Payroll Fellow
userPassword: ttimmurBan
uid: Adriana_Brummitt
givenName: Adriana
mail: Adriana_Brummitt@example.com
carLicense: W60QRX9
departmentNumber: 6588
employeeType: Temp
homePhone: +1 818 413-2831
initials: A. B.
mobile: +1 408 192-2443
pager: +1 303 460-9559
manager: cn=Hendrik Lormor
secretary: cn=Bhupinder Scammerhorn
roomNumber: 5310

dn: cn=Ivette Eckhart, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ivette Eckhart
sn: Eckhart
description: This is Ivette Eckhart's description
facsimileTelephoneNumber: +1 408 149-2483
l: San Francisco
ou: Product Testing
postalAddress: example$Product Testing$Dept # 744
telephoneNumber: +1 804 294-4481
title: Master Product Testing Accountant
userPassword: trahkcEett
uid: Ivette_Eckhart
givenName: Ivette
mail: Ivette_Eckhart@example.com
carLicense: BOANDJ5
departmentNumber: 1352
employeeType: Contract
homePhone: +1 206 867-5183
initials: I. E.
mobile: +1 408 229-7675
pager: +1 303 170-9372
manager: cn=Marylynne Kurczak
secretary: cn=Dilpreet Putnam
roomNumber: 5576

dn: cn=Fraser Naor, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Fraser Naor
sn: Naor
description: This is Fraser Naor's description
facsimileTelephoneNumber: +1 415 435-8678
l: Armonk
ou: Human Resources
postalAddress: example$Human Resources$Dept # 239
telephoneNumber: +1 818 327-8628
title: Associate Human Resources Pinhead
userPassword: roaNresarF
uid: Fraser_Naor
givenName: Fraser
mail: Fraser_Naor@example.com
carLicense: WR7H1ZY
departmentNumber: 8573
employeeType: Contract
homePhone: +1 303 498-3230
initials: F. N.
mobile: +1 408 956-2720
pager: +1 818 450-9361
manager: cn=Abdullah Sutphen
secretary: cn=Paulinus Henneberger
roomNumber: 4586

dn: cn=Cherice Hysler, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Cherice Hysler
sn: Hysler
description: This is Cherice Hysler's description
facsimileTelephoneNumber: +1 206 989-8948
l: Palo Alto
ou: Janitorial
postalAddress: example$Janitorial$Dept # 230
telephoneNumber: +1 510 714-9547
title: Supreme Janitorial Director
userPassword: relsyHecir
uid: Cherice_Hysler
givenName: Cherice
mail: Cherice_Hysler@example.com
carLicense: 4XDBRVE
departmentNumber: 6877
employeeType: Normal
homePhone: +1 510 836-7582
initials: C. H.
mobile: +1 804 125-5705
pager: +1 804 180-1364
manager: cn=Ardelle Dryer
secretary: cn=Terrell D'Antonio
roomNumber: 8269

dn: cn=Rosemonde St.Germain, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rosemonde St.Germain
sn: St.Germain
description: This is Rosemonde St.Germain's description
facsimileTelephoneNumber: +1 71 227-9241
l: Redmond
ou: Product Development
postalAddress: example$Product Development$Dept # 207
telephoneNumber: +1 206 310-2724
title: Junior Product Development Stooge
userPassword: niamreG.tS
uid: Rosemonde_St.Germain
givenName: Rosemonde
mail: Rosemonde_St.Germain@example.com
carLicense: JZ0X19C
departmentNumber: 4957
employeeType: Normal
homePhone: +1 206 289-1031
initials: R. S.
mobile: +1 408 394-5151
pager: +1 415 946-2203
manager: cn=Afton Rtpbuild
secretary: cn=Gabriella Hann
roomNumber: 2031

dn: cn=Kien-Nghiep Ranahan, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kien-Nghiep Ranahan
sn: Ranahan
description: This is Kien-Nghiep Ranahan's description
facsimileTelephoneNumber: +1 71 441-9506
l: Cupertino
ou: Planning
postalAddress: example$Planning$Dept # 666
telephoneNumber: +1 213 289-1356
title: Junior Planning Stooge
userPassword: nahanaRpei
uid: Kien-Nghiep_Ranahan
givenName: Kien-Nghiep
mail: Kien-Nghiep_Ranahan@example.com
carLicense: ZT8PY6H
departmentNumber: 764
employeeType: Temp
homePhone: +1 510 609-7714
initials: K. R.
mobile: +1 213 512-8114
pager: +1 213 459-8684
manager: cn=Jean-Yves Seay
secretary: cn=Wynnie Matsuzawa
roomNumber: 613

dn: cn=Dorisa Batchelder, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dorisa Batchelder
sn: Batchelder
description: This is Dorisa Batchelder's description
facsimileTelephoneNumber: +1 510 454-2409
l: Fremont
ou: Management
postalAddress: example$Management$Dept # 751
telephoneNumber: +1 408 813-2879
title: Junior Management Evangelist
userPassword: redlehctaB
uid: Dorisa_Batchelder
givenName: Dorisa
mail: Dorisa_Batchelder@example.com
carLicense: EWR304Z
departmentNumber: 7186
employeeType: Normal
homePhone: +1 804 277-4974
initials: D. B.
mobile: +1 818 410-1396
pager: +1 818 798-6674
manager: cn=Barbara Swails
secretary: cn=Clevon Miasek
roomNumber: 115

dn: cn=Dodie Australia, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dodie Australia
sn: Australia
description: This is Dodie Australia's description
facsimileTelephoneNumber: +1 818 483-6600
l: Santa Clara
ou: Administrative
postalAddress: example$Administrative$Dept # 310
telephoneNumber: +1 408 630-2959
title: Associate Administrative Writer
userPassword: ailartsuAe
uid: Dodie_Australia
givenName: Dodie
mail: Dodie_Australia@example.com
carLicense: K6FX4EA
departmentNumber: 3837
employeeType: Employee
homePhone: +1 804 791-1955
initials: D. A.
mobile: +1 206 881-2826
pager: +1 213 387-7159
manager: cn=Guglielma Oman
secretary: cn=Catherine Coste
roomNumber: 3434

dn: cn=Sharity Overcash, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sharity Overcash
sn: Overcash
description: This is Sharity Overcash's description
facsimileTelephoneNumber: +1 206 422-7168
l: Cambridge
ou: Janitorial
postalAddress: example$Janitorial$Dept # 274
telephoneNumber: +1 818 423-6507
title: Junior Janitorial Janitor
userPassword: hsacrevOyt
uid: Sharity_Overcash
givenName: Sharity
mail: Sharity_Overcash@example.com
carLicense: 2ZU7J65
departmentNumber: 2845
employeeType: Employee
homePhone: +1 213 120-8577
initials: S. O.
mobile: +1 206 716-9529
pager: +1 206 908-2942
manager: cn=Anet Nevrela
secretary: cn=Thanh-Hoa Hysler
roomNumber: 7742

dn: cn=Ema Decasper, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ema Decasper
sn: Decasper
description: This is Ema Decasper's description
facsimileTelephoneNumber: +1 408 654-4688
l: Orem
ou: Janitorial
postalAddress: example$Janitorial$Dept # 654
telephoneNumber: +1 408 143-5292
title: Senior Janitorial Dictator
userPassword: repsaceDam
uid: Ema_Decasper
givenName: Ema
mail: Ema_Decasper@example.com
carLicense: MRKL93I
departmentNumber: 4821
employeeType: Contract
homePhone: +1 818 820-8509
initials: E. D.
mobile: +1 303 830-2198
pager: +1 303 373-8951
manager: cn=Hubert Monterosso-Wood
secretary: cn=Cordelie Bourland
roomNumber: 4037

dn: cn=Kinna Frankcom, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kinna Frankcom
sn: Frankcom
description: This is Kinna Frankcom's description
facsimileTelephoneNumber: +1 804 988-8473
l: Emeryville
ou: Accounting
postalAddress: example$Accounting$Dept # 868
telephoneNumber: +1 818 672-5752
title: Chief Accounting Madonna
userPassword: mocknarFan
uid: Kinna_Frankcom
givenName: Kinna
mail: Kinna_Frankcom@example.com
carLicense: 1M5KX16
departmentNumber: 1552
employeeType: Normal
homePhone: +1 818 515-6816
initials: K. F.
mobile: +1 213 724-2917
pager: +1 213 853-5072
manager: cn=Adaline Castronova
secretary: cn=Samual Devlin
roomNumber: 7212

dn: cn=Thom Littau, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Thom Littau
sn: Littau
description: This is Thom Littau's description
facsimileTelephoneNumber: +1 213 437-8097
l: Palo Alto
ou: Janitorial
postalAddress: example$Janitorial$Dept # 713
telephoneNumber: +1 303 292-3348
title: Associate Janitorial Pinhead
userPassword: uattiLmohT
uid: Thom_Littau
givenName: Thom
mail: Thom_Littau@example.com
carLicense: UPLF680
departmentNumber: 1383
employeeType: Temp
homePhone: +1 303 484-8297
initials: T. L.
mobile: +1 71 551-7795
pager: +1 206 304-3576
manager: cn=Hack-Hoo Hopf
secretary: cn=Pirooz Frankcom
roomNumber: 1642

dn: cn=Shana Polulack, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shana Polulack
sn: Polulack
description: This is Shana Polulack's description
facsimileTelephoneNumber: +1 510 674-7584
l: San Mateo
ou: Product Testing
postalAddress: example$Product Testing$Dept # 564
telephoneNumber: +1 206 792-8167
title: Associate Product Testing Developer
userPassword: kcaluloPan
uid: Shana_Polulack
givenName: Shana
mail: Shana_Polulack@example.com
carLicense: RUT1TGQ
departmentNumber: 5436
employeeType: Manager
homePhone: +1 408 951-4182
initials: S. P.
mobile: +1 804 370-6607
pager: +1 408 278-5227
manager: cn=Emilda Pifko
secretary: cn=Leena Cassar
roomNumber: 3276

dn: cn=Rivalee Letsome, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rivalee Letsome
sn: Letsome
description: This is Rivalee Letsome's description
facsimileTelephoneNumber: +1 206 348-5919
l: Palo Alto
ou: Product Development
postalAddress: example$Product Development$Dept # 413
telephoneNumber: +1 408 273-4354
title: Master Product Development Manager
userPassword: emosteLeel
uid: Rivalee_Letsome
givenName: Rivalee
mail: Rivalee_Letsome@example.com
carLicense: 23EUZEN
departmentNumber: 8348
employeeType: Employee
homePhone: +1 303 488-9887
initials: R. L.
mobile: +1 408 310-7111
pager: +1 408 289-1053
manager: cn=Ramon Marco
secretary: cn=Nurhan Lobianco
roomNumber: 1488

dn: cn=Bernardina Juscesak, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bernardina Juscesak
sn: Juscesak
description: This is Bernardina Juscesak's description
facsimileTelephoneNumber: +1 510 197-8837
l: Redwood Shores
ou: Accounting
postalAddress: example$Accounting$Dept # 327
telephoneNumber: +1 510 946-3336
title: Elite Accounting Dictator
userPassword: kasecsuJan
uid: Bernardina_Juscesak
givenName: Bernardina
mail: Bernardina_Juscesak@example.com
carLicense: MV649YK
departmentNumber: 4230
employeeType: Contract
homePhone: +1 303 527-7531
initials: B. J.
mobile: +1 818 661-5367
pager: +1 213 301-2474
manager: cn=Neala Bridgeford
secretary: cn=Antonie Liew
roomNumber: 4837

dn: cn=Annadiane Axberg, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Annadiane Axberg
sn: Axberg
description: This is Annadiane Axberg's description
facsimileTelephoneNumber: +1 71 559-3741
l: Alameda
ou: Payroll
postalAddress: example$Payroll$Dept # 545
telephoneNumber: +1 206 507-8005
title: Chief Payroll Stooge
userPassword: grebxAenai
uid: Annadiane_Axberg
givenName: Annadiane
mail: Annadiane_Axberg@example.com
carLicense: LUT4HGU
departmentNumber: 7433
employeeType: Contract
homePhone: +1 303 865-6381
initials: A. A.
mobile: +1 415 801-6258
pager: +1 303 733-6064
manager: cn=Berangere Walker
secretary: cn=Vahid Hylarides
roomNumber: 3756

dn: cn=Fu-Shin Cantlie, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Fu-Shin Cantlie
sn: Cantlie
description: This is Fu-Shin Cantlie's description
facsimileTelephoneNumber: +1 510 953-3859
l: Cambridge
ou: Product Testing
postalAddress: example$Product Testing$Dept # 286
telephoneNumber: +1 206 313-2050
title: Supreme Product Testing Sales Rep
userPassword: eiltnaCnih
uid: Fu-Shin_Cantlie
givenName: Fu-Shin
mail: Fu-Shin_Cantlie@example.com
carLicense: M8S7K9C
departmentNumber: 8318
employeeType: Normal
homePhone: +1 206 216-6429
initials: F. C.
mobile: +1 213 974-6973
pager: +1 510 472-2562
manager: cn=Jimson Brantley
secretary: cn=Etty Castillo
roomNumber: 7956

dn: cn=Tedda Langenberg, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tedda Langenberg
sn: Langenberg
description: This is Tedda Langenberg's description
facsimileTelephoneNumber: +1 71 730-8246
l: Cupertino
ou: Accounting
postalAddress: example$Accounting$Dept # 929
telephoneNumber: +1 510 358-8817
title: Senior Accounting Manager
userPassword: grebnegnaL
uid: Tedda_Langenberg
givenName: Tedda
mail: Tedda_Langenberg@example.com
carLicense: 21R9LZO
departmentNumber: 9330
employeeType: Employee
homePhone: +1 213 457-6359
initials: T. L.
mobile: +1 510 400-7885
pager: +1 804 425-3098
manager: cn=Ron Sanzone
secretary: cn=Zuben Lukers
roomNumber: 91

dn: cn=Brana Suda, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Brana Suda
sn: Suda
description: This is Brana Suda's description
facsimileTelephoneNumber: +1 510 924-3977
l: Emeryville
ou: Planning
postalAddress: example$Planning$Dept # 942
telephoneNumber: +1 818 460-4131
title: Master Planning Accountant
userPassword: aduSanarB
uid: Brana_Suda
givenName: Brana
mail: Brana_Suda@example.com
carLicense: M50B1H0
departmentNumber: 4903
employeeType: Employee
homePhone: +1 206 907-3010
initials: B. S.
mobile: +1 71 388-7659
pager: +1 408 832-9280
manager: cn=Wendy Amini
secretary: cn=Mika Schmoe
roomNumber: 5050

dn: cn=Eadith Fradette, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Eadith Fradette
sn: Fradette
description: This is Eadith Fradette's description
facsimileTelephoneNumber: +1 818 220-1528
l: San Jose
ou: Administrative
postalAddress: example$Administrative$Dept # 889
telephoneNumber: +1 818 970-1343
title: Senior Administrative Vice President
userPassword: ettedarFht
uid: Eadith_Fradette
givenName: Eadith
mail: Eadith_Fradette@example.com
carLicense: 9KWU50X
departmentNumber: 9660
employeeType: Contract
homePhone: +1 804 824-9292
initials: E. F.
mobile: +1 415 727-1332
pager: +1 510 128-9094
manager: cn=Darren Talbot
secretary: cn=Phoenix Van Phil
roomNumber: 3334

dn: cn=Clio Iyengar, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Clio Iyengar
sn: Iyengar
description: This is Clio Iyengar's description
facsimileTelephoneNumber: +1 206 683-9934
l: Cupertino
ou: Administrative
postalAddress: example$Administrative$Dept # 178
telephoneNumber: +1 818 430-9658
title: Chief Administrative Grunt
userPassword: ragneyIoil
uid: Clio_Iyengar
givenName: Clio
mail: Clio_Iyengar@example.com
carLicense: 54FIVQO
departmentNumber: 1660
employeeType: Employee
homePhone: +1 213 169-4863
initials: C. I.
mobile: +1 206 491-7672
pager: +1 71 531-5105
manager: cn=Lujanka Meisner
secretary: cn=Natassia Perreault
roomNumber: 2974

dn: cn=Motaz Mrozinski, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Motaz Mrozinski
sn: Mrozinski
description: This is Motaz Mrozinski's description
facsimileTelephoneNumber: +1 804 406-5061
l: San Jose
ou: Payroll
postalAddress: example$Payroll$Dept # 894
telephoneNumber: +1 213 539-4117
title: Supreme Payroll Mascot
userPassword: iksnizorMz
uid: Motaz_Mrozinski
givenName: Motaz
mail: Motaz_Mrozinski@example.com
carLicense: O98UZA3
departmentNumber: 8943
employeeType: Employee
homePhone: +1 415 714-8328
initials: M. M.
mobile: +1 415 300-4474
pager: +1 303 266-4788
manager: cn=Lucien Mihan
secretary: cn=Mirilla Sobiesiak
roomNumber: 3718

dn: cn=Jose Woodward-Jack, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jose Woodward-Jack
sn: Woodward-Jack
description: This is Jose Woodward-Jack's description
facsimileTelephoneNumber: +1 303 205-2281
l: Mountain View
ou: Administrative
postalAddress: example$Administrative$Dept # 346
telephoneNumber: +1 415 177-7564
title: Junior Administrative Manager
userPassword: kcaJ-drawd
uid: Jose_Woodward-Jack
givenName: Jose
mail: Jose_Woodward-Jack@example.com
carLicense: 4XWHTCY
departmentNumber: 7557
employeeType: Employee
homePhone: +1 415 241-9728
initials: J. W.
mobile: +1 415 173-2547
pager: +1 415 609-3295
manager: cn=Schaffer Milloy
secretary: cn=Naser Michaels
roomNumber: 6069

dn: cn=Dyane Kluger, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dyane Kluger
sn: Kluger
description: This is Dyane Kluger's description
facsimileTelephoneNumber: +1 415 234-3632
l: Redwood Shores
ou: Administrative
postalAddress: example$Administrative$Dept # 64
telephoneNumber: +1 510 634-4698
title: Chief Administrative Admin
userPassword: regulKenay
uid: Dyane_Kluger
givenName: Dyane
mail: Dyane_Kluger@example.com
carLicense: EWQY2L7
departmentNumber: 7143
employeeType: Temp
homePhone: +1 408 442-1794
initials: D. K.
mobile: +1 804 340-5393
pager: +1 213 895-5852
manager: cn=Electra Kato
secretary: cn=Chi Alberse
roomNumber: 5974

dn: cn=Esmeralda Mahonen, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Esmeralda Mahonen
sn: Mahonen
description: This is Esmeralda Mahonen's description
facsimileTelephoneNumber: +1 71 138-4348
l: Cambridge
ou: Peons
postalAddress: example$Peons$Dept # 825
telephoneNumber: +1 206 997-5694
title: Associate Peons Figurehead
userPassword: nenohaMadl
uid: Esmeralda_Mahonen
givenName: Esmeralda
mail: Esmeralda_Mahonen@example.com
carLicense: GWXKRMK
departmentNumber: 627
employeeType: Manager
homePhone: +1 206 483-6026
initials: E. M.
mobile: +1 510 679-9847
pager: +1 408 514-7670
manager: cn=Fouad Brydon
secretary: cn=Lebbie Mulroney
roomNumber: 2793

dn: cn=Dominga Whitehurst, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dominga Whitehurst
sn: Whitehurst
description: This is Dominga Whitehurst's description
facsimileTelephoneNumber: +1 213 256-8990
l: Mountain View
ou: Management
postalAddress: example$Management$Dept # 58
telephoneNumber: +1 408 806-4907
title: Master Management Architect
userPassword: tsruhetihW
uid: Dominga_Whitehurst
givenName: Dominga
mail: Dominga_Whitehurst@example.com
carLicense: Z3CGDJZ
departmentNumber: 9137
employeeType: Contract
homePhone: +1 71 699-1081
initials: D. W.
mobile: +1 71 588-2210
pager: +1 415 207-4707
manager: cn=Ashlie Constantinescu
secretary: cn=Catlee Purohit
roomNumber: 5411

dn: cn=Lusa Barsch, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lusa Barsch
sn: Barsch
description: This is Lusa Barsch's description
facsimileTelephoneNumber: +1 303 992-6949
l: Emeryville
ou: Product Testing
postalAddress: example$Product Testing$Dept # 545
telephoneNumber: +1 303 543-7858
title: Supreme Product Testing Artist
userPassword: hcsraBasuL
uid: Lusa_Barsch
givenName: Lusa
mail: Lusa_Barsch@example.com
carLicense: AGDA0GA
departmentNumber: 6239
employeeType: Manager
homePhone: +1 510 840-8962
initials: L. B.
mobile: +1 415 261-5608
pager: +1 415 252-7480
manager: cn=Pic Silwer
secretary: cn=Shlomo Klein
roomNumber: 70

dn: cn=Vilas Pastorek, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Vilas Pastorek
sn: Pastorek
description: This is Vilas Pastorek's description
facsimileTelephoneNumber: +1 818 644-9466
l: Alameda
ou: Human Resources
postalAddress: example$Human Resources$Dept # 542
telephoneNumber: +1 408 817-3212
title: Elite Human Resources Admin
userPassword: kerotsaPsa
uid: Vilas_Pastorek
givenName: Vilas
mail: Vilas_Pastorek@example.com
carLicense: NYOHGI6
departmentNumber: 2558
employeeType: Temp
homePhone: +1 71 218-9597
initials: V. P.
mobile: +1 71 644-1561
pager: +1 818 618-9594
manager: cn=Elke Schultz
secretary: cn=Sherryl Brys
roomNumber: 5677

dn: cn=Randene O'Toole, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Randene O'Toole
sn: O'Toole
description: This is Randene O'Toole's description
facsimileTelephoneNumber: +1 303 409-7211
l: Armonk
ou: Product Testing
postalAddress: example$Product Testing$Dept # 582
telephoneNumber: +1 206 690-2493
title: Chief Product Testing Assistant
userPassword: elooT'Oene
uid: Randene_O'Toole
givenName: Randene
mail: Randene_O'Toole@example.com
carLicense: 0OML3RV
departmentNumber: 5465
employeeType: Contract
homePhone: +1 415 321-7409
initials: R. O.
mobile: +1 408 116-2472
pager: +1 303 759-3332
manager: cn=Kai-Wai McNally
secretary: cn=Jun Flewelling
roomNumber: 1771

dn: cn=Korrie Sist, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Korrie Sist
sn: Sist
description: This is Korrie Sist's description
facsimileTelephoneNumber: +1 71 539-8066
l: Sunnyvale
ou: Accounting
postalAddress: example$Accounting$Dept # 873
telephoneNumber: +1 818 769-4028
title: Junior Accounting Vice President
userPassword: tsiSeirroK
uid: Korrie_Sist
givenName: Korrie
mail: Korrie_Sist@example.com
carLicense: UGBCAOW
departmentNumber: 7471
employeeType: Contract
homePhone: +1 818 597-4171
initials: K. S.
mobile: +1 71 724-1606
pager: +1 303 856-1901
manager: cn=Norah Narasimhan
secretary: cn=Helenelizabeth Rabzel
roomNumber: 5415

dn: cn=Dulci Armstrong, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dulci Armstrong
sn: Armstrong
description: This is Dulci Armstrong's description
facsimileTelephoneNumber: +1 303 511-4654
l: Menlo Park
ou: Human Resources
postalAddress: example$Human Resources$Dept # 272
telephoneNumber: +1 71 431-3711
title: Supreme Human Resources Yahoo
userPassword: gnortsmrAi
uid: Dulci_Armstrong
givenName: Dulci
mail: Dulci_Armstrong@example.com
carLicense: 1E2YUTW
departmentNumber: 323
employeeType: Temp
homePhone: +1 804 862-9466
initials: D. A.
mobile: +1 415 767-6314
pager: +1 818 623-4482
manager: cn=Ott Burleigh
secretary: cn=Brock Guillaume
roomNumber: 421

dn: cn=Gillan Ress, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gillan Ress
sn: Ress
description: This is Gillan Ress's description
facsimileTelephoneNumber: +1 818 798-5635
l: Sunnyvale
ou: Management
postalAddress: example$Management$Dept # 640
telephoneNumber: +1 415 414-9375
title: Elite Management Technician
userPassword: sseRnalliG
uid: Gillan_Ress
givenName: Gillan
mail: Gillan_Ress@example.com
carLicense: 0H8LH9Z
departmentNumber: 2077
employeeType: Contract
homePhone: +1 818 304-3376
initials: G. R.
mobile: +1 818 167-9570
pager: +1 303 338-7776
manager: cn=Violet Ainsworth
secretary: cn=Horst Schryburt
roomNumber: 4160

dn: cn=Conny Rufino, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Conny Rufino
sn: Rufino
description: This is Conny Rufino's description
facsimileTelephoneNumber: +1 303 248-4094
l: Redmond
ou: Management
postalAddress: example$Management$Dept # 833
telephoneNumber: +1 206 482-9073
title: Chief Management Engineer
userPassword: onifuRynno
uid: Conny_Rufino
givenName: Conny
mail: Conny_Rufino@example.com
carLicense: 4C1O7IN
departmentNumber: 9683
employeeType: Employee
homePhone: +1 818 497-4776
initials: C. R.
mobile: +1 206 217-3025
pager: +1 415 462-6384
manager: cn=Hetty Nevison
secretary: cn=Lari Stctest
roomNumber: 8179

dn: cn=Marleen FWPtools, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marleen FWPtools
sn: FWPtools
description: This is Marleen FWPtools's description
facsimileTelephoneNumber: +1 303 778-4309
l: Mountain View
ou: Product Development
postalAddress: example$Product Development$Dept # 247
telephoneNumber: +1 510 715-1783
title: Associate Product Development President
userPassword: slootPWFne
uid: Marleen_FWPtools
givenName: Marleen
mail: Marleen_FWPtools@example.com
carLicense: WUQ4LSG
departmentNumber: 3881
employeeType: Temp
homePhone: +1 408 196-4785
initials: M. F.
mobile: +1 206 877-2976
pager: +1 818 433-5574
manager: cn=Leshia Sommerdorf
secretary: cn=Gale  Borozny
roomNumber: 8246

dn: cn=Aviva Deslandes, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Aviva Deslandes
sn: Deslandes
description: This is Aviva Deslandes's description
facsimileTelephoneNumber: +1 206 995-9649
l: Armonk
ou: Planning
postalAddress: example$Planning$Dept # 582
telephoneNumber: +1 71 503-6775
title: Associate Planning Accountant
userPassword: sednalseDa
uid: Aviva_Deslandes
givenName: Aviva
mail: Aviva_Deslandes@example.com
carLicense: 6Q8FJHO
departmentNumber: 5225
employeeType: Employee
homePhone: +1 510 122-7820
initials: A. D.
mobile: +1 415 883-1763
pager: +1 804 146-9427
manager: cn=Nertie Kechichian
secretary: cn=Pauly Rudisill
roomNumber: 8041

dn: cn=Wileen Logarajah, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Wileen Logarajah
sn: Logarajah
description: This is Wileen Logarajah's description
facsimileTelephoneNumber: +1 818 323-6643
l: Menlo Park
ou: Peons
postalAddress: example$Peons$Dept # 559
telephoneNumber: +1 510 971-5342
title: Associate Peons Architect
userPassword: hajaragoLn
uid: Wileen_Logarajah
givenName: Wileen
mail: Wileen_Logarajah@example.com
carLicense: VO5KKO6
departmentNumber: 5171
employeeType: Contract
homePhone: +1 71 613-2879
initials: W. L.
mobile: +1 415 857-8255
pager: +1 71 403-9536
manager: cn=Brittan Maunu
secretary: cn=Jessamyn Gallegos
roomNumber: 4047

dn: cn=Emogene Florence, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Emogene Florence
sn: Florence
description: This is Emogene Florence's description
facsimileTelephoneNumber: +1 804 192-2290
l: Redwood Shores
ou: Product Development
postalAddress: example$Product Development$Dept # 804
telephoneNumber: +1 213 146-8522
title: Senior Product Development Madonna
userPassword: ecnerolFen
uid: Emogene_Florence
givenName: Emogene
mail: Emogene_Florence@example.com
carLicense: F3SWANH
departmentNumber: 9790
employeeType: Contract
homePhone: +1 303 421-7847
initials: E. F.
mobile: +1 206 833-7435
pager: +1 415 393-7829
manager: cn=Kaitlyn Przybycien
secretary: cn=Thea Kokoska
roomNumber: 2817

dn: cn=Emelia Mote, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Emelia Mote
sn: Mote
description: This is Emelia Mote's description
facsimileTelephoneNumber: +1 818 813-7269
l: Santa Clara
ou: Peons
postalAddress: example$Peons$Dept # 775
telephoneNumber: +1 71 746-8313
title: Chief Peons Evangelist
userPassword: etoMailemE
uid: Emelia_Mote
givenName: Emelia
mail: Emelia_Mote@example.com
carLicense: 7CSXD60
departmentNumber: 9064
employeeType: Manager
homePhone: +1 71 467-2178
initials: E. M.
mobile: +1 408 942-9846
pager: +1 804 135-5252
manager: cn=Ashok Tahir
secretary: cn=Dita Estey
roomNumber: 7255

dn: cn=Harrietta McGarry, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Harrietta McGarry
sn: McGarry
description: This is Harrietta McGarry's description
facsimileTelephoneNumber: +1 804 825-5556
l: Sunnyvale
ou: Product Development
postalAddress: example$Product Development$Dept # 685
telephoneNumber: +1 206 692-1375
title: Junior Product Development Architect
userPassword: yrraGcMatt
uid: Harrietta_McGarry
givenName: Harrietta
mail: Harrietta_McGarry@example.com
carLicense: UREW39H
departmentNumber: 9511
employeeType: Contract
homePhone: +1 206 567-8178
initials: H. M.
mobile: +1 71 751-8182
pager: +1 408 740-5933
manager: cn=Deepak Moen
secretary: cn=Hillary Szabo
roomNumber: 1564

dn: cn=Mardi Hosier, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mardi Hosier
sn: Hosier
description: This is Mardi Hosier's description
facsimileTelephoneNumber: +1 818 986-7690
l: Menlo Park
ou: Janitorial
postalAddress: example$Janitorial$Dept # 885
telephoneNumber: +1 510 231-5597
title: Master Janitorial Consultant
userPassword: reisoHidra
uid: Mardi_Hosier
givenName: Mardi
mail: Mardi_Hosier@example.com
carLicense: TG3MHCD
departmentNumber: 1504
employeeType: Manager
homePhone: +1 303 436-3978
initials: M. H.
mobile: +1 415 703-2374
pager: +1 510 715-9620
manager: cn=Glynis Readling
secretary: cn=Jastinder Sokolowski
roomNumber: 1897

dn: cn=Natasja Omura, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Natasja Omura
sn: Omura
description: This is Natasja Omura's description
facsimileTelephoneNumber: +1 213 583-5434
l: San Mateo
ou: Human Resources
postalAddress: example$Human Resources$Dept # 645
telephoneNumber: +1 818 691-2933
title: Supreme Human Resources Dictator
userPassword: arumOajsat
uid: Natasja_Omura
givenName: Natasja
mail: Natasja_Omura@example.com
carLicense: OQSGXN6
departmentNumber: 4997
employeeType: Temp
homePhone: +1 213 258-7791
initials: N. O.
mobile: +1 71 238-3159
pager: +1 213 603-7155
manager: cn=Lishe Da Gama
secretary: cn=Der-Chang Gopaul
roomNumber: 4402

dn: cn=Ciel Ghorashy, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ciel Ghorashy
sn: Ghorashy
description: This is Ciel Ghorashy's description
facsimileTelephoneNumber: +1 804 977-3660
l: Cupertino
ou: Product Testing
postalAddress: example$Product Testing$Dept # 276
telephoneNumber: +1 415 689-6453
title: Elite Product Testing Stooge
userPassword: yhsarohGle
uid: Ciel_Ghorashy
givenName: Ciel
mail: Ciel_Ghorashy@example.com
carLicense: OMJGBXM
departmentNumber: 9175
employeeType: Employee
homePhone: +1 213 862-8793
initials: C. G.
mobile: +1 804 902-1620
pager: +1 818 786-1258
manager: cn=Mersey Morden
secretary: cn=Hafeezah Fodell
roomNumber: 92

dn: cn=Derick McNitt, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Derick McNitt
sn: McNitt
description: This is Derick McNitt's description
facsimileTelephoneNumber: +1 818 113-9552
l: Menlo Park
ou: Product Development
postalAddress: example$Product Development$Dept # 463
telephoneNumber: +1 206 910-1467
title: Senior Product Development Engineer
userPassword: ttiNcMkcir
uid: Derick_McNitt
givenName: Derick
mail: Derick_McNitt@example.com
carLicense: 4PDV6J7
departmentNumber: 4317
employeeType: Contract
homePhone: +1 206 808-6633
initials: D. M.
mobile: +1 415 795-3783
pager: +1 206 414-3820
manager: cn=Vera Vanderhelm
secretary: cn=Remy Friedberg
roomNumber: 997

dn: cn=Hermann Hammel, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Hermann Hammel
sn: Hammel
description: This is Hermann Hammel's description
facsimileTelephoneNumber: +1 415 843-3404
l: Menlo Park
ou: Planning
postalAddress: example$Planning$Dept # 124
telephoneNumber: +1 818 874-9081
title: Senior Planning Vice President
userPassword: lemmaHnnam
uid: Hermann_Hammel
givenName: Hermann
mail: Hermann_Hammel@example.com
carLicense: 0ZWAXUF
departmentNumber: 7105
employeeType: Normal
homePhone: +1 510 218-3986
initials: H. H.
mobile: +1 818 385-3986
pager: +1 804 188-2368
manager: cn=Kien Deugo
secretary: cn=Izumi Berrisford
roomNumber: 1983

dn: cn=Weringh Bennefeld, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Weringh Bennefeld
sn: Bennefeld
description: This is Weringh Bennefeld's description
facsimileTelephoneNumber: +1 213 292-2935
l: Emeryville
ou: Peons
postalAddress: example$Peons$Dept # 10
telephoneNumber: +1 206 879-8616
title: Chief Peons Madonna
userPassword: dlefenneBh
uid: Weringh_Bennefeld
givenName: Weringh
mail: Weringh_Bennefeld@example.com
carLicense: TZCORI9
departmentNumber: 356
employeeType: Normal
homePhone: +1 206 290-6512
initials: W. B.
mobile: +1 71 378-5716
pager: +1 818 629-7598
manager: cn=Eveleen Coppedge
secretary: cn=Hulda Giekes
roomNumber: 8917

dn: cn=Jackson Finley, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jackson Finley
sn: Finley
description: This is Jackson Finley's description
facsimileTelephoneNumber: +1 804 681-9655
l: Milpitas
ou: Product Development
postalAddress: example$Product Development$Dept # 980
telephoneNumber: +1 213 293-1895
title: Supreme Product Development Yahoo
userPassword: yelniFnosk
uid: Jackson_Finley
givenName: Jackson
mail: Jackson_Finley@example.com
carLicense: 0K2SXZO
departmentNumber: 6521
employeeType: Temp
homePhone: +1 213 173-2127
initials: J. F.
mobile: +1 213 798-3466
pager: +1 415 648-1843
manager: cn=Ashley Moree
secretary: cn=Ekaterina Adhem
roomNumber: 3539

dn: cn=Fianna Herman, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Fianna Herman
sn: Herman
description: This is Fianna Herman's description
facsimileTelephoneNumber: +1 206 356-2173
l: Redmond
ou: Planning
postalAddress: example$Planning$Dept # 656
telephoneNumber: +1 408 107-4282
title: Senior Planning Dictator
userPassword: namreHanna
uid: Fianna_Herman
givenName: Fianna
mail: Fianna_Herman@example.com
carLicense: YGE8PG8
departmentNumber: 6047
employeeType: Manager
homePhone: +1 415 774-2912
initials: F. H.
mobile: +1 510 960-4071
pager: +1 408 495-4476
manager: cn=Paulina Toth
secretary: cn=Cherlyn Diradmin
roomNumber: 9571

dn: cn=Tad Bolduc, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tad Bolduc
sn: Bolduc
description: This is Tad Bolduc's description
facsimileTelephoneNumber: +1 408 468-5854
l: Sunnyvale
ou: Payroll
postalAddress: example$Payroll$Dept # 385
telephoneNumber: +1 510 466-1350
title: Elite Payroll Janitor
userPassword: cudloBdaT
uid: Tad_Bolduc
givenName: Tad
mail: Tad_Bolduc@example.com
carLicense: PVDSMSZ
departmentNumber: 7001
employeeType: Contract
homePhone: +1 415 532-9111
initials: T. B.
mobile: +1 206 419-9095
pager: +1 510 552-3827
manager: cn=Celisse McLawhon
secretary: cn=Ranson Robitaille
roomNumber: 6774

dn: cn=Mustafa Goodfellow, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mustafa Goodfellow
sn: Goodfellow
description: This is Mustafa Goodfellow's description
facsimileTelephoneNumber: +1 804 747-3201
l: Mountain View
ou: Product Testing
postalAddress: example$Product Testing$Dept # 832
telephoneNumber: +1 206 759-5792
title: Elite Product Testing Visionary
userPassword: wollefdooG
uid: Mustafa_Goodfellow
givenName: Mustafa
mail: Mustafa_Goodfellow@example.com
carLicense: T7H8WEV
departmentNumber: 1733
employeeType: Temp
homePhone: +1 213 707-6191
initials: M. G.
mobile: +1 206 708-2540
pager: +1 804 267-1608
manager: cn=Georgianne Keehn
secretary: cn=Jianli Feil
roomNumber: 5129

dn: cn=Davida Waloff, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Davida Waloff
sn: Waloff
description: This is Davida Waloff's description
facsimileTelephoneNumber: +1 804 575-1950
l: Cambridge
ou: Planning
postalAddress: example$Planning$Dept # 226
telephoneNumber: +1 71 574-4002
title: Elite Planning Czar
userPassword: ffolaWadiv
uid: Davida_Waloff
givenName: Davida
mail: Davida_Waloff@example.com
carLicense: 72PCPKG
departmentNumber: 935
employeeType: Temp
homePhone: +1 804 251-5967
initials: D. W.
mobile: +1 510 857-7901
pager: +1 510 205-5810
manager: cn=Naohiko Gostanian
secretary: cn=Anje Borodajluk
roomNumber: 1127

dn: cn=Gupta Dantu, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gupta Dantu
sn: Dantu
description: This is Gupta Dantu's description
facsimileTelephoneNumber: +1 206 711-1440
l: Cupertino
ou: Product Development
postalAddress: example$Product Development$Dept # 169
telephoneNumber: +1 415 967-4634
title: Master Product Development Director
userPassword: utnaDatpuG
uid: Gupta_Dantu
givenName: Gupta
mail: Gupta_Dantu@example.com
carLicense: A2GSAY5
departmentNumber: 8255
employeeType: Temp
homePhone: +1 510 606-7627
initials: G. D.
mobile: +1 510 952-7193
pager: +1 213 121-6362
manager: cn=Berta Admin-mtv
secretary: cn=Gladi IRCMTL
roomNumber: 3448

dn: cn=Benefits Mukherjee, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Benefits Mukherjee
sn: Mukherjee
description: This is Benefits Mukherjee's description
facsimileTelephoneNumber: +1 213 968-8685
l: Fremont
ou: Management
postalAddress: example$Management$Dept # 793
telephoneNumber: +1 71 161-9882
title: Supreme Management Technician
userPassword: eejrehkuMs
uid: Benefits_Mukherjee
givenName: Benefits
mail: Benefits_Mukherjee@example.com
carLicense: OVBO65A
departmentNumber: 6506
employeeType: Manager
homePhone: +1 213 925-2365
initials: B. M.
mobile: +1 408 801-9786
pager: +1 415 305-8600
manager: cn=Mala Bunzey
secretary: cn=Anabelle Boyce
roomNumber: 5054

dn: cn=Omar Dokken, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Omar Dokken
sn: Dokken
description: This is Omar Dokken's description
facsimileTelephoneNumber: +1 408 499-9111
l: Redmond
ou: Peons
postalAddress: example$Peons$Dept # 540
telephoneNumber: +1 408 352-4507
title: Senior Peons Assistant
userPassword: nekkoDramO
uid: Omar_Dokken
givenName: Omar
mail: Omar_Dokken@example.com
carLicense: I1UEWPP
departmentNumber: 6095
employeeType: Contract
homePhone: +1 818 121-6046
initials: O. D.
mobile: +1 71 392-7484
pager: +1 303 306-4750
manager: cn=Kjell Groulx
secretary: cn=Faustine Leavitt
roomNumber: 3423

dn: cn=Phillis Ganguly, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Phillis Ganguly
sn: Ganguly
description: This is Phillis Ganguly's description
facsimileTelephoneNumber: +1 408 352-5183
l: Menlo Park
ou: Administrative
postalAddress: example$Administrative$Dept # 988
telephoneNumber: +1 303 456-1127
title: Supreme Administrative Writer
userPassword: ylugnaGsil
uid: Phillis_Ganguly
givenName: Phillis
mail: Phillis_Ganguly@example.com
carLicense: WGF2X9I
departmentNumber: 1524
employeeType: Contract
homePhone: +1 415 632-7072
initials: P. G.
mobile: +1 415 963-2037
pager: +1 804 409-2986
manager: cn=Tova Cuddy
secretary: cn=Willy Zwick
roomNumber: 1992

dn: cn=Vahe Schlachter, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Vahe Schlachter
sn: Schlachter
description: This is Vahe Schlachter's description
facsimileTelephoneNumber: +1 71 205-6557
l: Redmond
ou: Administrative
postalAddress: example$Administrative$Dept # 491
telephoneNumber: +1 303 978-4355
title: Chief Administrative Yahoo
userPassword: rethcalhcS
uid: Vahe_Schlachter
givenName: Vahe
mail: Vahe_Schlachter@example.com
carLicense: BO15FKQ
departmentNumber: 3671
employeeType: Contract
homePhone: +1 213 835-8797
initials: V. S.
mobile: +1 303 337-1436
pager: +1 818 603-7610
manager: cn=Melodie Beswick
secretary: cn=Mat Madigan
roomNumber: 4002

dn: cn=Ede Abrahim, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ede Abrahim
sn: Abrahim
description: This is Ede Abrahim's description
facsimileTelephoneNumber: +1 804 968-1719
l: Cambridge
ou: Janitorial
postalAddress: example$Janitorial$Dept # 719
telephoneNumber: +1 415 845-4115
title: Senior Janitorial Developer
userPassword: miharbAedE
uid: Ede_Abrahim
givenName: Ede
mail: Ede_Abrahim@example.com
carLicense: S4QCVN9
departmentNumber: 8351
employeeType: Manager
homePhone: +1 818 833-7793
initials: E. A.
mobile: +1 303 314-5571
pager: +1 408 164-1519
manager: cn=Lilah Cuffle
secretary: cn=Verina Letsome
roomNumber: 3620

dn: cn=Liesbeth Burkepile, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Liesbeth Burkepile
sn: Burkepile
description: This is Liesbeth Burkepile's description
facsimileTelephoneNumber: +1 303 203-9722
l: Palo Alto
ou: Planning
postalAddress: example$Planning$Dept # 413
telephoneNumber: +1 303 106-5961
title: Junior Planning Accountant
userPassword: elipekruBh
uid: Liesbeth_Burkepile
givenName: Liesbeth
mail: Liesbeth_Burkepile@example.com
carLicense: AW7OL9W
departmentNumber: 5512
employeeType: Normal
homePhone: +1 415 529-3794
initials: L. B.
mobile: +1 510 278-1822
pager: +1 71 399-2751
manager: cn=Nert Sponagle
secretary: cn=Bela Moledina
roomNumber: 185

dn: cn=Karan Betcher, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Karan Betcher
sn: Betcher
description: This is Karan Betcher's description
facsimileTelephoneNumber: +1 818 977-5803
l: San Mateo
ou: Management
postalAddress: example$Management$Dept # 91
telephoneNumber: +1 206 348-7764
title: Associate Management Writer
userPassword: rehcteBnar
uid: Karan_Betcher
givenName: Karan
mail: Karan_Betcher@example.com
carLicense: BVFA25X
departmentNumber: 3472
employeeType: Normal
homePhone: +1 206 467-2902
initials: K. B.
mobile: +1 415 885-7956
pager: +1 510 192-2376
manager: cn=Marjan Jaworsky
secretary: cn=Renelle Letchworth
roomNumber: 4389

dn: cn=Anup Schlobohm, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Anup Schlobohm
sn: Schlobohm
description: This is Anup Schlobohm's description
facsimileTelephoneNumber: +1 206 332-8863
l: Mountain View
ou: Product Testing
postalAddress: example$Product Testing$Dept # 675
telephoneNumber: +1 71 552-9836
title: Chief Product Testing Janitor
userPassword: mhobolhcSp
uid: Anup_Schlobohm
givenName: Anup
mail: Anup_Schlobohm@example.com
carLicense: 0FB1HB8
departmentNumber: 2304
employeeType: Manager
homePhone: +1 303 565-9859
initials: A. S.
mobile: +1 415 473-6448
pager: +1 804 368-3192
manager: cn=Fabien Lafleur
secretary: cn=Tish Sova
roomNumber: 2336

dn: cn=Elsi Frankenberger, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Elsi Frankenberger
sn: Frankenberger
description: This is Elsi Frankenberger's description
facsimileTelephoneNumber: +1 213 178-9870
l: Sunnyvale
ou: Management
postalAddress: example$Management$Dept # 446
telephoneNumber: +1 303 876-3308
title: Master Management President
userPassword: regrebnekn
uid: Elsi_Frankenberger
givenName: Elsi
mail: Elsi_Frankenberger@example.com
carLicense: LV0Z44B
departmentNumber: 7834
employeeType: Temp
homePhone: +1 71 774-4353
initials: E. F.
mobile: +1 213 306-9166
pager: +1 510 399-4748
manager: cn=Adorne Bennison
secretary: cn=Pauly Philion
roomNumber: 3478

dn: cn=Nasser Barcza, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Nasser Barcza
sn: Barcza
description: This is Nasser Barcza's description
facsimileTelephoneNumber: +1 213 546-3662
l: San Mateo
ou: Management
postalAddress: example$Management$Dept # 305
telephoneNumber: +1 408 287-2998
title: Supreme Management Vice President
userPassword: azcraBress
uid: Nasser_Barcza
givenName: Nasser
mail: Nasser_Barcza@example.com
carLicense: 953UC3Y
departmentNumber: 8332
employeeType: Contract
homePhone: +1 213 820-3768
initials: N. B.
mobile: +1 415 123-1094
pager: +1 818 667-4905
manager: cn=Malanie Svo
secretary: cn=Phan Dumas
roomNumber: 4887

dn: cn=Afton Desharnais, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Afton Desharnais
sn: Desharnais
description: This is Afton Desharnais's description
facsimileTelephoneNumber: +1 804 705-5987
l: San Mateo
ou: Human Resources
postalAddress: example$Human Resources$Dept # 464
telephoneNumber: +1 415 512-4395
title: Junior Human Resources Architect
userPassword: sianrahseD
uid: Afton_Desharnais
givenName: Afton
mail: Afton_Desharnais@example.com
carLicense: KB99IQT
departmentNumber: 9595
employeeType: Employee
homePhone: +1 804 305-3482
initials: A. D.
mobile: +1 213 477-3182
pager: +1 415 429-3345
manager: cn=Saeed Derosa
secretary: cn=Tory Dionne
roomNumber: 2322

dn: cn=Regina Pizzanelli, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Regina Pizzanelli
sn: Pizzanelli
description: This is Regina Pizzanelli's description
facsimileTelephoneNumber: +1 408 259-2282
l: San Jose
ou: Management
postalAddress: example$Management$Dept # 855
telephoneNumber: +1 415 843-6135
title: Elite Management Assistant
userPassword: illenazziP
uid: Regina_Pizzanelli
givenName: Regina
mail: Regina_Pizzanelli@example.com
carLicense: MVNK7DJ
departmentNumber: 6548
employeeType: Normal
homePhone: +1 303 719-3887
initials: R. P.
mobile: +1 415 853-7286
pager: +1 303 380-5628
manager: cn=Amrik Saulnier
secretary: cn=Tsuyoshi Brisby
roomNumber: 737

dn: cn=Krissy Ottco, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Krissy Ottco
sn: Ottco
description: This is Krissy Ottco's description
facsimileTelephoneNumber: +1 206 522-8708
l: Milpitas
ou: Product Development
postalAddress: example$Product Development$Dept # 335
telephoneNumber: +1 804 123-3823
title: Elite Product Development Visionary
userPassword: octtOyssir
uid: Krissy_Ottco
givenName: Krissy
mail: Krissy_Ottco@example.com
carLicense: EHTC6B4
departmentNumber: 3193
employeeType: Temp
homePhone: +1 415 606-5286
initials: K. O.
mobile: +1 71 191-1403
pager: +1 510 612-1380
manager: cn=Hedi Cotuna
secretary: cn=Rona Loggins
roomNumber: 6504

dn: cn=Yogi Maliepaard, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Yogi Maliepaard
sn: Maliepaard
description: This is Yogi Maliepaard's description
facsimileTelephoneNumber: +1 206 427-2189
l: San Jose
ou: Product Development
postalAddress: example$Product Development$Dept # 448
telephoneNumber: +1 206 502-5949
title: Supreme Product Development Stooge
userPassword: draapeilaM
uid: Yogi_Maliepaard
givenName: Yogi
mail: Yogi_Maliepaard@example.com
carLicense: VX7BJUU
departmentNumber: 5968
employeeType: Normal
homePhone: +1 804 945-2170
initials: Y. M.
mobile: +1 408 502-5633
pager: +1 415 610-5572
manager: cn=Dre Yuhanna
secretary: cn=Marthe Stodart
roomNumber: 2076

dn: cn=Devin Colwell, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Devin Colwell
sn: Colwell
description: This is Devin Colwell's description
facsimileTelephoneNumber: +1 510 857-4693
l: Cambridge
ou: Planning
postalAddress: example$Planning$Dept # 252
telephoneNumber: +1 818 459-9361
title: Chief Planning Engineer
userPassword: llewloCniv
uid: Devin_Colwell
givenName: Devin
mail: Devin_Colwell@example.com
carLicense: BG1NMSO
departmentNumber: 1324
employeeType: Manager
homePhone: +1 213 316-4005
initials: D. C.
mobile: +1 213 356-6732
pager: +1 818 366-9678
manager: cn=Modestine Windsor
secretary: cn=Doretta LeTarte
roomNumber: 8706

dn: cn=Alan Montelli, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Alan Montelli
sn: Montelli
description: This is Alan Montelli's description
facsimileTelephoneNumber: +1 804 683-1981
l: Cupertino
ou: Management
postalAddress: example$Management$Dept # 769
telephoneNumber: +1 303 334-8615
title: Senior Management Stooge
userPassword: illetnoMna
uid: Alan_Montelli
givenName: Alan
mail: Alan_Montelli@example.com
carLicense: AJJKB89
departmentNumber: 9248
employeeType: Contract
homePhone: +1 408 950-5244
initials: A. M.
mobile: +1 213 827-4891
pager: +1 415 769-5696
manager: cn=Murielle Mansbridge
secretary: cn=Katrine Cwirzen
roomNumber: 751

dn: cn=Yolanda Schrang, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Yolanda Schrang
sn: Schrang
description: This is Yolanda Schrang's description
facsimileTelephoneNumber: +1 303 747-5524
l: Fremont
ou: Payroll
postalAddress: example$Payroll$Dept # 108
telephoneNumber: +1 415 127-7924
title: Master Payroll Writer
userPassword: gnarhcSadn
uid: Yolanda_Schrang
givenName: Yolanda
mail: Yolanda_Schrang@example.com
carLicense: L6XGOUU
departmentNumber: 3381
employeeType: Manager
homePhone: +1 804 245-2347
initials: Y. S.
mobile: +1 206 296-1523
pager: +1 415 350-4164
manager: cn=Qainfo Reporting
secretary: cn=Sharri McQuarrie
roomNumber: 3483

dn: cn=Burt Bigley, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Burt Bigley
sn: Bigley
description: This is Burt Bigley's description
facsimileTelephoneNumber: +1 213 394-9377
l: Milpitas
ou: Administrative
postalAddress: example$Administrative$Dept # 566
telephoneNumber: +1 415 122-3561
title: Supreme Administrative Mascot
userPassword: yelgiBtruB
uid: Burt_Bigley
givenName: Burt
mail: Burt_Bigley@example.com
carLicense: G7U8J5R
departmentNumber: 8712
employeeType: Manager
homePhone: +1 206 649-5420
initials: B. B.
mobile: +1 206 352-2127
pager: +1 818 113-1971
manager: cn=Evey Keates
secretary: cn=Sophey Broberg
roomNumber: 5458

dn: cn=Katja Remillard, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Katja Remillard
sn: Remillard
description: This is Katja Remillard's description
facsimileTelephoneNumber: +1 408 840-3448
l: Sunnyvale
ou: Human Resources
postalAddress: example$Human Resources$Dept # 157
telephoneNumber: +1 303 728-4601
title: Senior Human Resources Czar
userPassword: drallimeRa
uid: Katja_Remillard
givenName: Katja
mail: Katja_Remillard@example.com
carLicense: F3J9QRE
departmentNumber: 8810
employeeType: Employee
homePhone: +1 408 119-3253
initials: K. R.
mobile: +1 804 464-8689
pager: +1 303 891-1853
manager: cn=Umeko Hagerty
secretary: cn=Farid Hemens-Davis
roomNumber: 3521

dn: cn=Michaela Creighton, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Michaela Creighton
sn: Creighton
description: This is Michaela Creighton's description
facsimileTelephoneNumber: +1 206 466-3460
l: Milpitas
ou: Administrative
postalAddress: example$Administrative$Dept # 262
telephoneNumber: +1 510 639-9600
title: Senior Administrative Figurehead
userPassword: nothgierCa
uid: Michaela_Creighton
givenName: Michaela
mail: Michaela_Creighton@example.com
carLicense: VBGO6JL
departmentNumber: 5642
employeeType: Manager
homePhone: +1 71 885-3371
initials: M. C.
mobile: +1 804 164-6067
pager: +1 804 453-1117
manager: cn=Rubina Maguire
secretary: cn=Roberta Dennen
roomNumber: 2566

dn: cn=Lora-Lee Bowser, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lora-Lee Bowser
sn: Bowser
description: This is Lora-Lee Bowser's description
facsimileTelephoneNumber: +1 510 678-3239
l: Cupertino
ou: Product Development
postalAddress: example$Product Development$Dept # 323
telephoneNumber: +1 408 614-8676
title: Associate Product Development Janitor
userPassword: reswoBeeL-
uid: Lora-Lee_Bowser
givenName: Lora-Lee
mail: Lora-Lee_Bowser@example.com
carLicense: KO8NPE3
departmentNumber: 422
employeeType: Contract
homePhone: +1 818 558-9697
initials: L. B.
mobile: +1 303 206-9542
pager: +1 303 297-2416
manager: cn=Tilak Skalski
secretary: cn=Lisette Chotkowski
roomNumber: 9474

dn: cn=Utpala Chaar, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Utpala Chaar
sn: Chaar
description: This is Utpala Chaar's description
facsimileTelephoneNumber: +1 818 643-8405
l: Orem
ou: Janitorial
postalAddress: example$Janitorial$Dept # 1
telephoneNumber: +1 303 815-7238
title: Elite Janitorial Czar
userPassword: raahCalapt
uid: Utpala_Chaar
givenName: Utpala
mail: Utpala_Chaar@example.com
carLicense: I8U4SX2
departmentNumber: 9051
employeeType: Manager
homePhone: +1 804 492-7075
initials: U. C.
mobile: +1 415 492-8050
pager: +1 408 629-5000
manager: cn=Thanh-Tinh Tsao
secretary: cn=Blancha Tebbe
roomNumber: 6257

dn: cn=Dael Cowling, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dael Cowling
sn: Cowling
description: This is Dael Cowling's description
facsimileTelephoneNumber: +1 303 168-3958
l: Redwood Shores
ou: Product Testing
postalAddress: example$Product Testing$Dept # 834
telephoneNumber: +1 303 695-8623
title: Supreme Product Testing Dictator
userPassword: gnilwoClea
uid: Dael_Cowling
givenName: Dael
mail: Dael_Cowling@example.com
carLicense: MJQ89G9
departmentNumber: 4843
employeeType: Normal
homePhone: +1 71 392-3443
initials: D. C.
mobile: +1 206 634-1798
pager: +1 303 658-3373
manager: cn=Richelle Alford
secretary: cn=Rory Sldisk
roomNumber: 374

dn: cn=Jobi Angvall, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jobi Angvall
sn: Angvall
description: This is Jobi Angvall's description
facsimileTelephoneNumber: +1 510 298-7779
l: San Jose
ou: Management
postalAddress: example$Management$Dept # 226
telephoneNumber: +1 213 890-1992
title: Senior Management Artist
userPassword: llavgnAibo
uid: Jobi_Angvall
givenName: Jobi
mail: Jobi_Angvall@example.com
carLicense: F4NXKW4
departmentNumber: 5133
employeeType: Temp
homePhone: +1 408 608-4956
initials: J. A.
mobile: +1 213 457-5573
pager: +1 804 294-9687
manager: cn=Blondy Skelly
secretary: cn=Marylinda Epps
roomNumber: 5391

dn: cn=Ginny Hewett, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ginny Hewett
sn: Hewett
description: This is Ginny Hewett's description
facsimileTelephoneNumber: +1 303 259-9993
l: Redwood Shores
ou: Administrative
postalAddress: example$Administrative$Dept # 651
telephoneNumber: +1 213 178-6055
title: Associate Administrative Evangelist
userPassword: tteweHynni
uid: Ginny_Hewett
givenName: Ginny
mail: Ginny_Hewett@example.com
carLicense: Y0F9PB0
departmentNumber: 8553
employeeType: Contract
homePhone: +1 213 794-5847
initials: G. H.
mobile: +1 303 109-4919
pager: +1 818 642-1734
manager: cn=Carmel Milston
secretary: cn=Ariadne Belrango
roomNumber: 7434

dn: cn=Nabil Kirn, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Nabil Kirn
sn: Kirn
description: This is Nabil Kirn's description
facsimileTelephoneNumber: +1 213 412-5415
l: San Mateo
ou: Administrative
postalAddress: example$Administrative$Dept # 87
telephoneNumber: +1 213 248-7293
title: Elite Administrative Fellow
userPassword: nriKlibaN
uid: Nabil_Kirn
givenName: Nabil
mail: Nabil_Kirn@example.com
carLicense: CWUAKSZ
departmentNumber: 5881
employeeType: Employee
homePhone: +1 71 744-6287
initials: N. K.
mobile: +1 415 473-2310
pager: +1 510 538-2296
manager: cn=Jewel McIntyre
secretary: cn=Karly Purohit
roomNumber: 3033

dn: cn=JoAnne Hopcroft, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: JoAnne Hopcroft
sn: Hopcroft
description: This is JoAnne Hopcroft's description
facsimileTelephoneNumber: +1 303 463-5145
l: Milpitas
ou: Janitorial
postalAddress: example$Janitorial$Dept # 692
telephoneNumber: +1 206 948-9516
title: Supreme Janitorial Dictator
userPassword: tforcpoHen
uid: JoAnne_Hopcroft
givenName: JoAnne
mail: JoAnne_Hopcroft@example.com
carLicense: Z8YN8PI
departmentNumber: 2258
employeeType: Contract
homePhone: +1 303 623-3589
initials: J. H.
mobile: +1 510 549-9016
pager: +1 206 359-3508
manager: cn=Helmuth Geary
secretary: cn=Erik Whitlock
roomNumber: 9923

dn: cn=Mila Baumberg, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mila Baumberg
sn: Baumberg
description: This is Mila Baumberg's description
facsimileTelephoneNumber: +1 818 197-9158
l: Cupertino
ou: Planning
postalAddress: example$Planning$Dept # 953
telephoneNumber: +1 818 161-5343
title: Master Planning Madonna
userPassword: grebmuaBal
uid: Mila_Baumberg
givenName: Mila
mail: Mila_Baumberg@example.com
carLicense: 7G67VHN
departmentNumber: 3826
employeeType: Contract
homePhone: +1 510 237-2168
initials: M. B.
mobile: +1 213 799-8421
pager: +1 415 857-5821
manager: cn=Cyb Pillsworth
secretary: cn=Morgen Beaudin
roomNumber: 3522

dn: cn=Myranda Eisenhart, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Myranda Eisenhart
sn: Eisenhart
description: This is Myranda Eisenhart's description
facsimileTelephoneNumber: +1 510 527-9770
l: San Jose
ou: Administrative
postalAddress: example$Administrative$Dept # 42
telephoneNumber: +1 415 449-7845
title: Elite Administrative Sales Rep
userPassword: trahnesiEa
uid: Myranda_Eisenhart
givenName: Myranda
mail: Myranda_Eisenhart@example.com
carLicense: RCMB6TM
departmentNumber: 7331
employeeType: Contract
homePhone: +1 71 191-8459
initials: M. E.
mobile: +1 303 520-1257
pager: +1 213 887-6597
manager: cn=Merridie Brent
secretary: cn=Shigeru Marson
roomNumber: 6049

dn: cn=Daphne Allahyari, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Daphne Allahyari
sn: Allahyari
description: This is Daphne Allahyari's description
facsimileTelephoneNumber: +1 818 642-6904
l: Cambridge
ou: Product Testing
postalAddress: example$Product Testing$Dept # 479
telephoneNumber: +1 510 656-7442
title: Supreme Product Testing Consultant
userPassword: irayhallAe
uid: Daphne_Allahyari
givenName: Daphne
mail: Daphne_Allahyari@example.com
carLicense: 08O0BC9
departmentNumber: 8086
employeeType: Employee
homePhone: +1 818 490-2262
initials: D. A.
mobile: +1 206 904-3960
pager: +1 206 732-8246
manager: cn=Raoul Fricker
secretary: cn=J-Francois Magnusson
roomNumber: 5163

dn: cn=Keslie Prystie, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Keslie Prystie
sn: Prystie
description: This is Keslie Prystie's description
facsimileTelephoneNumber: +1 213 905-2687
l: Armonk
ou: Planning
postalAddress: example$Planning$Dept # 941
telephoneNumber: +1 408 293-6382
title: Senior Planning Madonna
userPassword: eitsyrPeil
uid: Keslie_Prystie
givenName: Keslie
mail: Keslie_Prystie@example.com
carLicense: ZE6AZMJ
departmentNumber: 9532
employeeType: Manager
homePhone: +1 818 400-2965
initials: K. P.
mobile: +1 206 535-5526
pager: +1 818 235-7279
manager: cn=Agnese Greenstreet
secretary: cn=Robbi Kessel
roomNumber: 3590

dn: cn=Ramakant Wolowidnyk, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ramakant Wolowidnyk
sn: Wolowidnyk
description: This is Ramakant Wolowidnyk's description
facsimileTelephoneNumber: +1 303 605-7657
l: San Francisco
ou: Planning
postalAddress: example$Planning$Dept # 374
telephoneNumber: +1 206 798-6334
title: Supreme Planning Writer
userPassword: kyndiwoloW
uid: Ramakant_Wolowidnyk
givenName: Ramakant
mail: Ramakant_Wolowidnyk@example.com
carLicense: 3NFHPK7
departmentNumber: 3756
employeeType: Normal
homePhone: +1 408 715-6209
initials: R. W.
mobile: +1 510 545-9954
pager: +1 213 721-2968
manager: cn=Jelene Kaid
secretary: cn=Marj Gebhart
roomNumber: 7771

dn: cn=Suvanee Shireman, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Suvanee Shireman
sn: Shireman
description: This is Suvanee Shireman's description
facsimileTelephoneNumber: +1 303 662-2889
l: Santa Clara
ou: Management
postalAddress: example$Management$Dept # 366
telephoneNumber: +1 415 884-9426
title: Senior Management Architect
userPassword: namerihSee
uid: Suvanee_Shireman
givenName: Suvanee
mail: Suvanee_Shireman@example.com
carLicense: 6QA5L94
departmentNumber: 885
employeeType: Normal
homePhone: +1 206 120-8373
initials: S. S.
mobile: +1 804 668-8228
pager: +1 510 356-7450
manager: cn=Gladys Armstrong
secretary: cn=Tilly Loyd
roomNumber: 2692

dn: cn=Ardavan Bascombe, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ardavan Bascombe
sn: Bascombe
description: This is Ardavan Bascombe's description
facsimileTelephoneNumber: +1 303 549-9122
l: Santa Clara
ou: Product Development
postalAddress: example$Product Development$Dept # 661
telephoneNumber: +1 71 654-2263
title: Elite Product Development Admin
userPassword: ebmocsaBna
uid: Ardavan_Bascombe
givenName: Ardavan
mail: Ardavan_Bascombe@example.com
carLicense: 4C8XBAH
departmentNumber: 914
employeeType: Manager
homePhone: +1 804 628-2697
initials: A. B.
mobile: +1 408 397-7309
pager: +1 510 960-7193
manager: cn=Beulah Dagg
secretary: cn=Aurea Hobesh
roomNumber: 7151

dn: cn=Ginger Hedke, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ginger Hedke
sn: Hedke
description: This is Ginger Hedke's description
facsimileTelephoneNumber: +1 415 810-7678
l: Redwood Shores
ou: Payroll
postalAddress: example$Payroll$Dept # 237
telephoneNumber: +1 206 419-7010
title: Chief Payroll Assistant
userPassword: ekdeHregni
uid: Ginger_Hedke
givenName: Ginger
mail: Ginger_Hedke@example.com
carLicense: 61VAMPJ
departmentNumber: 4251
employeeType: Manager
homePhone: +1 71 378-8773
initials: G. H.
mobile: +1 206 555-7152
pager: +1 510 725-2039
manager: cn=Alika Gallagher
secretary: cn=Elsey Syed
roomNumber: 6007

dn: cn=Kunie Guilfoyle, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kunie Guilfoyle
sn: Guilfoyle
description: This is Kunie Guilfoyle's description
facsimileTelephoneNumber: +1 408 199-6163
l: Redwood Shores
ou: Product Testing
postalAddress: example$Product Testing$Dept # 139
telephoneNumber: +1 804 600-9134
title: Master Product Testing Admin
userPassword: elyofliuGe
uid: Kunie_Guilfoyle
givenName: Kunie
mail: Kunie_Guilfoyle@example.com
carLicense: A26Q6FO
departmentNumber: 9757
employeeType: Normal
homePhone: +1 213 266-1884
initials: K. G.
mobile: +1 408 797-8829
pager: +1 804 823-4894
manager: cn=Massoud Jeffries
secretary: cn=Damita Folwell
roomNumber: 6966

dn: cn=Martino Beauvais, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Martino Beauvais
sn: Beauvais
description: This is Martino Beauvais's description
facsimileTelephoneNumber: +1 303 974-4718
l: Mountain View
ou: Payroll
postalAddress: example$Payroll$Dept # 639
telephoneNumber: +1 303 579-5047
title: Supreme Payroll Sales Rep
userPassword: siavuaeBon
uid: Martino_Beauvais
givenName: Martino
mail: Martino_Beauvais@example.com
carLicense: 4CNZX95
departmentNumber: 516
employeeType: Normal
homePhone: +1 303 890-6690
initials: M. B.
mobile: +1 71 829-8473
pager: +1 71 991-2362
manager: cn=Wojciech Aidarous
secretary: cn=Karlene Akens
roomNumber: 7462

dn: cn=Dix Krienke, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dix Krienke
sn: Krienke
description: This is Dix Krienke's description
facsimileTelephoneNumber: +1 408 795-1449
l: Cambridge
ou: Product Development
postalAddress: example$Product Development$Dept # 350
telephoneNumber: +1 206 296-9826
title: Master Product Development Stooge
userPassword: ekneirKxiD
uid: Dix_Krienke
givenName: Dix
mail: Dix_Krienke@example.com
carLicense: 1E61ZTG
departmentNumber: 3571
employeeType: Employee
homePhone: +1 213 496-7662
initials: D. K.
mobile: +1 408 172-8287
pager: +1 804 262-5517
manager: cn=Koji Kwok
secretary: cn=Alberta Dilallo
roomNumber: 5976

dn: cn=Fscocos Quinones, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Fscocos Quinones
sn: Quinones
description: This is Fscocos Quinones's description
facsimileTelephoneNumber: +1 206 994-6478
l: Cupertino
ou: Product Testing
postalAddress: example$Product Testing$Dept # 755
telephoneNumber: +1 818 186-5139
title: Chief Product Testing Technician
userPassword: senoniuQso
uid: Fscocos_Quinones
givenName: Fscocos
mail: Fscocos_Quinones@example.com
carLicense: XF0ZI8W
departmentNumber: 9130
employeeType: Manager
homePhone: +1 804 422-1605
initials: F. Q.
mobile: +1 510 664-7654
pager: +1 303 363-3484
manager: cn=Nelle Mitchelson
secretary: cn=Remo Doucette
roomNumber: 9628

dn: cn=Kylie Kruziak, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kylie Kruziak
sn: Kruziak
description: This is Kylie Kruziak's description
facsimileTelephoneNumber: +1 415 161-9392
l: Palo Alto
ou: Product Testing
postalAddress: example$Product Testing$Dept # 24
telephoneNumber: +1 71 925-8608
title: Supreme Product Testing President
userPassword: kaizurKeil
uid: Kylie_Kruziak
givenName: Kylie
mail: Kylie_Kruziak@example.com
carLicense: U4SHQZ2
departmentNumber: 9909
employeeType: Employee
homePhone: +1 818 320-7405
initials: K. K.
mobile: +1 818 310-6474
pager: +1 818 959-3262
manager: cn=Arlena Syrett
secretary: cn=Coretta Chaves
roomNumber: 9184

dn: cn=Jodie Lauzon, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jodie Lauzon
sn: Lauzon
description: This is Jodie Lauzon's description
facsimileTelephoneNumber: +1 213 553-3445
l: Palo Alto
ou: Human Resources
postalAddress: example$Human Resources$Dept # 413
telephoneNumber: +1 510 556-6671
title: Supreme Human Resources Admin
userPassword: nozuaLeido
uid: Jodie_Lauzon
givenName: Jodie
mail: Jodie_Lauzon@example.com
carLicense: YVG4UDJ
departmentNumber: 2818
employeeType: Normal
homePhone: +1 206 383-7260
initials: J. L.
mobile: +1 213 802-7667
pager: +1 408 908-6859
manager: cn=Carrissa Traynor
secretary: cn=Annemie Lugwig
roomNumber: 9234

dn: cn=Alyda Fouke, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Alyda Fouke
sn: Fouke
description: This is Alyda Fouke's description
facsimileTelephoneNumber: +1 510 280-8208
l: San Jose
ou: Human Resources
postalAddress: example$Human Resources$Dept # 14
telephoneNumber: +1 206 334-6223
title: Elite Human Resources Punk
userPassword: ekuoFadylA
uid: Alyda_Fouke
givenName: Alyda
mail: Alyda_Fouke@example.com
carLicense: RL967Y6
departmentNumber: 1497
employeeType: Employee
homePhone: +1 510 808-6888
initials: A. F.
mobile: +1 818 997-1396
pager: +1 408 931-7848
manager: cn=Anet Boggan
secretary: cn=Berton Kinahan
roomNumber: 8645

dn: cn=Arif Thierry, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Arif Thierry
sn: Thierry
description: This is Arif Thierry's description
facsimileTelephoneNumber: +1 818 513-4366
l: Mountain View
ou: Payroll
postalAddress: example$Payroll$Dept # 47
telephoneNumber: +1 510 662-1162
title: Associate Payroll Manager
userPassword: yrreihTfir
uid: Arif_Thierry
givenName: Arif
mail: Arif_Thierry@example.com
carLicense: MEAI3TM
departmentNumber: 287
employeeType: Temp
homePhone: +1 415 989-5796
initials: A. T.
mobile: +1 71 284-9875
pager: +1 510 790-6449
manager: cn=Annabel Stanfield
secretary: cn=Arts Filion
roomNumber: 9951

dn: cn=Heath O'Dea, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Heath O'Dea
sn: O'Dea
description: This is Heath O'Dea's description
facsimileTelephoneNumber: +1 510 833-2140
l: Redwood Shores
ou: Product Testing
postalAddress: example$Product Testing$Dept # 620
telephoneNumber: +1 804 673-9849
title: Elite Product Testing Dictator
userPassword: aeD'OhtaeH
uid: Heath_O'Dea
givenName: Heath
mail: Heath_O'Dea@example.com
carLicense: VH3L76Y
departmentNumber: 487
employeeType: Employee
homePhone: +1 71 988-5161
initials: H. O.
mobile: +1 804 872-5119
pager: +1 303 249-5518
manager: cn=Wileen MacKay
secretary: cn=Anastasie Operator
roomNumber: 7428

dn: cn=Ofella Majury, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ofella Majury
sn: Majury
description: This is Ofella Majury's description
facsimileTelephoneNumber: +1 303 349-8252
l: San Jose
ou: Janitorial
postalAddress: example$Janitorial$Dept # 432
telephoneNumber: +1 213 408-5993
title: Chief Janitorial Visionary
userPassword: yrujaMalle
uid: Ofella_Majury
givenName: Ofella
mail: Ofella_Majury@example.com
carLicense: WVM5S3Y
departmentNumber: 520
employeeType: Manager
homePhone: +1 415 351-1419
initials: O. M.
mobile: +1 510 684-4488
pager: +1 408 961-1013
manager: cn=Maryrose Ilic
secretary: cn=Elna Horton
roomNumber: 9652

dn: cn=Farrand Knickerbocker, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Farrand Knickerbocker
sn: Knickerbocker
description: This is Farrand Knickerbocker's description
facsimileTelephoneNumber: +1 818 152-8899
l: Menlo Park
ou: Payroll
postalAddress: example$Payroll$Dept # 228
telephoneNumber: +1 415 489-9212
title: Associate Payroll Accountant
userPassword: rekcobrekc
uid: Farrand_Knickerbocker
givenName: Farrand
mail: Farrand_Knickerbocker@example.com
carLicense: GTJKTHK
departmentNumber: 1737
employeeType: Normal
homePhone: +1 71 844-5944
initials: F. K.
mobile: +1 408 965-2100
pager: +1 71 537-7105
manager: cn=Bnrecad Horemans
secretary: cn=Teri Coord
roomNumber: 2153

dn: cn=Liam Jesshope, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Liam Jesshope
sn: Jesshope
description: This is Liam Jesshope's description
facsimileTelephoneNumber: +1 510 228-3078
l: Palo Alto
ou: Accounting
postalAddress: example$Accounting$Dept # 520
telephoneNumber: +1 510 186-8605
title: Senior Accounting Consultant
userPassword: epohsseJma
uid: Liam_Jesshope
givenName: Liam
mail: Liam_Jesshope@example.com
carLicense: PW6YJY7
departmentNumber: 2447
employeeType: Contract
homePhone: +1 818 714-2254
initials: L. J.
mobile: +1 804 448-4211
pager: +1 415 415-3323
manager: cn=Maryjo Shute
secretary: cn=Manish Baenziger
roomNumber: 4774

dn: cn=Adelina Shieff, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Adelina Shieff
sn: Shieff
description: This is Adelina Shieff's description
facsimileTelephoneNumber: +1 408 494-5483
l: Menlo Park
ou: Product Testing
postalAddress: example$Product Testing$Dept # 779
telephoneNumber: +1 804 501-5991
title: Associate Product Testing Assistant
userPassword: ffeihSanil
uid: Adelina_Shieff
givenName: Adelina
mail: Adelina_Shieff@example.com
carLicense: WF07JGM
departmentNumber: 1371
employeeType: Normal
homePhone: +1 408 768-8664
initials: A. S.
mobile: +1 818 744-3006
pager: +1 303 268-9521
manager: cn=Ravi Rutherford
secretary: cn=Kusum Bourk
roomNumber: 9415

dn: cn=Franky Tull, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Franky Tull
sn: Tull
description: This is Franky Tull's description
facsimileTelephoneNumber: +1 303 412-8742
l: Redwood Shores
ou: Planning
postalAddress: example$Planning$Dept # 144
telephoneNumber: +1 510 767-8721
title: Master Planning President
userPassword: lluTyknarF
uid: Franky_Tull
givenName: Franky
mail: Franky_Tull@example.com
carLicense: WUUSKTK
departmentNumber: 8225
employeeType: Temp
homePhone: +1 804 539-4320
initials: F. T.
mobile: +1 804 665-1698
pager: +1 818 227-1393
manager: cn=Colene Parkinson
secretary: cn=Hall Wilde
roomNumber: 6019

dn: cn=Shiroshi Dalmard, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shiroshi Dalmard
sn: Dalmard
description: This is Shiroshi Dalmard's description
facsimileTelephoneNumber: +1 804 257-4308
l: Redwood Shores
ou: Product Development
postalAddress: example$Product Development$Dept # 80
telephoneNumber: +1 415 204-4697
title: Supreme Product Development Dictator
userPassword: dramlaDihs
uid: Shiroshi_Dalmard
givenName: Shiroshi
mail: Shiroshi_Dalmard@example.com
carLicense: 02MM7OW
departmentNumber: 4858
employeeType: Contract
homePhone: +1 303 761-2255
initials: S. D.
mobile: +1 408 398-9992
pager: +1 206 239-1591
manager: cn=Olympe Flury
secretary: cn=Ryman Poulsen
roomNumber: 3037

dn: cn=Tandi Stewart, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tandi Stewart
sn: Stewart
description: This is Tandi Stewart's description
facsimileTelephoneNumber: +1 415 796-9826
l: San Mateo
ou: Human Resources
postalAddress: example$Human Resources$Dept # 882
telephoneNumber: +1 408 569-2234
title: Associate Human Resources Consultant
userPassword: trawetSidn
uid: Tandi_Stewart
givenName: Tandi
mail: Tandi_Stewart@example.com
carLicense: NYPHGM3
departmentNumber: 762
employeeType: Temp
homePhone: +1 510 778-1872
initials: T. S.
mobile: +1 303 995-4360
pager: +1 818 989-3433
manager: cn=Marjory Wiebe
secretary: cn=Serene Uhl
roomNumber: 5940

dn: cn=Stephan Fradette, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Stephan Fradette
sn: Fradette
description: This is Stephan Fradette's description
facsimileTelephoneNumber: +1 818 200-8162
l: Cambridge
ou: Administrative
postalAddress: example$Administrative$Dept # 786
telephoneNumber: +1 303 741-1783
title: Senior Administrative Developer
userPassword: ettedarFna
uid: Stephan_Fradette
givenName: Stephan
mail: Stephan_Fradette@example.com
carLicense: TFGXTS4
departmentNumber: 2833
employeeType: Contract
homePhone: +1 408 362-4386
initials: S. F.
mobile: +1 71 219-7853
pager: +1 408 162-7901
manager: cn=Brittni Devera
secretary: cn=Andras Hoadley
roomNumber: 6187

dn: cn=Merrielle Nunnally, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Merrielle Nunnally
sn: Nunnally
description: This is Merrielle Nunnally's description
facsimileTelephoneNumber: +1 408 254-3956
l: Emeryville
ou: Product Testing
postalAddress: example$Product Testing$Dept # 170
telephoneNumber: +1 71 971-4713
title: Master Product Testing Punk
userPassword: yllannuNel
uid: Merrielle_Nunnally
givenName: Merrielle
mail: Merrielle_Nunnally@example.com
carLicense: IN6H6K5
departmentNumber: 237
employeeType: Manager
homePhone: +1 818 849-8694
initials: M. N.
mobile: +1 415 596-4059
pager: +1 415 837-8969
manager: cn=Cecil Datta
secretary: cn=Lorena Hagan
roomNumber: 9003

dn: cn=Catriona Saini, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Catriona Saini
sn: Saini
description: This is Catriona Saini's description
facsimileTelephoneNumber: +1 213 727-5463
l: Mountain View
ou: Management
postalAddress: example$Management$Dept # 344
telephoneNumber: +1 818 369-8258
title: Elite Management Admin
userPassword: iniaSanoir
uid: Catriona_Saini
givenName: Catriona
mail: Catriona_Saini@example.com
carLicense: 0B7DRF0
departmentNumber: 1850
employeeType: Contract
homePhone: +1 804 578-7448
initials: C. S.
mobile: +1 510 795-3588
pager: +1 206 443-2856
manager: cn=Sileas Glasa
secretary: cn=Celie Schledwitz
roomNumber: 8697

dn: cn=Der-Chang Rastelli, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Der-Chang Rastelli
sn: Rastelli
description: This is Der-Chang Rastelli's description
facsimileTelephoneNumber: +1 213 227-8896
l: Cambridge
ou: Payroll
postalAddress: example$Payroll$Dept # 398
telephoneNumber: +1 415 109-2123
title: Supreme Payroll Figurehead
userPassword: illetsaRgn
uid: Der-Chang_Rastelli
givenName: Der-Chang
mail: Der-Chang_Rastelli@example.com
carLicense: HC5MWZP
departmentNumber: 3952
employeeType: Contract
homePhone: +1 510 941-8108
initials: D. R.
mobile: +1 408 450-2834
pager: +1 818 510-9237
manager: cn=Adrian Dao
secretary: cn=Gama Coghlan
roomNumber: 6998

dn: cn=Tawnya Oswald, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tawnya Oswald
sn: Oswald
description: This is Tawnya Oswald's description
facsimileTelephoneNumber: +1 206 113-6862
l: Mountain View
ou: Product Testing
postalAddress: example$Product Testing$Dept # 719
telephoneNumber: +1 818 275-3879
title: Supreme Product Testing President
userPassword: dlawsOaynw
uid: Tawnya_Oswald
givenName: Tawnya
mail: Tawnya_Oswald@example.com
carLicense: HY5PK6J
departmentNumber: 9652
employeeType: Normal
homePhone: +1 71 112-6723
initials: T. O.
mobile: +1 408 587-4180
pager: +1 71 459-1613
manager: cn=Chi MacLaren
secretary: cn=Milan Swact
roomNumber: 9200

dn: cn=Thad Justus, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Thad Justus
sn: Justus
description: This is Thad Justus's description
facsimileTelephoneNumber: +1 818 965-2820
l: Mountain View
ou: Janitorial
postalAddress: example$Janitorial$Dept # 307
telephoneNumber: +1 804 697-3065
title: Supreme Janitorial Artist
userPassword: sutsuJdahT
uid: Thad_Justus
givenName: Thad
mail: Thad_Justus@example.com
carLicense: 4YTUL59
departmentNumber: 2679
employeeType: Contract
homePhone: +1 303 735-5394
initials: T. J.
mobile: +1 818 535-7304
pager: +1 818 767-2559
manager: cn=Joeann Moritz
secretary: cn=Carlee Samalot
roomNumber: 6109

dn: cn=Angie Yowell, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Angie Yowell
sn: Yowell
description: This is Angie Yowell's description
facsimileTelephoneNumber: +1 408 229-5917
l: Cupertino
ou: Accounting
postalAddress: example$Accounting$Dept # 655
telephoneNumber: +1 415 494-9452
title: Chief Accounting Visionary
userPassword: llewoYeign
uid: Angie_Yowell
givenName: Angie
mail: Angie_Yowell@example.com
carLicense: UPO11H1
departmentNumber: 9440
employeeType: Manager
homePhone: +1 408 406-9974
initials: A. Y.
mobile: +1 804 493-8084
pager: +1 71 299-2553
manager: cn=Kimberlyn Guertin
secretary: cn=Glad Madl
roomNumber: 7692

dn: cn=Lavina Waespe, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lavina Waespe
sn: Waespe
description: This is Lavina Waespe's description
facsimileTelephoneNumber: +1 804 212-2416
l: Sunnyvale
ou: Product Development
postalAddress: example$Product Development$Dept # 300
telephoneNumber: +1 213 846-6534
title: Senior Product Development Punk
userPassword: epseaWaniv
uid: Lavina_Waespe
givenName: Lavina
mail: Lavina_Waespe@example.com
carLicense: 7ENZE1Z
departmentNumber: 8427
employeeType: Manager
homePhone: +1 804 483-6826
initials: L. W.
mobile: +1 415 464-1188
pager: +1 818 347-8673
manager: cn=Donita Peter
secretary: cn=Anwar Petzold
roomNumber: 5216

dn: cn=Melisenda Hilberman, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Melisenda Hilberman
sn: Hilberman
description: This is Melisenda Hilberman's description
facsimileTelephoneNumber: +1 213 255-6605
l: Palo Alto
ou: Product Development
postalAddress: example$Product Development$Dept # 983
telephoneNumber: +1 818 514-2332
title: Supreme Product Development Architect
userPassword: namrebliHa
uid: Melisenda_Hilberman
givenName: Melisenda
mail: Melisenda_Hilberman@example.com
carLicense: IZ6IJ0A
departmentNumber: 3493
employeeType: Contract
homePhone: +1 510 796-9780
initials: M. H.
mobile: +1 804 759-3588
pager: +1 71 367-6646
manager: cn=Ayn MacHattie
secretary: cn=Therine Jodoin
roomNumber: 612

dn: cn=Fancy Cadzow, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Fancy Cadzow
sn: Cadzow
description: This is Fancy Cadzow's description
facsimileTelephoneNumber: +1 71 944-1379
l: San Francisco
ou: Payroll
postalAddress: example$Payroll$Dept # 177
telephoneNumber: +1 213 344-4751
title: Master Payroll Fellow
userPassword: wozdaCycna
uid: Fancy_Cadzow
givenName: Fancy
mail: Fancy_Cadzow@example.com
carLicense: NOTKMH9
departmentNumber: 7054
employeeType: Employee
homePhone: +1 408 735-8563
initials: F. C.
mobile: +1 71 337-4872
pager: +1 408 426-7911
manager: cn=Aurea Ruel
secretary: cn=Rhianon Loghry
roomNumber: 6845

dn: cn=Teriann Fazel, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Teriann Fazel
sn: Fazel
description: This is Teriann Fazel's description
facsimileTelephoneNumber: +1 818 869-2312
l: Sunnyvale
ou: Janitorial
postalAddress: example$Janitorial$Dept # 380
telephoneNumber: +1 408 636-1109
title: Senior Janitorial Fellow
userPassword: lezaFnnair
uid: Teriann_Fazel
givenName: Teriann
mail: Teriann_Fazel@example.com
carLicense: OVHKQ4X
departmentNumber: 2267
employeeType: Manager
homePhone: +1 818 665-1495
initials: T. F.
mobile: +1 206 959-2924
pager: +1 303 921-6004
manager: cn=Trixie Miner
secretary: cn=Millisent Scholman
roomNumber: 1181

dn: cn=Tyne Zug, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tyne Zug
sn: Zug
description: This is Tyne Zug's description
facsimileTelephoneNumber: +1 71 254-6386
l: Orem
ou: Planning
postalAddress: example$Planning$Dept # 197
telephoneNumber: +1 818 327-3956
title: Master Planning Manager
userPassword: guZenyT
uid: Tyne_Zug
givenName: Tyne
mail: Tyne_Zug@example.com
carLicense: 390JP4Y
departmentNumber: 8107
employeeType: Employee
homePhone: +1 818 658-8996
initials: T. Z.
mobile: +1 818 974-4450
pager: +1 408 886-2502
manager: cn=Edeline Jubb
secretary: cn=Berenice Dornback
roomNumber: 1390

dn: cn=Ly-Khanh Choi, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ly-Khanh Choi
sn: Choi
description: This is Ly-Khanh Choi's description
facsimileTelephoneNumber: +1 206 125-7982
l: Menlo Park
ou: Payroll
postalAddress: example$Payroll$Dept # 458
telephoneNumber: +1 71 170-8245
title: Supreme Payroll Czar
userPassword: iohChnahK-
uid: Ly-Khanh_Choi
givenName: Ly-Khanh
mail: Ly-Khanh_Choi@example.com
carLicense: 9R4OQZ9
departmentNumber: 9481
employeeType: Employee
homePhone: +1 818 918-2219
initials: L. C.
mobile: +1 415 742-5578
pager: +1 510 793-6375
manager: cn=Edlene Tassi
secretary: cn=Douglas Gelo
roomNumber: 6434

dn: cn=Kat Mitalas, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kat Mitalas
sn: Mitalas
description: This is Kat Mitalas's description
facsimileTelephoneNumber: +1 408 983-9652
l: Orem
ou: Payroll
postalAddress: example$Payroll$Dept # 112
telephoneNumber: +1 510 814-5888
title: Supreme Payroll Stooge
userPassword: salatiMtaK
uid: Kat_Mitalas
givenName: Kat
mail: Kat_Mitalas@example.com
carLicense: OQN4VP2
departmentNumber: 7481
employeeType: Temp
homePhone: +1 206 722-9764
initials: K. M.
mobile: +1 303 400-7643
pager: +1 303 750-7022
manager: cn=Janenna Maginley
secretary: cn=Rori Bothwell
roomNumber: 532

dn: cn=Klaas Queries, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Klaas Queries
sn: Queries
description: This is Klaas Queries's description
facsimileTelephoneNumber: +1 818 468-6697
l: San Jose
ou: Human Resources
postalAddress: example$Human Resources$Dept # 425
telephoneNumber: +1 818 728-3329
title: Master Human Resources Manager
userPassword: seireuQsaa
uid: Klaas_Queries
givenName: Klaas
mail: Klaas_Queries@example.com
carLicense: 89PL5N7
departmentNumber: 5341
employeeType: Manager
homePhone: +1 71 566-5992
initials: K. Q.
mobile: +1 408 304-7523
pager: +1 415 924-1691
manager: cn=Halie Haley
secretary: cn=Doria Ault
roomNumber: 2711

dn: cn=Shanta Zoppel, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shanta Zoppel
sn: Zoppel
description: This is Shanta Zoppel's description
facsimileTelephoneNumber: +1 206 521-9564
l: Palo Alto
ou: Administrative
postalAddress: example$Administrative$Dept # 823
telephoneNumber: +1 303 691-2138
title: Junior Administrative Punk
userPassword: leppoZatna
uid: Shanta_Zoppel
givenName: Shanta
mail: Shanta_Zoppel@example.com
carLicense: RGDOHL0
departmentNumber: 9602
employeeType: Employee
homePhone: +1 818 305-5351
initials: S. Z.
mobile: +1 213 530-8707
pager: +1 206 979-9001
manager: cn=Arline Brockmeyer
secretary: cn=Didi Schartmann
roomNumber: 5886

dn: cn=Demeter Sezer, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Demeter Sezer
sn: Sezer
description: This is Demeter Sezer's description
facsimileTelephoneNumber: +1 415 208-3431
l: Santa Clara
ou: Janitorial
postalAddress: example$Janitorial$Dept # 263
telephoneNumber: +1 71 241-1051
title: Associate Janitorial President
userPassword: rezeSretem
uid: Demeter_Sezer
givenName: Demeter
mail: Demeter_Sezer@example.com
carLicense: RB2GPVI
departmentNumber: 7914
employeeType: Manager
homePhone: +1 206 240-8869
initials: D. S.
mobile: +1 71 918-7015
pager: +1 71 577-4497
manager: cn=Diju Wong
secretary: cn=Caria Pickles
roomNumber: 7244

dn: cn=Glynda Wolowidnyk, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Glynda Wolowidnyk
sn: Wolowidnyk
description: This is Glynda Wolowidnyk's description
facsimileTelephoneNumber: +1 206 652-7450
l: Redmond
ou: Peons
postalAddress: example$Peons$Dept # 355
telephoneNumber: +1 415 196-3248
title: Elite Peons Figurehead
userPassword: kyndiwoloW
uid: Glynda_Wolowidnyk
givenName: Glynda
mail: Glynda_Wolowidnyk@example.com
carLicense: 7A92E5S
departmentNumber: 8752
employeeType: Contract
homePhone: +1 408 684-5926
initials: G. W.
mobile: +1 804 715-4376
pager: +1 303 624-5785
manager: cn=Bora Werth
secretary: cn=Nurettin Heroux
roomNumber: 2685

dn: cn=Joice Viau, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Joice Viau
sn: Viau
description: This is Joice Viau's description
facsimileTelephoneNumber: +1 804 225-7580
l: Fremont
ou: Janitorial
postalAddress: example$Janitorial$Dept # 686
telephoneNumber: +1 818 337-5421
title: Master Janitorial Director
userPassword: uaiVecioJ
uid: Joice_Viau
givenName: Joice
mail: Joice_Viau@example.com
carLicense: UGKBZPR
departmentNumber: 5241
employeeType: Manager
homePhone: +1 408 801-1068
initials: J. V.
mobile: +1 303 511-3988
pager: +1 818 333-1392
manager: cn=Carley Couron
secretary: cn=Ginette Covey
roomNumber: 8522

dn: cn=Zonda Birkett, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zonda Birkett
sn: Birkett
description: This is Zonda Birkett's description
facsimileTelephoneNumber: +1 415 549-6310
l: Milpitas
ou: Management
postalAddress: example$Management$Dept # 977
telephoneNumber: +1 804 872-7461
title: Supreme Management Visionary
userPassword: ttekriBadn
uid: Zonda_Birkett
givenName: Zonda
mail: Zonda_Birkett@example.com
carLicense: UYID95B
departmentNumber: 7456
employeeType: Employee
homePhone: +1 71 114-2576
initials: Z. B.
mobile: +1 818 559-6329
pager: +1 206 722-1833
manager: cn=Sheryl Gaylord
secretary: cn=Pia Heckman
roomNumber: 4339

dn: cn=Marybelle Passin, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marybelle Passin
sn: Passin
description: This is Marybelle Passin's description
facsimileTelephoneNumber: +1 408 723-9290
l: San Jose
ou: Management
postalAddress: example$Management$Dept # 577
telephoneNumber: +1 206 411-3396
title: Junior Management Dictator
userPassword: nissaPelle
uid: Marybelle_Passin
givenName: Marybelle
mail: Marybelle_Passin@example.com
carLicense: CRW20D4
departmentNumber: 7051
employeeType: Manager
homePhone: +1 71 853-1337
initials: M. P.
mobile: +1 804 919-4814
pager: +1 213 853-5051
manager: cn=Manon Cheeseman
secretary: cn=Previn Zalite
roomNumber: 6139

dn: cn=Sharai Pezzoli, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sharai Pezzoli
sn: Pezzoli
description: This is Sharai Pezzoli's description
facsimileTelephoneNumber: +1 415 830-1006
l: Redwood Shores
ou: Administrative
postalAddress: example$Administrative$Dept # 189
telephoneNumber: +1 408 590-5082
title: Supreme Administrative Admin
userPassword: ilozzePiar
uid: Sharai_Pezzoli
givenName: Sharai
mail: Sharai_Pezzoli@example.com
carLicense: 0PWVA7A
departmentNumber: 7431
employeeType: Employee
homePhone: +1 804 420-7444
initials: S. P.
mobile: +1 213 339-2074
pager: +1 510 162-6549
manager: cn=Candice Vlahos
secretary: cn=Sucha MacNeill
roomNumber: 901

dn: cn=Carly Smuda, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Carly Smuda
sn: Smuda
description: This is Carly Smuda's description
facsimileTelephoneNumber: +1 71 366-3162
l: Orem
ou: Peons
postalAddress: example$Peons$Dept # 366
telephoneNumber: +1 213 495-5466
title: Master Peons Architect
userPassword: adumSylraC
uid: Carly_Smuda
givenName: Carly
mail: Carly_Smuda@example.com
carLicense: E4V6O6O
departmentNumber: 3482
employeeType: Normal
homePhone: +1 804 616-3406
initials: C. S.
mobile: +1 415 479-8184
pager: +1 510 594-2190
manager: cn=Stefa Zaharychuk
secretary: cn=Edwin Surazski
roomNumber: 5581

dn: cn=Jinny Stds, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jinny Stds
sn: Stds
description: This is Jinny Stds's description
facsimileTelephoneNumber: +1 415 299-2965
l: Redmond
ou: Janitorial
postalAddress: example$Janitorial$Dept # 984
telephoneNumber: +1 71 208-9503
title: Master Janitorial Writer
userPassword: sdtSynniJ
uid: Jinny_Stds
givenName: Jinny
mail: Jinny_Stds@example.com
carLicense: Z24L06X
departmentNumber: 9136
employeeType: Normal
homePhone: +1 408 889-9149
initials: J. S.
mobile: +1 818 130-1783
pager: +1 415 320-2448
manager: cn=Alka Liew
secretary: cn=Lonnie Zoellner
roomNumber: 1475

dn: cn=Ellis Barsky, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ellis Barsky
sn: Barsky
description: This is Ellis Barsky's description
facsimileTelephoneNumber: +1 213 614-2840
l: San Mateo
ou: Peons
postalAddress: example$Peons$Dept # 503
telephoneNumber: +1 804 887-2089
title: Associate Peons Admin
userPassword: yksraBsill
uid: Ellis_Barsky
givenName: Ellis
mail: Ellis_Barsky@example.com
carLicense: Q7SP1UW
departmentNumber: 7909
employeeType: Normal
homePhone: +1 510 275-1317
initials: E. B.
mobile: +1 408 545-3968
pager: +1 408 544-2462
manager: cn=Cleto Ntelpac
secretary: cn=Fifi Bostelmann
roomNumber: 8633

dn: cn=Rosette Elkington, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rosette Elkington
sn: Elkington
description: This is Rosette Elkington's description
facsimileTelephoneNumber: +1 71 643-5767
l: Redmond
ou: Payroll
postalAddress: example$Payroll$Dept # 76
telephoneNumber: +1 206 974-9881
title: Senior Payroll Dictator
userPassword: notgniklEe
uid: Rosette_Elkington
givenName: Rosette
mail: Rosette_Elkington@example.com
carLicense: 0ISB4RP
departmentNumber: 3157
employeeType: Contract
homePhone: +1 408 364-9270
initials: R. E.
mobile: +1 408 750-2001
pager: +1 818 566-5697
manager: cn=Chander Frodsham
secretary: cn=Rowe Fiegel
roomNumber: 6207

dn: cn=Corene Rankin, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Corene Rankin
sn: Rankin
description: This is Corene Rankin's description
facsimileTelephoneNumber: +1 303 988-9737
l: San Francisco
ou: Accounting
postalAddress: example$Accounting$Dept # 873
telephoneNumber: +1 818 726-7769
title: Master Accounting Developer
userPassword: niknaRener
uid: Corene_Rankin
givenName: Corene
mail: Corene_Rankin@example.com
carLicense: FA5KWY0
departmentNumber: 474
employeeType: Manager
homePhone: +1 818 605-9872
initials: C. R.
mobile: +1 71 482-1976
pager: +1 206 520-5056
manager: cn=Cristine Widows
secretary: cn=Parker Calistro
roomNumber: 9503

dn: cn=Marget Timm, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marget Timm
sn: Timm
description: This is Marget Timm's description
facsimileTelephoneNumber: +1 213 616-9877
l: Menlo Park
ou: Janitorial
postalAddress: example$Janitorial$Dept # 535
telephoneNumber: +1 415 864-3126
title: Junior Janitorial Writer
userPassword: mmiTtegraM
uid: Marget_Timm
givenName: Marget
mail: Marget_Timm@example.com
carLicense: 74V4UTL
departmentNumber: 4455
employeeType: Normal
homePhone: +1 206 510-5409
initials: M. T.
mobile: +1 408 657-8524
pager: +1 415 884-3865
manager: cn=Ros Poff
secretary: cn=Kaycee Schyving
roomNumber: 6458

dn: cn=Philip Ruyant, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Philip Ruyant
sn: Ruyant
description: This is Philip Ruyant's description
facsimileTelephoneNumber: +1 818 271-7057
l: Milpitas
ou: Human Resources
postalAddress: example$Human Resources$Dept # 435
telephoneNumber: +1 415 756-5089
title: Elite Human Resources Pinhead
userPassword: tnayuRpili
uid: Philip_Ruyant
givenName: Philip
mail: Philip_Ruyant@example.com
carLicense: RTAKY71
departmentNumber: 7012
employeeType: Normal
homePhone: +1 206 258-2838
initials: P. R.
mobile: +1 510 228-2479
pager: +1 415 614-6100
manager: cn=Merrill Naugle
secretary: cn=Trista Naro
roomNumber: 4167

dn: cn=Magdalena Noel, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Magdalena Noel
sn: Noel
description: This is Magdalena Noel's description
facsimileTelephoneNumber: +1 71 932-1171
l: Milpitas
ou: Peons
postalAddress: example$Peons$Dept # 406
telephoneNumber: +1 206 159-1131
title: Master Peons Visionary
userPassword: leoNanelad
uid: Magdalena_Noel
givenName: Magdalena
mail: Magdalena_Noel@example.com
carLicense: 4HKGRVG
departmentNumber: 2354
employeeType: Normal
homePhone: +1 804 810-2355
initials: M. N.
mobile: +1 804 783-5560
pager: +1 510 772-5833
manager: cn=Barb Heppes
secretary: cn=Lucy Crabtree
roomNumber: 781

dn: cn=Mady Hogue, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mady Hogue
sn: Hogue
description: This is Mady Hogue's description
facsimileTelephoneNumber: +1 206 529-6732
l: Cupertino
ou: Administrative
postalAddress: example$Administrative$Dept # 525
telephoneNumber: +1 213 111-4880
title: Master Administrative Writer
userPassword: eugoHydaM
uid: Mady_Hogue
givenName: Mady
mail: Mady_Hogue@example.com
carLicense: BD2HXMN
departmentNumber: 3545
employeeType: Contract
homePhone: +1 213 517-3203
initials: M. H.
mobile: +1 804 926-2015
pager: +1 510 204-4242
manager: cn=Lotta Geldrez
secretary: cn=Nanni Donovan
roomNumber: 9575

dn: cn=Ivie Malee, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ivie Malee
sn: Malee
description: This is Ivie Malee's description
facsimileTelephoneNumber: +1 303 588-8656
l: San Francisco
ou: Product Development
postalAddress: example$Product Development$Dept # 241
telephoneNumber: +1 408 459-3942
title: Junior Product Development Fellow
userPassword: eelaMeivI
uid: Ivie_Malee
givenName: Ivie
mail: Ivie_Malee@example.com
carLicense: BXL4TZB
departmentNumber: 9117
employeeType: Contract
homePhone: +1 71 141-8943
initials: I. M.
mobile: +1 510 827-1920
pager: +1 303 800-1158
manager: cn=Turus Emami
secretary: cn=Japan Layne
roomNumber: 8232

dn: cn=Coursdev Smits, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Coursdev Smits
sn: Smits
description: This is Coursdev Smits's description
facsimileTelephoneNumber: +1 510 146-3293
l: Fremont
ou: Administrative
postalAddress: example$Administrative$Dept # 902
telephoneNumber: +1 206 607-6996
title: Supreme Administrative Janitor
userPassword: stimSvedsr
uid: Coursdev_Smits
givenName: Coursdev
mail: Coursdev_Smits@example.com
carLicense: OLALO2R
departmentNumber: 8026
employeeType: Normal
homePhone: +1 415 578-1146
initials: C. S.
mobile: +1 206 773-1314
pager: +1 510 963-2309
manager: cn=Renny Naismith
secretary: cn=Therese Hazlett
roomNumber: 3927

dn: cn=Gelais Dixon, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gelais Dixon
sn: Dixon
description: This is Gelais Dixon's description
facsimileTelephoneNumber: +1 206 717-6773
l: Santa Clara
ou: Product Development
postalAddress: example$Product Development$Dept # 942
telephoneNumber: +1 206 128-1695
title: Elite Product Development Pinhead
userPassword: noxiDsiale
uid: Gelais_Dixon
givenName: Gelais
mail: Gelais_Dixon@example.com
carLicense: JZ7407S
departmentNumber: 1534
employeeType: Manager
homePhone: +1 408 860-8272
initials: G. D.
mobile: +1 804 127-4751
pager: +1 818 526-9668
manager: cn=Cecco Plsntp
secretary: cn=Dhanvinder Averette
roomNumber: 8012

dn: cn=Action Laberge, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Action Laberge
sn: Laberge
description: This is Action Laberge's description
facsimileTelephoneNumber: +1 804 931-9330
l: Emeryville
ou: Product Development
postalAddress: example$Product Development$Dept # 773
telephoneNumber: +1 510 920-2422
title: Junior Product Development Consultant
userPassword: egrebaLnoi
uid: Action_Laberge
givenName: Action
mail: Action_Laberge@example.com
carLicense: 1A6ICKQ
departmentNumber: 7424
employeeType: Normal
homePhone: +1 408 694-1770
initials: A. L.
mobile: +1 408 329-4881
pager: +1 408 147-1193
manager: cn=Fastmer Cusson
secretary: cn=Salim Ruth
roomNumber: 3048

dn: cn=Sarita Fouke, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sarita Fouke
sn: Fouke
description: This is Sarita Fouke's description
facsimileTelephoneNumber: +1 415 540-5752
l: Redwood Shores
ou: Planning
postalAddress: example$Planning$Dept # 752
telephoneNumber: +1 408 989-8492
title: Elite Planning Mascot
userPassword: ekuoFatira
uid: Sarita_Fouke
givenName: Sarita
mail: Sarita_Fouke@example.com
carLicense: 9LJNKZI
departmentNumber: 7065
employeeType: Employee
homePhone: +1 818 978-8910
initials: S. F.
mobile: +1 818 295-5678
pager: +1 415 723-3513
manager: cn=Deniece Whetzel
secretary: cn=Florella Sharma
roomNumber: 4270

dn: cn=Belle Moxley, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Belle Moxley
sn: Moxley
description: This is Belle Moxley's description
facsimileTelephoneNumber: +1 818 574-4599
l: Menlo Park
ou: Accounting
postalAddress: example$Accounting$Dept # 351
telephoneNumber: +1 213 498-9024
title: Supreme Accounting Fellow
userPassword: yelxoMelle
uid: Belle_Moxley
givenName: Belle
mail: Belle_Moxley@example.com
carLicense: YLW046G
departmentNumber: 8213
employeeType: Temp
homePhone: +1 213 615-6463
initials: B. M.
mobile: +1 303 800-1965
pager: +1 510 400-4379
manager: cn=Marion Querengesser
secretary: cn=Pawel McMinn
roomNumber: 4667

dn: cn=MaryLou Kenlan, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: MaryLou Kenlan
sn: Kenlan
description: This is MaryLou Kenlan's description
facsimileTelephoneNumber: +1 206 599-1562
l: Fremont
ou: Product Development
postalAddress: example$Product Development$Dept # 569
telephoneNumber: +1 303 508-1876
title: Supreme Product Development Admin
userPassword: nalneKuoLy
uid: MaryLou_Kenlan
givenName: MaryLou
mail: MaryLou_Kenlan@example.com
carLicense: N97KWQO
departmentNumber: 2369
employeeType: Manager
homePhone: +1 510 671-7364
initials: M. K.
mobile: +1 818 866-6445
pager: +1 415 195-3197
manager: cn=Chandal Ibsen
secretary: cn=Go McClure
roomNumber: 5806

dn: cn=Nettie Holthaus, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Nettie Holthaus
sn: Holthaus
description: This is Nettie Holthaus's description
facsimileTelephoneNumber: +1 510 259-5175
l: Fremont
ou: Product Development
postalAddress: example$Product Development$Dept # 167
telephoneNumber: +1 804 256-5440
title: Elite Product Development Writer
userPassword: suahtloHei
uid: Nettie_Holthaus
givenName: Nettie
mail: Nettie_Holthaus@example.com
carLicense: NIWXD5L
departmentNumber: 652
employeeType: Employee
homePhone: +1 71 941-2382
initials: N. H.
mobile: +1 415 582-2954
pager: +1 71 405-2199
manager: cn=Shellie McClure
secretary: cn=Therese Kenlan
roomNumber: 5899

dn: cn=Linea Stansby, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Linea Stansby
sn: Stansby
description: This is Linea Stansby's description
facsimileTelephoneNumber: +1 510 983-8963
l: San Jose
ou: Janitorial
postalAddress: example$Janitorial$Dept # 773
telephoneNumber: +1 415 258-7985
title: Supreme Janitorial Warrior
userPassword: ybsnatSaen
uid: Linea_Stansby
givenName: Linea
mail: Linea_Stansby@example.com
carLicense: DHX8UYK
departmentNumber: 1335
employeeType: Temp
homePhone: +1 408 284-6360
initials: L. S.
mobile: +1 804 513-7755
pager: +1 303 949-6182
manager: cn=Goldy Osman
secretary: cn=Ronica Brushey
roomNumber: 1184

dn: cn=Carlyn Schissel, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Carlyn Schissel
sn: Schissel
description: This is Carlyn Schissel's description
facsimileTelephoneNumber: +1 818 362-5920
l: San Jose
ou: Administrative
postalAddress: example$Administrative$Dept # 851
telephoneNumber: +1 303 993-9907
title: Associate Administrative Yahoo
userPassword: lessihcSny
uid: Carlyn_Schissel
givenName: Carlyn
mail: Carlyn_Schissel@example.com
carLicense: NWK3GBB
departmentNumber: 659
employeeType: Temp
homePhone: +1 213 480-6322
initials: C. S.
mobile: +1 206 604-3711
pager: +1 510 675-3434
manager: cn=Nerti Van Holst
secretary: cn=Enriqueta Arvin
roomNumber: 3203

dn: cn=Marv Sponchia, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marv Sponchia
sn: Sponchia
description: This is Marv Sponchia's description
facsimileTelephoneNumber: +1 213 439-6793
l: Redmond
ou: Management
postalAddress: example$Management$Dept # 423
telephoneNumber: +1 408 572-2798
title: Elite Management Manager
userPassword: aihcnopSvr
uid: Marv_Sponchia
givenName: Marv
mail: Marv_Sponchia@example.com
carLicense: G32423U
departmentNumber: 5738
employeeType: Normal
homePhone: +1 408 219-4278
initials: M. S.
mobile: +1 71 646-9163
pager: +1 510 644-3275
manager: cn=Sotos Krull
secretary: cn=Tally Petillion
roomNumber: 9108

dn: cn=Saman Kosarski, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Saman Kosarski
sn: Kosarski
description: This is Saman Kosarski's description
facsimileTelephoneNumber: +1 206 731-9107
l: Menlo Park
ou: Administrative
postalAddress: example$Administrative$Dept # 780
telephoneNumber: +1 408 630-6494
title: Elite Administrative Consultant
userPassword: iksrasoKna
uid: Saman_Kosarski
givenName: Saman
mail: Saman_Kosarski@example.com
carLicense: J5160RX
departmentNumber: 6354
employeeType: Normal
homePhone: +1 71 952-2587
initials: S. K.
mobile: +1 510 129-8844
pager: +1 510 536-8503
manager: cn=Bertina Marengere
secretary: cn=Jeremy Sparksman
roomNumber: 2245

dn: cn=Guglielma Elwood, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Guglielma Elwood
sn: Elwood
description: This is Guglielma Elwood's description
facsimileTelephoneNumber: +1 408 841-1943
l: Milpitas
ou: Product Development
postalAddress: example$Product Development$Dept # 832
telephoneNumber: +1 804 899-6944
title: Chief Product Development Accountant
userPassword: doowlEamle
uid: Guglielma_Elwood
givenName: Guglielma
mail: Guglielma_Elwood@example.com
carLicense: 3CQSO3V
departmentNumber: 8084
employeeType: Contract
homePhone: +1 818 412-6174
initials: G. E.
mobile: +1 213 540-9204
pager: +1 303 594-4194
manager: cn=Myriam McNeal
secretary: cn=Emma Brummitt
roomNumber: 632

dn: cn=Viviana Ozyetis, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Viviana Ozyetis
sn: Ozyetis
description: This is Viviana Ozyetis's description
facsimileTelephoneNumber: +1 510 337-8866
l: Cambridge
ou: Product Testing
postalAddress: example$Product Testing$Dept # 164
telephoneNumber: +1 408 170-5723
title: Associate Product Testing Architect
userPassword: siteyzOana
uid: Viviana_Ozyetis
givenName: Viviana
mail: Viviana_Ozyetis@example.com
carLicense: 2DNLFJR
departmentNumber: 8190
employeeType: Employee
homePhone: +1 415 197-6709
initials: V. O.
mobile: +1 408 950-8170
pager: +1 408 719-6699
manager: cn=Merb Streight
secretary: cn=Metrics Wagle
roomNumber: 1292

dn: cn=Dominique Slaa, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dominique Slaa
sn: Slaa
description: This is Dominique Slaa's description
facsimileTelephoneNumber: +1 818 774-9206
l: Redwood Shores
ou: Peons
postalAddress: example$Peons$Dept # 2
telephoneNumber: +1 206 412-7150
title: Supreme Peons Punk
userPassword: aalSeuqini
uid: Dominique_Slaa
givenName: Dominique
mail: Dominique_Slaa@example.com
carLicense: XTSKRPQ
departmentNumber: 6917
employeeType: Temp
homePhone: +1 71 850-2200
initials: D. S.
mobile: +1 303 265-8608
pager: +1 415 693-6163
manager: cn=Dorothy Revis
secretary: cn=Foad Jowett
roomNumber: 2546

dn: cn=Amalea Strandberg, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Amalea Strandberg
sn: Strandberg
description: This is Amalea Strandberg's description
facsimileTelephoneNumber: +1 818 317-8408
l: Redwood Shores
ou: Administrative
postalAddress: example$Administrative$Dept # 307
telephoneNumber: +1 206 614-2202
title: Senior Administrative Director
userPassword: grebdnartS
uid: Amalea_Strandberg
givenName: Amalea
mail: Amalea_Strandberg@example.com
carLicense: 2Y3OUF9
departmentNumber: 791
employeeType: Employee
homePhone: +1 303 580-5453
initials: A. S.
mobile: +1 71 225-3720
pager: +1 408 628-3095
manager: cn=Yves Rishy-Maharaj
secretary: cn=Jewelle McGonigal
roomNumber: 9591

dn: cn=Ranea Crooks, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ranea Crooks
sn: Crooks
description: This is Ranea Crooks's description
facsimileTelephoneNumber: +1 71 561-5330
l: Cupertino
ou: Janitorial
postalAddress: example$Janitorial$Dept # 144
telephoneNumber: +1 303 494-2601
title: Chief Janitorial Manager
userPassword: skoorCaena
uid: Ranea_Crooks
givenName: Ranea
mail: Ranea_Crooks@example.com
carLicense: BBV8RGI
departmentNumber: 3493
employeeType: Employee
homePhone: +1 804 854-8696
initials: R. C.
mobile: +1 303 515-9934
pager: +1 206 348-8147
manager: cn=Jewelle Matatall
secretary: cn=Jeralee Orfano
roomNumber: 8942

dn: cn=Glynnis Sobkow, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Glynnis Sobkow
sn: Sobkow
description: This is Glynnis Sobkow's description
facsimileTelephoneNumber: +1 408 453-8293
l: Palo Alto
ou: Product Testing
postalAddress: example$Product Testing$Dept # 181
telephoneNumber: +1 408 620-5102
title: Senior Product Testing Visionary
userPassword: wokboSsinn
uid: Glynnis_Sobkow
givenName: Glynnis
mail: Glynnis_Sobkow@example.com
carLicense: ZBQ0O3V
departmentNumber: 9925
employeeType: Normal
homePhone: +1 818 410-7723
initials: G. S.
mobile: +1 206 146-7249
pager: +1 408 948-3112
manager: cn=Drucill Karibian
secretary: cn=Demi Suprick
roomNumber: 5556

dn: cn=James Peng, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: James Peng
sn: Peng
description: This is James Peng's description
facsimileTelephoneNumber: +1 818 710-8953
l: Orem
ou: Product Development
postalAddress: example$Product Development$Dept # 366
telephoneNumber: +1 415 589-9938
title: Associate Product Development Artist
userPassword: gnePsemaJ
uid: James_Peng
givenName: James
mail: James_Peng@example.com
carLicense: 0GDFBAM
departmentNumber: 9113
employeeType: Employee
homePhone: +1 213 146-3663
initials: J. P.
mobile: +1 804 348-4649
pager: +1 510 893-2162
manager: cn=Leni Petrie
secretary: cn=Donny Tonelli
roomNumber: 9158

dn: cn=Jean-Roch Absi, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jean-Roch Absi
sn: Absi
description: This is Jean-Roch Absi's description
facsimileTelephoneNumber: +1 510 378-7514
l: Santa Clara
ou: Product Development
postalAddress: example$Product Development$Dept # 567
telephoneNumber: +1 818 995-6472
title: Master Product Development Warrior
userPassword: isbAhcoR-n
uid: Jean-Roch_Absi
givenName: Jean-Roch
mail: Jean-Roch_Absi@example.com
carLicense: 6M8IH9T
departmentNumber: 8148
employeeType: Normal
homePhone: +1 510 484-4295
initials: J. A.
mobile: +1 206 404-8391
pager: +1 408 588-5896
manager: cn=Yao Fanus
secretary: cn=Yoshi Clites
roomNumber: 2118

dn: cn=Humberto Marco, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Humberto Marco
sn: Marco
description: This is Humberto Marco's description
facsimileTelephoneNumber: +1 303 868-2565
l: Armonk
ou: Planning
postalAddress: example$Planning$Dept # 402
telephoneNumber: +1 804 569-2380
title: Supreme Planning Visionary
userPassword: ocraMotreb
uid: Humberto_Marco
givenName: Humberto
mail: Humberto_Marco@example.com
carLicense: 28ZPBER
departmentNumber: 8232
employeeType: Temp
homePhone: +1 415 527-8669
initials: H. M.
mobile: +1 71 729-9850
pager: +1 206 229-8942
manager: cn=Gray Koelbl
secretary: cn=Rafiq Lazure
roomNumber: 3919

dn: cn=Samual Widuch, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Samual Widuch
sn: Widuch
description: This is Samual Widuch's description
facsimileTelephoneNumber: +1 206 206-3404
l: San Jose
ou: Product Development
postalAddress: example$Product Development$Dept # 163
telephoneNumber: +1 415 429-7771
title: Chief Product Development Fellow
userPassword: hcudiWlaum
uid: Samual_Widuch
givenName: Samual
mail: Samual_Widuch@example.com
carLicense: 9TK4361
departmentNumber: 6412
employeeType: Contract
homePhone: +1 206 406-7980
initials: S. W.
mobile: +1 415 378-7065
pager: +1 415 199-6764
manager: cn=Gupta Kinch
secretary: cn=Elsey Perreault
roomNumber: 2163

dn: cn=Devon Hamner, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Devon Hamner
sn: Hamner
description: This is Devon Hamner's description
facsimileTelephoneNumber: +1 408 827-4091
l: Milpitas
ou: Product Testing
postalAddress: example$Product Testing$Dept # 988
telephoneNumber: +1 213 899-1290
title: Elite Product Testing Developer
userPassword: renmaHnove
uid: Devon_Hamner
givenName: Devon
mail: Devon_Hamner@example.com
carLicense: SIOACF2
departmentNumber: 7530
employeeType: Temp
homePhone: +1 71 772-7250
initials: D. H.
mobile: +1 804 923-4986
pager: +1 303 312-9904
manager: cn=Stormy Stocks
secretary: cn=Lawrence Lacasse
roomNumber: 6125

dn: cn=Wassim Seiple, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Wassim Seiple
sn: Seiple
description: This is Wassim Seiple's description
facsimileTelephoneNumber: +1 804 180-7682
l: Palo Alto
ou: Janitorial
postalAddress: example$Janitorial$Dept # 612
telephoneNumber: +1 510 186-6122
title: Senior Janitorial Punk
userPassword: elpieSmiss
uid: Wassim_Seiple
givenName: Wassim
mail: Wassim_Seiple@example.com
carLicense: ZR6625O
departmentNumber: 4996
employeeType: Manager
homePhone: +1 303 544-3306
initials: W. S.
mobile: +1 71 220-3481
pager: +1 408 697-3557
manager: cn=Radomir Beshir
secretary: cn=Franky Sunatori
roomNumber: 9169

dn: cn=Corabelle Willette, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Corabelle Willette
sn: Willette
description: This is Corabelle Willette's description
facsimileTelephoneNumber: +1 415 935-8013
l: Armonk
ou: Management
postalAddress: example$Management$Dept # 853
telephoneNumber: +1 213 698-8314
title: Associate Management Vice President
userPassword: ettelliWel
uid: Corabelle_Willette
givenName: Corabelle
mail: Corabelle_Willette@example.com
carLicense: 3ROV6N5
departmentNumber: 9164
employeeType: Temp
homePhone: +1 818 126-4550
initials: C. W.
mobile: +1 206 873-9350
pager: +1 303 411-4796
manager: cn=Lou Basinger
secretary: cn=Chrissy Banks
roomNumber: 7251

dn: cn=Yuko Leggett, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Yuko Leggett
sn: Leggett
description: This is Yuko Leggett's description
facsimileTelephoneNumber: +1 804 885-1583
l: San Jose
ou: Product Testing
postalAddress: example$Product Testing$Dept # 782
telephoneNumber: +1 71 174-6937
title: Junior Product Testing Writer
userPassword: tteggeLoku
uid: Yuko_Leggett
givenName: Yuko
mail: Yuko_Leggett@example.com
carLicense: SRJ5384
departmentNumber: 4270
employeeType: Contract
homePhone: +1 206 531-1475
initials: Y. L.
mobile: +1 408 580-9956
pager: +1 303 653-9114
manager: cn=Nanon Stiles
secretary: cn=Patch Muselik
roomNumber: 934

dn: cn=Marya Eakes, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marya Eakes
sn: Eakes
description: This is Marya Eakes's description
facsimileTelephoneNumber: +1 510 277-6974
l: Sunnyvale
ou: Product Development
postalAddress: example$Product Development$Dept # 385
telephoneNumber: +1 71 259-6949
title: Junior Product Development Sales Rep
userPassword: sekaEayraM
uid: Marya_Eakes
givenName: Marya
mail: Marya_Eakes@example.com
carLicense: D0GLPT0
departmentNumber: 1140
employeeType: Normal
homePhone: +1 804 428-1056
initials: M. E.
mobile: +1 818 140-7212
pager: +1 303 491-7423
manager: cn=Henry Anolik
secretary: cn=Atique Honbarrier
roomNumber: 3463

dn: cn=Pete Shames, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Pete Shames
sn: Shames
description: This is Pete Shames's description
facsimileTelephoneNumber: +1 213 876-4561
l: Armonk
ou: Accounting
postalAddress: example$Accounting$Dept # 175
telephoneNumber: +1 71 791-7119
title: Elite Accounting Evangelist
userPassword: semahSeteP
uid: Pete_Shames
givenName: Pete
mail: Pete_Shames@example.com
carLicense: S3AOJS2
departmentNumber: 2852
employeeType: Employee
homePhone: +1 510 160-8575
initials: P. S.
mobile: +1 804 489-5534
pager: +1 818 585-5593
manager: cn=Sibelle Wilczewski
secretary: cn=Franklyn Asghar
roomNumber: 1154

dn: cn=Carolan Rangel, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Carolan Rangel
sn: Rangel
description: This is Carolan Rangel's description
facsimileTelephoneNumber: +1 71 233-1472
l: Sunnyvale
ou: Peons
postalAddress: example$Peons$Dept # 494
telephoneNumber: +1 415 171-9753
title: Senior Peons Dictator
userPassword: legnaRnalo
uid: Carolan_Rangel
givenName: Carolan
mail: Carolan_Rangel@example.com
carLicense: XQIJM1O
departmentNumber: 2990
employeeType: Manager
homePhone: +1 804 259-7816
initials: C. R.
mobile: +1 303 907-6627
pager: +1 303 136-1379
manager: cn=Dyanne Feeley
secretary: cn=Kaycee Banigan
roomNumber: 5953

dn: cn=Rodrigo Howse, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rodrigo Howse
sn: Howse
description: This is Rodrigo Howse's description
facsimileTelephoneNumber: +1 818 502-5698
l: Santa Clara
ou: Management
postalAddress: example$Management$Dept # 401
telephoneNumber: +1 213 824-8436
title: Junior Management Madonna
userPassword: eswoHogird
uid: Rodrigo_Howse
givenName: Rodrigo
mail: Rodrigo_Howse@example.com
carLicense: OBWWW45
departmentNumber: 6277
employeeType: Normal
homePhone: +1 804 301-7119
initials: R. H.
mobile: +1 415 304-7146
pager: +1 804 601-8474
manager: cn=Shabbir Owen
secretary: cn=Helmuth Preville
roomNumber: 8577

dn: cn=Shandie Tomlinson, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shandie Tomlinson
sn: Tomlinson
description: This is Shandie Tomlinson's description
facsimileTelephoneNumber: +1 213 176-3399
l: Armonk
ou: Product Development
postalAddress: example$Product Development$Dept # 97
telephoneNumber: +1 71 155-7520
title: Elite Product Development Figurehead
userPassword: nosnilmoTe
uid: Shandie_Tomlinson
givenName: Shandie
mail: Shandie_Tomlinson@example.com
carLicense: B46G33M
departmentNumber: 4361
employeeType: Employee
homePhone: +1 303 684-3227
initials: S. T.
mobile: +1 71 567-8483
pager: +1 818 555-1567
manager: cn=Malethia Vermette
secretary: cn=Crysta Spriggs
roomNumber: 6587

dn: cn=Ashien Breslin, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ashien Breslin
sn: Breslin
description: This is Ashien Breslin's description
facsimileTelephoneNumber: +1 303 580-1145
l: Santa Clara
ou: Janitorial
postalAddress: example$Janitorial$Dept # 674
telephoneNumber: +1 71 204-9152
title: Master Janitorial Czar
userPassword: nilserBnei
uid: Ashien_Breslin
givenName: Ashien
mail: Ashien_Breslin@example.com
carLicense: HUSVSSW
departmentNumber: 2283
employeeType: Temp
homePhone: +1 510 549-9456
initials: A. B.
mobile: +1 213 401-7401
pager: +1 206 535-6074
manager: cn=Faunie Gascho
secretary: cn=Rebekkah Batchoun
roomNumber: 30

dn: cn=Eachelle Fabrizio, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Eachelle Fabrizio
sn: Fabrizio
description: This is Eachelle Fabrizio's description
facsimileTelephoneNumber: +1 206 659-6659
l: Armonk
ou: Janitorial
postalAddress: example$Janitorial$Dept # 742
telephoneNumber: +1 510 450-3176
title: Elite Janitorial Janitor
userPassword: oizirbaFel
uid: Eachelle_Fabrizio
givenName: Eachelle
mail: Eachelle_Fabrizio@example.com
carLicense: D4UM5QY
departmentNumber: 1508
employeeType: Temp
homePhone: +1 510 488-1339
initials: E. F.
mobile: +1 818 921-2770
pager: +1 303 104-8410
manager: cn=Stephana O'Grady
secretary: cn=Okan Hameed
roomNumber: 1302

dn: cn=Britni Cwirzen, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Britni Cwirzen
sn: Cwirzen
description: This is Britni Cwirzen's description
facsimileTelephoneNumber: +1 510 623-7079
l: Orem
ou: Janitorial
postalAddress: example$Janitorial$Dept # 600
telephoneNumber: +1 510 267-1155
title: Supreme Janitorial Assistant
userPassword: nezriwCint
uid: Britni_Cwirzen
givenName: Britni
mail: Britni_Cwirzen@example.com
carLicense: 1475JM0
departmentNumber: 7848
employeeType: Normal
homePhone: +1 71 158-4673
initials: B. C.
mobile: +1 510 865-3190
pager: +1 415 324-1704
manager: cn=Pooh Fastpack
secretary: cn=Edwin Hamori
roomNumber: 5726

dn: cn=Bertie MacNeill, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bertie MacNeill
sn: MacNeill
description: This is Bertie MacNeill's description
facsimileTelephoneNumber: +1 804 681-3840
l: Cupertino
ou: Product Testing
postalAddress: example$Product Testing$Dept # 121
telephoneNumber: +1 804 144-5078
title: Senior Product Testing Dictator
userPassword: llieNcaMei
uid: Bertie_MacNeill
givenName: Bertie
mail: Bertie_MacNeill@example.com
carLicense: Q6L3Y0O
departmentNumber: 5832
employeeType: Employee
homePhone: +1 303 902-6450
initials: B. M.
mobile: +1 818 400-6535
pager: +1 818 337-4870
manager: cn=Willamina Shafik
secretary: cn=Anatola Brockmann
roomNumber: 8713

dn: cn=Doreen Howse, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Doreen Howse
sn: Howse
description: This is Doreen Howse's description
facsimileTelephoneNumber: +1 206 426-2076
l: San Jose
ou: Payroll
postalAddress: example$Payroll$Dept # 477
telephoneNumber: +1 303 633-2753
title: Elite Payroll Visionary
userPassword: eswoHneero
uid: Doreen_Howse
givenName: Doreen
mail: Doreen_Howse@example.com
carLicense: PX33SPM
departmentNumber: 6990
employeeType: Normal
homePhone: +1 206 540-1747
initials: D. H.
mobile: +1 206 830-8753
pager: +1 71 489-3580
manager: cn=Didani Kovarik
secretary: cn=Horacio Kunkel
roomNumber: 6572

dn: cn=Emanuel Kryski, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Emanuel Kryski
sn: Kryski
description: This is Emanuel Kryski's description
facsimileTelephoneNumber: +1 818 575-1776
l: Sunnyvale
ou: Janitorial
postalAddress: example$Janitorial$Dept # 401
telephoneNumber: +1 206 302-8490
title: Supreme Janitorial Artist
userPassword: iksyrKleun
uid: Emanuel_Kryski
givenName: Emanuel
mail: Emanuel_Kryski@example.com
carLicense: YBDPZXD
departmentNumber: 3185
employeeType: Manager
homePhone: +1 408 402-7437
initials: E. K.
mobile: +1 804 539-3118
pager: +1 415 637-6578
manager: cn=Berna Hofstetter
secretary: cn=Ling-Zhong Ingersoll
roomNumber: 8680

dn: cn=Ingeberg Uecker, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ingeberg Uecker
sn: Uecker
description: This is Ingeberg Uecker's description
facsimileTelephoneNumber: +1 510 645-1887
l: Fremont
ou: Peons
postalAddress: example$Peons$Dept # 374
telephoneNumber: +1 213 264-6668
title: Elite Peons Developer
userPassword: rekceUgreb
uid: Ingeberg_Uecker
givenName: Ingeberg
mail: Ingeberg_Uecker@example.com
carLicense: KKROWG7
departmentNumber: 5843
employeeType: Normal
homePhone: +1 510 909-6303
initials: I. U.
mobile: +1 213 341-7760
pager: +1 408 926-7613
manager: cn=Sashenka Anderson
secretary: cn=Gail Momtahan
roomNumber: 7472

dn: cn=Lorine Kinch, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lorine Kinch
sn: Kinch
description: This is Lorine Kinch's description
facsimileTelephoneNumber: +1 408 260-7615
l: Cupertino
ou: Peons
postalAddress: example$Peons$Dept # 387
telephoneNumber: +1 303 471-6222
title: Supreme Peons Engineer
userPassword: hcniKeniro
uid: Lorine_Kinch
givenName: Lorine
mail: Lorine_Kinch@example.com
carLicense: EQ622AQ
departmentNumber: 6450
employeeType: Temp
homePhone: +1 818 643-2710
initials: L. K.
mobile: +1 415 508-5241
pager: +1 510 895-8472
manager: cn=Grace Dieu
secretary: cn=Micaela Naor
roomNumber: 4015

dn: cn=Tab Danko, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tab Danko
sn: Danko
description: This is Tab Danko's description
facsimileTelephoneNumber: +1 510 538-4771
l: Orem
ou: Administrative
postalAddress: example$Administrative$Dept # 669
telephoneNumber: +1 415 622-6699
title: Associate Administrative Yahoo
userPassword: oknaDbaT
uid: Tab_Danko
givenName: Tab
mail: Tab_Danko@example.com
carLicense: XB3M2XY
departmentNumber: 1320
employeeType: Employee
homePhone: +1 206 386-9666
initials: T. D.
mobile: +1 206 296-7053
pager: +1 804 647-2654
manager: cn=Ingaborg Burruss
secretary: cn=Hilde McCaugherty
roomNumber: 1597

dn: cn=Marlon Totino, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marlon Totino
sn: Totino
description: This is Marlon Totino's description
facsimileTelephoneNumber: +1 510 766-2749
l: Orem
ou: Human Resources
postalAddress: example$Human Resources$Dept # 299
telephoneNumber: +1 804 210-1623
title: Elite Human Resources Yahoo
userPassword: onitoTnolr
uid: Marlon_Totino
givenName: Marlon
mail: Marlon_Totino@example.com
carLicense: OZUM13H
departmentNumber: 726
employeeType: Employee
homePhone: +1 818 641-7683
initials: M. T.
mobile: +1 510 638-7831
pager: +1 206 109-7099
manager: cn=Nahum Xavier
secretary: cn=Patra Pancholy
roomNumber: 6781

dn: cn=Geri Clendening, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Geri Clendening
sn: Clendening
description: This is Geri Clendening's description
facsimileTelephoneNumber: +1 804 546-8102
l: Milpitas
ou: Planning
postalAddress: example$Planning$Dept # 9
telephoneNumber: +1 206 946-7570
title: Senior Planning Developer
userPassword: gninednelC
uid: Geri_Clendening
givenName: Geri
mail: Geri_Clendening@example.com
carLicense: YSRDKUW
departmentNumber: 626
employeeType: Temp
homePhone: +1 213 664-2462
initials: G. C.
mobile: +1 415 182-6899
pager: +1 213 706-7011
manager: cn=Pavla Sharman
secretary: cn=Flore Piecowye
roomNumber: 2093

dn: cn=Jozsef Cricker, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jozsef Cricker
sn: Cricker
description: This is Jozsef Cricker's description
facsimileTelephoneNumber: +1 213 738-2340
l: Mountain View
ou: Accounting
postalAddress: example$Accounting$Dept # 378
telephoneNumber: +1 71 289-1183
title: Junior Accounting Director
userPassword: rekcirCfes
uid: Jozsef_Cricker
givenName: Jozsef
mail: Jozsef_Cricker@example.com
carLicense: FMI2DBR
departmentNumber: 3888
employeeType: Employee
homePhone: +1 71 155-8180
initials: J. C.
mobile: +1 510 173-6447
pager: +1 303 395-8165
manager: cn=Jodi Planting
secretary: cn=Debi Kiernan
roomNumber: 7477

dn: cn=Brinn Cicci, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Brinn Cicci
sn: Cicci
description: This is Brinn Cicci's description
facsimileTelephoneNumber: +1 818 402-7322
l: Mountain View
ou: Administrative
postalAddress: example$Administrative$Dept # 18
telephoneNumber: +1 303 766-5895
title: Associate Administrative Manager
userPassword: icciCnnirB
uid: Brinn_Cicci
givenName: Brinn
mail: Brinn_Cicci@example.com
carLicense: SMMAQ8X
departmentNumber: 9613
employeeType: Temp
homePhone: +1 510 595-5078
initials: B. C.
mobile: +1 213 229-3849
pager: +1 415 826-5456
manager: cn=Millicent Majmudar
secretary: cn=Truus Kot
roomNumber: 3350

dn: cn=Indiana Hedrick, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Indiana Hedrick
sn: Hedrick
description: This is Indiana Hedrick's description
facsimileTelephoneNumber: +1 206 467-7225
l: Alameda
ou: Product Development
postalAddress: example$Product Development$Dept # 408
telephoneNumber: +1 510 743-2784
title: Supreme Product Development Janitor
userPassword: kcirdeHana
uid: Indiana_Hedrick
givenName: Indiana
mail: Indiana_Hedrick@example.com
carLicense: L1M55RO
departmentNumber: 9323
employeeType: Employee
homePhone: +1 206 485-1077
initials: I. H.
mobile: +1 510 717-7167
pager: +1 415 351-6790
manager: cn=Cosetta Canavan
secretary: cn=Cheuk Steeves
roomNumber: 9602

dn: cn=Shaw Karwowski, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shaw Karwowski
sn: Karwowski
description: This is Shaw Karwowski's description
facsimileTelephoneNumber: +1 510 254-1850
l: San Francisco
ou: Administrative
postalAddress: example$Administrative$Dept # 81
telephoneNumber: +1 818 961-8470
title: Master Administrative Writer
userPassword: ikswowraKw
uid: Shaw_Karwowski
givenName: Shaw
mail: Shaw_Karwowski@example.com
carLicense: WS0GFAI
departmentNumber: 7332
employeeType: Manager
homePhone: +1 415 429-2328
initials: S. K.
mobile: +1 408 215-2303
pager: +1 71 402-3905
manager: cn=Lanita Tarof
secretary: cn=Sabine Karr
roomNumber: 7633

dn: cn=Trish Hyman, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Trish Hyman
sn: Hyman
description: This is Trish Hyman's description
facsimileTelephoneNumber: +1 71 536-8500
l: San Francisco
ou: Administrative
postalAddress: example$Administrative$Dept # 476
telephoneNumber: +1 510 222-5063
title: Supreme Administrative Consultant
userPassword: namyHhsirT
uid: Trish_Hyman
givenName: Trish
mail: Trish_Hyman@example.com
carLicense: B7LG58P
departmentNumber: 5752
employeeType: Contract
homePhone: +1 510 735-3567
initials: T. H.
mobile: +1 213 727-1295
pager: +1 510 825-1188
manager: cn=Natalina Rowley
secretary: cn=Gaynor Bragado
roomNumber: 2395

dn: cn=Biddie Reith, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Biddie Reith
sn: Reith
description: This is Biddie Reith's description
facsimileTelephoneNumber: +1 303 575-3669
l: Menlo Park
ou: Administrative
postalAddress: example$Administrative$Dept # 378
telephoneNumber: +1 804 998-5469
title: Elite Administrative Yahoo
userPassword: htieReiddi
uid: Biddie_Reith
givenName: Biddie
mail: Biddie_Reith@example.com
carLicense: MMQ52N2
departmentNumber: 864
employeeType: Temp
homePhone: +1 71 387-4945
initials: B. R.
mobile: +1 213 580-4713
pager: +1 818 698-6851
manager: cn=Chester Balascak
secretary: cn=Mellisa Parsloe
roomNumber: 8220

dn: cn=Tracey Sgornikov, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tracey Sgornikov
sn: Sgornikov
description: This is Tracey Sgornikov's description
facsimileTelephoneNumber: +1 206 740-4930
l: Fremont
ou: Peons
postalAddress: example$Peons$Dept # 819
telephoneNumber: +1 71 706-6210
title: Chief Peons Vice President
userPassword: vokinrogSy
uid: Tracey_Sgornikov
givenName: Tracey
mail: Tracey_Sgornikov@example.com
carLicense: 27MTRHA
departmentNumber: 5277
employeeType: Temp
homePhone: +1 415 748-5665
initials: T. S.
mobile: +1 71 107-1408
pager: +1 213 116-5969
manager: cn=Sara-ann Livezey
secretary: cn=Kathi Alfred
roomNumber: 9674

dn: cn=Maggy Maduri, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Maggy Maduri
sn: Maduri
description: This is Maggy Maduri's description
facsimileTelephoneNumber: +1 213 447-2272
l: San Mateo
ou: Product Testing
postalAddress: example$Product Testing$Dept # 555
telephoneNumber: +1 818 180-2121
title: Master Product Testing Engineer
userPassword: irudaMygga
uid: Maggy_Maduri
givenName: Maggy
mail: Maggy_Maduri@example.com
carLicense: TNLLXG6
departmentNumber: 7262
employeeType: Contract
homePhone: +1 213 175-6512
initials: M. M.
mobile: +1 71 714-4757
pager: +1 415 289-2777
manager: cn=Sarajane Beauvais
secretary: cn=Hoa-Van Bowden
roomNumber: 1059

dn: cn=Rochell Reva, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rochell Reva
sn: Reva
description: This is Rochell Reva's description
facsimileTelephoneNumber: +1 213 365-3487
l: Emeryville
ou: Payroll
postalAddress: example$Payroll$Dept # 138
telephoneNumber: +1 206 483-6808
title: Associate Payroll Yahoo
userPassword: aveRllehco
uid: Rochell_Reva
givenName: Rochell
mail: Rochell_Reva@example.com
carLicense: SAOEA2M
departmentNumber: 8588
employeeType: Manager
homePhone: +1 408 867-6779
initials: R. R.
mobile: +1 510 753-5758
pager: +1 71 116-1834
manager: cn=Saied Kuehn
secretary: cn=Janenna Cloutier
roomNumber: 1439

dn: cn=Karilynn Hungle, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Karilynn Hungle
sn: Hungle
description: This is Karilynn Hungle's description
facsimileTelephoneNumber: +1 415 407-3988
l: Mountain View
ou: Janitorial
postalAddress: example$Janitorial$Dept # 230
telephoneNumber: +1 408 536-7299
title: Master Janitorial Pinhead
userPassword: elgnuHnnyl
uid: Karilynn_Hungle
givenName: Karilynn
mail: Karilynn_Hungle@example.com
carLicense: WU2Q8VC
departmentNumber: 7102
employeeType: Normal
homePhone: +1 408 806-2471
initials: K. H.
mobile: +1 213 619-4680
pager: +1 415 749-1643
manager: cn=Lolly Hamelin
secretary: cn=Marjorie Sidor
roomNumber: 6995

dn: cn=Delcine Aksel, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Delcine Aksel
sn: Aksel
description: This is Delcine Aksel's description
facsimileTelephoneNumber: +1 206 563-3474
l: Cambridge
ou: Administrative
postalAddress: example$Administrative$Dept # 394
telephoneNumber: +1 213 450-8763
title: Master Administrative Janitor
userPassword: leskAenicl
uid: Delcine_Aksel
givenName: Delcine
mail: Delcine_Aksel@example.com
carLicense: IMAH7I3
departmentNumber: 9679
employeeType: Contract
homePhone: +1 213 479-2206
initials: D. A.
mobile: +1 71 851-6931
pager: +1 213 406-9050
manager: cn=Dewi Hallman
secretary: cn=Grant Gentes
roomNumber: 5955

dn: cn=Hpone Skerry, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Hpone Skerry
sn: Skerry
description: This is Hpone Skerry's description
facsimileTelephoneNumber: +1 303 148-7675
l: Redmond
ou: Janitorial
postalAddress: example$Janitorial$Dept # 458
telephoneNumber: +1 71 568-4643
title: Elite Janitorial Stooge
userPassword: yrrekSenop
uid: Hpone_Skerry
givenName: Hpone
mail: Hpone_Skerry@example.com
carLicense: CZMYESU
departmentNumber: 5364
employeeType: Manager
homePhone: +1 408 118-6258
initials: H. S.
mobile: +1 804 904-7291
pager: +1 818 470-7449
manager: cn=Mary-Ellen Vempati
secretary: cn=Alvin Goetz
roomNumber: 9655

dn: cn=Tosca Belley, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tosca Belley
sn: Belley
description: This is Tosca Belley's description
facsimileTelephoneNumber: +1 303 222-4953
l: Santa Clara
ou: Management
postalAddress: example$Management$Dept # 491
telephoneNumber: +1 510 584-9011
title: Junior Management Sales Rep
userPassword: yelleBacso
uid: Tosca_Belley
givenName: Tosca
mail: Tosca_Belley@example.com
carLicense: ZQOM5MA
departmentNumber: 4909
employeeType: Employee
homePhone: +1 510 733-4242
initials: T. B.
mobile: +1 213 253-9601
pager: +1 71 445-2483
manager: cn=Radomir Kingan
secretary: cn=Livvy Zakarow
roomNumber: 2237

dn: cn=Andree Benefits, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Andree Benefits
sn: Benefits
description: This is Andree Benefits's description
facsimileTelephoneNumber: +1 818 420-5771
l: Sunnyvale
ou: Human Resources
postalAddress: example$Human Resources$Dept # 593
telephoneNumber: +1 818 366-7679
title: Supreme Human Resources Mascot
userPassword: stifeneBee
uid: Andree_Benefits
givenName: Andree
mail: Andree_Benefits@example.com
carLicense: 1J373OI
departmentNumber: 8939
employeeType: Manager
homePhone: +1 213 865-3912
initials: A. B.
mobile: +1 206 379-9531
pager: +1 408 680-8167
manager: cn=Wen-Kai Licandro
secretary: cn=Nick Goertzen
roomNumber: 3822

dn: cn=Joyous Vahdat, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Joyous Vahdat
sn: Vahdat
description: This is Joyous Vahdat's description
facsimileTelephoneNumber: +1 71 226-5208
l: San Francisco
ou: Accounting
postalAddress: example$Accounting$Dept # 451
telephoneNumber: +1 408 841-3211
title: Associate Accounting Mascot
userPassword: tadhaVsuoy
uid: Joyous_Vahdat
givenName: Joyous
mail: Joyous_Vahdat@example.com
carLicense: U13IIMB
departmentNumber: 6836
employeeType: Contract
homePhone: +1 213 709-2972
initials: J. V.
mobile: +1 71 641-5362
pager: +1 213 960-4165
manager: cn=Randie Bowler
secretary: cn=Leonida Plamondon
roomNumber: 4827

dn: cn=Hermione Cooke, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Hermione Cooke
sn: Cooke
description: This is Hermione Cooke's description
facsimileTelephoneNumber: +1 303 421-3165
l: Milpitas
ou: Product Testing
postalAddress: example$Product Testing$Dept # 80
telephoneNumber: +1 213 342-4314
title: Associate Product Testing Evangelist
userPassword: ekooCenoim
uid: Hermione_Cooke
givenName: Hermione
mail: Hermione_Cooke@example.com
carLicense: 4LFKTD3
departmentNumber: 4715
employeeType: Employee
homePhone: +1 408 756-8850
initials: H. C.
mobile: +1 71 866-4175
pager: +1 804 835-7472
manager: cn=Selma Han
secretary: cn=Geri Sandner
roomNumber: 9347

dn: cn=Adriana McFeely, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Adriana McFeely
sn: McFeely
description: This is Adriana McFeely's description
facsimileTelephoneNumber: +1 804 232-3810
l: San Francisco
ou: Peons
postalAddress: example$Peons$Dept # 443
telephoneNumber: +1 206 837-4130
title: Junior Peons Vice President
userPassword: yleeFcMana
uid: Adriana_McFeely
givenName: Adriana
mail: Adriana_McFeely@example.com
carLicense: XVRFQ03
departmentNumber: 5443
employeeType: Temp
homePhone: +1 818 186-2710
initials: A. M.
mobile: +1 804 152-9483
pager: +1 206 986-9151
manager: cn=Phuong Goyal
secretary: cn=Jordanna Frobel
roomNumber: 3399

dn: cn=Christal Haig, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Christal Haig
sn: Haig
description: This is Christal Haig's description
facsimileTelephoneNumber: +1 818 424-7571
l: San Mateo
ou: Management
postalAddress: example$Management$Dept # 21
telephoneNumber: +1 206 460-2357
title: Chief Management Vice President
userPassword: giaHlatsir
uid: Christal_Haig
givenName: Christal
mail: Christal_Haig@example.com
carLicense: TI8IBH4
departmentNumber: 6729
employeeType: Employee
homePhone: +1 818 763-6515
initials: C. H.
mobile: +1 415 481-3761
pager: +1 213 504-8977
manager: cn=Kerrill Tufford
secretary: cn=Phan Chacko
roomNumber: 72

dn: cn=Maitreya Kruger, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Maitreya Kruger
sn: Kruger
description: This is Maitreya Kruger's description
facsimileTelephoneNumber: +1 510 972-1213
l: Sunnyvale
ou: Planning
postalAddress: example$Planning$Dept # 318
telephoneNumber: +1 71 593-4533
title: Junior Planning Technician
userPassword: regurKayer
uid: Maitreya_Kruger
givenName: Maitreya
mail: Maitreya_Kruger@example.com
carLicense: XU80AAG
departmentNumber: 2132
employeeType: Manager
homePhone: +1 303 151-1291
initials: M. K.
mobile: +1 408 645-4317
pager: +1 206 893-3658
manager: cn=Russell Leima
secretary: cn=Maynard Blackshire
roomNumber: 7642

dn: cn=Laurie Mcshane, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Laurie Mcshane
sn: Mcshane
description: This is Laurie Mcshane's description
facsimileTelephoneNumber: +1 804 526-3497
l: Alameda
ou: Peons
postalAddress: example$Peons$Dept # 188
telephoneNumber: +1 206 401-5905
title: Junior Peons Director
userPassword: enahscMeir
uid: Laurie_Mcshane
givenName: Laurie
mail: Laurie_Mcshane@example.com
carLicense: MT9DAQW
departmentNumber: 3956
employeeType: Manager
homePhone: +1 206 171-3450
initials: L. M.
mobile: +1 804 732-9316
pager: +1 408 344-3526
manager: cn=Nadim Hagerty
secretary: cn=Dalenna Wagner
roomNumber: 9656

dn: cn=Lynda Mototsune, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lynda Mototsune
sn: Mototsune
description: This is Lynda Mototsune's description
facsimileTelephoneNumber: +1 415 376-8738
l: Fremont
ou: Planning
postalAddress: example$Planning$Dept # 894
telephoneNumber: +1 71 892-7223
title: Associate Planning Assistant
userPassword: enustotoMa
uid: Lynda_Mototsune
givenName: Lynda
mail: Lynda_Mototsune@example.com
carLicense: 979VNH2
departmentNumber: 8775
employeeType: Temp
homePhone: +1 303 995-8927
initials: L. M.
mobile: +1 206 129-4437
pager: +1 510 486-6142
manager: cn=Gokul Ludviksen
secretary: cn=Corina Hutt
roomNumber: 1940

dn: cn=Lucilia Krodel, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lucilia Krodel
sn: Krodel
description: This is Lucilia Krodel's description
facsimileTelephoneNumber: +1 206 202-8883
l: Redmond
ou: Janitorial
postalAddress: example$Janitorial$Dept # 895
telephoneNumber: +1 510 593-7649
title: Master Janitorial Architect
userPassword: ledorKaili
uid: Lucilia_Krodel
givenName: Lucilia
mail: Lucilia_Krodel@example.com
carLicense: F2XN6JO
departmentNumber: 765
employeeType: Normal
homePhone: +1 510 560-9551
initials: L. K.
mobile: +1 804 299-5573
pager: +1 804 157-1827
manager: cn=Virgie Samsonenko
secretary: cn=Nona Armenta
roomNumber: 299

dn: cn=Roger Fowlston, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Roger Fowlston
sn: Fowlston
description: This is Roger Fowlston's description
facsimileTelephoneNumber: +1 303 187-8025
l: Mountain View
ou: Management
postalAddress: example$Management$Dept # 112
telephoneNumber: +1 804 451-2104
title: Supreme Management Dictator
userPassword: notslwoFre
uid: Roger_Fowlston
givenName: Roger
mail: Roger_Fowlston@example.com
carLicense: 9Q99OAO
departmentNumber: 9007
employeeType: Contract
homePhone: +1 408 586-2902
initials: R. F.
mobile: +1 818 410-4661
pager: +1 510 790-6793
manager: cn=Natividad Hurd
secretary: cn=Deri Falquero
roomNumber: 3244

dn: cn=Kaminsky Dennen, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kaminsky Dennen
sn: Dennen
description: This is Kaminsky Dennen's description
facsimileTelephoneNumber: +1 415 344-9342
l: Cambridge
ou: Management
postalAddress: example$Management$Dept # 214
telephoneNumber: +1 415 174-6019
title: Master Management Vice President
userPassword: nenneDyksn
uid: Kaminsky_Dennen
givenName: Kaminsky
mail: Kaminsky_Dennen@example.com
carLicense: 1YRG7Z7
departmentNumber: 8616
employeeType: Manager
homePhone: +1 71 176-2078
initials: K. D.
mobile: +1 303 519-5191
pager: +1 213 184-1795
manager: cn=Giambattista Bottoms
secretary: cn=Ashley Rotenberg
roomNumber: 9959

dn: cn=Gilbert Vertolli, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gilbert Vertolli
sn: Vertolli
description: This is Gilbert Vertolli's description
facsimileTelephoneNumber: +1 303 866-9523
l: Mountain View
ou: Product Development
postalAddress: example$Product Development$Dept # 159
telephoneNumber: +1 71 913-8787
title: Junior Product Development President
userPassword: illotreVtr
uid: Gilbert_Vertolli
givenName: Gilbert
mail: Gilbert_Vertolli@example.com
carLicense: IJLNM0X
departmentNumber: 5590
employeeType: Contract
homePhone: +1 408 246-9967
initials: G. V.
mobile: +1 213 960-7422
pager: +1 206 427-5903
manager: cn=Marrissa Makohoniuk
secretary: cn=Coralie Stampfl
roomNumber: 8011

dn: cn=Pierrette Stern, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Pierrette Stern
sn: Stern
description: This is Pierrette Stern's description
facsimileTelephoneNumber: +1 510 153-7113
l: Palo Alto
ou: Management
postalAddress: example$Management$Dept # 743
telephoneNumber: +1 804 572-2868
title: Master Management Visionary
userPassword: nretSetter
uid: Pierrette_Stern
givenName: Pierrette
mail: Pierrette_Stern@example.com
carLicense: KGFVA7O
departmentNumber: 8957
employeeType: Normal
homePhone: +1 408 239-1926
initials: P. S.
mobile: +1 303 296-2747
pager: +1 804 168-7675
manager: cn=Noyes Hawes
secretary: cn=Anneliese Mattiussi
roomNumber: 2470

dn: cn=Marie-Nadine Finnighan, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marie-Nadine Finnighan
sn: Finnighan
description: This is Marie-Nadine Finnighan's description
facsimileTelephoneNumber: +1 804 785-6184
l: Cambridge
ou: Payroll
postalAddress: example$Payroll$Dept # 569
telephoneNumber: +1 804 727-5093
title: Elite Payroll Artist
userPassword: nahginniFe
uid: Marie-Nadine_Finnighan
givenName: Marie-Nadine
mail: Marie-Nadine_Finnighan@example.com
carLicense: G7UHEG9
departmentNumber: 9749
employeeType: Contract
homePhone: +1 206 551-2746
initials: M. F.
mobile: +1 71 171-6790
pager: +1 818 353-2532
manager: cn=Wai-Bun Wegrowicz
secretary: cn=Gaston Paton
roomNumber: 8909

dn: cn=Blinnie Maidenhead, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Blinnie Maidenhead
sn: Maidenhead
description: This is Blinnie Maidenhead's description
facsimileTelephoneNumber: +1 206 836-2712
l: San Jose
ou: Planning
postalAddress: example$Planning$Dept # 882
telephoneNumber: +1 408 950-9394
title: Master Planning Figurehead
userPassword: daehnediaM
uid: Blinnie_Maidenhead
givenName: Blinnie
mail: Blinnie_Maidenhead@example.com
carLicense: 5OGK27G
departmentNumber: 2256
employeeType: Temp
homePhone: +1 206 880-6960
initials: B. M.
mobile: +1 303 794-9864
pager: +1 510 571-1785
manager: cn=Ynes Agostino
secretary: cn=Kimberley Stodart
roomNumber: 6736

dn: cn=Dwain McCue, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dwain McCue
sn: McCue
description: This is Dwain McCue's description
facsimileTelephoneNumber: +1 71 905-4092
l: Redmond
ou: Administrative
postalAddress: example$Administrative$Dept # 177
telephoneNumber: +1 510 633-3803
title: Master Administrative Admin
userPassword: euCcMniawD
uid: Dwain_McCue
givenName: Dwain
mail: Dwain_McCue@example.com
carLicense: YIT3VTT
departmentNumber: 9685
employeeType: Normal
homePhone: +1 303 100-4605
initials: D. M.
mobile: +1 804 859-5183
pager: +1 818 871-2674
manager: cn=Nerissa Hosseini
secretary: cn=Chelsae Zug
roomNumber: 7909

dn: cn=Jennie Millspaugh, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jennie Millspaugh
sn: Millspaugh
description: This is Jennie Millspaugh's description
facsimileTelephoneNumber: +1 213 627-7126
l: San Jose
ou: Accounting
postalAddress: example$Accounting$Dept # 802
telephoneNumber: +1 818 624-9763
title: Elite Accounting Manager
userPassword: hguapslliM
uid: Jennie_Millspaugh
givenName: Jennie
mail: Jennie_Millspaugh@example.com
carLicense: P5FHZDG
departmentNumber: 5435
employeeType: Employee
homePhone: +1 804 351-9325
initials: J. M.
mobile: +1 206 399-8733
pager: +1 71 110-3934
manager: cn=Redgie Lojewski
secretary: cn=Ichiro Lindow
roomNumber: 9667

dn: cn=Annabel Planthara, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Annabel Planthara
sn: Planthara
description: This is Annabel Planthara's description
facsimileTelephoneNumber: +1 804 783-4207
l: Alameda
ou: Product Development
postalAddress: example$Product Development$Dept # 285
telephoneNumber: +1 804 691-1923
title: Master Product Development Assistant
userPassword: arahtnalPl
uid: Annabel_Planthara
givenName: Annabel
mail: Annabel_Planthara@example.com
carLicense: TIMAAPO
departmentNumber: 9779
employeeType: Normal
homePhone: +1 510 454-9254
initials: A. P.
mobile: +1 303 755-3875
pager: +1 818 581-8706
manager: cn=Haig De Leon
secretary: cn=Anett Krenn
roomNumber: 8145

dn: cn=Sarajane Shtivelman, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sarajane Shtivelman
sn: Shtivelman
description: This is Sarajane Shtivelman's description
facsimileTelephoneNumber: +1 804 134-3746
l: San Francisco
ou: Product Testing
postalAddress: example$Product Testing$Dept # 252
telephoneNumber: +1 71 144-2666
title: Senior Product Testing Technician
userPassword: namlevithS
uid: Sarajane_Shtivelman
givenName: Sarajane
mail: Sarajane_Shtivelman@example.com
carLicense: CWJ9V9X
departmentNumber: 8381
employeeType: Normal
homePhone: +1 510 670-2961
initials: S. S.
mobile: +1 415 326-4228
pager: +1 213 254-6548
manager: cn=Dee Nunez
secretary: cn=Marguerita Marshman
roomNumber: 6517

dn: cn=Matthew Kurolapnik, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Matthew Kurolapnik
sn: Kurolapnik
description: This is Matthew Kurolapnik's description
facsimileTelephoneNumber: +1 71 280-4567
l: Mountain View
ou: Planning
postalAddress: example$Planning$Dept # 614
telephoneNumber: +1 71 986-5573
title: Elite Planning Vice President
userPassword: kinpaloruK
uid: Matthew_Kurolapnik
givenName: Matthew
mail: Matthew_Kurolapnik@example.com
carLicense: 578455X
departmentNumber: 2947
employeeType: Employee
homePhone: +1 71 117-5562
initials: M. K.
mobile: +1 804 988-1132
pager: +1 303 423-4026
manager: cn=Dido Linke
secretary: cn=Lanie MacGregor
roomNumber: 1273

dn: cn=Ardra Boyer, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ardra Boyer
sn: Boyer
description: This is Ardra Boyer's description
facsimileTelephoneNumber: +1 303 930-9326
l: San Francisco
ou: Administrative
postalAddress: example$Administrative$Dept # 529
telephoneNumber: +1 510 467-7202
title: Chief Administrative Mascot
userPassword: reyoBardrA
uid: Ardra_Boyer
givenName: Ardra
mail: Ardra_Boyer@example.com
carLicense: U8MLFI5
departmentNumber: 8495
employeeType: Employee
homePhone: +1 804 158-9355
initials: A. B.
mobile: +1 804 234-5788
pager: +1 804 986-7673
manager: cn=Belle Daudin
secretary: cn=Aurelia Kee
roomNumber: 5432

dn: cn=Henrietta Litt, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Henrietta Litt
sn: Litt
description: This is Henrietta Litt's description
facsimileTelephoneNumber: +1 408 621-6067
l: Sunnyvale
ou: Product Development
postalAddress: example$Product Development$Dept # 201
telephoneNumber: +1 213 699-1133
title: Supreme Product Development Janitor
userPassword: ttiLatteir
uid: Henrietta_Litt
givenName: Henrietta
mail: Henrietta_Litt@example.com
carLicense: 6BO1UC0
departmentNumber: 122
employeeType: Manager
homePhone: +1 303 433-6757
initials: H. L.
mobile: +1 303 715-4558
pager: +1 408 110-4443
manager: cn=Julianna Pipkins
secretary: cn=Addie Groetsema
roomNumber: 8776

dn: cn=Ashli Gowan, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ashli Gowan
sn: Gowan
description: This is Ashli Gowan's description
facsimileTelephoneNumber: +1 71 435-8571
l: Fremont
ou: Peons
postalAddress: example$Peons$Dept # 639
telephoneNumber: +1 415 874-4530
title: Senior Peons Grunt
userPassword: nawoGilhsA
uid: Ashli_Gowan
givenName: Ashli
mail: Ashli_Gowan@example.com
carLicense: 9GZEW1O
departmentNumber: 9043
employeeType: Temp
homePhone: +1 510 988-6319
initials: A. G.
mobile: +1 804 813-6018
pager: +1 408 169-7798
manager: cn=Nissie Hr
secretary: cn=Fitzroy Sanche
roomNumber: 4186

dn: cn=Rosa Haren, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rosa Haren
sn: Haren
description: This is Rosa Haren's description
facsimileTelephoneNumber: +1 804 627-5203
l: Mountain View
ou: Human Resources
postalAddress: example$Human Resources$Dept # 444
telephoneNumber: +1 206 874-2444
title: Supreme Human Resources Consultant
userPassword: neraHasoR
uid: Rosa_Haren
givenName: Rosa
mail: Rosa_Haren@example.com
carLicense: ZIAZ88H
departmentNumber: 6486
employeeType: Manager
homePhone: +1 408 915-8966
initials: R. H.
mobile: +1 206 648-1469
pager: +1 408 520-3881
manager: cn=Birendra Soong
secretary: cn=Tineke Kashani-nia
roomNumber: 3424

dn: cn=Sherryl Hubley, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sherryl Hubley
sn: Hubley
description: This is Sherryl Hubley's description
facsimileTelephoneNumber: +1 415 345-9742
l: Palo Alto
ou: Product Development
postalAddress: example$Product Development$Dept # 301
telephoneNumber: +1 71 513-4816
title: Supreme Product Development Manager
userPassword: yelbuHlyrr
uid: Sherryl_Hubley
givenName: Sherryl
mail: Sherryl_Hubley@example.com
carLicense: QDMPWW5
departmentNumber: 1350
employeeType: Employee
homePhone: +1 303 577-4676
initials: S. H.
mobile: +1 303 298-5535
pager: +1 408 158-6998
manager: cn=Pauly Marum
secretary: cn=Calypso Jaques
roomNumber: 9962

dn: cn=Evy McCartin, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Evy McCartin
sn: McCartin
description: This is Evy McCartin's description
facsimileTelephoneNumber: +1 303 501-6790
l: Mountain View
ou: Administrative
postalAddress: example$Administrative$Dept # 684
telephoneNumber: +1 818 269-1491
title: Associate Administrative Consultant
userPassword: nitraCcMyv
uid: Evy_McCartin
givenName: Evy
mail: Evy_McCartin@example.com
carLicense: MX13PEC
departmentNumber: 7744
employeeType: Normal
homePhone: +1 303 594-3087
initials: E. M.
mobile: +1 303 677-6050
pager: +1 71 977-4179
manager: cn=Alaine Noris
secretary: cn=Tonya Tupas
roomNumber: 9770

dn: cn=Vikki Loker, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Vikki Loker
sn: Loker
description: This is Vikki Loker's description
facsimileTelephoneNumber: +1 303 803-1484
l: Menlo Park
ou: Product Development
postalAddress: example$Product Development$Dept # 392
telephoneNumber: +1 415 484-9793
title: Associate Product Development Artist
userPassword: rekoLikkiV
uid: Vikki_Loker
givenName: Vikki
mail: Vikki_Loker@example.com
carLicense: RPETBKN
departmentNumber: 1086
employeeType: Contract
homePhone: +1 804 481-7397
initials: V. L.
mobile: +1 206 438-5732
pager: +1 303 527-1871
manager: cn=Iwan Kuo
secretary: cn=Mersey Toulson
roomNumber: 1350

dn: cn=Gillan Enns, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gillan Enns
sn: Enns
description: This is Gillan Enns's description
facsimileTelephoneNumber: +1 415 945-7529
l: Redwood Shores
ou: Product Testing
postalAddress: example$Product Testing$Dept # 109
telephoneNumber: +1 510 942-9016
title: Supreme Product Testing Admin
userPassword: snnEnalliG
uid: Gillan_Enns
givenName: Gillan
mail: Gillan_Enns@example.com
carLicense: Z1X1R73
departmentNumber: 5104
employeeType: Contract
homePhone: +1 804 698-3875
initials: G. E.
mobile: +1 71 899-7262
pager: +1 818 291-8114
manager: cn=Ivo Chong
secretary: cn=Gerrard Brownlie
roomNumber: 8775

dn: cn=Abigale Buggie, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Abigale Buggie
sn: Buggie
description: This is Abigale Buggie's description
facsimileTelephoneNumber: +1 206 340-5088
l: Fremont
ou: Payroll
postalAddress: example$Payroll$Dept # 6
telephoneNumber: +1 206 119-9817
title: Elite Payroll Consultant
userPassword: eigguBelag
uid: Abigale_Buggie
givenName: Abigale
mail: Abigale_Buggie@example.com
carLicense: 5N4DTWL
departmentNumber: 4104
employeeType: Contract
homePhone: +1 206 749-8565
initials: A. B.
mobile: +1 510 883-2759
pager: +1 415 287-1338
manager: cn=Larysa Dikens
secretary: cn=Shae Schmadtke
roomNumber: 9324

dn: cn=Salomi Marconi, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Salomi Marconi
sn: Marconi
description: This is Salomi Marconi's description
facsimileTelephoneNumber: +1 818 129-1241
l: Milpitas
ou: Administrative
postalAddress: example$Administrative$Dept # 36
telephoneNumber: +1 818 196-4698
title: Supreme Administrative Figurehead
userPassword: inocraMimo
uid: Salomi_Marconi
givenName: Salomi
mail: Salomi_Marconi@example.com
carLicense: 35YLW6D
departmentNumber: 8455
employeeType: Manager
homePhone: +1 213 223-7980
initials: S. M.
mobile: +1 415 127-5127
pager: +1 303 832-4116
manager: cn=Cicily Meridew
secretary: cn=Lincoln Mathis
roomNumber: 219

dn: cn=Lan Satta, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lan Satta
sn: Satta
description: This is Lan Satta's description
facsimileTelephoneNumber: +1 818 933-5138
l: San Mateo
ou: Planning
postalAddress: example$Planning$Dept # 978
telephoneNumber: +1 303 394-9267
title: Supreme Planning Director
userPassword: attaSnaL
uid: Lan_Satta
givenName: Lan
mail: Lan_Satta@example.com
carLicense: H4OVY9V
departmentNumber: 5965
employeeType: Manager
homePhone: +1 303 427-9685
initials: L. S.
mobile: +1 804 744-6028
pager: +1 213 859-6999
manager: cn=Keep Nerem
secretary: cn=Silvestro Nordstrom
roomNumber: 4454

dn: cn=Millicent Kirn, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Millicent Kirn
sn: Kirn
description: This is Millicent Kirn's description
facsimileTelephoneNumber: +1 213 354-4566
l: Palo Alto
ou: Administrative
postalAddress: example$Administrative$Dept # 851
telephoneNumber: +1 71 635-7932
title: Elite Administrative Dictator
userPassword: nriKtnecil
uid: Millicent_Kirn
givenName: Millicent
mail: Millicent_Kirn@example.com
carLicense: 2853Z9D
departmentNumber: 7301
employeeType: Temp
homePhone: +1 71 807-9201
initials: M. K.
mobile: +1 303 730-7705
pager: +1 415 180-3311
manager: cn=Martita Murash
secretary: cn=Nissy Kunecke
roomNumber: 6255

dn: cn=Kelley Cooney, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kelley Cooney
sn: Cooney
description: This is Kelley Cooney's description
facsimileTelephoneNumber: +1 818 408-9273
l: Alameda
ou: Administrative
postalAddress: example$Administrative$Dept # 749
telephoneNumber: +1 71 842-3720
title: Senior Administrative Warrior
userPassword: yenooCyell
uid: Kelley_Cooney
givenName: Kelley
mail: Kelley_Cooney@example.com
carLicense: 2H40R3M
departmentNumber: 396
employeeType: Contract
homePhone: +1 71 925-7895
initials: K. C.
mobile: +1 303 526-7796
pager: +1 206 176-1833
manager: cn=Frayda Eteminan
secretary: cn=Cecilia McNerlan
roomNumber: 4888

dn: cn=Parveen Eller, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Parveen Eller
sn: Eller
description: This is Parveen Eller's description
facsimileTelephoneNumber: +1 510 114-4717
l: Emeryville
ou: Administrative
postalAddress: example$Administrative$Dept # 794
telephoneNumber: +1 71 103-8525
title: Associate Administrative Mascot
userPassword: rellEneevr
uid: Parveen_Eller
givenName: Parveen
mail: Parveen_Eller@example.com
carLicense: OJOIV49
departmentNumber: 2052
employeeType: Manager
homePhone: +1 71 229-8361
initials: P. E.
mobile: +1 415 234-3951
pager: +1 71 377-4448
manager: cn=StClair Copeman
secretary: cn=Hanni Syrett
roomNumber: 8936

dn: cn=Otha Meyerink, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Otha Meyerink
sn: Meyerink
description: This is Otha Meyerink's description
facsimileTelephoneNumber: +1 510 897-1978
l: Menlo Park
ou: Administrative
postalAddress: example$Administrative$Dept # 364
telephoneNumber: +1 408 923-1988
title: Master Administrative Visionary
userPassword: knireyeMah
uid: Otha_Meyerink
givenName: Otha
mail: Otha_Meyerink@example.com
carLicense: ITY2RL7
departmentNumber: 4075
employeeType: Contract
homePhone: +1 804 593-2750
initials: O. M.
mobile: +1 71 479-5687
pager: +1 818 421-2877
manager: cn=Farzad Skopliak
secretary: cn=Fairy Dunnett
roomNumber: 7326

dn: cn=Node Tran, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Node Tran
sn: Tran
description: This is Node Tran's description
facsimileTelephoneNumber: +1 408 902-1433
l: Redwood Shores
ou: Product Development
postalAddress: example$Product Development$Dept # 303
telephoneNumber: +1 415 415-5826
title: Junior Product Development Grunt
userPassword: narTedoN
uid: Node_Tran
givenName: Node
mail: Node_Tran@example.com
carLicense: SFRU6LI
departmentNumber: 7978
employeeType: Temp
homePhone: +1 71 941-3886
initials: N. T.
mobile: +1 213 931-1271
pager: +1 818 585-6809
manager: cn=Mahmoud Moxley
secretary: cn=Byron Tomasetti
roomNumber: 2684

dn: cn=Ranique Eansor, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ranique Eansor
sn: Eansor
description: This is Ranique Eansor's description
facsimileTelephoneNumber: +1 804 217-2312
l: Armonk
ou: Administrative
postalAddress: example$Administrative$Dept # 391
telephoneNumber: +1 206 605-5168
title: Senior Administrative Assistant
userPassword: rosnaEeuqi
uid: Ranique_Eansor
givenName: Ranique
mail: Ranique_Eansor@example.com
carLicense: 68GVLF9
departmentNumber: 7152
employeeType: Normal
homePhone: +1 804 490-8997
initials: R. E.
mobile: +1 303 714-5366
pager: +1 213 696-9564
manager: cn=Modestia Loudiadis
secretary: cn=Svenn-Erik Pols
roomNumber: 5162

dn: cn=Chesteen Zhong, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Chesteen Zhong
sn: Zhong
description: This is Chesteen Zhong's description
facsimileTelephoneNumber: +1 804 314-7419
l: San Mateo
ou: Product Development
postalAddress: example$Product Development$Dept # 643
telephoneNumber: +1 510 427-3733
title: Senior Product Development Writer
userPassword: gnohZneets
uid: Chesteen_Zhong
givenName: Chesteen
mail: Chesteen_Zhong@example.com
carLicense: 97VKVD5
departmentNumber: 4398
employeeType: Employee
homePhone: +1 510 910-2117
initials: C. Z.
mobile: +1 510 641-8237
pager: +1 415 337-9215
manager: cn=Xuong Reynolds
secretary: cn=Ira Ghossein
roomNumber: 8094

dn: cn=Davis Debord, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Davis Debord
sn: Debord
description: This is Davis Debord's description
facsimileTelephoneNumber: +1 408 323-5635
l: Armonk
ou: Product Development
postalAddress: example$Product Development$Dept # 149
telephoneNumber: +1 818 957-1715
title: Associate Product Development Admin
userPassword: drobeDsiva
uid: Davis_Debord
givenName: Davis
mail: Davis_Debord@example.com
carLicense: LDK4N4L
departmentNumber: 9531
employeeType: Normal
homePhone: +1 303 244-8371
initials: D. D.
mobile: +1 408 909-7766
pager: +1 415 378-5985
manager: cn=Hiroshi Cescon
secretary: cn=Dutch Musca
roomNumber: 7850

dn: cn=Selma Tilton, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Selma Tilton
sn: Tilton
description: This is Selma Tilton's description
facsimileTelephoneNumber: +1 415 920-1214
l: Santa Clara
ou: Management
postalAddress: example$Management$Dept # 938
telephoneNumber: +1 415 792-8542
title: Senior Management Assistant
userPassword: notliTamle
uid: Selma_Tilton
givenName: Selma
mail: Selma_Tilton@example.com
carLicense: FJVVNUV
departmentNumber: 6062
employeeType: Normal
homePhone: +1 510 189-7668
initials: S. T.
mobile: +1 415 680-3984
pager: +1 415 203-7578
manager: cn=Jeri Kotval
secretary: cn=Penelope Clenney
roomNumber: 5030

dn: cn=Annabell Id, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Annabell Id
sn: Id
description: This is Annabell Id's description
facsimileTelephoneNumber: +1 71 508-7128
l: San Jose
ou: Human Resources
postalAddress: example$Human Resources$Dept # 148
telephoneNumber: +1 510 235-2601
title: Associate Human Resources Manager
userPassword: dIllebannA
uid: Annabell_Id
givenName: Annabell
mail: Annabell_Id@example.com
carLicense: YO9DVM1
departmentNumber: 874
employeeType: Normal
homePhone: +1 213 225-6368
initials: A. I.
mobile: +1 408 603-7170
pager: +1 206 892-2838
manager: cn=Bunny Wealch
secretary: cn=John-Paul Kumamoto
roomNumber: 7214

dn: cn=May Rhodenizer, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: May Rhodenizer
sn: Rhodenizer
description: This is May Rhodenizer's description
facsimileTelephoneNumber: +1 408 683-5414
l: Palo Alto
ou: Product Testing
postalAddress: example$Product Testing$Dept # 297
telephoneNumber: +1 71 232-2613
title: Elite Product Testing Writer
userPassword: rezinedohR
uid: May_Rhodenizer
givenName: May
mail: May_Rhodenizer@example.com
carLicense: U2V5OOU
departmentNumber: 6124
employeeType: Manager
homePhone: +1 213 185-3444
initials: M. R.
mobile: +1 71 203-1683
pager: +1 206 557-9126
manager: cn=Serge Michaels
secretary: cn=Kerrill Noone
roomNumber: 9338

dn: cn=Juline Dolezal, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Juline Dolezal
sn: Dolezal
description: This is Juline Dolezal's description
facsimileTelephoneNumber: +1 818 920-7493
l: Redmond
ou: Accounting
postalAddress: example$Accounting$Dept # 423
telephoneNumber: +1 415 302-8192
title: Master Accounting Pinhead
userPassword: lazeloDeni
uid: Juline_Dolezal
givenName: Juline
mail: Juline_Dolezal@example.com
carLicense: GSVXNVK
departmentNumber: 6333
employeeType: Contract
homePhone: +1 804 551-1568
initials: J. D.
mobile: +1 818 745-9995
pager: +1 415 654-3636
manager: cn=Delisle Lacelle
secretary: cn=Godfrey Dikaitis
roomNumber: 3440

dn: cn=Katrinka Gehm, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Katrinka Gehm
sn: Gehm
description: This is Katrinka Gehm's description
facsimileTelephoneNumber: +1 303 506-6410
l: San Mateo
ou: Janitorial
postalAddress: example$Janitorial$Dept # 476
telephoneNumber: +1 213 166-4215
title: Supreme Janitorial Accountant
userPassword: mheGaknirt
uid: Katrinka_Gehm
givenName: Katrinka
mail: Katrinka_Gehm@example.com
carLicense: C8BJ9L5
departmentNumber: 4205
employeeType: Employee
homePhone: +1 213 721-7282
initials: K. G.
mobile: +1 415 203-8518
pager: +1 303 140-8063
manager: cn=Breanne Mayer
secretary: cn=Marja Temp
roomNumber: 7493

dn: cn=Jill Izbinsky, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jill Izbinsky
sn: Izbinsky
description: This is Jill Izbinsky's description
facsimileTelephoneNumber: +1 804 124-8196
l: Cupertino
ou: Peons
postalAddress: example$Peons$Dept # 949
telephoneNumber: +1 818 196-3978
title: Supreme Peons Visionary
userPassword: yksnibzIll
uid: Jill_Izbinsky
givenName: Jill
mail: Jill_Izbinsky@example.com
carLicense: FKQUUDE
departmentNumber: 8916
employeeType: Normal
homePhone: +1 206 334-6701
initials: J. I.
mobile: +1 415 185-1127
pager: +1 510 325-5233
manager: cn=Thia Feith
secretary: cn=Petrina Nardiello
roomNumber: 9881

dn: cn=Li Avellaneda, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Li Avellaneda
sn: Avellaneda
description: This is Li Avellaneda's description
facsimileTelephoneNumber: +1 206 807-3164
l: Cambridge
ou: Product Testing
postalAddress: example$Product Testing$Dept # 198
telephoneNumber: +1 303 748-5498
title: Senior Product Testing Mascot
userPassword: adenallevA
uid: Li_Avellaneda
givenName: Li
mail: Li_Avellaneda@example.com
carLicense: OKU2I9B
departmentNumber: 2635
employeeType: Employee
homePhone: +1 213 593-9363
initials: L. A.
mobile: +1 510 403-2172
pager: +1 303 730-2192
manager: cn=Tara Jone
secretary: cn=Heloise Kazmierczak
roomNumber: 5823

dn: cn=Dianemarie Laughridge, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dianemarie Laughridge
sn: Laughridge
description: This is Dianemarie Laughridge's description
facsimileTelephoneNumber: +1 818 504-5158
l: Sunnyvale
ou: Human Resources
postalAddress: example$Human Resources$Dept # 816
telephoneNumber: +1 804 304-8130
title: Junior Human Resources Sales Rep
userPassword: egdirhguaL
uid: Dianemarie_Laughridge
givenName: Dianemarie
mail: Dianemarie_Laughridge@example.com
carLicense: NYKDGE8
departmentNumber: 1855
employeeType: Employee
homePhone: +1 213 616-7746
initials: D. L.
mobile: +1 510 674-3046
pager: +1 213 989-5209
manager: cn=Buford Eperjesy
secretary: cn=Trudey Watson
roomNumber: 5014

dn: cn=Mohammad Astor, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mohammad Astor
sn: Astor
description: This is Mohammad Astor's description
facsimileTelephoneNumber: +1 818 653-9826
l: San Mateo
ou: Human Resources
postalAddress: example$Human Resources$Dept # 824
telephoneNumber: +1 510 222-2291
title: Associate Human Resources Manager
userPassword: rotsAdamma
uid: Mohammad_Astor
givenName: Mohammad
mail: Mohammad_Astor@example.com
carLicense: N3MF78B
departmentNumber: 2775
employeeType: Normal
homePhone: +1 71 602-3440
initials: M. A.
mobile: +1 213 816-6357
pager: +1 804 460-3565
manager: cn=Didani Hawthorne
secretary: cn=Dinah Lilleniit
roomNumber: 4353

dn: cn=Rudy Mansouri, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rudy Mansouri
sn: Mansouri
description: This is Rudy Mansouri's description
facsimileTelephoneNumber: +1 303 506-6510
l: Cupertino
ou: Payroll
postalAddress: example$Payroll$Dept # 800
telephoneNumber: +1 408 845-5262
title: Supreme Payroll Janitor
userPassword: iruosnaMyd
uid: Rudy_Mansouri
givenName: Rudy
mail: Rudy_Mansouri@example.com
carLicense: UZQNBHL
departmentNumber: 9176
employeeType: Manager
homePhone: +1 408 617-2198
initials: R. M.
mobile: +1 303 165-9565
pager: +1 804 778-4649
manager: cn=Audy Bossett
secretary: cn=Susil Hartsell
roomNumber: 5926

dn: cn=Gursharan Rangaswami, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gursharan Rangaswami
sn: Rangaswami
description: This is Gursharan Rangaswami's description
facsimileTelephoneNumber: +1 510 992-4238
l: Fremont
ou: Payroll
postalAddress: example$Payroll$Dept # 358
telephoneNumber: +1 206 648-5076
title: Master Payroll Accountant
userPassword: imawsagnaR
uid: Gursharan_Rangaswami
givenName: Gursharan
mail: Gursharan_Rangaswami@example.com
carLicense: UQQ60LT
departmentNumber: 5115
employeeType: Employee
homePhone: +1 804 505-2501
initials: G. R.
mobile: +1 206 748-3690
pager: +1 415 929-3504
manager: cn=Santiago Santella
secretary: cn=Sissy Crowe
roomNumber: 8604

dn: cn=Jojo Menechian, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jojo Menechian
sn: Menechian
description: This is Jojo Menechian's description
facsimileTelephoneNumber: +1 206 448-9037
l: Menlo Park
ou: Accounting
postalAddress: example$Accounting$Dept # 717
telephoneNumber: +1 213 578-3384
title: Senior Accounting President
userPassword: naihceneMo
uid: Jojo_Menechian
givenName: Jojo
mail: Jojo_Menechian@example.com
carLicense: D923FG7
departmentNumber: 8600
employeeType: Normal
homePhone: +1 206 468-6675
initials: J. M.
mobile: +1 408 154-6743
pager: +1 415 885-5555
manager: cn=Kinman Hamlin
secretary: cn=Siouxie Valko
roomNumber: 4150

dn: cn=Subhash Petrick, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Subhash Petrick
sn: Petrick
description: This is Subhash Petrick's description
facsimileTelephoneNumber: +1 818 509-7242
l: Redmond
ou: Management
postalAddress: example$Management$Dept # 667
telephoneNumber: +1 206 187-1104
title: Chief Management Janitor
userPassword: kcirtePhsa
uid: Subhash_Petrick
givenName: Subhash
mail: Subhash_Petrick@example.com
carLicense: BKBKQWP
departmentNumber: 5296
employeeType: Temp
homePhone: +1 415 436-8437
initials: S. P.
mobile: +1 804 410-9723
pager: +1 408 653-1269
manager: cn=Lian-Hong McMurray
secretary: cn=Fausto Averett
roomNumber: 4509

dn: cn=Yoshi Figura, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Yoshi Figura
sn: Figura
description: This is Yoshi Figura's description
facsimileTelephoneNumber: +1 213 632-5838
l: Orem
ou: Payroll
postalAddress: example$Payroll$Dept # 839
telephoneNumber: +1 71 254-7900
title: Supreme Payroll Vice President
userPassword: arugiFihso
uid: Yoshi_Figura
givenName: Yoshi
mail: Yoshi_Figura@example.com
carLicense: IFG2GJE
departmentNumber: 6641
employeeType: Manager
homePhone: +1 804 629-7643
initials: Y. F.
mobile: +1 510 779-5227
pager: +1 71 858-2055
manager: cn=Vino Gruszczynski
secretary: cn=Nj Kyoung
roomNumber: 5364

dn: cn=Furrukh Efstration, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Furrukh Efstration
sn: Efstration
description: This is Furrukh Efstration's description
facsimileTelephoneNumber: +1 510 172-2272
l: Emeryville
ou: Peons
postalAddress: example$Peons$Dept # 210
telephoneNumber: +1 510 119-6867
title: Supreme Peons Director
userPassword: noitartsfE
uid: Furrukh_Efstration
givenName: Furrukh
mail: Furrukh_Efstration@example.com
carLicense: BOM8Y8Q
departmentNumber: 7218
employeeType: Contract
homePhone: +1 415 349-6730
initials: F. E.
mobile: +1 510 163-8441
pager: +1 303 582-9024
manager: cn=Tarrah Gorman
secretary: cn=Kynthia Luke
roomNumber: 820

dn: cn=Emelda Cutrufello, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Emelda Cutrufello
sn: Cutrufello
description: This is Emelda Cutrufello's description
facsimileTelephoneNumber: +1 206 288-8759
l: Santa Clara
ou: Peons
postalAddress: example$Peons$Dept # 114
telephoneNumber: +1 71 914-5291
title: Master Peons Architect
userPassword: ollefurtuC
uid: Emelda_Cutrufello
givenName: Emelda
mail: Emelda_Cutrufello@example.com
carLicense: P28KQCO
departmentNumber: 148
employeeType: Normal
homePhone: +1 206 548-7516
initials: E. C.
mobile: +1 408 157-7825
pager: +1 818 767-2029
manager: cn=Robina Sudbey
secretary: cn=Gigi Nettles
roomNumber: 3117

dn: cn=Mahmut Seagle, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mahmut Seagle
sn: Seagle
description: This is Mahmut Seagle's description
facsimileTelephoneNumber: +1 510 369-5702
l: Palo Alto
ou: Product Testing
postalAddress: example$Product Testing$Dept # 282
telephoneNumber: +1 213 380-4082
title: Master Product Testing Developer
userPassword: elgaeStumh
uid: Mahmut_Seagle
givenName: Mahmut
mail: Mahmut_Seagle@example.com
carLicense: TPPCCFM
departmentNumber: 7551
employeeType: Normal
homePhone: +1 408 441-8718
initials: M. S.
mobile: +1 213 886-5489
pager: +1 804 586-5495
manager: cn=Florina Fuqua
secretary: cn=Gerladina Peckel
roomNumber: 8669

dn: cn=Witold Mayfield, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Witold Mayfield
sn: Mayfield
description: This is Witold Mayfield's description
facsimileTelephoneNumber: +1 408 951-8728
l: Redmond
ou: Product Testing
postalAddress: example$Product Testing$Dept # 334
telephoneNumber: +1 415 238-4018
title: Junior Product Testing Engineer
userPassword: dleifyaMdl
uid: Witold_Mayfield
givenName: Witold
mail: Witold_Mayfield@example.com
carLicense: 4BBQR74
departmentNumber: 224
employeeType: Employee
homePhone: +1 71 811-1199
initials: W. M.
mobile: +1 818 417-5436
pager: +1 303 928-5168
manager: cn=Camellia Jalilvand
secretary: cn=Lelah Hedman
roomNumber: 7119

dn: cn=Calvin Austin, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Calvin Austin
sn: Austin
description: This is Calvin Austin's description
facsimileTelephoneNumber: +1 804 725-6802
l: Cambridge
ou: Planning
postalAddress: example$Planning$Dept # 794
telephoneNumber: +1 303 681-3693
title: Chief Planning Evangelist
userPassword: nitsuAnivl
uid: Calvin_Austin
givenName: Calvin
mail: Calvin_Austin@example.com
carLicense: 3AQMUIF
departmentNumber: 7123
employeeType: Normal
homePhone: +1 303 186-9756
initials: C. A.
mobile: +1 213 867-7124
pager: +1 206 334-8708
manager: cn=Bedford Knecht
secretary: cn=Dimitrios Blethen
roomNumber: 8104

dn: cn=Rea Somani, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rea Somani
sn: Somani
description: This is Rea Somani's description
facsimileTelephoneNumber: +1 510 531-2640
l: Menlo Park
ou: Peons
postalAddress: example$Peons$Dept # 40
telephoneNumber: +1 804 816-1641
title: Elite Peons Warrior
userPassword: inamoSaeR
uid: Rea_Somani
givenName: Rea
mail: Rea_Somani@example.com
carLicense: TJ7SV9H
departmentNumber: 2243
employeeType: Normal
homePhone: +1 818 223-6631
initials: R. S.
mobile: +1 303 820-9339
pager: +1 510 857-4120
manager: cn=Swee-Joo Thirugnanam
secretary: cn=Basheer Ballou
roomNumber: 2461

dn: cn=Zilvia Welch, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zilvia Welch
sn: Welch
description: This is Zilvia Welch's description
facsimileTelephoneNumber: +1 408 485-5775
l: Alameda
ou: Product Development
postalAddress: example$Product Development$Dept # 683
telephoneNumber: +1 510 860-2192
title: Chief Product Development Madonna
userPassword: hcleWaivli
uid: Zilvia_Welch
givenName: Zilvia
mail: Zilvia_Welch@example.com
carLicense: 8RSUTQX
departmentNumber: 7527
employeeType: Manager
homePhone: +1 510 946-1265
initials: Z. W.
mobile: +1 415 405-2811
pager: +1 213 804-8281
manager: cn=Shellie Chappuis
secretary: cn=Randhir Avellaneda
roomNumber: 5097

dn: cn=Joshi Foest, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Joshi Foest
sn: Foest
description: This is Joshi Foest's description
facsimileTelephoneNumber: +1 408 177-6240
l: Menlo Park
ou: Product Development
postalAddress: example$Product Development$Dept # 322
telephoneNumber: +1 818 203-5533
title: Chief Product Development Accountant
userPassword: tseoFihsoJ
uid: Joshi_Foest
givenName: Joshi
mail: Joshi_Foest@example.com
carLicense: HYF1MLE
departmentNumber: 8340
employeeType: Temp
homePhone: +1 206 619-9391
initials: J. F.
mobile: +1 213 782-3248
pager: +1 71 218-2782
manager: cn=Mack Tilson
secretary: cn=Amargo Oplinger
roomNumber: 1210

dn: cn=Elton Prymack, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Elton Prymack
sn: Prymack
description: This is Elton Prymack's description
facsimileTelephoneNumber: +1 818 145-8035
l: Redwood Shores
ou: Human Resources
postalAddress: example$Human Resources$Dept # 129
telephoneNumber: +1 415 698-8490
title: Junior Human Resources Warrior
userPassword: kcamyrPnot
uid: Elton_Prymack
givenName: Elton
mail: Elton_Prymack@example.com
carLicense: RD9AS5V
departmentNumber: 9654
employeeType: Temp
homePhone: +1 415 546-3251
initials: E. P.
mobile: +1 804 879-3838
pager: +1 818 946-4816
manager: cn=Marissa Collette
secretary: cn=Seungchul Cantlie
roomNumber: 8671

dn: cn=Croix Flatley, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Croix Flatley
sn: Flatley
description: This is Croix Flatley's description
facsimileTelephoneNumber: +1 213 892-8922
l: Mountain View
ou: Planning
postalAddress: example$Planning$Dept # 84
telephoneNumber: +1 408 339-6244
title: Senior Planning Manager
userPassword: yeltalFxio
uid: Croix_Flatley
givenName: Croix
mail: Croix_Flatley@example.com
carLicense: NE73BFS
departmentNumber: 4959
employeeType: Temp
homePhone: +1 206 166-3370
initials: C. F.
mobile: +1 818 483-1276
pager: +1 213 982-6149
manager: cn=Vishwa Nikfarjam
secretary: cn=Detlef Clifford
roomNumber: 5834

dn: cn=Moira Gratton, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Moira Gratton
sn: Gratton
description: This is Moira Gratton's description
facsimileTelephoneNumber: +1 510 618-3622
l: San Jose
ou: Administrative
postalAddress: example$Administrative$Dept # 687
telephoneNumber: +1 213 323-6709
title: Junior Administrative Artist
userPassword: nottarGari
uid: Moira_Gratton
givenName: Moira
mail: Moira_Gratton@example.com
carLicense: 55V2FPP
departmentNumber: 8374
employeeType: Normal
homePhone: +1 71 909-1127
initials: M. G.
mobile: +1 303 447-7682
pager: +1 303 572-1884
manager: cn=Robyn Guercioni
secretary: cn=Turus Fiore
roomNumber: 4886

dn: cn=Balaji Juni, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Balaji Juni
sn: Juni
description: This is Balaji Juni's description
facsimileTelephoneNumber: +1 804 785-3878
l: Cambridge
ou: Human Resources
postalAddress: example$Human Resources$Dept # 109
telephoneNumber: +1 303 270-8919
title: Associate Human Resources Admin
userPassword: inuJijalaB
uid: Balaji_Juni
givenName: Balaji
mail: Balaji_Juni@example.com
carLicense: BJC0G3I
departmentNumber: 6640
employeeType: Temp
homePhone: +1 303 941-5268
initials: B. J.
mobile: +1 213 538-1956
pager: +1 71 818-4960
manager: cn=Eydie Wartman
secretary: cn=Reza Kingston
roomNumber: 8654

dn: cn=Marlaine Hasan, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marlaine Hasan
sn: Hasan
description: This is Marlaine Hasan's description
facsimileTelephoneNumber: +1 804 374-5175
l: Redwood Shores
ou: Management
postalAddress: example$Management$Dept # 119
telephoneNumber: +1 804 512-2308
title: Master Management Punk
userPassword: nasaHenial
uid: Marlaine_Hasan
givenName: Marlaine
mail: Marlaine_Hasan@example.com
carLicense: D01S2SI
departmentNumber: 8147
employeeType: Contract
homePhone: +1 818 861-8628
initials: M. H.
mobile: +1 818 424-6590
pager: +1 804 981-4753
manager: cn=Haig Nagaraj
secretary: cn=Enid Addona
roomNumber: 1500

dn: cn=Leonard Stirling, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Leonard Stirling
sn: Stirling
description: This is Leonard Stirling's description
facsimileTelephoneNumber: +1 206 584-9006
l: Emeryville
ou: Peons
postalAddress: example$Peons$Dept # 944
telephoneNumber: +1 213 970-8571
title: Junior Peons Visionary
userPassword: gnilritSdr
uid: Leonard_Stirling
givenName: Leonard
mail: Leonard_Stirling@example.com
carLicense: 9BV9JCM
departmentNumber: 6868
employeeType: Contract
homePhone: +1 415 337-1577
initials: L. S.
mobile: +1 818 378-8522
pager: +1 818 514-6014
manager: cn=Charis Jackman
secretary: cn=Sibeal Wakim
roomNumber: 6765

dn: cn=Karin Kester, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Karin Kester
sn: Kester
description: This is Karin Kester's description
facsimileTelephoneNumber: +1 71 287-9218
l: San Francisco
ou: Management
postalAddress: example$Management$Dept # 988
telephoneNumber: +1 510 835-4905
title: Senior Management Evangelist
userPassword: retseKnira
uid: Karin_Kester
givenName: Karin
mail: Karin_Kester@example.com
carLicense: BTHEO1D
departmentNumber: 1594
employeeType: Contract
homePhone: +1 510 254-2955
initials: K. K.
mobile: +1 213 569-5127
pager: +1 408 605-7295
manager: cn=Elwira Sparksman
secretary: cn=Susy Kaley
roomNumber: 2671

dn: cn=Vanny Merciline, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Vanny Merciline
sn: Merciline
description: This is Vanny Merciline's description
facsimileTelephoneNumber: +1 303 808-4673
l: Fremont
ou: Product Development
postalAddress: example$Product Development$Dept # 359
telephoneNumber: +1 213 495-7834
title: Junior Product Development Architect
userPassword: enilicreMy
uid: Vanny_Merciline
givenName: Vanny
mail: Vanny_Merciline@example.com
carLicense: SHDRP12
departmentNumber: 6353
employeeType: Employee
homePhone: +1 206 538-4173
initials: V. M.
mobile: +1 303 401-9774
pager: +1 303 427-4803
manager: cn=Vo Saberi
secretary: cn=Karrie Wenzel
roomNumber: 1263

dn: cn=Papagena Komenda, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Papagena Komenda
sn: Komenda
description: This is Papagena Komenda's description
facsimileTelephoneNumber: +1 510 321-2378
l: Emeryville
ou: Payroll
postalAddress: example$Payroll$Dept # 826
telephoneNumber: +1 71 979-9837
title: Elite Payroll Technician
userPassword: adnemoKane
uid: Papagena_Komenda
givenName: Papagena
mail: Papagena_Komenda@example.com
carLicense: X60QQ78
departmentNumber: 742
employeeType: Contract
homePhone: +1 213 195-6047
initials: P. K.
mobile: +1 510 285-7873
pager: +1 415 910-7281
manager: cn=Zein Ong
secretary: cn=Rurick Fok
roomNumber: 6066

dn: cn=Theresa Birkett, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Theresa Birkett
sn: Birkett
description: This is Theresa Birkett's description
facsimileTelephoneNumber: +1 510 525-2392
l: Cupertino
ou: Product Testing
postalAddress: example$Product Testing$Dept # 481
telephoneNumber: +1 71 331-5423
title: Senior Product Testing Developer
userPassword: ttekriBase
uid: Theresa_Birkett
givenName: Theresa
mail: Theresa_Birkett@example.com
carLicense: QAWB46X
departmentNumber: 7168
employeeType: Contract
homePhone: +1 818 189-1014
initials: T. B.
mobile: +1 510 764-7739
pager: +1 818 764-4424
manager: cn=America Kashani-nia
secretary: cn=Allister Siperco
roomNumber: 7044

dn: cn=Andria Suyama, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Andria Suyama
sn: Suyama
description: This is Andria Suyama's description
facsimileTelephoneNumber: +1 206 160-9576
l: Milpitas
ou: Management
postalAddress: example$Management$Dept # 844
telephoneNumber: +1 818 161-3750
title: Supreme Management Director
userPassword: amayuSaird
uid: Andria_Suyama
givenName: Andria
mail: Andria_Suyama@example.com
carLicense: 38HQLSZ
departmentNumber: 2625
employeeType: Manager
homePhone: +1 71 617-9880
initials: A. S.
mobile: +1 71 927-7279
pager: +1 206 784-9362
manager: cn=Jacky Capindale
secretary: cn=Doll Doda
roomNumber: 3736

dn: cn=Weber Lalu, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Weber Lalu
sn: Lalu
description: This is Weber Lalu's description
facsimileTelephoneNumber: +1 415 332-9905
l: Alameda
ou: Management
postalAddress: example$Management$Dept # 987
telephoneNumber: +1 510 533-8334
title: Associate Management Admin
userPassword: ulaLrebeW
uid: Weber_Lalu
givenName: Weber
mail: Weber_Lalu@example.com
carLicense: NA6BWB4
departmentNumber: 6679
employeeType: Employee
homePhone: +1 71 993-8133
initials: W. L.
mobile: +1 804 707-9054
pager: +1 510 258-1035
manager: cn=Annis Rogan
secretary: cn=Giulietta Douet
roomNumber: 8548

dn: cn=Marilee Mir, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marilee Mir
sn: Mir
description: This is Marilee Mir's description
facsimileTelephoneNumber: +1 213 593-6952
l: Sunnyvale
ou: Human Resources
postalAddress: example$Human Resources$Dept # 463
telephoneNumber: +1 303 321-6311
title: Elite Human Resources Yahoo
userPassword: riMeeliraM
uid: Marilee_Mir
givenName: Marilee
mail: Marilee_Mir@example.com
carLicense: FIX7BKH
departmentNumber: 805
employeeType: Contract
homePhone: +1 213 920-3744
initials: M. M.
mobile: +1 206 275-2272
pager: +1 818 114-5451
manager: cn=Wren Csaszar
secretary: cn=Annarbor Kusyk
roomNumber: 2052

dn: cn=Marnie Ayoubzadeh, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marnie Ayoubzadeh
sn: Ayoubzadeh
description: This is Marnie Ayoubzadeh's description
facsimileTelephoneNumber: +1 804 558-6542
l: Menlo Park
ou: Management
postalAddress: example$Management$Dept # 246
telephoneNumber: +1 804 147-4120
title: Chief Management Czar
userPassword: hedazbuoyA
uid: Marnie_Ayoubzadeh
givenName: Marnie
mail: Marnie_Ayoubzadeh@example.com
carLicense: M3TCV65
departmentNumber: 6689
employeeType: Normal
homePhone: +1 213 950-5445
initials: M. A.
mobile: +1 408 836-4250
pager: +1 71 918-4622
manager: cn=Haley Hassey
secretary: cn=Erin McNicol
roomNumber: 1422

dn: cn=Thakor De-Boer, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Thakor De-Boer
sn: De-Boer
description: This is Thakor De-Boer's description
facsimileTelephoneNumber: +1 804 940-8307
l: San Francisco
ou: Administrative
postalAddress: example$Administrative$Dept # 463
telephoneNumber: +1 804 663-7224
title: Elite Administrative Janitor
userPassword: reoB-eDrok
uid: Thakor_De-Boer
givenName: Thakor
mail: Thakor_De-Boer@example.com
carLicense: 2A05SHH
departmentNumber: 6011
employeeType: Normal
homePhone: +1 71 417-4571
initials: T. D.
mobile: +1 303 769-5021
pager: +1 804 701-8613
manager: cn=Hester Brogden
secretary: cn=Winston Cogan
roomNumber: 3437

dn: cn=Hpone Syed, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Hpone Syed
sn: Syed
description: This is Hpone Syed's description
facsimileTelephoneNumber: +1 510 324-3423
l: San Jose
ou: Planning
postalAddress: example$Planning$Dept # 568
telephoneNumber: +1 71 960-5888
title: Supreme Planning Assistant
userPassword: deySenopH
uid: Hpone_Syed
givenName: Hpone
mail: Hpone_Syed@example.com
carLicense: IGP7BGY
departmentNumber: 9648
employeeType: Manager
homePhone: +1 303 891-3307
initials: H. S.
mobile: +1 303 912-8874
pager: +1 408 952-9101
manager: cn=Becca Simkin
secretary: cn=Fidelity Ervi
roomNumber: 3880

dn: cn=Leslie Hilbig, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Leslie Hilbig
sn: Hilbig
description: This is Leslie Hilbig's description
facsimileTelephoneNumber: +1 818 365-5871
l: San Jose
ou: Accounting
postalAddress: example$Accounting$Dept # 948
telephoneNumber: +1 510 963-3120
title: Junior Accounting Director
userPassword: gibliHeils
uid: Leslie_Hilbig
givenName: Leslie
mail: Leslie_Hilbig@example.com
carLicense: J80UH5X
departmentNumber: 6156
employeeType: Contract
homePhone: +1 71 234-3282
initials: L. H.
mobile: +1 415 168-2586
pager: +1 71 686-7093
manager: cn=Nha Wolski
secretary: cn=Unreg Miksik
roomNumber: 2475

dn: cn=Nyssa Priede, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Nyssa Priede
sn: Priede
description: This is Nyssa Priede's description
facsimileTelephoneNumber: +1 408 554-1398
l: Fremont
ou: Payroll
postalAddress: example$Payroll$Dept # 958
telephoneNumber: +1 804 549-8146
title: Associate Payroll Janitor
userPassword: edeirPassy
uid: Nyssa_Priede
givenName: Nyssa
mail: Nyssa_Priede@example.com
carLicense: 2MFWFQW
departmentNumber: 3728
employeeType: Temp
homePhone: +1 303 396-6569
initials: N. P.
mobile: +1 510 832-6542
pager: +1 415 658-1044
manager: cn=Caryl Gostanian
secretary: cn=Seana Dost
roomNumber: 8299

dn: cn=Eddy Brodowski, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Eddy Brodowski
sn: Brodowski
description: This is Eddy Brodowski's description
facsimileTelephoneNumber: +1 818 179-7369
l: Alameda
ou: Janitorial
postalAddress: example$Janitorial$Dept # 683
telephoneNumber: +1 804 353-2325
title: Chief Janitorial Assistant
userPassword: ikswodorBy
uid: Eddy_Brodowski
givenName: Eddy
mail: Eddy_Brodowski@example.com
carLicense: GAM09AM
departmentNumber: 2358
employeeType: Employee
homePhone: +1 818 416-2826
initials: E. B.
mobile: +1 303 373-7443
pager: +1 408 533-8374
manager: cn=Lance Ohmayer
secretary: cn=Joo-Euin Waidler
roomNumber: 9895

dn: cn=Aaccf Phung, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Aaccf Phung
sn: Phung
description: This is Aaccf Phung's description
facsimileTelephoneNumber: +1 408 578-7033
l: Cambridge
ou: Peons
postalAddress: example$Peons$Dept # 491
telephoneNumber: +1 510 710-4916
title: Elite Peons Stooge
userPassword: gnuhPfccaA
uid: Aaccf_Phung
givenName: Aaccf
mail: Aaccf_Phung@example.com
carLicense: 1SDWMRY
departmentNumber: 912
employeeType: Normal
homePhone: +1 415 935-3472
initials: A. P.
mobile: +1 415 977-6339
pager: +1 818 332-2001
manager: cn=Annamarie Scarrow
secretary: cn=Bess Newland
roomNumber: 7129

dn: cn=Ginni Fougere, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ginni Fougere
sn: Fougere
description: This is Ginni Fougere's description
facsimileTelephoneNumber: +1 804 985-9211
l: Redmond
ou: Peons
postalAddress: example$Peons$Dept # 361
telephoneNumber: +1 804 686-6994
title: Elite Peons President
userPassword: ereguoFinn
uid: Ginni_Fougere
givenName: Ginni
mail: Ginni_Fougere@example.com
carLicense: DHV2H5A
departmentNumber: 6789
employeeType: Temp
homePhone: +1 804 764-7821
initials: G. F.
mobile: +1 415 478-2421
pager: +1 804 926-6211
manager: cn=Marshal Zoppel
secretary: cn=Fitzgerald Bernardo
roomNumber: 8674

dn: cn=Kien-Nghiep SVM-BNRMTVA, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kien-Nghiep SVM-BNRMTVA
sn: SVM-BNRMTVA
description: This is Kien-Nghiep SVM-BNRMTVA's description
facsimileTelephoneNumber: +1 213 306-2758
l: Emeryville
ou: Accounting
postalAddress: example$Accounting$Dept # 888
telephoneNumber: +1 818 163-9758
title: Senior Accounting President
userPassword: AVTMRNB-MV
uid: Kien-Nghiep_SVM-BNRMTVA
givenName: Kien-Nghiep
mail: Kien-Nghiep_SVM-BNRMTVA@example.com
carLicense: 5LRQFJD
departmentNumber: 5645
employeeType: Contract
homePhone: +1 206 674-6551
initials: K. S.
mobile: +1 213 667-9460
pager: +1 804 539-3892
manager: cn=Focus Freyermuth
secretary: cn=Siusan Galligan
roomNumber: 1101

dn: cn=Valli Haney, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Valli Haney
sn: Haney
description: This is Valli Haney's description
facsimileTelephoneNumber: +1 71 483-8348
l: Cambridge
ou: Peons
postalAddress: example$Peons$Dept # 44
telephoneNumber: +1 804 452-6380
title: Junior Peons President
userPassword: yenaHillaV
uid: Valli_Haney
givenName: Valli
mail: Valli_Haney@example.com
carLicense: WCRC4BV
departmentNumber: 4252
employeeType: Temp
homePhone: +1 510 404-1536
initials: V. H.
mobile: +1 206 904-6817
pager: +1 415 196-3277
manager: cn=Petunia Colantonio
secretary: cn=Pippy Gouhara
roomNumber: 5884

dn: cn=Gabbey Vieger, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gabbey Vieger
sn: Vieger
description: This is Gabbey Vieger's description
facsimileTelephoneNumber: +1 303 499-1918
l: Redwood Shores
ou: Management
postalAddress: example$Management$Dept # 569
telephoneNumber: +1 415 845-4066
title: Chief Management Technician
userPassword: regeiVyebb
uid: Gabbey_Vieger
givenName: Gabbey
mail: Gabbey_Vieger@example.com
carLicense: Z6PRY74
departmentNumber: 4878
employeeType: Manager
homePhone: +1 510 716-2017
initials: G. V.
mobile: +1 303 557-1453
pager: +1 408 932-5598
manager: cn=Teresina Banfalvi
secretary: cn=Katharyn Fluet
roomNumber: 5955

dn: cn=Zero Kannemann, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zero Kannemann
sn: Kannemann
description: This is Zero Kannemann's description
facsimileTelephoneNumber: +1 213 777-1612
l: Palo Alto
ou: Human Resources
postalAddress: example$Human Resources$Dept # 149
telephoneNumber: +1 71 463-2081
title: Chief Human Resources Assistant
userPassword: nnamennaKo
uid: Zero_Kannemann
givenName: Zero
mail: Zero_Kannemann@example.com
carLicense: JBUT5UW
departmentNumber: 5783
employeeType: Employee
homePhone: +1 213 857-8985
initials: Z. K.
mobile: +1 510 913-8725
pager: +1 408 538-5125
manager: cn=Lidio Halpenny
secretary: cn=Wai-Leung Lanoue
roomNumber: 582

dn: cn=Zhengyu Shull, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zhengyu Shull
sn: Shull
description: This is Zhengyu Shull's description
facsimileTelephoneNumber: +1 415 765-3239
l: Armonk
ou: Peons
postalAddress: example$Peons$Dept # 625
telephoneNumber: +1 206 257-9486
title: Associate Peons Fellow
userPassword: lluhSuygne
uid: Zhengyu_Shull
givenName: Zhengyu
mail: Zhengyu_Shull@example.com
carLicense: WI63BDD
departmentNumber: 2157
employeeType: Manager
homePhone: +1 408 676-2349
initials: Z. S.
mobile: +1 804 695-2705
pager: +1 71 666-7092
manager: cn=Dyke Ozer
secretary: cn=Myriam Johnsen
roomNumber: 29

dn: cn=Kikelia Voight, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kikelia Voight
sn: Voight
description: This is Kikelia Voight's description
facsimileTelephoneNumber: +1 303 739-7104
l: Sunnyvale
ou: Planning
postalAddress: example$Planning$Dept # 837
telephoneNumber: +1 804 439-7733
title: Elite Planning Grunt
userPassword: thgioVaile
uid: Kikelia_Voight
givenName: Kikelia
mail: Kikelia_Voight@example.com
carLicense: C23U2CJ
departmentNumber: 3322
employeeType: Normal
homePhone: +1 213 469-6364
initials: K. V.
mobile: +1 415 698-8309
pager: +1 303 566-3751
manager: cn=Marybelle McCaugherty
secretary: cn=Carlotta Tieu
roomNumber: 852

dn: cn=Rama Ploof, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rama Ploof
sn: Ploof
description: This is Rama Ploof's description
facsimileTelephoneNumber: +1 206 540-1575
l: Mountain View
ou: Janitorial
postalAddress: example$Janitorial$Dept # 329
telephoneNumber: +1 510 102-9947
title: Associate Janitorial Sales Rep
userPassword: foolPamaR
uid: Rama_Ploof
givenName: Rama
mail: Rama_Ploof@example.com
carLicense: BIZED1P
departmentNumber: 7650
employeeType: Employee
homePhone: +1 818 723-5409
initials: R. P.
mobile: +1 206 487-7886
pager: +1 71 625-1587
manager: cn=Ginelle Iantaffi
secretary: cn=Tallie Marcotte
roomNumber: 2931

dn: cn=Shaker Marengere, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shaker Marengere
sn: Marengere
description: This is Shaker Marengere's description
facsimileTelephoneNumber: +1 408 123-7329
l: Fremont
ou: Payroll
postalAddress: example$Payroll$Dept # 423
telephoneNumber: +1 510 602-9780
title: Elite Payroll Pinhead
userPassword: eregneraMr
uid: Shaker_Marengere
givenName: Shaker
mail: Shaker_Marengere@example.com
carLicense: 38LRCSX
departmentNumber: 3078
employeeType: Manager
homePhone: +1 510 299-9437
initials: S. M.
mobile: +1 415 190-2235
pager: +1 303 401-2705
manager: cn=Franz Gobeli
secretary: cn=Tyronda Saward
roomNumber: 737

dn: cn=Rebecca Albea, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rebecca Albea
sn: Albea
description: This is Rebecca Albea's description
facsimileTelephoneNumber: +1 303 637-1044
l: Menlo Park
ou: Human Resources
postalAddress: example$Human Resources$Dept # 192
telephoneNumber: +1 415 884-6035
title: Junior Human Resources Madonna
userPassword: aeblAacceb
uid: Rebecca_Albea
givenName: Rebecca
mail: Rebecca_Albea@example.com
carLicense: OWI2NJK
departmentNumber: 1269
employeeType: Normal
homePhone: +1 804 739-3562
initials: R. A.
mobile: +1 206 745-7709
pager: +1 71 449-9590
manager: cn=Hailee Theis
secretary: cn=Sydel Kirkby
roomNumber: 8861

dn: cn=Linnie Caruth, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Linnie Caruth
sn: Caruth
description: This is Linnie Caruth's description
facsimileTelephoneNumber: +1 804 932-5140
l: Santa Clara
ou: Product Development
postalAddress: example$Product Development$Dept # 234
telephoneNumber: +1 213 568-5430
title: Chief Product Development Sales Rep
userPassword: hturaCeinn
uid: Linnie_Caruth
givenName: Linnie
mail: Linnie_Caruth@example.com
carLicense: GEW2B1F
departmentNumber: 5657
employeeType: Manager
homePhone: +1 71 982-4988
initials: L. C.
mobile: +1 303 434-7153
pager: +1 206 223-9585
manager: cn=Kylynn Kuzbary
secretary: cn=Cyrine Malhi
roomNumber: 9028

dn: cn=Kat Golaszewski, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kat Golaszewski
sn: Golaszewski
description: This is Kat Golaszewski's description
facsimileTelephoneNumber: +1 206 786-8217
l: Palo Alto
ou: Product Development
postalAddress: example$Product Development$Dept # 697
telephoneNumber: +1 213 542-2548
title: Associate Product Development Director
userPassword: ikswezsalo
uid: Kat_Golaszewski
givenName: Kat
mail: Kat_Golaszewski@example.com
carLicense: R59IY87
departmentNumber: 6186
employeeType: Manager
homePhone: +1 213 241-1609
initials: K. G.
mobile: +1 213 663-6489
pager: +1 415 405-6871
manager: cn=Lavinia Melfi
secretary: cn=Joella Adam
roomNumber: 700

dn: cn=Inga Graessley, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Inga Graessley
sn: Graessley
description: This is Inga Graessley's description
facsimileTelephoneNumber: +1 804 377-1997
l: San Jose
ou: Janitorial
postalAddress: example$Janitorial$Dept # 194
telephoneNumber: +1 415 457-6662
title: Master Janitorial Director
userPassword: yelssearGa
uid: Inga_Graessley
givenName: Inga
mail: Inga_Graessley@example.com
carLicense: BF65T92
departmentNumber: 6810
employeeType: Temp
homePhone: +1 818 971-7748
initials: I. G.
mobile: +1 71 452-3308
pager: +1 71 946-3546
manager: cn=Orelie Hardage
secretary: cn=Candie Watkinson
roomNumber: 5459

dn: cn=Dominica Rohal, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dominica Rohal
sn: Rohal
description: This is Dominica Rohal's description
facsimileTelephoneNumber: +1 206 802-8249
l: Mountain View
ou: Accounting
postalAddress: example$Accounting$Dept # 981
telephoneNumber: +1 213 812-3517
title: Supreme Accounting Manager
userPassword: lahoRacini
uid: Dominica_Rohal
givenName: Dominica
mail: Dominica_Rohal@example.com
carLicense: W4KWOTP
departmentNumber: 3781
employeeType: Temp
homePhone: +1 510 164-7017
initials: D. R.
mobile: +1 213 337-6509
pager: +1 213 877-6481
manager: cn=Ricca Woodley
secretary: cn=Concordia Bedoya
roomNumber: 6507

dn: cn=Meghann Loza, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Meghann Loza
sn: Loza
description: This is Meghann Loza's description
facsimileTelephoneNumber: +1 804 271-2532
l: Santa Clara
ou: Peons
postalAddress: example$Peons$Dept # 87
telephoneNumber: +1 408 804-6965
title: Chief Peons Dictator
userPassword: azoLnnahge
uid: Meghann_Loza
givenName: Meghann
mail: Meghann_Loza@example.com
carLicense: 6Z3292L
departmentNumber: 7735
employeeType: Manager
homePhone: +1 206 361-8241
initials: M. L.
mobile: +1 804 398-7507
pager: +1 804 509-3165
manager: cn=Andras Partin
secretary: cn=Izumi Loo
roomNumber: 4525

dn: cn=Bellanca Struzynski, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bellanca Struzynski
sn: Struzynski
description: This is Bellanca Struzynski's description
facsimileTelephoneNumber: +1 415 816-3119
l: Emeryville
ou: Planning
postalAddress: example$Planning$Dept # 298
telephoneNumber: +1 804 649-5046
title: Associate Planning Manager
userPassword: iksnyzurtS
uid: Bellanca_Struzynski
givenName: Bellanca
mail: Bellanca_Struzynski@example.com
carLicense: WMFGWYT
departmentNumber: 7130
employeeType: Manager
homePhone: +1 213 200-9388
initials: B. S.
mobile: +1 415 604-8909
pager: +1 303 834-9548
manager: cn=Tru-Fu Uberig
secretary: cn=Annabella Derbyshire
roomNumber: 2343

dn: cn=Agenia Kolesnik, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Agenia Kolesnik
sn: Kolesnik
description: This is Agenia Kolesnik's description
facsimileTelephoneNumber: +1 206 413-2686
l: Sunnyvale
ou: Planning
postalAddress: example$Planning$Dept # 879
telephoneNumber: +1 408 352-7329
title: Master Planning Artist
userPassword: kinseloKai
uid: Agenia_Kolesnik
givenName: Agenia
mail: Agenia_Kolesnik@example.com
carLicense: BAY0YS9
departmentNumber: 1898
employeeType: Temp
homePhone: +1 213 366-2657
initials: A. K.
mobile: +1 303 156-2098
pager: +1 408 898-8410
manager: cn=Zuzana Papageorges
secretary: cn=Stephannie Ciccarelli
roomNumber: 1627

dn: cn=Reiko Ketley, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Reiko Ketley
sn: Ketley
description: This is Reiko Ketley's description
facsimileTelephoneNumber: +1 804 650-6826
l: Cupertino
ou: Management
postalAddress: example$Management$Dept # 475
telephoneNumber: +1 71 858-6517
title: Associate Management Punk
userPassword: yelteKokie
uid: Reiko_Ketley
givenName: Reiko
mail: Reiko_Ketley@example.com
carLicense: BND7WEG
departmentNumber: 7303
employeeType: Temp
homePhone: +1 303 926-8826
initials: R. K.
mobile: +1 71 708-8679
pager: +1 303 389-5913
manager: cn=Powell Ramseyer
secretary: cn=Genna Antinucci
roomNumber: 7991

dn: cn=Arvin Blaauw, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Arvin Blaauw
sn: Blaauw
description: This is Arvin Blaauw's description
facsimileTelephoneNumber: +1 213 297-4759
l: Redwood Shores
ou: Janitorial
postalAddress: example$Janitorial$Dept # 488
telephoneNumber: +1 303 874-9708
title: Associate Janitorial Grunt
userPassword: wuaalBnivr
uid: Arvin_Blaauw
givenName: Arvin
mail: Arvin_Blaauw@example.com
carLicense: 14ZBO89
departmentNumber: 79
employeeType: Temp
homePhone: +1 510 946-7008
initials: A. B.
mobile: +1 71 503-1695
pager: +1 206 650-1475
manager: cn=Bobbie Magee
secretary: cn=Evaleen Lehtinen
roomNumber: 9404

dn: cn=Chocs Puddington, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Chocs Puddington
sn: Puddington
description: This is Chocs Puddington's description
facsimileTelephoneNumber: +1 804 923-8885
l: Milpitas
ou: Human Resources
postalAddress: example$Human Resources$Dept # 92
telephoneNumber: +1 415 497-1230
title: Elite Human Resources Madonna
userPassword: notgnidduP
uid: Chocs_Puddington
givenName: Chocs
mail: Chocs_Puddington@example.com
carLicense: 0UNFORR
departmentNumber: 3756
employeeType: Temp
homePhone: +1 510 728-8035
initials: C. P.
mobile: +1 408 415-7132
pager: +1 408 510-4236
manager: cn=Austin Scss
secretary: cn=Celinka Barnhill
roomNumber: 9156

dn: cn=Tiffany Kromer, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tiffany Kromer
sn: Kromer
description: This is Tiffany Kromer's description
facsimileTelephoneNumber: +1 206 440-6347
l: Redwood Shores
ou: Product Development
postalAddress: example$Product Development$Dept # 942
telephoneNumber: +1 818 265-2550
title: Senior Product Development Madonna
userPassword: remorKynaf
uid: Tiffany_Kromer
givenName: Tiffany
mail: Tiffany_Kromer@example.com
carLicense: VPX7TTC
departmentNumber: 557
employeeType: Employee
homePhone: +1 818 697-6303
initials: T. K.
mobile: +1 303 290-2555
pager: +1 818 413-4806
manager: cn=Sacto Finane
secretary: cn=Dulcine McNerney
roomNumber: 2503

dn: cn=Lashonda Yanosik, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lashonda Yanosik
sn: Yanosik
description: This is Lashonda Yanosik's description
facsimileTelephoneNumber: +1 804 608-5137
l: Santa Clara
ou: Payroll
postalAddress: example$Payroll$Dept # 224
telephoneNumber: +1 804 836-4936
title: Elite Payroll Writer
userPassword: kisonaYadn
uid: Lashonda_Yanosik
givenName: Lashonda
mail: Lashonda_Yanosik@example.com
carLicense: 8350DHV
departmentNumber: 5724
employeeType: Contract
homePhone: +1 303 773-4106
initials: L. Y.
mobile: +1 804 760-2949
pager: +1 206 180-6258
manager: cn=Julita Oaks
secretary: cn=Lonna Frodsham
roomNumber: 4559

dn: cn=Carri Scribner, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Carri Scribner
sn: Scribner
description: This is Carri Scribner's description
facsimileTelephoneNumber: +1 415 802-8287
l: Orem
ou: Planning
postalAddress: example$Planning$Dept # 690
telephoneNumber: +1 415 287-5089
title: Associate Planning Architect
userPassword: renbircSir
uid: Carri_Scribner
givenName: Carri
mail: Carri_Scribner@example.com
carLicense: 5R3S5VE
departmentNumber: 806
employeeType: Employee
homePhone: +1 206 444-3212
initials: C. S.
mobile: +1 408 672-4072
pager: +1 818 239-3450
manager: cn=Fitzroy Willis
secretary: cn=Fernando Merrill
roomNumber: 5624

dn: cn=Irita Bartkowska, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Irita Bartkowska
sn: Bartkowska
description: This is Irita Bartkowska's description
facsimileTelephoneNumber: +1 71 733-5083
l: Emeryville
ou: Peons
postalAddress: example$Peons$Dept # 560
telephoneNumber: +1 804 837-6263
title: Supreme Peons Admin
userPassword: akswoktraB
uid: Irita_Bartkowska
givenName: Irita
mail: Irita_Bartkowska@example.com
carLicense: ZELGQB7
departmentNumber: 4439
employeeType: Manager
homePhone: +1 206 682-1413
initials: I. B.
mobile: +1 303 312-3304
pager: +1 408 470-4943
manager: cn=Laureen Swinkels
secretary: cn=Anneliese Macoosh
roomNumber: 3008

dn: cn=Reynold Patchsqa, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Reynold Patchsqa
sn: Patchsqa
description: This is Reynold Patchsqa's description
facsimileTelephoneNumber: +1 818 698-9238
l: Orem
ou: Administrative
postalAddress: example$Administrative$Dept # 992
telephoneNumber: +1 408 208-5521
title: Associate Administrative Figurehead
userPassword: aqshctaPdl
uid: Reynold_Patchsqa
givenName: Reynold
mail: Reynold_Patchsqa@example.com
carLicense: WIW9S5F
departmentNumber: 4436
employeeType: Manager
homePhone: +1 510 909-6551
initials: R. P.
mobile: +1 408 577-2206
pager: +1 213 478-8025
manager: cn=Adie Voight
secretary: cn=Fiore Harapiak
roomNumber: 1042

dn: cn=Rosalynd Szkarlat, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rosalynd Szkarlat
sn: Szkarlat
description: This is Rosalynd Szkarlat's description
facsimileTelephoneNumber: +1 804 363-1089
l: Santa Clara
ou: Payroll
postalAddress: example$Payroll$Dept # 49
telephoneNumber: +1 510 527-9095
title: Junior Payroll Evangelist
userPassword: talrakzSdn
uid: Rosalynd_Szkarlat
givenName: Rosalynd
mail: Rosalynd_Szkarlat@example.com
carLicense: IY7212S
departmentNumber: 5064
employeeType: Temp
homePhone: +1 71 786-1102
initials: R. S.
mobile: +1 818 168-3571
pager: +1 408 130-7493
manager: cn=Caterina Bizga
secretary: cn=Beryl Vernon
roomNumber: 7593

dn: cn=Theodor Polulack, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Theodor Polulack
sn: Polulack
description: This is Theodor Polulack's description
facsimileTelephoneNumber: +1 415 220-6236
l: Milpitas
ou: Administrative
postalAddress: example$Administrative$Dept # 476
telephoneNumber: +1 510 304-4145
title: Master Administrative Manager
userPassword: kcaluloPro
uid: Theodor_Polulack
givenName: Theodor
mail: Theodor_Polulack@example.com
carLicense: NJRGXQF
departmentNumber: 6276
employeeType: Manager
homePhone: +1 206 881-7728
initials: T. P.
mobile: +1 213 546-4224
pager: +1 206 723-6787
manager: cn=Hesham Yearwood
secretary: cn=Marcus Zetts
roomNumber: 6806

dn: cn=Patricia Demarest, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Patricia Demarest
sn: Demarest
description: This is Patricia Demarest's description
facsimileTelephoneNumber: +1 206 711-5239
l: Orem
ou: Payroll
postalAddress: example$Payroll$Dept # 164
telephoneNumber: +1 510 918-6995
title: Supreme Payroll Punk
userPassword: tserameDai
uid: Patricia_Demarest
givenName: Patricia
mail: Patricia_Demarest@example.com
carLicense: CS0ECR3
departmentNumber: 1122
employeeType: Manager
homePhone: +1 206 597-6824
initials: P. D.
mobile: +1 510 954-3686
pager: +1 415 884-8219
manager: cn=Makam Strayhorn
secretary: cn=Jayendra Abi-Aad
roomNumber: 5479

dn: cn=Reagan Harrell, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Reagan Harrell
sn: Harrell
description: This is Reagan Harrell's description
facsimileTelephoneNumber: +1 818 165-2427
l: San Mateo
ou: Payroll
postalAddress: example$Payroll$Dept # 199
telephoneNumber: +1 818 475-4591
title: Master Payroll Warrior
userPassword: llerraHnag
uid: Reagan_Harrell
givenName: Reagan
mail: Reagan_Harrell@example.com
carLicense: TPNKX1P
departmentNumber: 3064
employeeType: Employee
homePhone: +1 818 613-2969
initials: R. H.
mobile: +1 303 622-4198
pager: +1 206 800-9491
manager: cn=Susy Beauvais
secretary: cn=Merlin Comm
roomNumber: 4109

dn: cn=Famke Bugajski, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Famke Bugajski
sn: Bugajski
description: This is Famke Bugajski's description
facsimileTelephoneNumber: +1 213 737-4703
l: Armonk
ou: Janitorial
postalAddress: example$Janitorial$Dept # 935
telephoneNumber: +1 206 275-2059
title: Master Janitorial Fellow
userPassword: iksjaguBek
uid: Famke_Bugajski
givenName: Famke
mail: Famke_Bugajski@example.com
carLicense: I3CTYL6
departmentNumber: 1653
employeeType: Manager
homePhone: +1 804 701-8927
initials: F. B.
mobile: +1 206 853-1058
pager: +1 510 275-9290
manager: cn=Raju Preville
secretary: cn=Arlen Sanks
roomNumber: 6023

dn: cn=Wenonah Dhir, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Wenonah Dhir
sn: Dhir
description: This is Wenonah Dhir's description
facsimileTelephoneNumber: +1 206 919-5991
l: San Francisco
ou: Product Development
postalAddress: example$Product Development$Dept # 577
telephoneNumber: +1 71 425-6513
title: Associate Product Development Dictator
userPassword: rihDhanone
uid: Wenonah_Dhir
givenName: Wenonah
mail: Wenonah_Dhir@example.com
carLicense: 4ABVEJ0
departmentNumber: 1009
employeeType: Employee
homePhone: +1 510 821-1590
initials: W. D.
mobile: +1 818 621-5714
pager: +1 415 696-2798
manager: cn=Manjit Torrell
secretary: cn=Melicent Shultz
roomNumber: 775

dn: cn=Marjie Hyjek, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marjie Hyjek
sn: Hyjek
description: This is Marjie Hyjek's description
facsimileTelephoneNumber: +1 818 770-4649
l: Armonk
ou: Payroll
postalAddress: example$Payroll$Dept # 296
telephoneNumber: +1 510 296-5543
title: Supreme Payroll Mascot
userPassword: kejyHeijra
uid: Marjie_Hyjek
givenName: Marjie
mail: Marjie_Hyjek@example.com
carLicense: UHATS76
departmentNumber: 1195
employeeType: Normal
homePhone: +1 71 132-6676
initials: M. H.
mobile: +1 206 928-9063
pager: +1 510 249-9787
manager: cn=Andrei Hadirahardjo
secretary: cn=Mougy Helgeland
roomNumber: 970

dn: cn=Wynnie Loader, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Wynnie Loader
sn: Loader
description: This is Wynnie Loader's description
facsimileTelephoneNumber: +1 415 889-4966
l: Palo Alto
ou: Peons
postalAddress: example$Peons$Dept # 357
telephoneNumber: +1 415 371-3305
title: Associate Peons Janitor
userPassword: redaoLeinn
uid: Wynnie_Loader
givenName: Wynnie
mail: Wynnie_Loader@example.com
carLicense: HKWL0D5
departmentNumber: 8295
employeeType: Manager
homePhone: +1 71 383-2582
initials: W. L.
mobile: +1 415 673-9559
pager: +1 804 655-8767
manager: cn=Marcela Wans
secretary: cn=Kassey Wojdylo
roomNumber: 6057

dn: cn=Leilah Bridenstine, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Leilah Bridenstine
sn: Bridenstine
description: This is Leilah Bridenstine's description
facsimileTelephoneNumber: +1 415 527-7283
l: San Jose
ou: Administrative
postalAddress: example$Administrative$Dept # 590
telephoneNumber: +1 818 835-4663
title: Chief Administrative Stooge
userPassword: enitsnedir
uid: Leilah_Bridenstine
givenName: Leilah
mail: Leilah_Bridenstine@example.com
carLicense: EBIG6UV
departmentNumber: 3567
employeeType: Employee
homePhone: +1 804 749-7634
initials: L. B.
mobile: +1 818 252-9362
pager: +1 213 727-6760
manager: cn=Nicoli Ginest
secretary: cn=Kelcey Kolappa
roomNumber: 1920

dn: cn=Kinman Meubus, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kinman Meubus
sn: Meubus
description: This is Kinman Meubus's description
facsimileTelephoneNumber: +1 303 466-1091
l: Cupertino
ou: Product Testing
postalAddress: example$Product Testing$Dept # 220
telephoneNumber: +1 510 374-1473
title: Master Product Testing Visionary
userPassword: subueMnamn
uid: Kinman_Meubus
givenName: Kinman
mail: Kinman_Meubus@example.com
carLicense: 1XLX2DR
departmentNumber: 4070
employeeType: Manager
homePhone: +1 303 971-5309
initials: K. M.
mobile: +1 510 182-6106
pager: +1 408 946-6399
manager: cn=Bethena Brasington
secretary: cn=Rebekah Dickeson
roomNumber: 8782

dn: cn=Melissa Krikorian, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Melissa Krikorian
sn: Krikorian
description: This is Melissa Krikorian's description
facsimileTelephoneNumber: +1 415 593-7969
l: Emeryville
ou: Planning
postalAddress: example$Planning$Dept # 433
telephoneNumber: +1 71 456-8371
title: Senior Planning Fellow
userPassword: nairokirKa
uid: Melissa_Krikorian
givenName: Melissa
mail: Melissa_Krikorian@example.com
carLicense: PS03ZCC
departmentNumber: 1223
employeeType: Temp
homePhone: +1 206 449-4391
initials: M. K.
mobile: +1 408 821-2471
pager: +1 415 545-2392
manager: cn=Nguyen Mansouri
secretary: cn=Vilma Lengel
roomNumber: 5954

dn: cn=Shay Ganguly, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shay Ganguly
sn: Ganguly
description: This is Shay Ganguly's description
facsimileTelephoneNumber: +1 818 576-2828
l: Cambridge
ou: Product Testing
postalAddress: example$Product Testing$Dept # 452
telephoneNumber: +1 510 802-9087
title: Junior Product Testing Visionary
userPassword: ylugnaGyah
uid: Shay_Ganguly
givenName: Shay
mail: Shay_Ganguly@example.com
carLicense: AXIBIRS
departmentNumber: 9430
employeeType: Normal
homePhone: +1 415 417-9977
initials: S. G.
mobile: +1 415 737-5161
pager: +1 818 408-4054
manager: cn=Elliot Komatsu
secretary: cn=Waneta Henderson
roomNumber: 5102

dn: cn=Lavonda Alles, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lavonda Alles
sn: Alles
description: This is Lavonda Alles's description
facsimileTelephoneNumber: +1 408 766-4928
l: Palo Alto
ou: Management
postalAddress: example$Management$Dept # 134
telephoneNumber: +1 415 751-3190
title: Master Management Visionary
userPassword: sellAadnov
uid: Lavonda_Alles
givenName: Lavonda
mail: Lavonda_Alles@example.com
carLicense: LX81N4Y
departmentNumber: 1269
employeeType: Contract
homePhone: +1 213 438-3842
initials: L. A.
mobile: +1 510 905-1104
pager: +1 303 558-8324
manager: cn=Jacek Das
secretary: cn=Moreen Essery
roomNumber: 3962

dn: cn=Darlene Bigley, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Darlene Bigley
sn: Bigley
description: This is Darlene Bigley's description
facsimileTelephoneNumber: +1 213 776-6233
l: Santa Clara
ou: Janitorial
postalAddress: example$Janitorial$Dept # 424
telephoneNumber: +1 804 132-4123
title: Chief Janitorial Mascot
userPassword: yelgiBenel
uid: Darlene_Bigley
givenName: Darlene
mail: Darlene_Bigley@example.com
carLicense: W94W8GP
departmentNumber: 4108
employeeType: Contract
homePhone: +1 408 668-9662
initials: D. B.
mobile: +1 415 434-3474
pager: +1 415 537-2839
manager: cn=Robin Credille
secretary: cn=Alejandra Wayling
roomNumber: 2383

dn: cn=Bruno Zunuzi, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bruno Zunuzi
sn: Zunuzi
description: This is Bruno Zunuzi's description
facsimileTelephoneNumber: +1 206 379-8074
l: Redwood Shores
ou: Administrative
postalAddress: example$Administrative$Dept # 730
telephoneNumber: +1 510 678-5567
title: Elite Administrative Evangelist
userPassword: izunuZonur
uid: Bruno_Zunuzi
givenName: Bruno
mail: Bruno_Zunuzi@example.com
carLicense: O0CIWCP
departmentNumber: 5979
employeeType: Normal
homePhone: +1 71 818-6549
initials: B. Z.
mobile: +1 818 363-2572
pager: +1 804 663-4075
manager: cn=Alica Beaudette
secretary: cn=Katherina Labrinos
roomNumber: 1575

dn: cn=Beana Hurteau, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Beana Hurteau
sn: Hurteau
description: This is Beana Hurteau's description
facsimileTelephoneNumber: +1 303 643-8202
l: Menlo Park
ou: Accounting
postalAddress: example$Accounting$Dept # 65
telephoneNumber: +1 71 348-7436
title: Associate Accounting Manager
userPassword: uaetruHana
uid: Beana_Hurteau
givenName: Beana
mail: Beana_Hurteau@example.com
carLicense: MSNQDP0
departmentNumber: 8713
employeeType: Normal
homePhone: +1 206 896-5863
initials: B. H.
mobile: +1 415 832-3335
pager: +1 303 999-9538
manager: cn=Dicky Kalyani
secretary: cn=Jessamyn Verrilli
roomNumber: 921

dn: cn=Zsa zsa Arnone, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zsa zsa Arnone
sn: Arnone
description: This is Zsa zsa Arnone's description
facsimileTelephoneNumber: +1 408 524-8595
l: Orem
ou: Administrative
postalAddress: example$Administrative$Dept # 885
telephoneNumber: +1 415 818-7186
title: Master Administrative Punk
userPassword: enonrAasza
uid: Zsa zsa_Arnone
givenName: Zsa zsa
mail: Zsa zsa_Arnone@example.com
carLicense: JA1PTPS
departmentNumber: 8385
employeeType: Contract
homePhone: +1 818 417-9630
initials: Z. A.
mobile: +1 510 982-3941
pager: +1 510 245-5695
manager: cn=Huy Mathias
secretary: cn=Georgiana Hallenbeck
roomNumber: 3460

dn: cn=Svr Pitts, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Svr Pitts
sn: Pitts
description: This is Svr Pitts's description
facsimileTelephoneNumber: +1 415 941-7757
l: Orem
ou: Human Resources
postalAddress: example$Human Resources$Dept # 105
telephoneNumber: +1 213 930-8296
title: Elite Human Resources Admin
userPassword: sttiPrvS
uid: Svr_Pitts
givenName: Svr
mail: Svr_Pitts@example.com
carLicense: RQVESWE
departmentNumber: 236
employeeType: Manager
homePhone: +1 408 933-2797
initials: S. P.
mobile: +1 818 998-2127
pager: +1 303 802-3242
manager: cn=Susie Serre
secretary: cn=Stergios Nuetzi
roomNumber: 3875

dn: cn=Pooh Somers, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Pooh Somers
sn: Somers
description: This is Pooh Somers's description
facsimileTelephoneNumber: +1 206 313-8132
l: San Francisco
ou: Payroll
postalAddress: example$Payroll$Dept # 743
telephoneNumber: +1 818 187-7558
title: Associate Payroll Warrior
userPassword: sremoShooP
uid: Pooh_Somers
givenName: Pooh
mail: Pooh_Somers@example.com
carLicense: 65VVW12
departmentNumber: 1202
employeeType: Temp
homePhone: +1 510 765-9719
initials: P. S.
mobile: +1 804 731-3541
pager: +1 818 460-8590
manager: cn=Eran Scates
secretary: cn=Dede Gilstorf
roomNumber: 1773

dn: cn=Pui-Wah McHale, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Pui-Wah McHale
sn: McHale
description: This is Pui-Wah McHale's description
facsimileTelephoneNumber: +1 303 673-8234
l: Sunnyvale
ou: Management
postalAddress: example$Management$Dept # 469
telephoneNumber: +1 206 982-7959
title: Elite Management Mascot
userPassword: elaHcMhaW-
uid: Pui-Wah_McHale
givenName: Pui-Wah
mail: Pui-Wah_McHale@example.com
carLicense: L3E2BGO
departmentNumber: 1893
employeeType: Normal
homePhone: +1 303 652-2670
initials: P. M.
mobile: +1 71 212-4626
pager: +1 415 890-1418
manager: cn=Glenine Speers
secretary: cn=Sosanna Borkowicz
roomNumber: 6920

dn: cn=Gerardo Micucci, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gerardo Micucci
sn: Micucci
description: This is Gerardo Micucci's description
facsimileTelephoneNumber: +1 303 289-4435
l: San Francisco
ou: Accounting
postalAddress: example$Accounting$Dept # 717
telephoneNumber: +1 818 390-5372
title: Master Accounting Technician
userPassword: iccuciModr
uid: Gerardo_Micucci
givenName: Gerardo
mail: Gerardo_Micucci@example.com
carLicense: FUVFUPX
departmentNumber: 8550
employeeType: Employee
homePhone: +1 804 550-2922
initials: G. M.
mobile: +1 206 160-5665
pager: +1 408 103-1416
manager: cn=Merla Mezzoiuso
secretary: cn=Grietje Erkel
roomNumber: 2625

dn: cn=Parkinson Aldridge, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Parkinson Aldridge
sn: Aldridge
description: This is Parkinson Aldridge's description
facsimileTelephoneNumber: +1 804 229-1425
l: San Mateo
ou: Peons
postalAddress: example$Peons$Dept # 609
telephoneNumber: +1 804 619-3292
title: Junior Peons Artist
userPassword: egdirdlAno
uid: Parkinson_Aldridge
givenName: Parkinson
mail: Parkinson_Aldridge@example.com
carLicense: OMIMFDE
departmentNumber: 929
employeeType: Employee
homePhone: +1 303 596-8980
initials: P. A.
mobile: +1 71 871-3752
pager: +1 510 315-9587
manager: cn=Marit Sezer
secretary: cn=Malgosia Deanda
roomNumber: 1252

dn: cn=Jeannine Cuddy, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jeannine Cuddy
sn: Cuddy
description: This is Jeannine Cuddy's description
facsimileTelephoneNumber: +1 303 187-3373
l: Redmond
ou: Accounting
postalAddress: example$Accounting$Dept # 535
telephoneNumber: +1 206 128-3964
title: Supreme Accounting Engineer
userPassword: ydduCeninn
uid: Jeannine_Cuddy
givenName: Jeannine
mail: Jeannine_Cuddy@example.com
carLicense: TPO84D7
departmentNumber: 3516
employeeType: Temp
homePhone: +1 206 664-2956
initials: J. C.
mobile: +1 213 322-4775
pager: +1 510 451-6191
manager: cn=Seline Dumouchel
secretary: cn=Penelope Merritt
roomNumber: 4118

dn: cn=Wynny Neuman, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Wynny Neuman
sn: Neuman
description: This is Wynny Neuman's description
facsimileTelephoneNumber: +1 303 177-8824
l: Orem
ou: Janitorial
postalAddress: example$Janitorial$Dept # 746
telephoneNumber: +1 408 469-1079
title: Chief Janitorial Engineer
userPassword: namueNynny
uid: Wynny_Neuman
givenName: Wynny
mail: Wynny_Neuman@example.com
carLicense: AAGR9YJ
departmentNumber: 3517
employeeType: Temp
homePhone: +1 408 189-9296
initials: W. N.
mobile: +1 818 146-2637
pager: +1 804 367-4966
manager: cn=Milena Guilford
secretary: cn=Norel Zitzmann
roomNumber: 5614

dn: cn=Minne Herrington, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Minne Herrington
sn: Herrington
description: This is Minne Herrington's description
facsimileTelephoneNumber: +1 408 660-3610
l: Orem
ou: Product Testing
postalAddress: example$Product Testing$Dept # 953
telephoneNumber: +1 818 484-6125
title: Elite Product Testing Technician
userPassword: notgnirreH
uid: Minne_Herrington
givenName: Minne
mail: Minne_Herrington@example.com
carLicense: 64XGONI
departmentNumber: 9271
employeeType: Manager
homePhone: +1 206 347-4089
initials: M. H.
mobile: +1 303 797-4126
pager: +1 213 191-5947
manager: cn=Sashenka Stratton
secretary: cn=Corella Sills
roomNumber: 9774

dn: cn=Bruce Crowe, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bruce Crowe
sn: Crowe
description: This is Bruce Crowe's description
facsimileTelephoneNumber: +1 71 452-5871
l: Redmond
ou: Peons
postalAddress: example$Peons$Dept # 764
telephoneNumber: +1 415 742-8040
title: Elite Peons Writer
userPassword: eworCecurB
uid: Bruce_Crowe
givenName: Bruce
mail: Bruce_Crowe@example.com
carLicense: S47E8B5
departmentNumber: 8154
employeeType: Manager
homePhone: +1 415 297-4855
initials: B. C.
mobile: +1 818 120-1713
pager: +1 408 406-9183
manager: cn=Guendolen Terwilligar
secretary: cn=Message Beers
roomNumber: 3612

dn: cn=Amos Colbert, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Amos Colbert
sn: Colbert
description: This is Amos Colbert's description
facsimileTelephoneNumber: +1 206 160-5525
l: Milpitas
ou: Accounting
postalAddress: example$Accounting$Dept # 853
telephoneNumber: +1 818 275-1824
title: Supreme Accounting Writer
userPassword: trebloCsom
uid: Amos_Colbert
givenName: Amos
mail: Amos_Colbert@example.com
carLicense: JIYCIV7
departmentNumber: 5967
employeeType: Contract
homePhone: +1 415 400-1423
initials: A. C.
mobile: +1 818 589-5302
pager: +1 818 604-2500
manager: cn=Sashenka Aboul-Magd
secretary: cn=Etta Calkins
roomNumber: 5557

dn: cn=Ann Tully, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ann Tully
sn: Tully
description: This is Ann Tully's description
facsimileTelephoneNumber: +1 213 206-5551
l: Cambridge
ou: Accounting
postalAddress: example$Accounting$Dept # 784
telephoneNumber: +1 415 208-1794
title: Supreme Accounting Admin
userPassword: ylluTnnA
uid: Ann_Tully
givenName: Ann
mail: Ann_Tully@example.com
carLicense: 3BM6G8F
departmentNumber: 1672
employeeType: Employee
homePhone: +1 213 246-2575
initials: A. T.
mobile: +1 213 889-1457
pager: +1 71 105-8458
manager: cn=Buddy Wingfield
secretary: cn=Minny Vahdat
roomNumber: 538

dn: cn=Sonny Pepe, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sonny Pepe
sn: Pepe
description: This is Sonny Pepe's description
facsimileTelephoneNumber: +1 303 696-1335
l: San Mateo
ou: Peons
postalAddress: example$Peons$Dept # 263
telephoneNumber: +1 818 255-2500
title: Associate Peons Janitor
userPassword: epePynnoS
uid: Sonny_Pepe
givenName: Sonny
mail: Sonny_Pepe@example.com
carLicense: FJ3ARDR
departmentNumber: 8739
employeeType: Temp
homePhone: +1 804 260-2967
initials: S. P.
mobile: +1 804 996-2579
pager: +1 71 431-1746
manager: cn=Joon Chen
secretary: cn=Linzie Linebarger
roomNumber: 1155

dn: cn=Laurna Barham, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Laurna Barham
sn: Barham
description: This is Laurna Barham's description
facsimileTelephoneNumber: +1 804 914-7533
l: Redwood Shores
ou: Human Resources
postalAddress: example$Human Resources$Dept # 27
telephoneNumber: +1 804 746-8767
title: Master Human Resources Grunt
userPassword: mahraBanru
uid: Laurna_Barham
givenName: Laurna
mail: Laurna_Barham@example.com
carLicense: 9W6J8D0
departmentNumber: 8884
employeeType: Normal
homePhone: +1 71 841-6532
initials: L. B.
mobile: +1 415 818-9707
pager: +1 303 710-6265
manager: cn=Mariya Mancini
secretary: cn=Rong-Chin Bowser
roomNumber: 7947

dn: cn=Blinny Brassem, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Blinny Brassem
sn: Brassem
description: This is Blinny Brassem's description
facsimileTelephoneNumber: +1 804 489-8374
l: Redwood Shores
ou: Human Resources
postalAddress: example$Human Resources$Dept # 611
telephoneNumber: +1 510 245-7901
title: Master Human Resources Dictator
userPassword: messarBynn
uid: Blinny_Brassem
givenName: Blinny
mail: Blinny_Brassem@example.com
carLicense: AL2I4UO
departmentNumber: 2597
employeeType: Manager
homePhone: +1 408 683-5120
initials: B. B.
mobile: +1 818 332-4217
pager: +1 71 342-3678
manager: cn=Aaron Odegaard
secretary: cn=Len Chouhan
roomNumber: 240

dn: cn=Hildagarde Naphan, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Hildagarde Naphan
sn: Naphan
description: This is Hildagarde Naphan's description
facsimileTelephoneNumber: +1 206 511-6335
l: Emeryville
ou: Payroll
postalAddress: example$Payroll$Dept # 487
telephoneNumber: +1 71 139-1287
title: Elite Payroll Figurehead
userPassword: nahpaNedra
uid: Hildagarde_Naphan
givenName: Hildagarde
mail: Hildagarde_Naphan@example.com
carLicense: HOHZ7A7
departmentNumber: 1836
employeeType: Contract
homePhone: +1 408 156-7583
initials: H. N.
mobile: +1 415 130-2609
pager: +1 415 275-6325
manager: cn=Amato Kurtz
secretary: cn=Prue Khatri
roomNumber: 3152

dn: cn=Delfin Labarge, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Delfin Labarge
sn: Labarge
description: This is Delfin Labarge's description
facsimileTelephoneNumber: +1 804 737-2330
l: Santa Clara
ou: Payroll
postalAddress: example$Payroll$Dept # 541
telephoneNumber: +1 510 882-3250
title: Senior Payroll Technician
userPassword: egrabaLnif
uid: Delfin_Labarge
givenName: Delfin
mail: Delfin_Labarge@example.com
carLicense: N9ZUDX9
departmentNumber: 6524
employeeType: Contract
homePhone: +1 303 121-6027
initials: D. L.
mobile: +1 408 905-7544
pager: +1 408 119-2332
manager: cn=Daveen Jeronimo
secretary: cn=Boris Lyman
roomNumber: 4080

dn: cn=Rubi MAINT, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rubi MAINT
sn: MAINT
description: This is Rubi MAINT's description
facsimileTelephoneNumber: +1 415 304-4441
l: San Francisco
ou: Product Testing
postalAddress: example$Product Testing$Dept # 844
telephoneNumber: +1 213 220-5777
title: Chief Product Testing Mascot
userPassword: TNIAMibuR
uid: Rubi_MAINT
givenName: Rubi
mail: Rubi_MAINT@example.com
carLicense: AOO04R0
departmentNumber: 1212
employeeType: Normal
homePhone: +1 415 822-4369
initials: R. M.
mobile: +1 408 768-7514
pager: +1 71 699-3787
manager: cn=Ajay Monforton
secretary: cn=Marice Almon
roomNumber: 3254

dn: cn=Muriel Oka, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Muriel Oka
sn: Oka
description: This is Muriel Oka's description
facsimileTelephoneNumber: +1 408 312-4431
l: Santa Clara
ou: Accounting
postalAddress: example$Accounting$Dept # 400
telephoneNumber: +1 804 119-4291
title: Master Accounting Engineer
userPassword: akOleiruM
uid: Muriel_Oka
givenName: Muriel
mail: Muriel_Oka@example.com
carLicense: SMK0IG5
departmentNumber: 493
employeeType: Normal
homePhone: +1 303 362-6176
initials: M. O.
mobile: +1 510 446-7985
pager: +1 71 506-3623
manager: cn=Tarah Kastner
secretary: cn=Mattie Gung
roomNumber: 1631

dn: cn=Blondelle Glasser, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Blondelle Glasser
sn: Glasser
description: This is Blondelle Glasser's description
facsimileTelephoneNumber: +1 804 996-9926
l: Mountain View
ou: Management
postalAddress: example$Management$Dept # 50
telephoneNumber: +1 804 499-2765
title: Supreme Management Yahoo
userPassword: ressalGell
uid: Blondelle_Glasser
givenName: Blondelle
mail: Blondelle_Glasser@example.com
carLicense: DZUB0VG
departmentNumber: 4548
employeeType: Manager
homePhone: +1 408 702-6355
initials: B. G.
mobile: +1 206 642-4016
pager: +1 408 610-1359
manager: cn=Murray Lappan
secretary: cn=Nazib Dingman
roomNumber: 8152

dn: cn=Marge Marouchos, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marge Marouchos
sn: Marouchos
description: This is Marge Marouchos's description
facsimileTelephoneNumber: +1 71 551-4436
l: San Mateo
ou: Product Development
postalAddress: example$Product Development$Dept # 217
telephoneNumber: +1 71 749-5826
title: Junior Product Development Sales Rep
userPassword: sohcuoraMe
uid: Marge_Marouchos
givenName: Marge
mail: Marge_Marouchos@example.com
carLicense: 72DUN23
departmentNumber: 4093
employeeType: Manager
homePhone: +1 818 406-9914
initials: M. M.
mobile: +1 415 855-9769
pager: +1 206 962-5794
manager: cn=Lin Tombul
secretary: cn=Arielle Herscovici
roomNumber: 1055

dn: cn=Chi-Keung Swartz, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Chi-Keung Swartz
sn: Swartz
description: This is Chi-Keung Swartz's description
facsimileTelephoneNumber: +1 206 569-5681
l: Palo Alto
ou: Payroll
postalAddress: example$Payroll$Dept # 796
telephoneNumber: +1 213 821-1433
title: Supreme Payroll Figurehead
userPassword: ztrawSgnue
uid: Chi-Keung_Swartz
givenName: Chi-Keung
mail: Chi-Keung_Swartz@example.com
carLicense: 4NL27ET
departmentNumber: 7242
employeeType: Manager
homePhone: +1 71 702-2781
initials: C. S.
mobile: +1 408 146-6036
pager: +1 71 191-6004
manager: cn=Valli Schlagenhauf
secretary: cn=Rici Massonneau
roomNumber: 2917

dn: cn=Tasia SVM-BNRMTVA, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tasia SVM-BNRMTVA
sn: SVM-BNRMTVA
description: This is Tasia SVM-BNRMTVA's description
facsimileTelephoneNumber: +1 408 528-5158
l: Sunnyvale
ou: Janitorial
postalAddress: example$Janitorial$Dept # 231
telephoneNumber: +1 408 323-7053
title: Junior Janitorial Figurehead
userPassword: AVTMRNB-MV
uid: Tasia_SVM-BNRMTVA
givenName: Tasia
mail: Tasia_SVM-BNRMTVA@example.com
carLicense: NY4N060
departmentNumber: 2767
employeeType: Manager
homePhone: +1 303 603-1682
initials: T. S.
mobile: +1 415 219-4966
pager: +1 303 402-4396
manager: cn=Xylia Deitiker
secretary: cn=Ludovika Ayres
roomNumber: 6079

dn: cn=Jack Jolliffe, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jack Jolliffe
sn: Jolliffe
description: This is Jack Jolliffe's description
facsimileTelephoneNumber: +1 206 463-1810
l: San Francisco
ou: Peons
postalAddress: example$Peons$Dept # 630
telephoneNumber: +1 408 928-7358
title: Senior Peons Consultant
userPassword: effilloJkc
uid: Jack_Jolliffe
givenName: Jack
mail: Jack_Jolliffe@example.com
carLicense: O1Z8TV9
departmentNumber: 6564
employeeType: Contract
homePhone: +1 408 399-2117
initials: J. J.
mobile: +1 818 236-3937
pager: +1 408 507-7328
manager: cn=Ross Missailidis
secretary: cn=Jany Hanser
roomNumber: 5644

dn: cn=Kathryn Schwaderer, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kathryn Schwaderer
sn: Schwaderer
description: This is Kathryn Schwaderer's description
facsimileTelephoneNumber: +1 408 322-7580
l: Menlo Park
ou: Planning
postalAddress: example$Planning$Dept # 993
telephoneNumber: +1 510 451-7822
title: Senior Planning Technician
userPassword: reredawhcS
uid: Kathryn_Schwaderer
givenName: Kathryn
mail: Kathryn_Schwaderer@example.com
carLicense: FME00W1
departmentNumber: 6103
employeeType: Employee
homePhone: +1 213 805-9060
initials: K. S.
mobile: +1 213 979-5400
pager: +1 206 230-6702
manager: cn=Anna-maria Renfro
secretary: cn=Quang-Trung Heybroek
roomNumber: 9592

dn: cn=Elton Churas, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Elton Churas
sn: Churas
description: This is Elton Churas's description
facsimileTelephoneNumber: +1 818 132-9892
l: Alameda
ou: Accounting
postalAddress: example$Accounting$Dept # 640
telephoneNumber: +1 303 880-8926
title: Master Accounting Mascot
userPassword: saruhCnotl
uid: Elton_Churas
givenName: Elton
mail: Elton_Churas@example.com
carLicense: 6MSE0G4
departmentNumber: 9944
employeeType: Employee
homePhone: +1 206 248-3924
initials: E. C.
mobile: +1 804 202-6373
pager: +1 206 382-8956
manager: cn=Susann Smulders
secretary: cn=Alfreda Scribner
roomNumber: 1279

dn: cn=Lenna Masse, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lenna Masse
sn: Masse
description: This is Lenna Masse's description
facsimileTelephoneNumber: +1 71 948-3740
l: Sunnyvale
ou: Janitorial
postalAddress: example$Janitorial$Dept # 241
telephoneNumber: +1 804 196-5082
title: Senior Janitorial Pinhead
userPassword: essaManneL
uid: Lenna_Masse
givenName: Lenna
mail: Lenna_Masse@example.com
carLicense: K63LETM
departmentNumber: 876
employeeType: Manager
homePhone: +1 408 282-9524
initials: L. M.
mobile: +1 415 885-7002
pager: +1 510 579-6621
manager: cn=Cooper Hollack
secretary: cn=Irc Schiegl
roomNumber: 45

dn: cn=Baines Jarboe, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Baines Jarboe
sn: Jarboe
description: This is Baines Jarboe's description
facsimileTelephoneNumber: +1 408 493-6631
l: Cambridge
ou: Janitorial
postalAddress: example$Janitorial$Dept # 273
telephoneNumber: +1 818 514-9417
title: Supreme Janitorial Engineer
userPassword: eobraJseni
uid: Baines_Jarboe
givenName: Baines
mail: Baines_Jarboe@example.com
carLicense: ZVHGOKW
departmentNumber: 205
employeeType: Contract
homePhone: +1 818 725-3631
initials: B. J.
mobile: +1 71 532-7345
pager: +1 408 484-7840
manager: cn=Lynette Pon
secretary: cn=Myron Irvine
roomNumber: 3607

dn: cn=Brigitte Lappan, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Brigitte Lappan
sn: Lappan
description: This is Brigitte Lappan's description
facsimileTelephoneNumber: +1 804 740-2837
l: San Jose
ou: Product Testing
postalAddress: example$Product Testing$Dept # 448
telephoneNumber: +1 408 616-4824
title: Associate Product Testing Writer
userPassword: nappaLetti
uid: Brigitte_Lappan
givenName: Brigitte
mail: Brigitte_Lappan@example.com
carLicense: ONIGC8W
departmentNumber: 8371
employeeType: Temp
homePhone: +1 206 896-5612
initials: B. L.
mobile: +1 408 705-1940
pager: +1 818 655-9906
manager: cn=Jenine Sayed
secretary: cn=Ernestine Zisu
roomNumber: 7395

dn: cn=Shan McNamara, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shan McNamara
sn: McNamara
description: This is Shan McNamara's description
facsimileTelephoneNumber: +1 408 278-8634
l: San Mateo
ou: Peons
postalAddress: example$Peons$Dept # 828
telephoneNumber: +1 71 428-3947
title: Senior Peons Yahoo
userPassword: aramaNcMna
uid: Shan_McNamara
givenName: Shan
mail: Shan_McNamara@example.com
carLicense: H800UZJ
departmentNumber: 6157
employeeType: Normal
homePhone: +1 804 115-3589
initials: S. M.
mobile: +1 213 645-1126
pager: +1 206 210-4418
manager: cn=Ammamaria Rains
secretary: cn=Tsing Millaire
roomNumber: 5359

dn: cn=Kristopher Gervaise, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kristopher Gervaise
sn: Gervaise
description: This is Kristopher Gervaise's description
facsimileTelephoneNumber: +1 408 358-3860
l: Cambridge
ou: Product Testing
postalAddress: example$Product Testing$Dept # 576
telephoneNumber: +1 804 485-3937
title: Master Product Testing Admin
userPassword: esiavreGre
uid: Kristopher_Gervaise
givenName: Kristopher
mail: Kristopher_Gervaise@example.com
carLicense: HURNQK1
departmentNumber: 9750
employeeType: Temp
homePhone: +1 510 673-8200
initials: K. G.
mobile: +1 206 213-1403
pager: +1 206 853-1047
manager: cn=Freek McElhone
secretary: cn=Anderea Caterina
roomNumber: 8167

dn: cn=Pinecrest Nizamuddin, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Pinecrest Nizamuddin
sn: Nizamuddin
description: This is Pinecrest Nizamuddin's description
facsimileTelephoneNumber: +1 213 268-6719
l: Milpitas
ou: Administrative
postalAddress: example$Administrative$Dept # 431
telephoneNumber: +1 206 997-2066
title: Senior Administrative Architect
userPassword: niddumaziN
uid: Pinecrest_Nizamuddin
givenName: Pinecrest
mail: Pinecrest_Nizamuddin@example.com
carLicense: 92AL2GK
departmentNumber: 569
employeeType: Temp
homePhone: +1 303 221-4735
initials: P. N.
mobile: +1 804 451-9720
pager: +1 213 461-2793
manager: cn=Lab Stanke
secretary: cn=Deborah Zagrodney
roomNumber: 9090

dn: cn=Sapphira Gurgenci, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sapphira Gurgenci
sn: Gurgenci
description: This is Sapphira Gurgenci's description
facsimileTelephoneNumber: +1 408 691-8785
l: San Francisco
ou: Product Development
postalAddress: example$Product Development$Dept # 501
telephoneNumber: +1 213 267-8152
title: Elite Product Development Consultant
userPassword: icnegruGar
uid: Sapphira_Gurgenci
givenName: Sapphira
mail: Sapphira_Gurgenci@example.com
carLicense: AYX8O7Q
departmentNumber: 3501
employeeType: Contract
homePhone: +1 408 991-3370
initials: S. G.
mobile: +1 206 586-3106
pager: +1 408 257-3189
manager: cn=Shannah Kwant
secretary: cn=Seyma Burrows
roomNumber: 7601

dn: cn=Phebe Courchesne, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Phebe Courchesne
sn: Courchesne
description: This is Phebe Courchesne's description
facsimileTelephoneNumber: +1 71 375-1979
l: Sunnyvale
ou: Product Testing
postalAddress: example$Product Testing$Dept # 606
telephoneNumber: +1 71 647-6824
title: Master Product Testing Yahoo
userPassword: ensehcruoC
uid: Phebe_Courchesne
givenName: Phebe
mail: Phebe_Courchesne@example.com
carLicense: FEB9JD1
departmentNumber: 2939
employeeType: Manager
homePhone: +1 408 787-9165
initials: P. C.
mobile: +1 818 955-8666
pager: +1 213 378-9440
manager: cn=Erkan Godowsky
secretary: cn=Padriac Norwood
roomNumber: 7633

dn: cn=Allsun Nolter, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Allsun Nolter
sn: Nolter
description: This is Allsun Nolter's description
facsimileTelephoneNumber: +1 71 470-8445
l: Emeryville
ou: Accounting
postalAddress: example$Accounting$Dept # 168
telephoneNumber: +1 408 187-7850
title: Elite Accounting Manager
userPassword: retloNnusl
uid: Allsun_Nolter
givenName: Allsun
mail: Allsun_Nolter@example.com
carLicense: AANBF8M
departmentNumber: 4139
employeeType: Contract
homePhone: +1 71 160-5067
initials: A. N.
mobile: +1 408 151-3182
pager: +1 206 639-6024
manager: cn=Selva Chee
secretary: cn=Dorri Cavan
roomNumber: 4350

dn: cn=Leonelle Abdo, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Leonelle Abdo
sn: Abdo
description: This is Leonelle Abdo's description
facsimileTelephoneNumber: +1 206 965-8823
l: Emeryville
ou: Product Testing
postalAddress: example$Product Testing$Dept # 912
telephoneNumber: +1 206 493-7340
title: Junior Product Testing Stooge
userPassword: odbAelleno
uid: Leonelle_Abdo
givenName: Leonelle
mail: Leonelle_Abdo@example.com
carLicense: FQBC5SN
departmentNumber: 5310
employeeType: Employee
homePhone: +1 206 181-2683
initials: L. A.
mobile: +1 71 149-9722
pager: +1 303 696-8266
manager: cn=Bosiljka Mittleider
secretary: cn=Umeko Papageorgiou
roomNumber: 1575

dn: cn=Amalia Crowder, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Amalia Crowder
sn: Crowder
description: This is Amalia Crowder's description
facsimileTelephoneNumber: +1 71 522-1890
l: Fremont
ou: Janitorial
postalAddress: example$Janitorial$Dept # 186
telephoneNumber: +1 818 325-3439
title: Master Janitorial Artist
userPassword: redworCail
uid: Amalia_Crowder
givenName: Amalia
mail: Amalia_Crowder@example.com
carLicense: FC6W628
departmentNumber: 75
employeeType: Normal
homePhone: +1 71 748-7975
initials: A. C.
mobile: +1 510 337-7154
pager: +1 303 441-3224
manager: cn=Angelo Lask
secretary: cn=Adrie Kaczmarek
roomNumber: 3772

dn: cn=Lonni Schellenberger, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lonni Schellenberger
sn: Schellenberger
description: This is Lonni Schellenberger's description
facsimileTelephoneNumber: +1 71 366-9151
l: Alameda
ou: Human Resources
postalAddress: example$Human Resources$Dept # 279
telephoneNumber: +1 510 193-1252
title: Junior Human Resources Visionary
userPassword: regrebnell
uid: Lonni_Schellenberger
givenName: Lonni
mail: Lonni_Schellenberger@example.com
carLicense: 17BC5GC
departmentNumber: 9439
employeeType: Normal
homePhone: +1 415 655-4002
initials: L. S.
mobile: +1 206 551-9580
pager: +1 408 847-3050
manager: cn=Dulciana Zeller
secretary: cn=Bailey Chaurasia
roomNumber: 837

dn: cn=Michie Stern, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Michie Stern
sn: Stern
description: This is Michie Stern's description
facsimileTelephoneNumber: +1 510 805-1860
l: Redmond
ou: Human Resources
postalAddress: example$Human Resources$Dept # 633
telephoneNumber: +1 818 201-2629
title: Chief Human Resources Fellow
userPassword: nretSeihci
uid: Michie_Stern
givenName: Michie
mail: Michie_Stern@example.com
carLicense: VLXXKHI
departmentNumber: 9632
employeeType: Manager
homePhone: +1 804 665-1017
initials: M. S.
mobile: +1 71 496-5888
pager: +1 71 990-2076
manager: cn=Domenic Racz
secretary: cn=Adelaida Sims
roomNumber: 3078

dn: cn=Sedat Seabrook, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sedat Seabrook
sn: Seabrook
description: This is Sedat Seabrook's description
facsimileTelephoneNumber: +1 510 142-5630
l: Alameda
ou: Human Resources
postalAddress: example$Human Resources$Dept # 611
telephoneNumber: +1 415 668-5765
title: Supreme Human Resources Director
userPassword: koorbaeSta
uid: Sedat_Seabrook
givenName: Sedat
mail: Sedat_Seabrook@example.com
carLicense: J0RFQMY
departmentNumber: 6654
employeeType: Temp
homePhone: +1 71 959-4365
initials: S. S.
mobile: +1 213 182-7712
pager: +1 408 834-6188
manager: cn=Klaus Lilleniit
secretary: cn=Gabey Rodschat
roomNumber: 6944

dn: cn=Ruben Bottomley, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ruben Bottomley
sn: Bottomley
description: This is Ruben Bottomley's description
facsimileTelephoneNumber: +1 804 374-6124
l: Fremont
ou: Product Development
postalAddress: example$Product Development$Dept # 721
telephoneNumber: +1 206 168-9537
title: Supreme Product Development Grunt
userPassword: yelmottoBn
uid: Ruben_Bottomley
givenName: Ruben
mail: Ruben_Bottomley@example.com
carLicense: XSDZRAM
departmentNumber: 6356
employeeType: Employee
homePhone: +1 415 832-6047
initials: R. B.
mobile: +1 206 957-3798
pager: +1 408 940-5832
manager: cn=Jaquelin Gros
secretary: cn=Utilla Watters
roomNumber: 9316

dn: cn=Fernandina Sherrard, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Fernandina Sherrard
sn: Sherrard
description: This is Fernandina Sherrard's description
facsimileTelephoneNumber: +1 415 583-7584
l: Mountain View
ou: Accounting
postalAddress: example$Accounting$Dept # 744
telephoneNumber: +1 71 284-3179
title: Senior Accounting Sales Rep
userPassword: drarrehSan
uid: Fernandina_Sherrard
givenName: Fernandina
mail: Fernandina_Sherrard@example.com
carLicense: IL4Z7WL
departmentNumber: 1484
employeeType: Contract
homePhone: +1 415 848-3023
initials: F. S.
mobile: +1 804 352-1012
pager: +1 408 544-7222
manager: cn=Eugine Melton
secretary: cn=Vania Guajardo
roomNumber: 713

dn: cn=Dasha Pallen, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dasha Pallen
sn: Pallen
description: This is Dasha Pallen's description
facsimileTelephoneNumber: +1 206 791-8512
l: Menlo Park
ou: Product Testing
postalAddress: example$Product Testing$Dept # 996
telephoneNumber: +1 818 587-6679
title: Master Product Testing Evangelist
userPassword: nellaPahsa
uid: Dasha_Pallen
givenName: Dasha
mail: Dasha_Pallen@example.com
carLicense: C037ZQQ
departmentNumber: 6785
employeeType: Manager
homePhone: +1 510 218-2016
initials: D. P.
mobile: +1 510 616-2869
pager: +1 818 116-6325
manager: cn=Bevvy Wilken
secretary: cn=Daffi Gentes
roomNumber: 3119

dn: cn=Claribel Al-Basi, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Claribel Al-Basi
sn: Al-Basi
description: This is Claribel Al-Basi's description
facsimileTelephoneNumber: +1 206 133-6183
l: Orem
ou: Payroll
postalAddress: example$Payroll$Dept # 864
telephoneNumber: +1 818 342-4711
title: Chief Payroll Assistant
userPassword: isaB-lAleb
uid: Claribel_Al-Basi
givenName: Claribel
mail: Claribel_Al-Basi@example.com
carLicense: G4U4HOJ
departmentNumber: 354
employeeType: Employee
homePhone: +1 804 800-1021
initials: C. A.
mobile: +1 213 940-5595
pager: +1 415 207-6626
manager: cn=Felton Jarboe
secretary: cn=Ignace Bilodeau
roomNumber: 5491

dn: cn=Kimberlyn Wegener, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kimberlyn Wegener
sn: Wegener
description: This is Kimberlyn Wegener's description
facsimileTelephoneNumber: +1 206 660-8544
l: Santa Clara
ou: Peons
postalAddress: example$Peons$Dept # 327
telephoneNumber: +1 408 845-5707
title: Associate Peons Warrior
userPassword: renegeWnyl
uid: Kimberlyn_Wegener
givenName: Kimberlyn
mail: Kimberlyn_Wegener@example.com
carLicense: PJA64VA
departmentNumber: 4445
employeeType: Contract
homePhone: +1 510 711-8510
initials: K. W.
mobile: +1 510 304-4281
pager: +1 303 140-8992
manager: cn=Lulu Dobbing
secretary: cn=Eadie Pharr
roomNumber: 1028

dn: cn=Biddie Auld, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Biddie Auld
sn: Auld
description: This is Biddie Auld's description
facsimileTelephoneNumber: +1 415 226-8217
l: Milpitas
ou: Accounting
postalAddress: example$Accounting$Dept # 874
telephoneNumber: +1 408 296-3963
title: Senior Accounting Assistant
userPassword: dluAeiddiB
uid: Biddie_Auld
givenName: Biddie
mail: Biddie_Auld@example.com
carLicense: SUMH0IQ
departmentNumber: 1075
employeeType: Contract
homePhone: +1 206 589-6926
initials: B. A.
mobile: +1 408 661-2122
pager: +1 71 378-4702
manager: cn=Ermina Mansbridge
secretary: cn=Phan Heikkila
roomNumber: 7646

dn: cn=Farhad Hutter, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Farhad Hutter
sn: Hutter
description: This is Farhad Hutter's description
facsimileTelephoneNumber: +1 818 601-9336
l: Sunnyvale
ou: Accounting
postalAddress: example$Accounting$Dept # 473
telephoneNumber: +1 206 300-9853
title: Associate Accounting Stooge
userPassword: rettuHdahr
uid: Farhad_Hutter
givenName: Farhad
mail: Farhad_Hutter@example.com
carLicense: NBPV6JX
departmentNumber: 1786
employeeType: Normal
homePhone: +1 408 560-7509
initials: F. H.
mobile: +1 303 188-3309
pager: +1 818 683-2951
manager: cn=Isoft Donohoe
secretary: cn=CrystalBay Eller
roomNumber: 828

dn: cn=Vikki Williford, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Vikki Williford
sn: Williford
description: This is Vikki Williford's description
facsimileTelephoneNumber: +1 510 982-7778
l: Orem
ou: Accounting
postalAddress: example$Accounting$Dept # 770
telephoneNumber: +1 303 222-3373
title: Supreme Accounting Madonna
userPassword: drofilliWi
uid: Vikki_Williford
givenName: Vikki
mail: Vikki_Williford@example.com
carLicense: XOKSQDZ
departmentNumber: 1267
employeeType: Temp
homePhone: +1 415 548-4527
initials: V. W.
mobile: +1 213 627-9947
pager: +1 804 243-3048
manager: cn=Helga Friton
secretary: cn=Rand MacNaughton
roomNumber: 2412

dn: cn=Jessy Blumer, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jessy Blumer
sn: Blumer
description: This is Jessy Blumer's description
facsimileTelephoneNumber: +1 303 478-2053
l: Redwood Shores
ou: Accounting
postalAddress: example$Accounting$Dept # 773
telephoneNumber: +1 510 174-3564
title: Elite Accounting Developer
userPassword: remulBysse
uid: Jessy_Blumer
givenName: Jessy
mail: Jessy_Blumer@example.com
carLicense: 6KGX6B4
departmentNumber: 5820
employeeType: Contract
homePhone: +1 818 560-4802
initials: J. B.
mobile: +1 818 792-8722
pager: +1 213 882-2569
manager: cn=Mark Rabiasz
secretary: cn=Quon Van Mansum
roomNumber: 1446

dn: cn=Suk-Yin Zegray, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Suk-Yin Zegray
sn: Zegray
description: This is Suk-Yin Zegray's description
facsimileTelephoneNumber: +1 804 763-2385
l: San Francisco
ou: Peons
postalAddress: example$Peons$Dept # 862
telephoneNumber: +1 408 276-2390
title: Associate Peons Artist
userPassword: yargeZniY-
uid: Suk-Yin_Zegray
givenName: Suk-Yin
mail: Suk-Yin_Zegray@example.com
carLicense: G9UYGCP
departmentNumber: 7962
employeeType: Temp
homePhone: +1 71 607-8825
initials: S. Z.
mobile: +1 206 618-1371
pager: +1 303 312-2783
manager: cn=Bobbie Babalola
secretary: cn=Robbin Demir
roomNumber: 3119

dn: cn=Lida Ocone, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lida Ocone
sn: Ocone
description: This is Lida Ocone's description
facsimileTelephoneNumber: +1 415 828-6128
l: Milpitas
ou: Product Development
postalAddress: example$Product Development$Dept # 27
telephoneNumber: +1 303 610-2363
title: Supreme Product Development Mascot
userPassword: enocOadiL
uid: Lida_Ocone
givenName: Lida
mail: Lida_Ocone@example.com
carLicense: 3K9Z6R1
departmentNumber: 2105
employeeType: Contract
homePhone: +1 206 618-9130
initials: L. O.
mobile: +1 804 282-7906
pager: +1 213 849-8442
manager: cn=Klaus Stevanovic
secretary: cn=Durali Seale
roomNumber: 5223

dn: cn=Cherin Wirth, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Cherin Wirth
sn: Wirth
description: This is Cherin Wirth's description
facsimileTelephoneNumber: +1 213 135-2429
l: Orem
ou: Planning
postalAddress: example$Planning$Dept # 318
telephoneNumber: +1 510 760-3713
title: Senior Planning Stooge
userPassword: htriWnireh
uid: Cherin_Wirth
givenName: Cherin
mail: Cherin_Wirth@example.com
carLicense: ZQDJ9R8
departmentNumber: 235
employeeType: Employee
homePhone: +1 303 909-8841
initials: C. W.
mobile: +1 213 428-8896
pager: +1 510 412-1388
manager: cn=Amitie Kuniyasu
secretary: cn=Myranda Yabe
roomNumber: 4222

dn: cn=Bachittar Vakili, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bachittar Vakili
sn: Vakili
description: This is Bachittar Vakili's description
facsimileTelephoneNumber: +1 303 140-2053
l: Redwood Shores
ou: Administrative
postalAddress: example$Administrative$Dept # 118
telephoneNumber: +1 303 506-4751
title: Master Administrative Admin
userPassword: ilikaVratt
uid: Bachittar_Vakili
givenName: Bachittar
mail: Bachittar_Vakili@example.com
carLicense: GFHKN13
departmentNumber: 4202
employeeType: Contract
homePhone: +1 415 382-3217
initials: B. V.
mobile: +1 303 594-8319
pager: +1 408 659-3245
manager: cn=Vijai Gateley
secretary: cn=Siana Lonergan
roomNumber: 9700

dn: cn=Pacific NeKueey, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Pacific NeKueey
sn: NeKueey
description: This is Pacific NeKueey's description
facsimileTelephoneNumber: +1 804 283-6147
l: Sunnyvale
ou: Administrative
postalAddress: example$Administrative$Dept # 103
telephoneNumber: +1 206 208-2628
title: Senior Administrative Figurehead
userPassword: yeeuKeNcif
uid: Pacific_NeKueey
givenName: Pacific
mail: Pacific_NeKueey@example.com
carLicense: N6VUI6B
departmentNumber: 5507
employeeType: Contract
homePhone: +1 206 158-4665
initials: P. N.
mobile: +1 510 433-2871
pager: +1 206 590-7629
manager: cn=Karissa Ciskowski
secretary: cn=Cad Jemczyk
roomNumber: 1934

dn: cn=Mary-Ellen Mickens, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mary-Ellen Mickens
sn: Mickens
description: This is Mary-Ellen Mickens's description
facsimileTelephoneNumber: +1 303 620-6201
l: San Jose
ou: Product Development
postalAddress: example$Product Development$Dept # 905
telephoneNumber: +1 213 360-3449
title: Junior Product Development Vice President
userPassword: snekciMnel
uid: Mary-Ellen_Mickens
givenName: Mary-Ellen
mail: Mary-Ellen_Mickens@example.com
carLicense: P2ZDXNB
departmentNumber: 558
employeeType: Manager
homePhone: +1 206 217-1707
initials: M. M.
mobile: +1 408 262-3493
pager: +1 818 863-5815
manager: cn=Hanny Leppert
secretary: cn=Cami Marleau
roomNumber: 2024

dn: cn=Cally Spragg, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Cally Spragg
sn: Spragg
description: This is Cally Spragg's description
facsimileTelephoneNumber: +1 415 903-7518
l: Mountain View
ou: Product Testing
postalAddress: example$Product Testing$Dept # 679
telephoneNumber: +1 804 741-5221
title: Associate Product Testing Stooge
userPassword: ggarpSylla
uid: Cally_Spragg
givenName: Cally
mail: Cally_Spragg@example.com
carLicense: YWEB6T6
departmentNumber: 1965
employeeType: Manager
homePhone: +1 818 133-5730
initials: C. S.
mobile: +1 206 741-5733
pager: +1 71 283-2270
manager: cn=Kacy Scarborough
secretary: cn=Clarey Fernandez
roomNumber: 9511

dn: cn=Maud Marmion, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Maud Marmion
sn: Marmion
description: This is Maud Marmion's description
facsimileTelephoneNumber: +1 303 902-5212
l: Mountain View
ou: Management
postalAddress: example$Management$Dept # 610
telephoneNumber: +1 408 892-3797
title: Supreme Management Madonna
userPassword: noimraMdua
uid: Maud_Marmion
givenName: Maud
mail: Maud_Marmion@example.com
carLicense: 6DZPEI0
departmentNumber: 6679
employeeType: Temp
homePhone: +1 510 696-4058
initials: M. M.
mobile: +1 303 502-7495
pager: +1 415 503-1607
manager: cn=Berni Haggarty
secretary: cn=Niz Doda
roomNumber: 5540

dn: cn=Staci Gostanian, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Staci Gostanian
sn: Gostanian
description: This is Staci Gostanian's description
facsimileTelephoneNumber: +1 71 367-9393
l: Mountain View
ou: Human Resources
postalAddress: example$Human Resources$Dept # 716
telephoneNumber: +1 804 734-3749
title: Master Human Resources President
userPassword: nainatsoGi
uid: Staci_Gostanian
givenName: Staci
mail: Staci_Gostanian@example.com
carLicense: VJ8F6B6
departmentNumber: 8004
employeeType: Employee
homePhone: +1 408 704-8269
initials: S. G.
mobile: +1 408 871-9499
pager: +1 71 568-2234
manager: cn=Frederique Ismail
secretary: cn=Shanna Lindow
roomNumber: 6136

dn: cn=Rahal Berna, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rahal Berna
sn: Berna
description: This is Rahal Berna's description
facsimileTelephoneNumber: +1 71 671-1322
l: Palo Alto
ou: Peons
postalAddress: example$Peons$Dept # 894
telephoneNumber: +1 818 476-4058
title: Master Peons Vice President
userPassword: anreBlahaR
uid: Rahal_Berna
givenName: Rahal
mail: Rahal_Berna@example.com
carLicense: QMS2WG7
departmentNumber: 6229
employeeType: Employee
homePhone: +1 213 187-4022
initials: R. B.
mobile: +1 510 819-4491
pager: +1 510 764-8561
manager: cn=Felicdad Elgar
secretary: cn=Bihari Racioppi
roomNumber: 8851

dn: cn=Margery Pillars, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Margery Pillars
sn: Pillars
description: This is Margery Pillars's description
facsimileTelephoneNumber: +1 206 637-6697
l: Palo Alto
ou: Peons
postalAddress: example$Peons$Dept # 484
telephoneNumber: +1 213 188-6823
title: Senior Peons Figurehead
userPassword: sralliPyre
uid: Margery_Pillars
givenName: Margery
mail: Margery_Pillars@example.com
carLicense: 11CFP1X
departmentNumber: 8595
employeeType: Normal
homePhone: +1 818 407-8340
initials: M. P.
mobile: +1 408 715-7716
pager: +1 303 913-9229
manager: cn=Phoebe Kirkwood
secretary: cn=Pak-Jong Widdis
roomNumber: 7509

dn: cn=Quality Amiot, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Quality Amiot
sn: Amiot
description: This is Quality Amiot's description
facsimileTelephoneNumber: +1 415 879-7072
l: San Jose
ou: Administrative
postalAddress: example$Administrative$Dept # 217
telephoneNumber: +1 415 743-2151
title: Senior Administrative Visionary
userPassword: toimAytila
uid: Quality_Amiot
givenName: Quality
mail: Quality_Amiot@example.com
carLicense: 2A1DL33
departmentNumber: 9035
employeeType: Contract
homePhone: +1 804 610-8868
initials: Q. A.
mobile: +1 415 544-1599
pager: +1 213 693-7865
manager: cn=Franky Richards
secretary: cn=Benthem Coe
roomNumber: 6242

dn: cn=Zahir Zelenka, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zahir Zelenka
sn: Zelenka
description: This is Zahir Zelenka's description
facsimileTelephoneNumber: +1 71 701-1569
l: Santa Clara
ou: Payroll
postalAddress: example$Payroll$Dept # 301
telephoneNumber: +1 213 575-1866
title: Junior Payroll Janitor
userPassword: akneleZrih
uid: Zahir_Zelenka
givenName: Zahir
mail: Zahir_Zelenka@example.com
carLicense: TBES61F
departmentNumber: 1654
employeeType: Normal
homePhone: +1 510 895-4340
initials: Z. Z.
mobile: +1 206 276-9998
pager: +1 71 732-7615
manager: cn=Ardelis Virani
secretary: cn=Andriana Kelland
roomNumber: 1564

dn: cn=Antonella Fredrickson, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Antonella Fredrickson
sn: Fredrickson
description: This is Antonella Fredrickson's description
facsimileTelephoneNumber: +1 510 790-1410
l: Cupertino
ou: Janitorial
postalAddress: example$Janitorial$Dept # 500
telephoneNumber: +1 818 492-3209
title: Chief Janitorial Warrior
userPassword: noskcirder
uid: Antonella_Fredrickson
givenName: Antonella
mail: Antonella_Fredrickson@example.com
carLicense: RX64PKO
departmentNumber: 5689
employeeType: Normal
homePhone: +1 415 736-1290
initials: A. F.
mobile: +1 415 513-2503
pager: +1 804 302-6510
manager: cn=Barlas Whitton
secretary: cn=Laurella Viau
roomNumber: 8757

dn: cn=Trinh Demchuk, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Trinh Demchuk
sn: Demchuk
description: This is Trinh Demchuk's description
facsimileTelephoneNumber: +1 818 213-6650
l: San Jose
ou: Product Development
postalAddress: example$Product Development$Dept # 452
telephoneNumber: +1 213 787-7790
title: Associate Product Development Figurehead
userPassword: kuhcmeDhni
uid: Trinh_Demchuk
givenName: Trinh
mail: Trinh_Demchuk@example.com
carLicense: XAX9HO5
departmentNumber: 8623
employeeType: Manager
homePhone: +1 804 205-4726
initials: T. D.
mobile: +1 408 276-8026
pager: +1 206 127-5276
manager: cn=Ceciley Khouderchan
secretary: cn=Mary-Ann Andrade
roomNumber: 2871

dn: cn=Zeb VanStaalduinen, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zeb VanStaalduinen
sn: VanStaalduinen
description: This is Zeb VanStaalduinen's description
facsimileTelephoneNumber: +1 415 936-1700
l: Redwood Shores
ou: Administrative
postalAddress: example$Administrative$Dept # 762
telephoneNumber: +1 415 204-3833
title: Junior Administrative Manager
userPassword: neniudlaat
uid: Zeb_VanStaalduinen
givenName: Zeb
mail: Zeb_VanStaalduinen@example.com
carLicense: BSOMKQW
departmentNumber: 6214
employeeType: Temp
homePhone: +1 303 341-2412
initials: Z. V.
mobile: +1 206 495-3240
pager: +1 804 470-9172
manager: cn=Stacie Overcash
secretary: cn=Sella Hashimoto
roomNumber: 1068

dn: cn=Ted Nagle, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ted Nagle
sn: Nagle
description: This is Ted Nagle's description
facsimileTelephoneNumber: +1 804 309-7376
l: San Jose
ou: Product Development
postalAddress: example$Product Development$Dept # 19
telephoneNumber: +1 408 750-3517
title: Junior Product Development Architect
userPassword: elgaNdeT
uid: Ted_Nagle
givenName: Ted
mail: Ted_Nagle@example.com
carLicense: 1TRKHV4
departmentNumber: 6383
employeeType: Manager
homePhone: +1 818 390-2285
initials: T. N.
mobile: +1 303 710-5797
pager: +1 818 474-5333
manager: cn=Bibbie Steene
secretary: cn=Irving Deguines
roomNumber: 2986

dn: cn=Clarissa Shankar, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Clarissa Shankar
sn: Shankar
description: This is Clarissa Shankar's description
facsimileTelephoneNumber: +1 303 244-7958
l: San Mateo
ou: Administrative
postalAddress: example$Administrative$Dept # 772
telephoneNumber: +1 303 887-7308
title: Chief Administrative Janitor
userPassword: raknahSass
uid: Clarissa_Shankar
givenName: Clarissa
mail: Clarissa_Shankar@example.com
carLicense: ZD2DCZ2
departmentNumber: 2933
employeeType: Manager
homePhone: +1 213 420-4842
initials: C. S.
mobile: +1 408 647-4909
pager: +1 510 946-9346
manager: cn=Dusan Malynowsky
secretary: cn=Fwp Trink
roomNumber: 4288

dn: cn=Querida Kauffeldt, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Querida Kauffeldt
sn: Kauffeldt
description: This is Querida Kauffeldt's description
facsimileTelephoneNumber: +1 408 372-5810
l: Santa Clara
ou: Peons
postalAddress: example$Peons$Dept # 201
telephoneNumber: +1 71 217-3988
title: Associate Peons Manager
userPassword: tdleffuaKa
uid: Querida_Kauffeldt
givenName: Querida
mail: Querida_Kauffeldt@example.com
carLicense: J95JMC4
departmentNumber: 5907
employeeType: Manager
homePhone: +1 510 683-1922
initials: Q. K.
mobile: +1 818 671-3371
pager: +1 213 139-6642
manager: cn=Ethelin Ajersch
secretary: cn=Muire Witzman
roomNumber: 136

dn: cn=Mireielle Snoke, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mireielle Snoke
sn: Snoke
description: This is Mireielle Snoke's description
facsimileTelephoneNumber: +1 804 621-8119
l: Cambridge
ou: Janitorial
postalAddress: example$Janitorial$Dept # 123
telephoneNumber: +1 415 937-3694
title: Supreme Janitorial Accountant
userPassword: ekonSellei
uid: Mireielle_Snoke
givenName: Mireielle
mail: Mireielle_Snoke@example.com
carLicense: AMBEMKN
departmentNumber: 3851
employeeType: Employee
homePhone: +1 415 378-1661
initials: M. S.
mobile: +1 408 890-7854
pager: +1 303 496-7013
manager: cn=Eva Meilleur
secretary: cn=Turgay Verch
roomNumber: 2976

dn: cn=Dpnlab Aziz, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dpnlab Aziz
sn: Aziz
description: This is Dpnlab Aziz's description
facsimileTelephoneNumber: +1 510 417-7276
l: San Francisco
ou: Human Resources
postalAddress: example$Human Resources$Dept # 592
telephoneNumber: +1 206 937-8375
title: Chief Human Resources Architect
userPassword: zizAbalnpD
uid: Dpnlab_Aziz
givenName: Dpnlab
mail: Dpnlab_Aziz@example.com
carLicense: 9A51V0Q
departmentNumber: 682
employeeType: Normal
homePhone: +1 206 790-7182
initials: D. A.
mobile: +1 206 122-9748
pager: +1 206 798-7925
manager: cn=Carol Maxsom
secretary: cn=Noemi Antinucci
roomNumber: 8736

dn: cn=Sangman Barkley, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sangman Barkley
sn: Barkley
description: This is Sangman Barkley's description
facsimileTelephoneNumber: +1 71 500-5644
l: Fremont
ou: Management
postalAddress: example$Management$Dept # 996
telephoneNumber: +1 408 311-1092
title: Supreme Management Architect
userPassword: yelkraBnam
uid: Sangman_Barkley
givenName: Sangman
mail: Sangman_Barkley@example.com
carLicense: Q6NXZYM
departmentNumber: 1502
employeeType: Temp
homePhone: +1 206 308-5959
initials: S. B.
mobile: +1 510 445-5807
pager: +1 71 371-2984
manager: cn=Yoshiaki Teo
secretary: cn=Faunie McDavitt
roomNumber: 9259

dn: cn=Berte Pinto-Lobo, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Berte Pinto-Lobo
sn: Pinto-Lobo
description: This is Berte Pinto-Lobo's description
facsimileTelephoneNumber: +1 415 421-8614
l: Sunnyvale
ou: Human Resources
postalAddress: example$Human Resources$Dept # 586
telephoneNumber: +1 303 613-7314
title: Elite Human Resources Consultant
userPassword: oboL-otniP
uid: Berte_Pinto-Lobo
givenName: Berte
mail: Berte_Pinto-Lobo@example.com
carLicense: SLA93UX
departmentNumber: 8395
employeeType: Temp
homePhone: +1 510 689-1651
initials: B. P.
mobile: +1 303 923-4172
pager: +1 818 454-1061
manager: cn=Vax Venier
secretary: cn=Dorella Gabbai
roomNumber: 6290

dn: cn=Charene Lawlis, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Charene Lawlis
sn: Lawlis
description: This is Charene Lawlis's description
facsimileTelephoneNumber: +1 303 652-1726
l: Milpitas
ou: Management
postalAddress: example$Management$Dept # 464
telephoneNumber: +1 303 648-1740
title: Master Management Admin
userPassword: silwaLener
uid: Charene_Lawlis
givenName: Charene
mail: Charene_Lawlis@example.com
carLicense: BDWA5OA
departmentNumber: 1826
employeeType: Normal
homePhone: +1 213 762-9479
initials: C. L.
mobile: +1 71 685-9600
pager: +1 510 925-1628
manager: cn=Aggie Davis
secretary: cn=Corenda Crowley
roomNumber: 2260

dn: cn=Electra McIntosh, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Electra McIntosh
sn: McIntosh
description: This is Electra McIntosh's description
facsimileTelephoneNumber: +1 71 826-9150
l: San Jose
ou: Janitorial
postalAddress: example$Janitorial$Dept # 324
telephoneNumber: +1 303 780-4869
title: Associate Janitorial Dictator
userPassword: hsotnIcMar
uid: Electra_McIntosh
givenName: Electra
mail: Electra_McIntosh@example.com
carLicense: G2N34UI
departmentNumber: 6666
employeeType: Employee
homePhone: +1 71 498-7760
initials: E. M.
mobile: +1 213 978-7520
pager: +1 408 921-1366
manager: cn=Bue Arnold
secretary: cn=Stephane Alexander
roomNumber: 812

dn: cn=Kaleena Drwiega, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kaleena Drwiega
sn: Drwiega
description: This is Kaleena Drwiega's description
facsimileTelephoneNumber: +1 213 556-1110
l: Milpitas
ou: Human Resources
postalAddress: example$Human Resources$Dept # 970
telephoneNumber: +1 213 592-2450
title: Elite Human Resources Stooge
userPassword: ageiwrDane
uid: Kaleena_Drwiega
givenName: Kaleena
mail: Kaleena_Drwiega@example.com
carLicense: TFWVQMD
departmentNumber: 7193
employeeType: Manager
homePhone: +1 818 367-1341
initials: K. D.
mobile: +1 71 947-7132
pager: +1 804 900-7534
manager: cn=Tammara Limeina
secretary: cn=Zeina Pickett
roomNumber: 434

dn: cn=Babs Grund, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Babs Grund
sn: Grund
description: This is Babs Grund's description
facsimileTelephoneNumber: +1 510 823-7408
l: Santa Clara
ou: Product Development
postalAddress: example$Product Development$Dept # 415
telephoneNumber: +1 415 512-3661
title: Associate Product Development Dictator
userPassword: dnurGsbaB
uid: Babs_Grund
givenName: Babs
mail: Babs_Grund@example.com
carLicense: UUZT6F2
departmentNumber: 1308
employeeType: Normal
homePhone: +1 510 358-5270
initials: B. G.
mobile: +1 71 620-7831
pager: +1 71 797-5416
manager: cn=Katuscha Plotter
secretary: cn=Georgianne Varley
roomNumber: 4867

dn: cn=Tsing Kenyon, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tsing Kenyon
sn: Kenyon
description: This is Tsing Kenyon's description
facsimileTelephoneNumber: +1 71 116-2416
l: Emeryville
ou: Management
postalAddress: example$Management$Dept # 987
telephoneNumber: +1 510 333-9492
title: Associate Management Mascot
userPassword: noyneKgnis
uid: Tsing_Kenyon
givenName: Tsing
mail: Tsing_Kenyon@example.com
carLicense: JA6N0PI
departmentNumber: 5759
employeeType: Employee
homePhone: +1 71 783-3243
initials: T. K.
mobile: +1 206 751-3172
pager: +1 415 153-1942
manager: cn=Doris De Anda
secretary: cn=Kimberli Rettie
roomNumber: 1812

dn: cn=Christin Franzky, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Christin Franzky
sn: Franzky
description: This is Christin Franzky's description
facsimileTelephoneNumber: +1 510 121-1960
l: Redmond
ou: Management
postalAddress: example$Management$Dept # 716
telephoneNumber: +1 206 481-8825
title: Chief Management Czar
userPassword: ykznarFnit
uid: Christin_Franzky
givenName: Christin
mail: Christin_Franzky@example.com
carLicense: U2SBSAF
departmentNumber: 7533
employeeType: Contract
homePhone: +1 818 581-4701
initials: C. F.
mobile: +1 213 596-2846
pager: +1 818 655-1122
manager: cn=Randolph Puent
secretary: cn=Jacquie Wayling
roomNumber: 1350

dn: cn=Tammi Kempster, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tammi Kempster
sn: Kempster
description: This is Tammi Kempster's description
facsimileTelephoneNumber: +1 415 692-7367
l: Emeryville
ou: Product Testing
postalAddress: example$Product Testing$Dept # 77
telephoneNumber: +1 804 494-2819
title: Master Product Testing Punk
userPassword: retspmeKim
uid: Tammi_Kempster
givenName: Tammi
mail: Tammi_Kempster@example.com
carLicense: 52BQDJJ
departmentNumber: 4827
employeeType: Temp
homePhone: +1 510 747-2181
initials: T. K.
mobile: +1 804 510-5316
pager: +1 804 698-7066
manager: cn=Igor Houle
secretary: cn=Nicolea Luzarraga
roomNumber: 6294

dn: cn=Carmelina Annable, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Carmelina Annable
sn: Annable
description: This is Carmelina Annable's description
facsimileTelephoneNumber: +1 213 894-2569
l: Emeryville
ou: Payroll
postalAddress: example$Payroll$Dept # 879
telephoneNumber: +1 206 423-3258
title: Senior Payroll Technician
userPassword: elbannAani
uid: Carmelina_Annable
givenName: Carmelina
mail: Carmelina_Annable@example.com
carLicense: I9A2DDS
departmentNumber: 4424
employeeType: Manager
homePhone: +1 510 458-6036
initials: C. A.
mobile: +1 408 135-9030
pager: +1 303 526-5109
manager: cn=Markus Penn
secretary: cn=Orenzo Zurl
roomNumber: 41

dn: cn=Henriette Furmaniak, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Henriette Furmaniak
sn: Furmaniak
description: This is Henriette Furmaniak's description
facsimileTelephoneNumber: +1 71 381-8641
l: Milpitas
ou: Payroll
postalAddress: example$Payroll$Dept # 143
telephoneNumber: +1 415 487-9860
title: Associate Payroll Evangelist
userPassword: kainamruFe
uid: Henriette_Furmaniak
givenName: Henriette
mail: Henriette_Furmaniak@example.com
carLicense: S681U1S
departmentNumber: 7396
employeeType: Employee
homePhone: +1 510 759-3104
initials: H. F.
mobile: +1 303 359-1627
pager: +1 303 591-8973
manager: cn=Pankaj Rakochy
secretary: cn=Lily Sergi
roomNumber: 8694

dn: cn=Gracie Waddell, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gracie Waddell
sn: Waddell
description: This is Gracie Waddell's description
facsimileTelephoneNumber: +1 510 574-3746
l: San Mateo
ou: Management
postalAddress: example$Management$Dept # 794
telephoneNumber: +1 408 874-8719
title: Master Management Technician
userPassword: lleddaWeic
uid: Gracie_Waddell
givenName: Gracie
mail: Gracie_Waddell@example.com
carLicense: QO92SJI
departmentNumber: 4612
employeeType: Contract
homePhone: +1 303 461-8149
initials: G. W.
mobile: +1 510 823-2368
pager: +1 303 223-8915
manager: cn=Kui Slusser
secretary: cn=Mick Villella
roomNumber: 1427

dn: cn=Roana Subsara, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Roana Subsara
sn: Subsara
description: This is Roana Subsara's description
facsimileTelephoneNumber: +1 408 331-2806
l: Redmond
ou: Peons
postalAddress: example$Peons$Dept # 893
telephoneNumber: +1 213 136-4389
title: Master Peons Manager
userPassword: arasbuSana
uid: Roana_Subsara
givenName: Roana
mail: Roana_Subsara@example.com
carLicense: TO5CJSA
departmentNumber: 7488
employeeType: Temp
homePhone: +1 213 416-6297
initials: R. S.
mobile: +1 408 316-7305
pager: +1 71 890-2897
manager: cn=Blaine Metheny
secretary: cn=Kirk Shackleford
roomNumber: 5080

dn: cn=Eachelle Greenfield, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Eachelle Greenfield
sn: Greenfield
description: This is Eachelle Greenfield's description
facsimileTelephoneNumber: +1 804 341-9966
l: San Jose
ou: Payroll
postalAddress: example$Payroll$Dept # 538
telephoneNumber: +1 804 389-8723
title: Chief Payroll Madonna
userPassword: dleifneerG
uid: Eachelle_Greenfield
givenName: Eachelle
mail: Eachelle_Greenfield@example.com
carLicense: C8Q1PWA
departmentNumber: 8187
employeeType: Contract
homePhone: +1 303 376-3297
initials: E. G.
mobile: +1 303 601-8532
pager: +1 818 361-9990
manager: cn=Marin Levo
secretary: cn=Ngai Piersol
roomNumber: 4123

dn: cn=Achamma Akyurekli, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Achamma Akyurekli
sn: Akyurekli
description: This is Achamma Akyurekli's description
facsimileTelephoneNumber: +1 71 296-2104
l: Cupertino
ou: Product Testing
postalAddress: example$Product Testing$Dept # 797
telephoneNumber: +1 206 754-7478
title: Junior Product Testing Grunt
userPassword: ilkeruykAa
uid: Achamma_Akyurekli
givenName: Achamma
mail: Achamma_Akyurekli@example.com
carLicense: M9DOR0U
departmentNumber: 9817
employeeType: Manager
homePhone: +1 206 801-8117
initials: A. A.
mobile: +1 408 356-5818
pager: +1 206 432-6175
manager: cn=Cloe Heinzman
secretary: cn=Ayda Khadbai
roomNumber: 1703

dn: cn=Yuan Dept, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Yuan Dept
sn: Dept
description: This is Yuan Dept's description
facsimileTelephoneNumber: +1 213 176-6815
l: Cambridge
ou: Management
postalAddress: example$Management$Dept # 726
telephoneNumber: +1 71 938-9616
title: Junior Management Stooge
userPassword: tpeDnauY
uid: Yuan_Dept
givenName: Yuan
mail: Yuan_Dept@example.com
carLicense: K42HCYP
departmentNumber: 6193
employeeType: Employee
homePhone: +1 206 493-9728
initials: Y. D.
mobile: +1 213 464-7273
pager: +1 408 405-3776
manager: cn=Nadeen Testagc
secretary: cn=Brittney Lasson
roomNumber: 3127

dn: cn=Darrell Jeffries, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Darrell Jeffries
sn: Jeffries
description: This is Darrell Jeffries's description
facsimileTelephoneNumber: +1 804 437-6265
l: San Francisco
ou: Payroll
postalAddress: example$Payroll$Dept # 520
telephoneNumber: +1 818 621-9829
title: Associate Payroll Vice President
userPassword: seirffeJll
uid: Darrell_Jeffries
givenName: Darrell
mail: Darrell_Jeffries@example.com
carLicense: 2MNPEY1
departmentNumber: 1683
employeeType: Normal
homePhone: +1 303 414-7312
initials: D. J.
mobile: +1 303 978-9308
pager: +1 510 100-5619
manager: cn=Gunter Chaikowsky
secretary: cn=Wing-Ki Hinds
roomNumber: 9309

dn: cn=Nora Kellogg, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Nora Kellogg
sn: Kellogg
description: This is Nora Kellogg's description
facsimileTelephoneNumber: +1 804 339-9336
l: Palo Alto
ou: Janitorial
postalAddress: example$Janitorial$Dept # 224
telephoneNumber: +1 804 755-9678
title: Associate Janitorial Mascot
userPassword: ggolleKaro
uid: Nora_Kellogg
givenName: Nora
mail: Nora_Kellogg@example.com
carLicense: SBRZ9Z3
departmentNumber: 8304
employeeType: Normal
homePhone: +1 71 790-5692
initials: N. K.
mobile: +1 408 133-4469
pager: +1 71 995-7187
manager: cn=Blanch Caudle
secretary: cn=Nga Hawryluk
roomNumber: 6925

dn: cn=Lorna Maciejewski, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lorna Maciejewski
sn: Maciejewski
description: This is Lorna Maciejewski's description
facsimileTelephoneNumber: +1 818 973-9335
l: Orem
ou: Management
postalAddress: example$Management$Dept # 298
telephoneNumber: +1 818 860-4997
title: Supreme Management Technician
userPassword: ikswejeica
uid: Lorna_Maciejewski
givenName: Lorna
mail: Lorna_Maciejewski@example.com
carLicense: 3746WMS
departmentNumber: 8407
employeeType: Normal
homePhone: +1 415 927-6953
initials: L. M.
mobile: +1 415 162-9536
pager: +1 415 272-3708
manager: cn=Usa Horton
secretary: cn=Kin Jachym
roomNumber: 9625

dn: cn=Tavis Bees, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tavis Bees
sn: Bees
description: This is Tavis Bees's description
facsimileTelephoneNumber: +1 213 563-5454
l: Fremont
ou: Administrative
postalAddress: example$Administrative$Dept # 831
telephoneNumber: +1 303 132-3422
title: Chief Administrative Stooge
userPassword: seeBsivaT
uid: Tavis_Bees
givenName: Tavis
mail: Tavis_Bees@example.com
carLicense: ZPVAGYM
departmentNumber: 5317
employeeType: Manager
homePhone: +1 818 168-1951
initials: T. B.
mobile: +1 206 942-2455
pager: +1 804 951-6491
manager: cn=Mehmet Kozlowski
secretary: cn=Fawnia Cozart
roomNumber: 801

dn: cn=Blancha Bejar, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Blancha Bejar
sn: Bejar
description: This is Blancha Bejar's description
facsimileTelephoneNumber: +1 510 638-9150
l: Menlo Park
ou: Administrative
postalAddress: example$Administrative$Dept # 388
telephoneNumber: +1 213 100-4750
title: Senior Administrative Fellow
userPassword: rajeBahcna
uid: Blancha_Bejar
givenName: Blancha
mail: Blancha_Bejar@example.com
carLicense: RY0J2ZW
departmentNumber: 6859
employeeType: Contract
homePhone: +1 408 563-2329
initials: B. B.
mobile: +1 510 556-2098
pager: +1 71 577-1582
manager: cn=Binnie Boisvert
secretary: cn=Anil Thuesen
roomNumber: 8932

dn: cn=Atique Hoscheid, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Atique Hoscheid
sn: Hoscheid
description: This is Atique Hoscheid's description
facsimileTelephoneNumber: +1 415 301-7675
l: Alameda
ou: Accounting
postalAddress: example$Accounting$Dept # 675
telephoneNumber: +1 206 723-9273
title: Associate Accounting Fellow
userPassword: diehcsoHeu
uid: Atique_Hoscheid
givenName: Atique
mail: Atique_Hoscheid@example.com
carLicense: TYEQ4YU
departmentNumber: 3523
employeeType: Manager
homePhone: +1 213 428-3966
initials: A. H.
mobile: +1 303 324-8556
pager: +1 408 804-9687
manager: cn=Page Katibian
secretary: cn=Conni Kapella
roomNumber: 409

dn: cn=Garland Donner, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Garland Donner
sn: Donner
description: This is Garland Donner's description
facsimileTelephoneNumber: +1 408 225-3831
l: Emeryville
ou: Human Resources
postalAddress: example$Human Resources$Dept # 934
telephoneNumber: +1 408 198-6277
title: Junior Human Resources Dictator
userPassword: rennoDdnal
uid: Garland_Donner
givenName: Garland
mail: Garland_Donner@example.com
carLicense: UKPQ6TP
departmentNumber: 3525
employeeType: Manager
homePhone: +1 804 136-4416
initials: G. D.
mobile: +1 415 506-2934
pager: +1 510 829-8068
manager: cn=Theresita Haslach
secretary: cn=Debbie Bennett
roomNumber: 9856

dn: cn=Pauli Spooner, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Pauli Spooner
sn: Spooner
description: This is Pauli Spooner's description
facsimileTelephoneNumber: +1 213 662-8018
l: San Francisco
ou: Administrative
postalAddress: example$Administrative$Dept # 565
telephoneNumber: +1 818 810-3512
title: Supreme Administrative President
userPassword: renoopSilu
uid: Pauli_Spooner
givenName: Pauli
mail: Pauli_Spooner@example.com
carLicense: C7FXNKP
departmentNumber: 8721
employeeType: Normal
homePhone: +1 818 905-5332
initials: P. S.
mobile: +1 415 269-1419
pager: +1 71 182-9029
manager: cn=Aloise McSheffrey
secretary: cn=Pat Turcot
roomNumber: 2471

dn: cn=Hal Erguven, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Hal Erguven
sn: Erguven
description: This is Hal Erguven's description
facsimileTelephoneNumber: +1 818 607-5047
l: Cupertino
ou: Janitorial
postalAddress: example$Janitorial$Dept # 727
telephoneNumber: +1 818 513-4628
title: Junior Janitorial Evangelist
userPassword: nevugrElaH
uid: Hal_Erguven
givenName: Hal
mail: Hal_Erguven@example.com
carLicense: G83WWRQ
departmentNumber: 2708
employeeType: Employee
homePhone: +1 408 529-9091
initials: H. E.
mobile: +1 71 249-8217
pager: +1 510 115-6064
manager: cn=Mitch Stroud
secretary: cn=Gracinda Schneiders
roomNumber: 337

dn: cn=Tri Streibel, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tri Streibel
sn: Streibel
description: This is Tri Streibel's description
facsimileTelephoneNumber: +1 408 428-9232
l: Cambridge
ou: Accounting
postalAddress: example$Accounting$Dept # 276
telephoneNumber: +1 206 512-2449
title: Supreme Accounting Czar
userPassword: lebiertSir
uid: Tri_Streibel
givenName: Tri
mail: Tri_Streibel@example.com
carLicense: 5RDIPP4
departmentNumber: 6292
employeeType: Contract
homePhone: +1 818 161-6473
initials: T. S.
mobile: +1 804 607-8237
pager: +1 510 747-8010
manager: cn=Dorothy Zonner
secretary: cn=Ame Pasher
roomNumber: 7074

dn: cn=Lorianna Chern, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lorianna Chern
sn: Chern
description: This is Lorianna Chern's description
facsimileTelephoneNumber: +1 303 351-4018
l: Fremont
ou: Accounting
postalAddress: example$Accounting$Dept # 739
telephoneNumber: +1 71 225-7973
title: Elite Accounting Pinhead
userPassword: nrehCannai
uid: Lorianna_Chern
givenName: Lorianna
mail: Lorianna_Chern@example.com
carLicense: NH1AY8N
departmentNumber: 5809
employeeType: Manager
homePhone: +1 71 595-9477
initials: L. C.
mobile: +1 415 831-4350
pager: +1 303 842-9088
manager: cn=Anne-corinne Monn
secretary: cn=Teena Vivian
roomNumber: 2080

dn: cn=Ava Rtpbuild, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ava Rtpbuild
sn: Rtpbuild
description: This is Ava Rtpbuild's description
facsimileTelephoneNumber: +1 804 326-2739
l: Mountain View
ou: Accounting
postalAddress: example$Accounting$Dept # 591
telephoneNumber: +1 303 107-7347
title: Master Accounting Architect
userPassword: dliubptRav
uid: Ava_Rtpbuild
givenName: Ava
mail: Ava_Rtpbuild@example.com
carLicense: BCQVE7T
departmentNumber: 4118
employeeType: Contract
homePhone: +1 408 349-5067
initials: A. R.
mobile: +1 818 415-2641
pager: +1 804 895-6044
manager: cn=Maxie Schmitt
secretary: cn=Fwpreg Goller
roomNumber: 7304

dn: cn=Claudia McMasters, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Claudia McMasters
sn: McMasters
description: This is Claudia McMasters's description
facsimileTelephoneNumber: +1 408 394-1817
l: Emeryville
ou: Product Development
postalAddress: example$Product Development$Dept # 449
telephoneNumber: +1 213 129-8408
title: Chief Product Development Technician
userPassword: sretsaMcMa
uid: Claudia_McMasters
givenName: Claudia
mail: Claudia_McMasters@example.com
carLicense: JPQO9E8
departmentNumber: 2709
employeeType: Employee
homePhone: +1 415 358-9077
initials: C. M.
mobile: +1 206 203-7937
pager: +1 415 233-4827
manager: cn=Pivert Cantlie
secretary: cn=Bennet Bilsborough
roomNumber: 3381

dn: cn=Attilio Cheshire, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Attilio Cheshire
sn: Cheshire
description: This is Attilio Cheshire's description
facsimileTelephoneNumber: +1 818 795-3209
l: Redmond
ou: Accounting
postalAddress: example$Accounting$Dept # 729
telephoneNumber: +1 71 793-1301
title: Elite Accounting Mascot
userPassword: erihsehCoi
uid: Attilio_Cheshire
givenName: Attilio
mail: Attilio_Cheshire@example.com
carLicense: O8ONIEG
departmentNumber: 966
employeeType: Temp
homePhone: +1 206 216-9992
initials: A. C.
mobile: +1 71 812-7193
pager: +1 408 656-5816
manager: cn=Ved Pittam
secretary: cn=Tresrch Flickinger
roomNumber: 5500

dn: cn=Jolyn Sturrock, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jolyn Sturrock
sn: Sturrock
description: This is Jolyn Sturrock's description
facsimileTelephoneNumber: +1 415 350-6718
l: Redmond
ou: Peons
postalAddress: example$Peons$Dept # 107
telephoneNumber: +1 818 723-8455
title: Master Peons Dictator
userPassword: kcorrutSny
uid: Jolyn_Sturrock
givenName: Jolyn
mail: Jolyn_Sturrock@example.com
carLicense: GWY1SNL
departmentNumber: 62
employeeType: Manager
homePhone: +1 206 158-7057
initials: J. S.
mobile: +1 818 574-2111
pager: +1 408 190-8741
manager: cn=Manda Santi
secretary: cn=Ling-Yue Smolin
roomNumber: 6698

dn: cn=Gerber Wefers, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gerber Wefers
sn: Wefers
description: This is Gerber Wefers's description
facsimileTelephoneNumber: +1 213 875-4086
l: Cupertino
ou: Product Testing
postalAddress: example$Product Testing$Dept # 512
telephoneNumber: +1 818 743-7160
title: Junior Product Testing Artist
userPassword: srefeWrebr
uid: Gerber_Wefers
givenName: Gerber
mail: Gerber_Wefers@example.com
carLicense: 3S3J6KQ
departmentNumber: 8016
employeeType: Manager
homePhone: +1 71 661-9616
initials: G. W.
mobile: +1 408 463-5608
pager: +1 408 293-1049
manager: cn=Tsugio Alms
secretary: cn=Leonanie Gregory
roomNumber: 94

dn: cn=May Mulqueen, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: May Mulqueen
sn: Mulqueen
description: This is May Mulqueen's description
facsimileTelephoneNumber: +1 213 224-4633
l: Mountain View
ou: Janitorial
postalAddress: example$Janitorial$Dept # 626
telephoneNumber: +1 408 546-2498
title: Associate Janitorial Stooge
userPassword: neeuqluMya
uid: May_Mulqueen
givenName: May
mail: May_Mulqueen@example.com
carLicense: W8W9FVI
departmentNumber: 7335
employeeType: Manager
homePhone: +1 818 569-7514
initials: M. M.
mobile: +1 213 692-2302
pager: +1 804 311-4290
manager: cn=Joellen Volker
secretary: cn=Katarina Alms
roomNumber: 3635

dn: cn=Klaus Wessenberg, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Klaus Wessenberg
sn: Wessenberg
description: This is Klaus Wessenberg's description
facsimileTelephoneNumber: +1 510 761-3757
l: Milpitas
ou: Human Resources
postalAddress: example$Human Resources$Dept # 998
telephoneNumber: +1 213 728-9369
title: Chief Human Resources Janitor
userPassword: grebnesseW
uid: Klaus_Wessenberg
givenName: Klaus
mail: Klaus_Wessenberg@example.com
carLicense: 89PC05L
departmentNumber: 9267
employeeType: Employee
homePhone: +1 303 177-7913
initials: K. W.
mobile: +1 408 302-2211
pager: +1 303 393-9789
manager: cn=Kien-Nghiep Chhabria
secretary: cn=Tyke Bennatt
roomNumber: 1510

dn: cn=Zero Astorino, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zero Astorino
sn: Astorino
description: This is Zero Astorino's description
facsimileTelephoneNumber: +1 206 749-3370
l: Sunnyvale
ou: Product Testing
postalAddress: example$Product Testing$Dept # 211
telephoneNumber: +1 804 389-4563
title: Senior Product Testing Stooge
userPassword: onirotsAor
uid: Zero_Astorino
givenName: Zero
mail: Zero_Astorino@example.com
carLicense: MAZB0EZ
departmentNumber: 8638
employeeType: Contract
homePhone: +1 206 963-3254
initials: Z. A.
mobile: +1 206 464-7408
pager: +1 206 210-3740
manager: cn=Emmalynne Van Schyndel
secretary: cn=Muni Maidens
roomNumber: 2686

dn: cn=Lou Pezzoni, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lou Pezzoni
sn: Pezzoni
description: This is Lou Pezzoni's description
facsimileTelephoneNumber: +1 213 731-5038
l: Armonk
ou: Management
postalAddress: example$Management$Dept # 372
telephoneNumber: +1 818 444-2323
title: Chief Management Stooge
userPassword: inozzePuoL
uid: Lou_Pezzoni
givenName: Lou
mail: Lou_Pezzoni@example.com
carLicense: 2WP98DC
departmentNumber: 1238
employeeType: Temp
homePhone: +1 408 604-7030
initials: L. P.
mobile: +1 213 624-5253
pager: +1 213 575-8354
manager: cn=Alys Booker
secretary: cn=Tan Albright
roomNumber: 8702

dn: cn=Nelie Beavis, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Nelie Beavis
sn: Beavis
description: This is Nelie Beavis's description
facsimileTelephoneNumber: +1 206 769-7925
l: Cupertino
ou: Product Testing
postalAddress: example$Product Testing$Dept # 318
telephoneNumber: +1 510 542-1388
title: Senior Product Testing Sales Rep
userPassword: sivaeBeile
uid: Nelie_Beavis
givenName: Nelie
mail: Nelie_Beavis@example.com
carLicense: NAEH8SQ
departmentNumber: 5361
employeeType: Normal
homePhone: +1 804 554-9766
initials: N. B.
mobile: +1 213 993-7114
pager: +1 206 955-1047
manager: cn=Elysha Poff
secretary: cn=Maris Govindasamy
roomNumber: 2030

dn: cn=Dita Stinson, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dita Stinson
sn: Stinson
description: This is Dita Stinson's description
facsimileTelephoneNumber: +1 71 262-1356
l: Sunnyvale
ou: Product Testing
postalAddress: example$Product Testing$Dept # 386
telephoneNumber: +1 303 869-4222
title: Supreme Product Testing Admin
userPassword: nosnitSati
uid: Dita_Stinson
givenName: Dita
mail: Dita_Stinson@example.com
carLicense: 5S9HDUI
departmentNumber: 7675
employeeType: Normal
homePhone: +1 510 148-7418
initials: D. S.
mobile: +1 415 530-1870
pager: +1 415 913-1652
manager: cn=Kieron Desmond
secretary: cn=Remington Kosiorska
roomNumber: 8364

dn: cn=Ijff Swails, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ijff Swails
sn: Swails
description: This is Ijff Swails's description
facsimileTelephoneNumber: +1 408 230-6699
l: Alameda
ou: Peons
postalAddress: example$Peons$Dept # 659
telephoneNumber: +1 804 920-5135
title: Elite Peons Engineer
userPassword: sliawSffjI
uid: Ijff_Swails
givenName: Ijff
mail: Ijff_Swails@example.com
carLicense: K53PGCQ
departmentNumber: 285
employeeType: Employee
homePhone: +1 408 231-4267
initials: I. S.
mobile: +1 818 439-9844
pager: +1 213 838-9388
manager: cn=Parminder Gougeon
secretary: cn=Raleigh Gallenbeck
roomNumber: 4580

dn: cn=Leon Kostyniuk, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Leon Kostyniuk
sn: Kostyniuk
description: This is Leon Kostyniuk's description
facsimileTelephoneNumber: +1 206 948-7104
l: Armonk
ou: Administrative
postalAddress: example$Administrative$Dept # 510
telephoneNumber: +1 510 228-7469
title: Associate Administrative Madonna
userPassword: kuinytsoKn
uid: Leon_Kostyniuk
givenName: Leon
mail: Leon_Kostyniuk@example.com
carLicense: 8FLR7LX
departmentNumber: 4349
employeeType: Employee
homePhone: +1 206 216-4381
initials: L. K.
mobile: +1 408 315-5005
pager: +1 818 788-4357
manager: cn=Viqar Traylor
secretary: cn=Gerianna Cousineau
roomNumber: 5375

dn: cn=Gabriel Isley, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gabriel Isley
sn: Isley
description: This is Gabriel Isley's description
facsimileTelephoneNumber: +1 303 806-2713
l: Sunnyvale
ou: Human Resources
postalAddress: example$Human Resources$Dept # 946
telephoneNumber: +1 408 964-7589
title: Supreme Human Resources Janitor
userPassword: yelsIleirb
uid: Gabriel_Isley
givenName: Gabriel
mail: Gabriel_Isley@example.com
carLicense: K86YHF5
departmentNumber: 6726
employeeType: Normal
homePhone: +1 303 751-6077
initials: G. I.
mobile: +1 206 439-1191
pager: +1 303 691-9922
manager: cn=Cefee Phan
secretary: cn=Belen Champsi
roomNumber: 9094

dn: cn=Tally Yandell, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tally Yandell
sn: Yandell
description: This is Tally Yandell's description
facsimileTelephoneNumber: +1 213 466-4179
l: Cambridge
ou: Payroll
postalAddress: example$Payroll$Dept # 73
telephoneNumber: +1 804 568-1143
title: Master Payroll Figurehead
userPassword: llednaYyll
uid: Tally_Yandell
givenName: Tally
mail: Tally_Yandell@example.com
carLicense: WA0Z8Y8
departmentNumber: 6442
employeeType: Employee
homePhone: +1 408 141-4736
initials: T. Y.
mobile: +1 818 290-8012
pager: +1 303 680-5641
manager: cn=Kathryne Silva
secretary: cn=Willy Rorie
roomNumber: 4444

dn: cn=Enis Fullager, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Enis Fullager
sn: Fullager
description: This is Enis Fullager's description
facsimileTelephoneNumber: +1 804 538-2070
l: Alameda
ou: Product Development
postalAddress: example$Product Development$Dept # 391
telephoneNumber: +1 71 726-4802
title: Chief Product Development Stooge
userPassword: regalluFsi
uid: Enis_Fullager
givenName: Enis
mail: Enis_Fullager@example.com
carLicense: T3ACW5H
departmentNumber: 5379
employeeType: Contract
homePhone: +1 206 919-9700
initials: E. F.
mobile: +1 818 223-9749
pager: +1 804 988-9659
manager: cn=Dynah Owensby
secretary: cn=Butch Chantal
roomNumber: 3261

dn: cn=Joly Tham, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Joly Tham
sn: Tham
description: This is Joly Tham's description
facsimileTelephoneNumber: +1 415 826-7425
l: Emeryville
ou: Payroll
postalAddress: example$Payroll$Dept # 873
telephoneNumber: +1 303 248-9970
title: Chief Payroll Evangelist
userPassword: mahTyloJ
uid: Joly_Tham
givenName: Joly
mail: Joly_Tham@example.com
carLicense: 24ZOPUT
departmentNumber: 181
employeeType: Manager
homePhone: +1 303 472-7110
initials: J. T.
mobile: +1 408 958-3175
pager: +1 804 283-2291
manager: cn=Arthur Diradmin
secretary: cn=Jessa Starsdps
roomNumber: 7894

dn: cn=Sydney Chenoweth, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sydney Chenoweth
sn: Chenoweth
description: This is Sydney Chenoweth's description
facsimileTelephoneNumber: +1 804 466-1282
l: Cupertino
ou: Payroll
postalAddress: example$Payroll$Dept # 941
telephoneNumber: +1 71 235-3143
title: Associate Payroll Czar
userPassword: htewonehCy
uid: Sydney_Chenoweth
givenName: Sydney
mail: Sydney_Chenoweth@example.com
carLicense: WGBDPP1
departmentNumber: 5361
employeeType: Contract
homePhone: +1 408 134-7132
initials: S. C.
mobile: +1 818 261-9758
pager: +1 213 254-6306
manager: cn=Helaine Hopper
secretary: cn=JoLee Schipper
roomNumber: 5967

dn: cn=Jeffrey Silieff, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jeffrey Silieff
sn: Silieff
description: This is Jeffrey Silieff's description
facsimileTelephoneNumber: +1 415 858-1535
l: Alameda
ou: Management
postalAddress: example$Management$Dept # 966
telephoneNumber: +1 415 630-5305
title: Senior Management Dictator
userPassword: ffeiliSyer
uid: Jeffrey_Silieff
givenName: Jeffrey
mail: Jeffrey_Silieff@example.com
carLicense: 895WWI4
departmentNumber: 4908
employeeType: Employee
homePhone: +1 71 579-7062
initials: J. S.
mobile: +1 408 404-2467
pager: +1 415 475-9295
manager: cn=Sami Kimler
secretary: cn=Faustina Sandberg
roomNumber: 1846

dn: cn=Kenny Michaels, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kenny Michaels
sn: Michaels
description: This is Kenny Michaels's description
facsimileTelephoneNumber: +1 818 657-3235
l: Cupertino
ou: Planning
postalAddress: example$Planning$Dept # 256
telephoneNumber: +1 71 874-9999
title: Senior Planning Punk
userPassword: sleahciMyn
uid: Kenny_Michaels
givenName: Kenny
mail: Kenny_Michaels@example.com
carLicense: 7VQOB9I
departmentNumber: 3466
employeeType: Employee
homePhone: +1 415 228-6489
initials: K. M.
mobile: +1 510 965-8849
pager: +1 818 867-1962
manager: cn=Guylaine Knapton
secretary: cn=Van-King Brett
roomNumber: 244

dn: cn=Partap Heller, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Partap Heller
sn: Heller
description: This is Partap Heller's description
facsimileTelephoneNumber: +1 818 220-6830
l: Armonk
ou: Accounting
postalAddress: example$Accounting$Dept # 363
telephoneNumber: +1 818 946-5877
title: Master Accounting Madonna
userPassword: relleHpatr
uid: Partap_Heller
givenName: Partap
mail: Partap_Heller@example.com
carLicense: H0HCYO6
departmentNumber: 5625
employeeType: Manager
homePhone: +1 213 623-6672
initials: P. H.
mobile: +1 408 316-1890
pager: +1 71 477-6018
manager: cn=Idalina Yohe
secretary: cn=Jaynell Quevillon
roomNumber: 1482

dn: cn=Mellie Ladouceur, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mellie Ladouceur
sn: Ladouceur
description: This is Mellie Ladouceur's description
facsimileTelephoneNumber: +1 206 693-4680
l: Cambridge
ou: Peons
postalAddress: example$Peons$Dept # 875
telephoneNumber: +1 510 169-8383
title: Chief Peons Grunt
userPassword: ruecuodaLe
uid: Mellie_Ladouceur
givenName: Mellie
mail: Mellie_Ladouceur@example.com
carLicense: NFG1QY6
departmentNumber: 5449
employeeType: Employee
homePhone: +1 510 806-6400
initials: M. L.
mobile: +1 206 797-4296
pager: +1 510 377-2981
manager: cn=Micah PATCOR
secretary: cn=Bhupinder Hiraki
roomNumber: 9379

dn: cn=Aloysia Running, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Aloysia Running
sn: Running
description: This is Aloysia Running's description
facsimileTelephoneNumber: +1 415 135-9978
l: Milpitas
ou: Accounting
postalAddress: example$Accounting$Dept # 203
telephoneNumber: +1 408 872-6236
title: Supreme Accounting Grunt
userPassword: gninnuRais
uid: Aloysia_Running
givenName: Aloysia
mail: Aloysia_Running@example.com
carLicense: 3KTA8R0
departmentNumber: 2824
employeeType: Employee
homePhone: +1 303 961-5246
initials: A. R.
mobile: +1 213 830-4008
pager: +1 510 361-3893
manager: cn=Hattie Chanco
secretary: cn=Wilfred Stumpf
roomNumber: 4337

dn: cn=Ellissa Herling, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ellissa Herling
sn: Herling
description: This is Ellissa Herling's description
facsimileTelephoneNumber: +1 206 340-9698
l: San Mateo
ou: Planning
postalAddress: example$Planning$Dept # 259
telephoneNumber: +1 303 377-6673
title: Master Planning Engineer
userPassword: gnilreHass
uid: Ellissa_Herling
givenName: Ellissa
mail: Ellissa_Herling@example.com
carLicense: ADZM7H9
departmentNumber: 4088
employeeType: Normal
homePhone: +1 415 848-8162
initials: E. H.
mobile: +1 510 482-7382
pager: +1 206 936-3580
manager: cn=Zbignew Merrick
secretary: cn=Jean McCurdy
roomNumber: 147

dn: cn=Valli Sigmon, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Valli Sigmon
sn: Sigmon
description: This is Valli Sigmon's description
facsimileTelephoneNumber: +1 818 423-8154
l: Redmond
ou: Peons
postalAddress: example$Peons$Dept # 333
telephoneNumber: +1 415 704-4892
title: Senior Peons Visionary
userPassword: nomgiSilla
uid: Valli_Sigmon
givenName: Valli
mail: Valli_Sigmon@example.com
carLicense: KVLVGUN
departmentNumber: 9890
employeeType: Normal
homePhone: +1 408 828-4293
initials: V. S.
mobile: +1 303 713-2307
pager: +1 510 164-7951
manager: cn=Zarrin Doyon
secretary: cn=Denice Tufford
roomNumber: 7410

dn: cn=Karol Standards, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Karol Standards
sn: Standards
description: This is Karol Standards's description
facsimileTelephoneNumber: +1 510 799-5003
l: Santa Clara
ou: Human Resources
postalAddress: example$Human Resources$Dept # 812
telephoneNumber: +1 408 771-5206
title: Chief Human Resources Dictator
userPassword: sdradnatSl
uid: Karol_Standards
givenName: Karol
mail: Karol_Standards@example.com
carLicense: GFT7952
departmentNumber: 9531
employeeType: Employee
homePhone: +1 415 426-1737
initials: K. S.
mobile: +1 303 785-6701
pager: +1 408 142-9151
manager: cn=Nelie Ananth
secretary: cn=Buddy Sheffey
roomNumber: 2785

dn: cn=Ronn Buchanan, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ronn Buchanan
sn: Buchanan
description: This is Ronn Buchanan's description
facsimileTelephoneNumber: +1 71 663-7181
l: San Francisco
ou: Management
postalAddress: example$Management$Dept # 155
telephoneNumber: +1 408 625-1957
title: Chief Management Technician
userPassword: nanahcuBnn
uid: Ronn_Buchanan
givenName: Ronn
mail: Ronn_Buchanan@example.com
carLicense: NQN6XAT
departmentNumber: 2350
employeeType: Temp
homePhone: +1 415 486-3839
initials: R. B.
mobile: +1 408 621-3994
pager: +1 71 925-1399
manager: cn=Dee dee Quante
secretary: cn=Manon Hildebrand
roomNumber: 8097

dn: cn=Loreta Sherali, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Loreta Sherali
sn: Sherali
description: This is Loreta Sherali's description
facsimileTelephoneNumber: +1 408 265-9408
l: Redwood Shores
ou: Product Development
postalAddress: example$Product Development$Dept # 939
telephoneNumber: +1 303 715-6655
title: Elite Product Development Dictator
userPassword: ilarehSate
uid: Loreta_Sherali
givenName: Loreta
mail: Loreta_Sherali@example.com
carLicense: 3I7II8B
departmentNumber: 6786
employeeType: Manager
homePhone: +1 804 226-8653
initials: L. S.
mobile: +1 206 310-6268
pager: +1 408 742-8228
manager: cn=Susi Telke
secretary: cn=Irice Paperno
roomNumber: 1030

dn: cn=Martha Soldera, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Martha Soldera
sn: Soldera
description: This is Martha Soldera's description
facsimileTelephoneNumber: +1 415 437-5683
l: San Francisco
ou: Janitorial
postalAddress: example$Janitorial$Dept # 809
telephoneNumber: +1 510 553-7486
title: Associate Janitorial Manager
userPassword: aredloSaht
uid: Martha_Soldera
givenName: Martha
mail: Martha_Soldera@example.com
carLicense: 9WN7ZDP
departmentNumber: 1418
employeeType: Temp
homePhone: +1 415 149-4197
initials: M. S.
mobile: +1 818 866-8309
pager: +1 804 776-2722
manager: cn=Lothar Meagher
secretary: cn=Margriet Essig
roomNumber: 9240

dn: cn=Rachelle Eyers, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rachelle Eyers
sn: Eyers
description: This is Rachelle Eyers's description
facsimileTelephoneNumber: +1 213 966-1936
l: Redmond
ou: Payroll
postalAddress: example$Payroll$Dept # 79
telephoneNumber: +1 804 109-3535
title: Junior Payroll Stooge
userPassword: sreyEelleh
uid: Rachelle_Eyers
givenName: Rachelle
mail: Rachelle_Eyers@example.com
carLicense: BOOQSXD
departmentNumber: 1910
employeeType: Contract
homePhone: +1 804 444-6218
initials: R. E.
mobile: +1 303 515-7109
pager: +1 415 694-1907
manager: cn=Atl-Sales Passin
secretary: cn=DeAnne Breuer
roomNumber: 833

dn: cn=Elyse Deatrick, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Elyse Deatrick
sn: Deatrick
description: This is Elyse Deatrick's description
facsimileTelephoneNumber: +1 818 499-4169
l: Santa Clara
ou: Product Testing
postalAddress: example$Product Testing$Dept # 272
telephoneNumber: +1 71 829-8603
title: Associate Product Testing Evangelist
userPassword: kcirtaeDes
uid: Elyse_Deatrick
givenName: Elyse
mail: Elyse_Deatrick@example.com
carLicense: SVKWQFG
departmentNumber: 427
employeeType: Normal
homePhone: +1 213 388-3585
initials: E. D.
mobile: +1 415 819-7678
pager: +1 415 682-6240
manager: cn=Danika Bailloux
secretary: cn=Purnam Eggleton
roomNumber: 3700

dn: cn=Zola Testsds, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zola Testsds
sn: Testsds
description: This is Zola Testsds's description
facsimileTelephoneNumber: +1 415 928-4013
l: Santa Clara
ou: Payroll
postalAddress: example$Payroll$Dept # 841
telephoneNumber: +1 71 345-8158
title: Chief Payroll Writer
userPassword: sdstseTalo
uid: Zola_Testsds
givenName: Zola
mail: Zola_Testsds@example.com
carLicense: XD87ET6
departmentNumber: 1258
employeeType: Normal
homePhone: +1 213 928-3383
initials: Z. T.
mobile: +1 804 903-8412
pager: +1 510 399-2722
manager: cn=Dwayne Firerobin
secretary: cn=Nicolea Testtools
roomNumber: 7297

dn: cn=Wendye Stillwell, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Wendye Stillwell
sn: Stillwell
description: This is Wendye Stillwell's description
facsimileTelephoneNumber: +1 213 318-7076
l: Redwood Shores
ou: Administrative
postalAddress: example$Administrative$Dept # 540
telephoneNumber: +1 206 822-6752
title: Junior Administrative President
userPassword: llewllitSe
uid: Wendye_Stillwell
givenName: Wendye
mail: Wendye_Stillwell@example.com
carLicense: 92QBPGR
departmentNumber: 636
employeeType: Manager
homePhone: +1 303 229-4142
initials: W. S.
mobile: +1 510 868-2412
pager: +1 415 633-1667
manager: cn=Cloris Chouinard
secretary: cn=Linh Hargadon
roomNumber: 2645

dn: cn=Beatrix Klaudt, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Beatrix Klaudt
sn: Klaudt
description: This is Beatrix Klaudt's description
facsimileTelephoneNumber: +1 206 749-8513
l: Menlo Park
ou: Peons
postalAddress: example$Peons$Dept # 188
telephoneNumber: +1 213 452-3220
title: Senior Peons Developer
userPassword: tdualKxirt
uid: Beatrix_Klaudt
givenName: Beatrix
mail: Beatrix_Klaudt@example.com
carLicense: H0A5PNJ
departmentNumber: 5780
employeeType: Employee
homePhone: +1 303 110-9650
initials: B. K.
mobile: +1 818 262-2202
pager: +1 408 698-8828
manager: cn=Maala D'Antonio
secretary: cn=Adela Bachecongi
roomNumber: 9361

dn: cn=Susy Ojerholm, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Susy Ojerholm
sn: Ojerholm
description: This is Susy Ojerholm's description
facsimileTelephoneNumber: +1 206 619-7003
l: San Jose
ou: Accounting
postalAddress: example$Accounting$Dept # 612
telephoneNumber: +1 213 469-6896
title: Master Accounting Evangelist
userPassword: mlohrejOys
uid: Susy_Ojerholm
givenName: Susy
mail: Susy_Ojerholm@example.com
carLicense: JRHY188
departmentNumber: 6207
employeeType: Contract
homePhone: +1 408 974-6966
initials: S. O.
mobile: +1 415 925-7650
pager: +1 303 841-5776
manager: cn=Rizwan Rfa
secretary: cn=Willette Beninger
roomNumber: 2753

dn: cn=Kare Lugwig, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kare Lugwig
sn: Lugwig
description: This is Kare Lugwig's description
facsimileTelephoneNumber: +1 71 122-2378
l: Armonk
ou: Management
postalAddress: example$Management$Dept # 563
telephoneNumber: +1 213 847-7383
title: Junior Management Dictator
userPassword: giwguLeraK
uid: Kare_Lugwig
givenName: Kare
mail: Kare_Lugwig@example.com
carLicense: GLPVSFU
departmentNumber: 3578
employeeType: Normal
homePhone: +1 213 103-9835
initials: K. L.
mobile: +1 71 478-8740
pager: +1 415 773-5205
manager: cn=Yate Venier
secretary: cn=Tomasine Wasserman
roomNumber: 4838

dn: cn=Sula Tanner, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sula Tanner
sn: Tanner
description: This is Sula Tanner's description
facsimileTelephoneNumber: +1 804 295-7586
l: San Mateo
ou: Product Testing
postalAddress: example$Product Testing$Dept # 471
telephoneNumber: +1 71 290-9074
title: Chief Product Testing Vice President
userPassword: rennaTaluS
uid: Sula_Tanner
givenName: Sula
mail: Sula_Tanner@example.com
carLicense: 0VQQ0HK
departmentNumber: 737
employeeType: Normal
homePhone: +1 415 237-6614
initials: S. T.
mobile: +1 206 850-6105
pager: +1 303 894-1732
manager: cn=Pierrette Spencer
secretary: cn=Svenn-Erik Kowalsky
roomNumber: 9742

dn: cn=Hermina Dickeson, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Hermina Dickeson
sn: Dickeson
description: This is Hermina Dickeson's description
facsimileTelephoneNumber: +1 818 451-7579
l: Sunnyvale
ou: Janitorial
postalAddress: example$Janitorial$Dept # 115
telephoneNumber: +1 415 741-4085
title: Junior Janitorial Czar
userPassword: nosekciDan
uid: Hermina_Dickeson
givenName: Hermina
mail: Hermina_Dickeson@example.com
carLicense: PKCPCT9
departmentNumber: 7530
employeeType: Temp
homePhone: +1 408 596-9759
initials: H. D.
mobile: +1 818 273-6370
pager: +1 213 923-8962
manager: cn=Natassia Soucie
secretary: cn=Orden Rakotomalala
roomNumber: 8119

dn: cn=Ari Adey, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ari Adey
sn: Adey
description: This is Ari Adey's description
facsimileTelephoneNumber: +1 415 543-1505
l: Fremont
ou: Product Testing
postalAddress: example$Product Testing$Dept # 395
telephoneNumber: +1 818 394-6297
title: Master Product Testing Czar
userPassword: yedAirA
uid: Ari_Adey
givenName: Ari
mail: Ari_Adey@example.com
carLicense: J9XM4PR
departmentNumber: 3231
employeeType: Normal
homePhone: +1 408 306-4874
initials: A. A.
mobile: +1 213 722-8365
pager: +1 206 325-2793
manager: cn=Analiese McMann
secretary: cn=Saundra Bennefeld
roomNumber: 7650

dn: cn=Victoria Leang, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Victoria Leang
sn: Leang
description: This is Victoria Leang's description
facsimileTelephoneNumber: +1 303 117-4601
l: Santa Clara
ou: Payroll
postalAddress: example$Payroll$Dept # 334
telephoneNumber: +1 415 975-4679
title: Master Payroll Writer
userPassword: gnaeLairot
uid: Victoria_Leang
givenName: Victoria
mail: Victoria_Leang@example.com
carLicense: PSJFH9X
departmentNumber: 9435
employeeType: Employee
homePhone: +1 510 209-6338
initials: V. L.
mobile: +1 303 857-3061
pager: +1 206 627-9915
manager: cn=Gwennie Klostermann
secretary: cn=Cristine Felske
roomNumber: 3111

dn: cn=Ronda Wilkerson, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ronda Wilkerson
sn: Wilkerson
description: This is Ronda Wilkerson's description
facsimileTelephoneNumber: +1 804 155-2937
l: Emeryville
ou: Peons
postalAddress: example$Peons$Dept # 721
telephoneNumber: +1 206 723-9225
title: Junior Peons Pinhead
userPassword: nosrekliWa
uid: Ronda_Wilkerson
givenName: Ronda
mail: Ronda_Wilkerson@example.com
carLicense: ZD3OJ20
departmentNumber: 3672
employeeType: Temp
homePhone: +1 804 586-6519
initials: R. W.
mobile: +1 818 363-8327
pager: +1 206 432-1897
manager: cn=Chitra Gartley
secretary: cn=Cecco Nordstrom
roomNumber: 3861

dn: cn=Russel Littau, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Russel Littau
sn: Littau
description: This is Russel Littau's description
facsimileTelephoneNumber: +1 71 915-4910
l: Mountain View
ou: Peons
postalAddress: example$Peons$Dept # 677
telephoneNumber: +1 415 889-5335
title: Supreme Peons President
userPassword: uattiLless
uid: Russel_Littau
givenName: Russel
mail: Russel_Littau@example.com
carLicense: VLLESG0
departmentNumber: 6651
employeeType: Normal
homePhone: +1 303 909-2504
initials: R. L.
mobile: +1 303 924-9552
pager: +1 818 728-2874
manager: cn=Jatinder Scarborough
secretary: cn=Tara Draffin
roomNumber: 8375

dn: cn=Sorin Sherif, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sorin Sherif
sn: Sherif
description: This is Sorin Sherif's description
facsimileTelephoneNumber: +1 804 118-9128
l: Fremont
ou: Janitorial
postalAddress: example$Janitorial$Dept # 445
telephoneNumber: +1 408 977-7192
title: Master Janitorial Punk
userPassword: firehSniro
uid: Sorin_Sherif
givenName: Sorin
mail: Sorin_Sherif@example.com
carLicense: 5VIK8EN
departmentNumber: 1036
employeeType: Normal
homePhone: +1 818 136-6030
initials: S. S.
mobile: +1 510 792-9261
pager: +1 71 353-9877
manager: cn=Terrell Peacocke
secretary: cn=Kathye Sandhar
roomNumber: 7862

dn: cn=Biplab Soldera, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Biplab Soldera
sn: Soldera
description: This is Biplab Soldera's description
facsimileTelephoneNumber: +1 206 112-3230
l: Redmond
ou: Planning
postalAddress: example$Planning$Dept # 603
telephoneNumber: +1 818 275-7258
title: Elite Planning Madonna
userPassword: aredloSbal
uid: Biplab_Soldera
givenName: Biplab
mail: Biplab_Soldera@example.com
carLicense: FS496ZQ
departmentNumber: 317
employeeType: Employee
homePhone: +1 206 807-9278
initials: B. S.
mobile: +1 303 703-5352
pager: +1 71 285-6965
manager: cn=Caryl Steen
secretary: cn=Ardeen Rabjohn
roomNumber: 3841

dn: cn=Kirbee Fleishman, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kirbee Fleishman
sn: Fleishman
description: This is Kirbee Fleishman's description
facsimileTelephoneNumber: +1 303 211-5706
l: Armonk
ou: Accounting
postalAddress: example$Accounting$Dept # 681
telephoneNumber: +1 303 348-3978
title: Supreme Accounting Madonna
userPassword: namhsielFe
uid: Kirbee_Fleishman
givenName: Kirbee
mail: Kirbee_Fleishman@example.com
carLicense: G4BNZ42
departmentNumber: 3552
employeeType: Employee
homePhone: +1 71 104-5707
initials: K. F.
mobile: +1 213 604-9318
pager: +1 71 560-2705
manager: cn=Danielle Bambach
secretary: cn=Dahlia Eckstein
roomNumber: 5724

dn: cn=Natassia Devarennes, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Natassia Devarennes
sn: Devarennes
description: This is Natassia Devarennes's description
facsimileTelephoneNumber: +1 213 510-5460
l: Alameda
ou: Payroll
postalAddress: example$Payroll$Dept # 197
telephoneNumber: +1 818 694-3052
title: Senior Payroll Technician
userPassword: senneraveD
uid: Natassia_Devarennes
givenName: Natassia
mail: Natassia_Devarennes@example.com
carLicense: K873P9G
departmentNumber: 4678
employeeType: Manager
homePhone: +1 415 950-5486
initials: N. D.
mobile: +1 408 324-4107
pager: +1 408 261-4010
manager: cn=Irvin Hinsdale
secretary: cn=Penny Outram
roomNumber: 2626

dn: cn=Geza Teniola, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Geza Teniola
sn: Teniola
description: This is Geza Teniola's description
facsimileTelephoneNumber: +1 510 448-5550
l: Fremont
ou: Management
postalAddress: example$Management$Dept # 830
telephoneNumber: +1 818 342-5948
title: Senior Management Yahoo
userPassword: aloineTaze
uid: Geza_Teniola
givenName: Geza
mail: Geza_Teniola@example.com
carLicense: 7P9IAAQ
departmentNumber: 5892
employeeType: Employee
homePhone: +1 303 840-4218
initials: G. T.
mobile: +1 510 901-2392
pager: +1 510 569-1997
manager: cn=Addi Pevzner
secretary: cn=Ruthe Hutchins
roomNumber: 1214

dn: cn=Pas Panger, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Pas Panger
sn: Panger
description: This is Pas Panger's description
facsimileTelephoneNumber: +1 818 338-5910
l: Emeryville
ou: Payroll
postalAddress: example$Payroll$Dept # 600
telephoneNumber: +1 408 218-7902
title: Chief Payroll Artist
userPassword: regnaPsaP
uid: Pas_Panger
givenName: Pas
mail: Pas_Panger@example.com
carLicense: 720HXUD
departmentNumber: 3584
employeeType: Temp
homePhone: +1 804 986-9454
initials: P. P.
mobile: +1 408 182-4550
pager: +1 206 188-7087
manager: cn=Elie Dubee
secretary: cn=Cheslie Bostock
roomNumber: 1025

dn: cn=Gunars Trader, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gunars Trader
sn: Trader
description: This is Gunars Trader's description
facsimileTelephoneNumber: +1 408 216-7540
l: Menlo Park
ou: Planning
postalAddress: example$Planning$Dept # 302
telephoneNumber: +1 510 768-6819
title: Master Planning Madonna
userPassword: redarTsran
uid: Gunars_Trader
givenName: Gunars
mail: Gunars_Trader@example.com
carLicense: HPYPGZJ
departmentNumber: 4629
employeeType: Temp
homePhone: +1 213 584-1198
initials: G. T.
mobile: +1 408 347-8551
pager: +1 71 425-9989
manager: cn=Kata Alfred
secretary: cn=Cariotta Zattiero
roomNumber: 6922

dn: cn=Jillayne Michelsen, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jillayne Michelsen
sn: Michelsen
description: This is Jillayne Michelsen's description
facsimileTelephoneNumber: +1 415 149-4718
l: San Jose
ou: Management
postalAddress: example$Management$Dept # 203
telephoneNumber: +1 510 847-7991
title: Supreme Management Artist
userPassword: neslehciMe
uid: Jillayne_Michelsen
givenName: Jillayne
mail: Jillayne_Michelsen@example.com
carLicense: JW0NJLZ
departmentNumber: 1629
employeeType: Manager
homePhone: +1 510 566-3934
initials: J. M.
mobile: +1 303 590-9559
pager: +1 213 112-5078
manager: cn=Condell Cranston
secretary: cn=Xenia Chapleau
roomNumber: 8583

dn: cn=Kalie Clough, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kalie Clough
sn: Clough
description: This is Kalie Clough's description
facsimileTelephoneNumber: +1 818 114-2105
l: Mountain View
ou: Payroll
postalAddress: example$Payroll$Dept # 574
telephoneNumber: +1 206 335-3186
title: Associate Payroll Vice President
userPassword: hguolCeila
uid: Kalie_Clough
givenName: Kalie
mail: Kalie_Clough@example.com
carLicense: BBILRZE
departmentNumber: 5335
employeeType: Employee
homePhone: +1 818 830-7942
initials: K. C.
mobile: +1 510 402-9085
pager: +1 206 628-8151
manager: cn=Petra Ledford
secretary: cn=Lauraine Paulin
roomNumber: 3572

dn: cn=Lowietje Koskinen, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lowietje Koskinen
sn: Koskinen
description: This is Lowietje Koskinen's description
facsimileTelephoneNumber: +1 415 610-7184
l: Fremont
ou: Product Development
postalAddress: example$Product Development$Dept # 338
telephoneNumber: +1 818 218-4648
title: Chief Product Development Sales Rep
userPassword: neniksoKej
uid: Lowietje_Koskinen
givenName: Lowietje
mail: Lowietje_Koskinen@example.com
carLicense: TU079TG
departmentNumber: 2364
employeeType: Contract
homePhone: +1 213 125-5802
initials: L. K.
mobile: +1 71 176-3460
pager: +1 408 271-5825
manager: cn=Jody Von Semmler
secretary: cn=Drucill Caleta
roomNumber: 1942

dn: cn=Dolly Zarate, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dolly Zarate
sn: Zarate
description: This is Dolly Zarate's description
facsimileTelephoneNumber: +1 415 249-5058
l: Fremont
ou: Planning
postalAddress: example$Planning$Dept # 181
telephoneNumber: +1 510 954-3865
title: Associate Planning Figurehead
userPassword: etaraZyllo
uid: Dolly_Zarate
givenName: Dolly
mail: Dolly_Zarate@example.com
carLicense: RRYYT6Y
departmentNumber: 6066
employeeType: Contract
homePhone: +1 804 673-6422
initials: D. Z.
mobile: +1 303 263-3964
pager: +1 303 386-5397
manager: cn=Analise Logntp
secretary: cn=Yodha Theriot
roomNumber: 1677

dn: cn=Kettie Roig, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kettie Roig
sn: Roig
description: This is Kettie Roig's description
facsimileTelephoneNumber: +1 303 720-1619
l: Cupertino
ou: Human Resources
postalAddress: example$Human Resources$Dept # 358
telephoneNumber: +1 303 513-2397
title: Senior Human Resources Admin
userPassword: gioReitteK
uid: Kettie_Roig
givenName: Kettie
mail: Kettie_Roig@example.com
carLicense: 99DJIXW
departmentNumber: 3346
employeeType: Contract
homePhone: +1 303 979-1218
initials: K. R.
mobile: +1 206 258-1558
pager: +1 818 557-4666
manager: cn=Lauretta Jaswal
secretary: cn=Sophia Valenziano
roomNumber: 4614

dn: cn=Cathee Cottrell, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Cathee Cottrell
sn: Cottrell
description: This is Cathee Cottrell's description
facsimileTelephoneNumber: +1 415 388-9625
l: Redmond
ou: Product Development
postalAddress: example$Product Development$Dept # 773
telephoneNumber: +1 804 855-7117
title: Senior Product Development Figurehead
userPassword: llerttoCee
uid: Cathee_Cottrell
givenName: Cathee
mail: Cathee_Cottrell@example.com
carLicense: A0J184M
departmentNumber: 6207
employeeType: Contract
homePhone: +1 71 220-9600
initials: C. C.
mobile: +1 206 723-4560
pager: +1 303 911-5420
manager: cn=Delmar Charlino
secretary: cn=Tetsumo McMillion
roomNumber: 5388

dn: cn=Anet Clapham, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Anet Clapham
sn: Clapham
description: This is Anet Clapham's description
facsimileTelephoneNumber: +1 213 803-3646
l: Cambridge
ou: Peons
postalAddress: example$Peons$Dept # 36
telephoneNumber: +1 510 788-4921
title: Supreme Peons Writer
userPassword: mahpalCten
uid: Anet_Clapham
givenName: Anet
mail: Anet_Clapham@example.com
carLicense: HX3P0G2
departmentNumber: 6210
employeeType: Employee
homePhone: +1 71 456-1198
initials: A. C.
mobile: +1 303 269-7859
pager: +1 213 265-6142
manager: cn=Ami Ching
secretary: cn=Sheldon Maleski
roomNumber: 3839

dn: cn=Larkin Remers, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Larkin Remers
sn: Remers
description: This is Larkin Remers's description
facsimileTelephoneNumber: +1 804 858-4790
l: Cambridge
ou: Administrative
postalAddress: example$Administrative$Dept # 543
telephoneNumber: +1 415 117-2576
title: Master Administrative Stooge
userPassword: sremeRnikr
uid: Larkin_Remers
givenName: Larkin
mail: Larkin_Remers@example.com
carLicense: VP6MJNV
departmentNumber: 665
employeeType: Temp
homePhone: +1 213 233-7341
initials: L. R.
mobile: +1 408 231-2216
pager: +1 415 524-3809
manager: cn=Kiersten Limbaugh
secretary: cn=Bharat Darcel
roomNumber: 2787

dn: cn=Tin Woodward-Jack, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tin Woodward-Jack
sn: Woodward-Jack
description: This is Tin Woodward-Jack's description
facsimileTelephoneNumber: +1 818 177-8005
l: Sunnyvale
ou: Human Resources
postalAddress: example$Human Resources$Dept # 350
telephoneNumber: +1 818 787-2067
title: Supreme Human Resources Architect
userPassword: kcaJ-drawd
uid: Tin_Woodward-Jack
givenName: Tin
mail: Tin_Woodward-Jack@example.com
carLicense: 3QBISPO
departmentNumber: 8199
employeeType: Manager
homePhone: +1 71 116-9381
initials: T. W.
mobile: +1 303 257-6901
pager: +1 408 229-3825
manager: cn=Ezmeralda Jago
secretary: cn=Augustin Allaway
roomNumber: 8846

dn: cn=Soyong Sandiford, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Soyong Sandiford
sn: Sandiford
description: This is Soyong Sandiford's description
facsimileTelephoneNumber: +1 71 284-4799
l: San Jose
ou: Janitorial
postalAddress: example$Janitorial$Dept # 953
telephoneNumber: +1 415 512-1280
title: Master Janitorial Technician
userPassword: drofidnaSg
uid: Soyong_Sandiford
givenName: Soyong
mail: Soyong_Sandiford@example.com
carLicense: U2VXM06
departmentNumber: 875
employeeType: Employee
homePhone: +1 415 810-9760
initials: S. S.
mobile: +1 303 589-6998
pager: +1 818 104-3267
manager: cn=Meghan Prodmfg
secretary: cn=Adelaide Ibach
roomNumber: 8549

dn: cn=Amii Hipson, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Amii Hipson
sn: Hipson
description: This is Amii Hipson's description
facsimileTelephoneNumber: +1 415 435-7189
l: Menlo Park
ou: Peons
postalAddress: example$Peons$Dept # 411
telephoneNumber: +1 213 402-7077
title: Elite Peons Figurehead
userPassword: nospiHiimA
uid: Amii_Hipson
givenName: Amii
mail: Amii_Hipson@example.com
carLicense: 3C2MIWS
departmentNumber: 8457
employeeType: Contract
homePhone: +1 415 931-9572
initials: A. H.
mobile: +1 303 208-9394
pager: +1 206 533-6902
manager: cn=Rhetta Liesenberg
secretary: cn=Allx Rantala
roomNumber: 5989

dn: cn=Misti Ramseyer, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Misti Ramseyer
sn: Ramseyer
description: This is Misti Ramseyer's description
facsimileTelephoneNumber: +1 510 161-7981
l: San Mateo
ou: Peons
postalAddress: example$Peons$Dept # 692
telephoneNumber: +1 818 207-5357
title: Supreme Peons Warrior
userPassword: reyesmaRit
uid: Misti_Ramseyer
givenName: Misti
mail: Misti_Ramseyer@example.com
carLicense: FR4H13P
departmentNumber: 9743
employeeType: Temp
homePhone: +1 415 742-2788
initials: M. R.
mobile: +1 415 701-8185
pager: +1 206 761-7230
manager: cn=Saba Vennos
secretary: cn=Kathryne Nagai
roomNumber: 2716

dn: cn=Ashli Borosch, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ashli Borosch
sn: Borosch
description: This is Ashli Borosch's description
facsimileTelephoneNumber: +1 804 743-7087
l: Cupertino
ou: Peons
postalAddress: example$Peons$Dept # 969
telephoneNumber: +1 408 388-7362
title: Senior Peons Warrior
userPassword: hcsoroBilh
uid: Ashli_Borosch
givenName: Ashli
mail: Ashli_Borosch@example.com
carLicense: Y5M28EV
departmentNumber: 6583
employeeType: Manager
homePhone: +1 303 846-8629
initials: A. B.
mobile: +1 213 915-9960
pager: +1 510 836-3825
manager: cn=Sissie MAINT
secretary: cn=Chen Peirce
roomNumber: 9109

dn: cn=Perle Pisani, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Perle Pisani
sn: Pisani
description: This is Perle Pisani's description
facsimileTelephoneNumber: +1 303 546-5638
l: Armonk
ou: Payroll
postalAddress: example$Payroll$Dept # 27
telephoneNumber: +1 213 452-3530
title: Elite Payroll Artist
userPassword: inasiPelre
uid: Perle_Pisani
givenName: Perle
mail: Perle_Pisani@example.com
carLicense: 95NVMCX
departmentNumber: 1320
employeeType: Normal
homePhone: +1 818 370-9640
initials: P. P.
mobile: +1 510 256-9020
pager: +1 818 401-8427
manager: cn=Marj Alzofon
secretary: cn=Bianka McCafferty
roomNumber: 1935

dn: cn=Ajit Kruziak, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ajit Kruziak
sn: Kruziak
description: This is Ajit Kruziak's description
facsimileTelephoneNumber: +1 510 354-6551
l: San Mateo
ou: Payroll
postalAddress: example$Payroll$Dept # 825
telephoneNumber: +1 415 103-1822
title: Elite Payroll Admin
userPassword: kaizurKtij
uid: Ajit_Kruziak
givenName: Ajit
mail: Ajit_Kruziak@example.com
carLicense: DLPPITG
departmentNumber: 1221
employeeType: Employee
homePhone: +1 408 261-2038
initials: A. K.
mobile: +1 415 101-1370
pager: +1 213 844-4207
manager: cn=Patti Brungardt
secretary: cn=Zarla Kyoung
roomNumber: 1811

dn: cn=Der Salomon, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Der Salomon
sn: Salomon
description: This is Der Salomon's description
facsimileTelephoneNumber: +1 408 488-7614
l: San Francisco
ou: Administrative
postalAddress: example$Administrative$Dept # 53
telephoneNumber: +1 818 130-5643
title: Supreme Administrative Stooge
userPassword: nomolaSreD
uid: Der_Salomon
givenName: Der
mail: Der_Salomon@example.com
carLicense: 217NI44
departmentNumber: 1363
employeeType: Contract
homePhone: +1 303 356-3857
initials: D. S.
mobile: +1 213 816-2420
pager: +1 71 563-2262
manager: cn=Aaren Lessin
secretary: cn=Tulip Adler
roomNumber: 5756

dn: cn=Anstice Chatel, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Anstice Chatel
sn: Chatel
description: This is Anstice Chatel's description
facsimileTelephoneNumber: +1 804 213-6878
l: San Mateo
ou: Product Development
postalAddress: example$Product Development$Dept # 69
telephoneNumber: +1 206 992-9408
title: Master Product Development Dictator
userPassword: letahCecit
uid: Anstice_Chatel
givenName: Anstice
mail: Anstice_Chatel@example.com
carLicense: SIIXYR0
departmentNumber: 1644
employeeType: Manager
homePhone: +1 510 234-8279
initials: A. C.
mobile: +1 71 326-4355
pager: +1 818 684-4808
manager: cn=Carran Frederick
secretary: cn=Gwenneth Conboy
roomNumber: 5898

dn: cn=Follick Tom, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Follick Tom
sn: Tom
description: This is Follick Tom's description
facsimileTelephoneNumber: +1 303 116-7145
l: Redwood Shores
ou: Janitorial
postalAddress: example$Janitorial$Dept # 681
telephoneNumber: +1 408 500-5612
title: Elite Janitorial Admin
userPassword: moTkcilloF
uid: Follick_Tom
givenName: Follick
mail: Follick_Tom@example.com
carLicense: TLW2Y9A
departmentNumber: 5789
employeeType: Normal
homePhone: +1 510 285-6385
initials: F. T.
mobile: +1 818 885-1048
pager: +1 408 951-6231
manager: cn=Fariborz Laviolette
secretary: cn=Liva Sebastian
roomNumber: 9167

dn: cn=Mary-Jane Breault, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mary-Jane Breault
sn: Breault
description: This is Mary-Jane Breault's description
facsimileTelephoneNumber: +1 415 995-2885
l: Mountain View
ou: Planning
postalAddress: example$Planning$Dept # 757
telephoneNumber: +1 408 647-4483
title: Chief Planning Technician
userPassword: tluaerBena
uid: Mary-Jane_Breault
givenName: Mary-Jane
mail: Mary-Jane_Breault@example.com
carLicense: WCWZDQI
departmentNumber: 3957
employeeType: Contract
homePhone: +1 415 425-3260
initials: M. B.
mobile: +1 303 426-8026
pager: +1 415 312-3046
manager: cn=Erlene Sylvie
secretary: cn=Marcos Schacham
roomNumber: 733

dn: cn=Grace Dmuchalsky, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Grace Dmuchalsky
sn: Dmuchalsky
description: This is Grace Dmuchalsky's description
facsimileTelephoneNumber: +1 71 687-7498
l: San Mateo
ou: Product Development
postalAddress: example$Product Development$Dept # 369
telephoneNumber: +1 804 342-9553
title: Elite Product Development Sales Rep
userPassword: ykslahcumD
uid: Grace_Dmuchalsky
givenName: Grace
mail: Grace_Dmuchalsky@example.com
carLicense: 755OD0D
departmentNumber: 4075
employeeType: Employee
homePhone: +1 818 699-6712
initials: G. D.
mobile: +1 213 758-9777
pager: +1 206 195-8748
manager: cn=Crystal Popovics
secretary: cn=Daryn Murock
roomNumber: 2198

dn: cn=Winne Chiu, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Winne Chiu
sn: Chiu
description: This is Winne Chiu's description
facsimileTelephoneNumber: +1 415 275-6077
l: Milpitas
ou: Peons
postalAddress: example$Peons$Dept # 651
telephoneNumber: +1 415 478-6184
title: Associate Peons Director
userPassword: uihCenniW
uid: Winne_Chiu
givenName: Winne
mail: Winne_Chiu@example.com
carLicense: CW6RJU2
departmentNumber: 4673
employeeType: Temp
homePhone: +1 415 408-4113
initials: W. C.
mobile: +1 71 332-6219
pager: +1 818 715-9563
manager: cn=Elbert Erichsen
secretary: cn=Khai Speer
roomNumber: 5558

dn: cn=Tanitansy Carmona, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tanitansy Carmona
sn: Carmona
description: This is Tanitansy Carmona's description
facsimileTelephoneNumber: +1 206 581-9777
l: Cambridge
ou: Peons
postalAddress: example$Peons$Dept # 545
telephoneNumber: +1 804 655-1179
title: Master Peons Manager
userPassword: anomraCysn
uid: Tanitansy_Carmona
givenName: Tanitansy
mail: Tanitansy_Carmona@example.com
carLicense: V12V2BZ
departmentNumber: 7687
employeeType: Employee
homePhone: +1 415 241-3229
initials: T. C.
mobile: +1 408 958-7516
pager: +1 818 234-8717
manager: cn=Laurette Legris
secretary: cn=Germ Vaserfirer
roomNumber: 6944

dn: cn=Zouheir Fisprod, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zouheir Fisprod
sn: Fisprod
description: This is Zouheir Fisprod's description
facsimileTelephoneNumber: +1 804 206-6561
l: Cupertino
ou: Product Development
postalAddress: example$Product Development$Dept # 550
telephoneNumber: +1 206 811-7606
title: Chief Product Development Mascot
userPassword: dorpsiFrie
uid: Zouheir_Fisprod
givenName: Zouheir
mail: Zouheir_Fisprod@example.com
carLicense: 3PRJVCA
departmentNumber: 6549
employeeType: Employee
homePhone: +1 213 621-4789
initials: Z. F.
mobile: +1 213 717-4374
pager: +1 818 383-5244
manager: cn=Norvie Eales
secretary: cn=Millie Siehl
roomNumber: 4490

dn: cn=John-Jr Saladna, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: John-Jr Saladna
sn: Saladna
description: This is John-Jr Saladna's description
facsimileTelephoneNumber: +1 71 164-5753
l: Mountain View
ou: Planning
postalAddress: example$Planning$Dept # 176
telephoneNumber: +1 510 901-9725
title: Supreme Planning Grunt
userPassword: andalaSrJ-
uid: John-Jr_Saladna
givenName: John-Jr
mail: John-Jr_Saladna@example.com
carLicense: 3JO3PVV
departmentNumber: 3382
employeeType: Employee
homePhone: +1 206 621-2628
initials: J. S.
mobile: +1 71 114-6947
pager: +1 818 601-2674
manager: cn=Minetta Sherif
secretary: cn=Kimberlyn Meskimen
roomNumber: 459

dn: cn=Susana Fougere, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Susana Fougere
sn: Fougere
description: This is Susana Fougere's description
facsimileTelephoneNumber: +1 213 454-9358
l: San Mateo
ou: Management
postalAddress: example$Management$Dept # 776
telephoneNumber: +1 510 480-5730
title: Master Management Janitor
userPassword: ereguoFana
uid: Susana_Fougere
givenName: Susana
mail: Susana_Fougere@example.com
carLicense: VSP4LUY
departmentNumber: 3606
employeeType: Employee
homePhone: +1 510 181-9611
initials: S. F.
mobile: +1 213 770-2904
pager: +1 408 415-5120
manager: cn=Meggi Dowding
secretary: cn=Josine Hubal
roomNumber: 5870

dn: cn=Takehiko Smits, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Takehiko Smits
sn: Smits
description: This is Takehiko Smits's description
facsimileTelephoneNumber: +1 408 346-8793
l: San Francisco
ou: Product Development
postalAddress: example$Product Development$Dept # 981
telephoneNumber: +1 804 705-8588
title: Senior Product Development Evangelist
userPassword: stimSokihe
uid: Takehiko_Smits
givenName: Takehiko
mail: Takehiko_Smits@example.com
carLicense: OMCXJMG
departmentNumber: 1408
employeeType: Contract
homePhone: +1 213 821-6140
initials: T. S.
mobile: +1 303 149-7395
pager: +1 408 574-8356
manager: cn=Verlyn Decapua
secretary: cn=Prafula Armentrout
roomNumber: 7433

dn: cn=Aarika Wolfman, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Aarika Wolfman
sn: Wolfman
description: This is Aarika Wolfman's description
facsimileTelephoneNumber: +1 510 421-3943
l: Cambridge
ou: Peons
postalAddress: example$Peons$Dept # 922
telephoneNumber: +1 71 544-8828
title: Supreme Peons Yahoo
userPassword: namfloWaki
uid: Aarika_Wolfman
givenName: Aarika
mail: Aarika_Wolfman@example.com
carLicense: XZ88GIK
departmentNumber: 2833
employeeType: Employee
homePhone: +1 213 859-5564
initials: A. W.
mobile: +1 804 713-4223
pager: +1 303 607-7149
manager: cn=Janio Wiltz
secretary: cn=Alfredo Boynton
roomNumber: 7541

dn: cn=Sabra Magee, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sabra Magee
sn: Magee
description: This is Sabra Magee's description
facsimileTelephoneNumber: +1 206 676-6546
l: Cambridge
ou: Peons
postalAddress: example$Peons$Dept # 122
telephoneNumber: +1 206 839-4408
title: Associate Peons Consultant
userPassword: eegaMarbaS
uid: Sabra_Magee
givenName: Sabra
mail: Sabra_Magee@example.com
carLicense: SMRV5GK
departmentNumber: 3108
employeeType: Temp
homePhone: +1 804 265-3642
initials: S. M.
mobile: +1 71 651-6097
pager: +1 213 287-8936
manager: cn=Harmi Cobran
secretary: cn=Valida Denomme
roomNumber: 8087

dn: cn=Marjan Kupferschmidt, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marjan Kupferschmidt
sn: Kupferschmidt
description: This is Marjan Kupferschmidt's description
facsimileTelephoneNumber: +1 213 692-8541
l: Armonk
ou: Administrative
postalAddress: example$Administrative$Dept # 541
telephoneNumber: +1 408 217-7188
title: Senior Administrative Fellow
userPassword: tdimhcsref
uid: Marjan_Kupferschmidt
givenName: Marjan
mail: Marjan_Kupferschmidt@example.com
carLicense: QGYGGLB
departmentNumber: 8065
employeeType: Contract
homePhone: +1 818 795-1037
initials: M. K.
mobile: +1 415 825-4455
pager: +1 408 105-7416
manager: cn=Magnolia Joyce
secretary: cn=Harriott Demren
roomNumber: 1223

dn: cn=Spicer Modafferi, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Spicer Modafferi
sn: Modafferi
description: This is Spicer Modafferi's description
facsimileTelephoneNumber: +1 408 328-6089
l: San Jose
ou: Administrative
postalAddress: example$Administrative$Dept # 846
telephoneNumber: +1 510 189-8308
title: Master Administrative Grunt
userPassword: ireffadoMr
uid: Spicer_Modafferi
givenName: Spicer
mail: Spicer_Modafferi@example.com
carLicense: 4KUKJ5D
departmentNumber: 4785
employeeType: Manager
homePhone: +1 804 278-1044
initials: S. M.
mobile: +1 206 849-2011
pager: +1 510 368-4746
manager: cn=Gabriellia Clancy
secretary: cn=Larysa Bmethods
roomNumber: 3133

dn: cn=Minnaminnie Lavers, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Minnaminnie Lavers
sn: Lavers
description: This is Minnaminnie Lavers's description
facsimileTelephoneNumber: +1 818 267-6325
l: San Jose
ou: Accounting
postalAddress: example$Accounting$Dept # 197
telephoneNumber: +1 818 513-3510
title: Master Accounting Yahoo
userPassword: srevaLeinn
uid: Minnaminnie_Lavers
givenName: Minnaminnie
mail: Minnaminnie_Lavers@example.com
carLicense: FI0MMND
departmentNumber: 5516
employeeType: Contract
homePhone: +1 206 657-8324
initials: M. L.
mobile: +1 71 119-4456
pager: +1 206 776-9058
manager: cn=Christi Minos
secretary: cn=Juli Ayres
roomNumber: 6299

dn: cn=Davis OFCPARM, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Davis OFCPARM
sn: OFCPARM
description: This is Davis OFCPARM's description
facsimileTelephoneNumber: +1 303 450-2426
l: Fremont
ou: Janitorial
postalAddress: example$Janitorial$Dept # 250
telephoneNumber: +1 818 838-6409
title: Master Janitorial Evangelist
userPassword: MRAPCFOsiv
uid: Davis_OFCPARM
givenName: Davis
mail: Davis_OFCPARM@example.com
carLicense: CC4Q350
departmentNumber: 8851
employeeType: Temp
homePhone: +1 818 603-3620
initials: D. O.
mobile: +1 818 757-8353
pager: +1 71 664-7052
manager: cn=Lilah Marting
secretary: cn=Hareton Piercey
roomNumber: 7599

dn: cn=Cordy Mahiger, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Cordy Mahiger
sn: Mahiger
description: This is Cordy Mahiger's description
facsimileTelephoneNumber: +1 510 967-3620
l: Redwood Shores
ou: Human Resources
postalAddress: example$Human Resources$Dept # 810
telephoneNumber: +1 818 304-8895
title: Senior Human Resources President
userPassword: regihaMydr
uid: Cordy_Mahiger
givenName: Cordy
mail: Cordy_Mahiger@example.com
carLicense: DGIUUIF
departmentNumber: 9385
employeeType: Normal
homePhone: +1 303 233-7844
initials: C. M.
mobile: +1 818 735-9720
pager: +1 213 482-6716
manager: cn=Iseabal Bryttan
secretary: cn=Rosella Delorenzi
roomNumber: 9567

dn: cn=Sherrie Ishak, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sherrie Ishak
sn: Ishak
description: This is Sherrie Ishak's description
facsimileTelephoneNumber: +1 213 222-2612
l: San Francisco
ou: Product Testing
postalAddress: example$Product Testing$Dept # 55
telephoneNumber: +1 303 465-7312
title: Associate Product Testing Admin
userPassword: kahsIeirre
uid: Sherrie_Ishak
givenName: Sherrie
mail: Sherrie_Ishak@example.com
carLicense: 8CJYTR1
departmentNumber: 2845
employeeType: Employee
homePhone: +1 804 973-9215
initials: S. I.
mobile: +1 415 681-2614
pager: +1 303 554-6124
manager: cn=Darwin Delroy
secretary: cn=Stirling Zargham
roomNumber: 7665

dn: cn=Andre Grills, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Andre Grills
sn: Grills
description: This is Andre Grills's description
facsimileTelephoneNumber: +1 415 186-9781
l: Redwood Shores
ou: Janitorial
postalAddress: example$Janitorial$Dept # 735
telephoneNumber: +1 206 810-7100
title: Supreme Janitorial Consultant
userPassword: sllirGerdn
uid: Andre_Grills
givenName: Andre
mail: Andre_Grills@example.com
carLicense: 3WHP5CH
departmentNumber: 8411
employeeType: Normal
homePhone: +1 303 354-3117
initials: A. G.
mobile: +1 71 782-9170
pager: +1 408 974-6830
manager: cn=Djordje Buckhoff
secretary: cn=Jaffer Trochu
roomNumber: 6853

dn: cn=Dorothee Anker, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dorothee Anker
sn: Anker
description: This is Dorothee Anker's description
facsimileTelephoneNumber: +1 213 854-9079
l: Alameda
ou: Management
postalAddress: example$Management$Dept # 815
telephoneNumber: +1 804 867-8999
title: Senior Management Mascot
userPassword: reknAeehto
uid: Dorothee_Anker
givenName: Dorothee
mail: Dorothee_Anker@example.com
carLicense: FLRUL35
departmentNumber: 5718
employeeType: Normal
homePhone: +1 71 176-6117
initials: D. A.
mobile: +1 71 779-2330
pager: +1 71 243-8423
manager: cn=Hengameh Popowicz
secretary: cn=Shela Mishina
roomNumber: 8072

dn: cn=Yosuf Vajentic, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Yosuf Vajentic
sn: Vajentic
description: This is Yosuf Vajentic's description
facsimileTelephoneNumber: +1 206 277-7099
l: San Jose
ou: Accounting
postalAddress: example$Accounting$Dept # 954
telephoneNumber: +1 213 207-7250
title: Junior Accounting Admin
userPassword: citnejaVfu
uid: Yosuf_Vajentic
givenName: Yosuf
mail: Yosuf_Vajentic@example.com
carLicense: IUX1PNF
departmentNumber: 9053
employeeType: Temp
homePhone: +1 510 486-8360
initials: Y. V.
mobile: +1 818 210-9416
pager: +1 415 596-7033
manager: cn=Tova Javallas
secretary: cn=Vithit Serre
roomNumber: 8330

dn: cn=Goldie Tanglao, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Goldie Tanglao
sn: Tanglao
description: This is Goldie Tanglao's description
facsimileTelephoneNumber: +1 804 831-5929
l: Santa Clara
ou: Accounting
postalAddress: example$Accounting$Dept # 310
telephoneNumber: +1 510 325-6263
title: Junior Accounting Vice President
userPassword: oalgnaTeid
uid: Goldie_Tanglao
givenName: Goldie
mail: Goldie_Tanglao@example.com
carLicense: 826ZCMC
departmentNumber: 2571
employeeType: Employee
homePhone: +1 71 925-7934
initials: G. T.
mobile: +1 303 371-2678
pager: +1 415 984-2018
manager: cn=Rajiv Glaser
secretary: cn=Imtiaz Winsborrow
roomNumber: 906

dn: cn=Kingsley Kawauchi, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kingsley Kawauchi
sn: Kawauchi
description: This is Kingsley Kawauchi's description
facsimileTelephoneNumber: +1 408 760-5114
l: San Francisco
ou: Janitorial
postalAddress: example$Janitorial$Dept # 519
telephoneNumber: +1 206 263-5681
title: Master Janitorial Consultant
userPassword: ihcuawaKye
uid: Kingsley_Kawauchi
givenName: Kingsley
mail: Kingsley_Kawauchi@example.com
carLicense: WOCI58R
departmentNumber: 6947
employeeType: Normal
homePhone: +1 804 395-9664
initials: K. K.
mobile: +1 804 216-1764
pager: +1 510 500-3660
manager: cn=Claudine Pipkins
secretary: cn=Clarence Hu
roomNumber: 5876

dn: cn=Daisi Lenathen, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Daisi Lenathen
sn: Lenathen
description: This is Daisi Lenathen's description
facsimileTelephoneNumber: +1 303 392-4879
l: Orem
ou: Human Resources
postalAddress: example$Human Resources$Dept # 748
telephoneNumber: +1 415 171-6359
title: Senior Human Resources President
userPassword: nehtaneLis
uid: Daisi_Lenathen
givenName: Daisi
mail: Daisi_Lenathen@example.com
carLicense: DQILRJE
departmentNumber: 7522
employeeType: Manager
homePhone: +1 213 288-2856
initials: D. L.
mobile: +1 408 189-7434
pager: +1 804 830-2227
manager: cn=Fikre Reckhard
secretary: cn=Chickie Swearingen
roomNumber: 8443

dn: cn=Kala Cohoe, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kala Cohoe
sn: Cohoe
description: This is Kala Cohoe's description
facsimileTelephoneNumber: +1 804 464-6621
l: Menlo Park
ou: Product Testing
postalAddress: example$Product Testing$Dept # 279
telephoneNumber: +1 408 445-8566
title: Junior Product Testing Czar
userPassword: eohoCalaK
uid: Kala_Cohoe
givenName: Kala
mail: Kala_Cohoe@example.com
carLicense: 83O6O9C
departmentNumber: 4001
employeeType: Manager
homePhone: +1 510 395-6239
initials: K. C.
mobile: +1 408 280-9013
pager: +1 408 456-6256
manager: cn=Dorice Tihanyi
secretary: cn=Jock Wandel
roomNumber: 1488

dn: cn=Ashraf Philbeck, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ashraf Philbeck
sn: Philbeck
description: This is Ashraf Philbeck's description
facsimileTelephoneNumber: +1 510 510-2082
l: Armonk
ou: Accounting
postalAddress: example$Accounting$Dept # 815
telephoneNumber: +1 303 409-9870
title: Chief Accounting Fellow
userPassword: kceblihPfa
uid: Ashraf_Philbeck
givenName: Ashraf
mail: Ashraf_Philbeck@example.com
carLicense: O9HJ38K
departmentNumber: 3333
employeeType: Contract
homePhone: +1 213 369-2997
initials: A. P.
mobile: +1 206 533-4383
pager: +1 510 627-6913
manager: cn=Brandea Critchley
secretary: cn=Simona Downs
roomNumber: 1927

dn: cn=Bud Luetchford, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bud Luetchford
sn: Luetchford
description: This is Bud Luetchford's description
facsimileTelephoneNumber: +1 408 815-1654
l: Milpitas
ou: Janitorial
postalAddress: example$Janitorial$Dept # 375
telephoneNumber: +1 206 938-7684
title: Junior Janitorial Punk
userPassword: drofhcteuL
uid: Bud_Luetchford
givenName: Bud
mail: Bud_Luetchford@example.com
carLicense: 4HOZS1C
departmentNumber: 2862
employeeType: Normal
homePhone: +1 408 697-2759
initials: B. L.
mobile: +1 415 988-7571
pager: +1 71 724-3491
manager: cn=Youwen Kirkby
secretary: cn=Wendy Orsini
roomNumber: 4401

dn: cn=Gee-Meng Howie, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gee-Meng Howie
sn: Howie
description: This is Gee-Meng Howie's description
facsimileTelephoneNumber: +1 303 729-1050
l: Redwood Shores
ou: Administrative
postalAddress: example$Administrative$Dept # 343
telephoneNumber: +1 804 261-8663
title: Junior Administrative Admin
userPassword: eiwoHgneM-
uid: Gee-Meng_Howie
givenName: Gee-Meng
mail: Gee-Meng_Howie@example.com
carLicense: GHFQY6Z
departmentNumber: 9631
employeeType: Manager
homePhone: +1 206 608-7175
initials: G. H.
mobile: +1 510 410-4154
pager: +1 408 723-2555
manager: cn=Faun Passin
secretary: cn=Jack Xpmbld
roomNumber: 3436

dn: cn=Phedra Darrimon, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Phedra Darrimon
sn: Darrimon
description: This is Phedra Darrimon's description
facsimileTelephoneNumber: +1 303 138-9353
l: Armonk
ou: Management
postalAddress: example$Management$Dept # 405
telephoneNumber: +1 818 405-8636
title: Elite Management Vice President
userPassword: nomirraDar
uid: Phedra_Darrimon
givenName: Phedra
mail: Phedra_Darrimon@example.com
carLicense: SH81DXM
departmentNumber: 2202
employeeType: Temp
homePhone: +1 408 271-1626
initials: P. D.
mobile: +1 408 464-4771
pager: +1 206 437-2201
manager: cn=Del Ermarkaryan
secretary: cn=Lulu Terwey
roomNumber: 4969

dn: cn=Elvira Nunn, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Elvira Nunn
sn: Nunn
description: This is Elvira Nunn's description
facsimileTelephoneNumber: +1 303 321-7280
l: Menlo Park
ou: Product Testing
postalAddress: example$Product Testing$Dept # 345
telephoneNumber: +1 206 655-9226
title: Elite Product Testing Developer
userPassword: nnuNarivlE
uid: Elvira_Nunn
givenName: Elvira
mail: Elvira_Nunn@example.com
carLicense: G60QAK0
departmentNumber: 2699
employeeType: Temp
homePhone: +1 510 573-4182
initials: E. N.
mobile: +1 818 935-2274
pager: +1 818 681-9250
manager: cn=Essie Csop
secretary: cn=Fredra Bourgault
roomNumber: 8155

dn: cn=Patti Rasmussen, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Patti Rasmussen
sn: Rasmussen
description: This is Patti Rasmussen's description
facsimileTelephoneNumber: +1 71 113-6095
l: Redwood Shores
ou: Management
postalAddress: example$Management$Dept # 791
telephoneNumber: +1 71 359-5237
title: Master Management Madonna
userPassword: nessumsaRi
uid: Patti_Rasmussen
givenName: Patti
mail: Patti_Rasmussen@example.com
carLicense: QWBN2VE
departmentNumber: 4044
employeeType: Manager
homePhone: +1 415 861-8446
initials: P. R.
mobile: +1 408 602-7137
pager: +1 415 248-6015
manager: cn=Anver Zivilik
secretary: cn=Marena Dorval
roomNumber: 9158

dn: cn=Kelsey Adjangba, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kelsey Adjangba
sn: Adjangba
description: This is Kelsey Adjangba's description
facsimileTelephoneNumber: +1 415 446-4800
l: Redmond
ou: Product Development
postalAddress: example$Product Development$Dept # 284
telephoneNumber: +1 213 302-5397
title: Senior Product Development Director
userPassword: abgnajdAye
uid: Kelsey_Adjangba
givenName: Kelsey
mail: Kelsey_Adjangba@example.com
carLicense: 42L6PYH
departmentNumber: 2664
employeeType: Contract
homePhone: +1 408 913-8280
initials: K. A.
mobile: +1 206 549-3065
pager: +1 408 744-1860
manager: cn=Evans Blaylock
secretary: cn=Wilie Fields
roomNumber: 8069

dn: cn=Flor Maidens, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Flor Maidens
sn: Maidens
description: This is Flor Maidens's description
facsimileTelephoneNumber: +1 818 357-8091
l: San Francisco
ou: Product Testing
postalAddress: example$Product Testing$Dept # 270
telephoneNumber: +1 408 181-7229
title: Senior Product Testing Consultant
userPassword: snediaMrol
uid: Flor_Maidens
givenName: Flor
mail: Flor_Maidens@example.com
carLicense: PX8C8S7
departmentNumber: 6989
employeeType: Employee
homePhone: +1 510 810-7529
initials: F. M.
mobile: +1 213 656-3768
pager: +1 510 985-6317
manager: cn=Steffen Deatherage
secretary: cn=Gretta Sherow
roomNumber: 2598

dn: cn=Derrik Branham, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Derrik Branham
sn: Branham
description: This is Derrik Branham's description
facsimileTelephoneNumber: +1 71 540-4555
l: San Mateo
ou: Management
postalAddress: example$Management$Dept # 489
telephoneNumber: +1 804 653-4502
title: Associate Management Vice President
userPassword: mahnarBkir
uid: Derrik_Branham
givenName: Derrik
mail: Derrik_Branham@example.com
carLicense: 42X0E0L
departmentNumber: 2307
employeeType: Manager
homePhone: +1 408 512-1260
initials: D. B.
mobile: +1 71 992-2538
pager: +1 818 198-6606
manager: cn=Nader Peate
secretary: cn=Vincente Cassar
roomNumber: 4204

dn: cn=Maurice Robles, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Maurice Robles
sn: Robles
description: This is Maurice Robles's description
facsimileTelephoneNumber: +1 213 175-1117
l: Milpitas
ou: Product Testing
postalAddress: example$Product Testing$Dept # 802
telephoneNumber: +1 804 335-9106
title: Master Product Testing Consultant
userPassword: selboRecir
uid: Maurice_Robles
givenName: Maurice
mail: Maurice_Robles@example.com
carLicense: I0RCX50
departmentNumber: 5445
employeeType: Manager
homePhone: +1 71 631-6192
initials: M. R.
mobile: +1 415 777-4720
pager: +1 408 898-9753
manager: cn=Dusty Hyte
secretary: cn=Sergei Mesquita
roomNumber: 1159

dn: cn=Radames Lentz, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Radames Lentz
sn: Lentz
description: This is Radames Lentz's description
facsimileTelephoneNumber: +1 213 325-2266
l: Redwood Shores
ou: Administrative
postalAddress: example$Administrative$Dept # 578
telephoneNumber: +1 415 844-6002
title: Supreme Administrative Visionary
userPassword: ztneLsemad
uid: Radames_Lentz
givenName: Radames
mail: Radames_Lentz@example.com
carLicense: LIUHL8A
departmentNumber: 9200
employeeType: Manager
homePhone: +1 510 835-4657
initials: R. L.
mobile: +1 510 755-4758
pager: +1 71 826-9342
manager: cn=Gerda Gulick
secretary: cn=Earle Stallings
roomNumber: 1670

dn: cn=Shelba Ketcham, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shelba Ketcham
sn: Ketcham
description: This is Shelba Ketcham's description
facsimileTelephoneNumber: +1 71 114-1567
l: Milpitas
ou: Administrative
postalAddress: example$Administrative$Dept # 14
telephoneNumber: +1 213 887-3220
title: Master Administrative Developer
userPassword: mahcteKabl
uid: Shelba_Ketcham
givenName: Shelba
mail: Shelba_Ketcham@example.com
carLicense: UZAO4F6
departmentNumber: 3536
employeeType: Manager
homePhone: +1 415 899-5339
initials: S. K.
mobile: +1 206 522-3282
pager: +1 415 197-6248
manager: cn=Herb McRuvie
secretary: cn=Norikazu Tognoni
roomNumber: 9741

dn: cn=Jacki Kryski, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jacki Kryski
sn: Kryski
description: This is Jacki Kryski's description
facsimileTelephoneNumber: +1 804 427-6956
l: Redmond
ou: Planning
postalAddress: example$Planning$Dept # 431
telephoneNumber: +1 415 611-5818
title: Master Planning Artist
userPassword: iksyrKikca
uid: Jacki_Kryski
givenName: Jacki
mail: Jacki_Kryski@example.com
carLicense: T2DOFBJ
departmentNumber: 3473
employeeType: Employee
homePhone: +1 213 134-6640
initials: J. K.
mobile: +1 213 658-7887
pager: +1 71 414-6419
manager: cn=Edie Arkesteijn
secretary: cn=Stefan Temp
roomNumber: 8814

dn: cn=Jill Claxton, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jill Claxton
sn: Claxton
description: This is Jill Claxton's description
facsimileTelephoneNumber: +1 303 860-8984
l: Emeryville
ou: Planning
postalAddress: example$Planning$Dept # 921
telephoneNumber: +1 206 527-1706
title: Supreme Planning Assistant
userPassword: notxalClli
uid: Jill_Claxton
givenName: Jill
mail: Jill_Claxton@example.com
carLicense: VO6ZLUZ
departmentNumber: 2000
employeeType: Normal
homePhone: +1 804 466-5758
initials: J. C.
mobile: +1 206 777-3124
pager: +1 213 356-7164
manager: cn=Benoit Corbitt
secretary: cn=Ardelle Strader
roomNumber: 6214

dn: cn=Audrye Casanova, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Audrye Casanova
sn: Casanova
description: This is Audrye Casanova's description
facsimileTelephoneNumber: +1 71 335-2611
l: Santa Clara
ou: Accounting
postalAddress: example$Accounting$Dept # 508
telephoneNumber: +1 510 794-3443
title: Master Accounting Stooge
userPassword: avonasaCey
uid: Audrye_Casanova
givenName: Audrye
mail: Audrye_Casanova@example.com
carLicense: IB8U70R
departmentNumber: 716
employeeType: Temp
homePhone: +1 71 744-9481
initials: A. C.
mobile: +1 818 445-3413
pager: +1 71 373-6330
manager: cn=Marella Gehring
secretary: cn=Lorrie StJames
roomNumber: 7349

dn: cn=Antonella Deek, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Antonella Deek
sn: Deek
description: This is Antonella Deek's description
facsimileTelephoneNumber: +1 408 612-7907
l: Santa Clara
ou: Human Resources
postalAddress: example$Human Resources$Dept # 440
telephoneNumber: +1 804 159-1602
title: Supreme Human Resources Architect
userPassword: keeDalleno
uid: Antonella_Deek
givenName: Antonella
mail: Antonella_Deek@example.com
carLicense: ETYGM3Y
departmentNumber: 5961
employeeType: Contract
homePhone: +1 206 245-7987
initials: A. D.
mobile: +1 415 420-2348
pager: +1 71 855-2985
manager: cn=Donna Rendon
secretary: cn=Shandy Saunderson
roomNumber: 6859

dn: cn=Vishwa Systest, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Vishwa Systest
sn: Systest
description: This is Vishwa Systest's description
facsimileTelephoneNumber: +1 71 985-7388
l: Redwood Shores
ou: Administrative
postalAddress: example$Administrative$Dept # 99
telephoneNumber: +1 415 236-8644
title: Elite Administrative Yahoo
userPassword: tsetsySawh
uid: Vishwa_Systest
givenName: Vishwa
mail: Vishwa_Systest@example.com
carLicense: F23TD75
departmentNumber: 7245
employeeType: Contract
homePhone: +1 213 535-1250
initials: V. S.
mobile: +1 213 467-1508
pager: +1 804 192-2068
manager: cn=Marlie Davalo
secretary: cn=Elena Kehler
roomNumber: 3332

dn: cn=Thuy Sorrentino, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Thuy Sorrentino
sn: Sorrentino
description: This is Thuy Sorrentino's description
facsimileTelephoneNumber: +1 303 661-5865
l: Redmond
ou: Peons
postalAddress: example$Peons$Dept # 563
telephoneNumber: +1 71 241-8155
title: Senior Peons Vice President
userPassword: onitnerroS
uid: Thuy_Sorrentino
givenName: Thuy
mail: Thuy_Sorrentino@example.com
carLicense: XT8H4LL
departmentNumber: 9503
employeeType: Normal
homePhone: +1 71 732-6008
initials: T. S.
mobile: +1 303 174-9452
pager: +1 804 736-5264
manager: cn=Ikram Somisetty
secretary: cn=Subu D'Anjou
roomNumber: 7065

dn: cn=Mika Connelly, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mika Connelly
sn: Connelly
description: This is Mika Connelly's description
facsimileTelephoneNumber: +1 213 466-9121
l: Santa Clara
ou: Human Resources
postalAddress: example$Human Resources$Dept # 195
telephoneNumber: +1 804 881-7779
title: Chief Human Resources Accountant
userPassword: yllennoCak
uid: Mika_Connelly
givenName: Mika
mail: Mika_Connelly@example.com
carLicense: 6V7IYY7
departmentNumber: 6921
employeeType: Contract
homePhone: +1 408 126-6367
initials: M. C.
mobile: +1 213 735-2246
pager: +1 818 819-6286
manager: cn=Redgie Hruska
secretary: cn=Nazi Webber
roomNumber: 7844

dn: cn=Gwenette Markovich, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gwenette Markovich
sn: Markovich
description: This is Gwenette Markovich's description
facsimileTelephoneNumber: +1 213 862-5730
l: Santa Clara
ou: Administrative
postalAddress: example$Administrative$Dept # 281
telephoneNumber: +1 213 710-5069
title: Senior Administrative Grunt
userPassword: hcivokraMe
uid: Gwenette_Markovich
givenName: Gwenette
mail: Gwenette_Markovich@example.com
carLicense: E6H2NGM
departmentNumber: 8458
employeeType: Manager
homePhone: +1 206 608-7706
initials: G. M.
mobile: +1 213 209-5846
pager: +1 510 218-9820
manager: cn=Melisande Lotochinski
secretary: cn=Barsha Gerynowicz
roomNumber: 156

dn: cn=Amjad Tavana, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Amjad Tavana
sn: Tavana
description: This is Amjad Tavana's description
facsimileTelephoneNumber: +1 303 127-2675
l: Sunnyvale
ou: Payroll
postalAddress: example$Payroll$Dept # 35
telephoneNumber: +1 213 414-7581
title: Master Payroll Developer
userPassword: anavaTdajm
uid: Amjad_Tavana
givenName: Amjad
mail: Amjad_Tavana@example.com
carLicense: QZQTXXZ
departmentNumber: 8566
employeeType: Manager
homePhone: +1 818 765-2743
initials: A. T.
mobile: +1 303 788-4053
pager: +1 303 559-1680
manager: cn=Tosca Thomlinson
secretary: cn=Helena Leiding
roomNumber: 169

dn: cn=Glendon Glasgow, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Glendon Glasgow
sn: Glasgow
description: This is Glendon Glasgow's description
facsimileTelephoneNumber: +1 415 108-6074
l: Redmond
ou: Janitorial
postalAddress: example$Janitorial$Dept # 954
telephoneNumber: +1 71 882-3826
title: Elite Janitorial Vice President
userPassword: wogsalGnod
uid: Glendon_Glasgow
givenName: Glendon
mail: Glendon_Glasgow@example.com
carLicense: MNCAS69
departmentNumber: 4364
employeeType: Employee
homePhone: +1 804 273-6358
initials: G. G.
mobile: +1 415 904-2866
pager: +1 213 813-1720
manager: cn=Nichole Hyte
secretary: cn=Vita Mayer
roomNumber: 1835

dn: cn=Candee Jamshidi, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Candee Jamshidi
sn: Jamshidi
description: This is Candee Jamshidi's description
facsimileTelephoneNumber: +1 408 844-3640
l: Palo Alto
ou: Product Development
postalAddress: example$Product Development$Dept # 858
telephoneNumber: +1 71 295-4351
title: Associate Product Development Punk
userPassword: idihsmaJee
uid: Candee_Jamshidi
givenName: Candee
mail: Candee_Jamshidi@example.com
carLicense: 3RETBS2
departmentNumber: 4681
employeeType: Normal
homePhone: +1 415 191-9303
initials: C. J.
mobile: +1 303 105-4529
pager: +1 213 204-6497
manager: cn=Wilhelmina Teitelbaum
secretary: cn=Lydie Pitton
roomNumber: 3702

dn: cn=Tai-Jen Leon, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tai-Jen Leon
sn: Leon
description: This is Tai-Jen Leon's description
facsimileTelephoneNumber: +1 303 881-2837
l: San Mateo
ou: Product Development
postalAddress: example$Product Development$Dept # 984
telephoneNumber: +1 303 739-3409
title: Supreme Product Development Writer
userPassword: noeLneJ-ia
uid: Tai-Jen_Leon
givenName: Tai-Jen
mail: Tai-Jen_Leon@example.com
carLicense: C28Y6H9
departmentNumber: 8912
employeeType: Normal
homePhone: +1 213 775-3850
initials: T. L.
mobile: +1 71 887-3972
pager: +1 804 229-3819
manager: cn=Bibi Marleau
secretary: cn=Kristan Gorfine
roomNumber: 1495

dn: cn=Phyl Arcouet, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Phyl Arcouet
sn: Arcouet
description: This is Phyl Arcouet's description
facsimileTelephoneNumber: +1 408 679-4960
l: Menlo Park
ou: Planning
postalAddress: example$Planning$Dept # 820
telephoneNumber: +1 408 491-8848
title: Chief Planning Evangelist
userPassword: teuocrAlyh
uid: Phyl_Arcouet
givenName: Phyl
mail: Phyl_Arcouet@example.com
carLicense: N0J4XE6
departmentNumber: 9253
employeeType: Employee
homePhone: +1 415 488-2625
initials: P. A.
mobile: +1 303 851-3126
pager: +1 213 747-2501
manager: cn=Renee Khosla
secretary: cn=Erlene Granic
roomNumber: 4682

dn: cn=Lesly Torbert, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lesly Torbert
sn: Torbert
description: This is Lesly Torbert's description
facsimileTelephoneNumber: +1 213 965-9953
l: Sunnyvale
ou: Product Testing
postalAddress: example$Product Testing$Dept # 979
telephoneNumber: +1 510 145-2379
title: Elite Product Testing Accountant
userPassword: trebroTyls
uid: Lesly_Torbert
givenName: Lesly
mail: Lesly_Torbert@example.com
carLicense: U1CQ5AP
departmentNumber: 8198
employeeType: Employee
homePhone: +1 213 972-3091
initials: L. T.
mobile: +1 206 708-6966
pager: +1 213 328-3883
manager: cn=Gerianna Fadel
secretary: cn=Jon Madigan
roomNumber: 8472

dn: cn=Wen-Kai Lilly, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Wen-Kai Lilly
sn: Lilly
description: This is Wen-Kai Lilly's description
facsimileTelephoneNumber: +1 804 688-8196
l: Redwood Shores
ou: Management
postalAddress: example$Management$Dept # 370
telephoneNumber: +1 71 673-1670
title: Junior Management Admin
userPassword: ylliLiaK-n
uid: Wen-Kai_Lilly
givenName: Wen-Kai
mail: Wen-Kai_Lilly@example.com
carLicense: A3NP7VV
departmentNumber: 8718
employeeType: Manager
homePhone: +1 510 775-7118
initials: W. L.
mobile: +1 71 731-2419
pager: +1 818 695-1076
manager: cn=Raymond Sheaffer
secretary: cn=Guillermo Harrison
roomNumber: 4720

dn: cn=Shanti Heffernan, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shanti Heffernan
sn: Heffernan
description: This is Shanti Heffernan's description
facsimileTelephoneNumber: +1 213 440-4982
l: Emeryville
ou: Management
postalAddress: example$Management$Dept # 240
telephoneNumber: +1 804 256-5413
title: Supreme Management Dictator
userPassword: nanreffeHi
uid: Shanti_Heffernan
givenName: Shanti
mail: Shanti_Heffernan@example.com
carLicense: TW94LE1
departmentNumber: 8437
employeeType: Temp
homePhone: +1 71 657-4760
initials: S. H.
mobile: +1 818 910-4409
pager: +1 415 652-5036
manager: cn=Estel Jacobson
secretary: cn=Hatti Grover
roomNumber: 4785

dn: cn=Claudetta Vetrie, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Claudetta Vetrie
sn: Vetrie
description: This is Claudetta Vetrie's description
facsimileTelephoneNumber: +1 213 831-3010
l: San Francisco
ou: Accounting
postalAddress: example$Accounting$Dept # 408
telephoneNumber: +1 206 564-6336
title: Elite Accounting Mascot
userPassword: eirteVatte
uid: Claudetta_Vetrie
givenName: Claudetta
mail: Claudetta_Vetrie@example.com
carLicense: 995LNFA
departmentNumber: 9473
employeeType: Temp
homePhone: +1 818 121-9394
initials: C. V.
mobile: +1 804 309-9257
pager: +1 818 127-6157
manager: cn=Ginger Plotter
secretary: cn=Gnni Anker
roomNumber: 8906

dn: cn=Narrima Ferraro, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Narrima Ferraro
sn: Ferraro
description: This is Narrima Ferraro's description
facsimileTelephoneNumber: +1 415 604-8636
l: Milpitas
ou: Product Testing
postalAddress: example$Product Testing$Dept # 940
telephoneNumber: +1 510 988-6734
title: Senior Product Testing Stooge
userPassword: orarreFami
uid: Narrima_Ferraro
givenName: Narrima
mail: Narrima_Ferraro@example.com
carLicense: H1ZMMNP
departmentNumber: 8163
employeeType: Temp
homePhone: +1 408 156-6261
initials: N. F.
mobile: +1 804 200-8860
pager: +1 206 255-2739
manager: cn=Syed Hooton
secretary: cn=Wren IRCMARKET
roomNumber: 9869

dn: cn=Jagdish Intemann, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jagdish Intemann
sn: Intemann
description: This is Jagdish Intemann's description
facsimileTelephoneNumber: +1 510 553-4156
l: Redwood Shores
ou: Product Testing
postalAddress: example$Product Testing$Dept # 204
telephoneNumber: +1 303 498-5597
title: Junior Product Testing Pinhead
userPassword: nnametnIhs
uid: Jagdish_Intemann
givenName: Jagdish
mail: Jagdish_Intemann@example.com
carLicense: SABGVXS
departmentNumber: 1279
employeeType: Manager
homePhone: +1 213 960-9320
initials: J. I.
mobile: +1 415 127-8311
pager: +1 804 870-4339
manager: cn=Kalli Trevethan
secretary: cn=Guenther Bowser
roomNumber: 578

dn: cn=Han-Co Hilliard, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Han-Co Hilliard
sn: Hilliard
description: This is Han-Co Hilliard's description
facsimileTelephoneNumber: +1 415 626-3663
l: Emeryville
ou: Management
postalAddress: example$Management$Dept # 970
telephoneNumber: +1 818 237-9531
title: Senior Management Vice President
userPassword: drailliHoC
uid: Han-Co_Hilliard
givenName: Han-Co
mail: Han-Co_Hilliard@example.com
carLicense: Q770WJ8
departmentNumber: 5265
employeeType: Contract
homePhone: +1 804 648-1132
initials: H. H.
mobile: +1 213 225-1074
pager: +1 206 187-4190
manager: cn=Varennes Donlon
secretary: cn=Nils Bladon
roomNumber: 1018

dn: cn=Tape Lamonde, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tape Lamonde
sn: Lamonde
description: This is Tape Lamonde's description
facsimileTelephoneNumber: +1 213 973-2554
l: Cambridge
ou: Product Testing
postalAddress: example$Product Testing$Dept # 208
telephoneNumber: +1 818 652-7569
title: Junior Product Testing Mascot
userPassword: ednomaLepa
uid: Tape_Lamonde
givenName: Tape
mail: Tape_Lamonde@example.com
carLicense: D7OUS6R
departmentNumber: 2397
employeeType: Manager
homePhone: +1 213 462-7286
initials: T. L.
mobile: +1 408 278-1808
pager: +1 303 141-3679
manager: cn=Karisa Delzer
secretary: cn=Cary Ottosson
roomNumber: 583

dn: cn=Bertina Simon, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bertina Simon
sn: Simon
description: This is Bertina Simon's description
facsimileTelephoneNumber: +1 804 334-9377
l: Emeryville
ou: Accounting
postalAddress: example$Accounting$Dept # 702
telephoneNumber: +1 303 374-4957
title: Senior Accounting Punk
userPassword: nomiSanitr
uid: Bertina_Simon
givenName: Bertina
mail: Bertina_Simon@example.com
carLicense: MLBFINM
departmentNumber: 9010
employeeType: Normal
homePhone: +1 408 869-9392
initials: B. S.
mobile: +1 804 371-2530
pager: +1 213 461-7236
manager: cn=Ninon Suh
secretary: cn=Rafa Moyer
roomNumber: 9367

dn: cn=Mable Scarffe, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mable Scarffe
sn: Scarffe
description: This is Mable Scarffe's description
facsimileTelephoneNumber: +1 213 193-6746
l: Menlo Park
ou: Administrative
postalAddress: example$Administrative$Dept # 358
telephoneNumber: +1 408 387-3910
title: Elite Administrative Stooge
userPassword: effracSelb
uid: Mable_Scarffe
givenName: Mable
mail: Mable_Scarffe@example.com
carLicense: P3H22ZQ
departmentNumber: 2955
employeeType: Employee
homePhone: +1 408 680-9649
initials: M. S.
mobile: +1 303 821-6922
pager: +1 213 877-2168
manager: cn=Blanch Azarshahi
secretary: cn=Milly Breton
roomNumber: 1622

dn: cn=Sheela Shishakly, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sheela Shishakly
sn: Shishakly
description: This is Sheela Shishakly's description
facsimileTelephoneNumber: +1 71 673-7431
l: Alameda
ou: Product Development
postalAddress: example$Product Development$Dept # 987
telephoneNumber: +1 804 762-2980
title: Supreme Product Development President
userPassword: ylkahsihSa
uid: Sheela_Shishakly
givenName: Sheela
mail: Sheela_Shishakly@example.com
carLicense: 5IFJPQ2
departmentNumber: 6433
employeeType: Employee
homePhone: +1 206 828-2857
initials: S. S.
mobile: +1 408 284-1761
pager: +1 415 489-4013
manager: cn=Tilmon Hume
secretary: cn=Den Klutts
roomNumber: 9152

dn: cn=Barbaraanne Dantzler, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Barbaraanne Dantzler
sn: Dantzler
description: This is Barbaraanne Dantzler's description
facsimileTelephoneNumber: +1 206 113-9209
l: Menlo Park
ou: Human Resources
postalAddress: example$Human Resources$Dept # 251
telephoneNumber: +1 408 773-2407
title: Chief Human Resources Mascot
userPassword: relztnaDen
uid: Barbaraanne_Dantzler
givenName: Barbaraanne
mail: Barbaraanne_Dantzler@example.com
carLicense: NLBLPQV
departmentNumber: 1728
employeeType: Normal
homePhone: +1 213 649-4726
initials: B. D.
mobile: +1 818 345-9799
pager: +1 818 812-7609
manager: cn=Elana Oziskender
secretary: cn=Morgen Somppi
roomNumber: 5920

dn: cn=Marijo Vilis, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marijo Vilis
sn: Vilis
description: This is Marijo Vilis's description
facsimileTelephoneNumber: +1 818 987-4698
l: Cupertino
ou: Peons
postalAddress: example$Peons$Dept # 180
telephoneNumber: +1 818 596-8525
title: Junior Peons Yahoo
userPassword: siliVojira
uid: Marijo_Vilis
givenName: Marijo
mail: Marijo_Vilis@example.com
carLicense: GV8EVPP
departmentNumber: 7763
employeeType: Manager
homePhone: +1 408 246-1705
initials: M. V.
mobile: +1 303 165-6594
pager: +1 510 255-5965
manager: cn=Carey Fronsee
secretary: cn=Kelsey Sohns
roomNumber: 1306

dn: cn=Helaine McHale, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Helaine McHale
sn: McHale
description: This is Helaine McHale's description
facsimileTelephoneNumber: +1 408 114-3298
l: Mountain View
ou: Planning
postalAddress: example$Planning$Dept # 358
telephoneNumber: +1 206 543-4860
title: Chief Planning Evangelist
userPassword: elaHcMenia
uid: Helaine_McHale
givenName: Helaine
mail: Helaine_McHale@example.com
carLicense: 3UBUU5W
departmentNumber: 1841
employeeType: Temp
homePhone: +1 408 144-3197
initials: H. M.
mobile: +1 303 109-7845
pager: +1 408 420-5460
manager: cn=Errol Mishina
secretary: cn=Twana McLawhon
roomNumber: 5704

dn: cn=Devonne Siegel, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Devonne Siegel
sn: Siegel
description: This is Devonne Siegel's description
facsimileTelephoneNumber: +1 804 930-1674
l: Alameda
ou: Product Development
postalAddress: example$Product Development$Dept # 58
telephoneNumber: +1 71 250-2535
title: Associate Product Development Stooge
userPassword: legeiSenno
uid: Devonne_Siegel
givenName: Devonne
mail: Devonne_Siegel@example.com
carLicense: VEEZJRC
departmentNumber: 5688
employeeType: Contract
homePhone: +1 510 539-1810
initials: D. S.
mobile: +1 510 474-7533
pager: +1 408 339-2999
manager: cn=Georgianne Tu
secretary: cn=Yoshiko Foucault
roomNumber: 5007

dn: cn=Cherlyn Stasaski, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Cherlyn Stasaski
sn: Stasaski
description: This is Cherlyn Stasaski's description
facsimileTelephoneNumber: +1 303 531-2122
l: Cupertino
ou: Accounting
postalAddress: example$Accounting$Dept # 305
telephoneNumber: +1 206 102-5841
title: Elite Accounting Visionary
userPassword: iksasatSny
uid: Cherlyn_Stasaski
givenName: Cherlyn
mail: Cherlyn_Stasaski@example.com
carLicense: 229221G
departmentNumber: 6792
employeeType: Normal
homePhone: +1 303 950-9071
initials: C. S.
mobile: +1 71 163-1113
pager: +1 415 356-6049
manager: cn=Sabina Harte
secretary: cn=Said Rizewiski
roomNumber: 9510

dn: cn=Bernetta Pena, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bernetta Pena
sn: Pena
description: This is Bernetta Pena's description
facsimileTelephoneNumber: +1 303 915-2046
l: Mountain View
ou: Payroll
postalAddress: example$Payroll$Dept # 651
telephoneNumber: +1 818 990-8944
title: Supreme Payroll Assistant
userPassword: anePattenr
uid: Bernetta_Pena
givenName: Bernetta
mail: Bernetta_Pena@example.com
carLicense: 5PEYCY4
departmentNumber: 6975
employeeType: Manager
homePhone: +1 408 429-9083
initials: B. P.
mobile: +1 213 624-1045
pager: +1 408 544-6654
manager: cn=Janeen Geuder
secretary: cn=Hengameh Gravitt
roomNumber: 6328

dn: cn=Kissiah Greszczuk, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kissiah Greszczuk
sn: Greszczuk
description: This is Kissiah Greszczuk's description
facsimileTelephoneNumber: +1 510 938-8051
l: Armonk
ou: Product Testing
postalAddress: example$Product Testing$Dept # 935
telephoneNumber: +1 408 928-6796
title: Supreme Product Testing Stooge
userPassword: kuzczserGh
uid: Kissiah_Greszczuk
givenName: Kissiah
mail: Kissiah_Greszczuk@example.com
carLicense: T8TE7ZE
departmentNumber: 3218
employeeType: Normal
homePhone: +1 510 643-8421
initials: K. G.
mobile: +1 408 709-4773
pager: +1 206 551-6228
manager: cn=Olusola Mattes
secretary: cn=Wanda Avery
roomNumber: 2052

dn: cn=Bello Toole, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bello Toole
sn: Toole
description: This is Bello Toole's description
facsimileTelephoneNumber: +1 804 808-5332
l: Alameda
ou: Planning
postalAddress: example$Planning$Dept # 580
telephoneNumber: +1 303 144-1714
title: Supreme Planning Director
userPassword: elooTolleB
uid: Bello_Toole
givenName: Bello
mail: Bello_Toole@example.com
carLicense: 3G2CJKZ
departmentNumber: 1080
employeeType: Contract
homePhone: +1 206 231-6545
initials: B. T.
mobile: +1 415 807-3208
pager: +1 804 971-6767
manager: cn=Sherwyn Verardi
secretary: cn=Kylie Unkefer
roomNumber: 5390

dn: cn=Elizabeth Aydin, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Elizabeth Aydin
sn: Aydin
description: This is Elizabeth Aydin's description
facsimileTelephoneNumber: +1 415 164-4898
l: Redwood Shores
ou: Product Testing
postalAddress: example$Product Testing$Dept # 257
telephoneNumber: +1 818 124-4732
title: Supreme Product Testing Vice President
userPassword: nidyAhteba
uid: Elizabeth_Aydin
givenName: Elizabeth
mail: Elizabeth_Aydin@example.com
carLicense: GWGJRNE
departmentNumber: 4864
employeeType: Contract
homePhone: +1 206 221-2740
initials: E. A.
mobile: +1 818 341-5047
pager: +1 510 567-5979
manager: cn=Monroe Bereza
secretary: cn=Prue Rastelli
roomNumber: 9869

dn: cn=Zahirul Holcombe, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zahirul Holcombe
sn: Holcombe
description: This is Zahirul Holcombe's description
facsimileTelephoneNumber: +1 818 324-1197
l: San Mateo
ou: Administrative
postalAddress: example$Administrative$Dept # 848
telephoneNumber: +1 408 272-3182
title: Master Administrative Manager
userPassword: ebmocloHlu
uid: Zahirul_Holcombe
givenName: Zahirul
mail: Zahirul_Holcombe@example.com
carLicense: N1PL49T
departmentNumber: 7494
employeeType: Contract
homePhone: +1 206 459-1493
initials: Z. H.
mobile: +1 804 957-2062
pager: +1 71 496-2453
manager: cn=Mehmud McWalters
secretary: cn=Julita Felicetti
roomNumber: 6113

dn: cn=Bellina Moledina, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bellina Moledina
sn: Moledina
description: This is Bellina Moledina's description
facsimileTelephoneNumber: +1 213 554-4858
l: Emeryville
ou: Product Development
postalAddress: example$Product Development$Dept # 458
telephoneNumber: +1 415 754-5468
title: Junior Product Development Evangelist
userPassword: anideloMan
uid: Bellina_Moledina
givenName: Bellina
mail: Bellina_Moledina@example.com
carLicense: HHJW54Z
departmentNumber: 6960
employeeType: Contract
homePhone: +1 71 853-4918
initials: B. M.
mobile: +1 415 769-3122
pager: +1 213 928-1483
manager: cn=Astra Brownfield
secretary: cn=Melody Rasberry
roomNumber: 8075

dn: cn=Lilin Grosse, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lilin Grosse
sn: Grosse
description: This is Lilin Grosse's description
facsimileTelephoneNumber: +1 206 793-1541
l: Cambridge
ou: Product Testing
postalAddress: example$Product Testing$Dept # 587
telephoneNumber: +1 213 968-8578
title: Elite Product Testing Technician
userPassword: essorGnili
uid: Lilin_Grosse
givenName: Lilin
mail: Lilin_Grosse@example.com
carLicense: J3PERBY
departmentNumber: 639
employeeType: Contract
homePhone: +1 71 694-2377
initials: L. G.
mobile: +1 510 522-4777
pager: +1 71 836-3761
manager: cn=Norikatsu Doskas
secretary: cn=Garland Gilliam
roomNumber: 9620

dn: cn=Inessa Domanico, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Inessa Domanico
sn: Domanico
description: This is Inessa Domanico's description
facsimileTelephoneNumber: +1 408 357-4921
l: Milpitas
ou: Human Resources
postalAddress: example$Human Resources$Dept # 624
telephoneNumber: +1 213 959-2815
title: Chief Human Resources Fellow
userPassword: ocinamoDas
uid: Inessa_Domanico
givenName: Inessa
mail: Inessa_Domanico@example.com
carLicense: 6YSOGV6
departmentNumber: 5439
employeeType: Normal
homePhone: +1 206 847-9333
initials: I. D.
mobile: +1 303 433-6682
pager: +1 206 383-4713
manager: cn=Mercer Kantor
secretary: cn=Dianne Janssen
roomNumber: 4005

dn: cn=Kylen Abedi, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kylen Abedi
sn: Abedi
description: This is Kylen Abedi's description
facsimileTelephoneNumber: +1 510 984-9118
l: Milpitas
ou: Human Resources
postalAddress: example$Human Resources$Dept # 316
telephoneNumber: +1 303 554-8893
title: Elite Human Resources Accountant
userPassword: idebAnelyK
uid: Kylen_Abedi
givenName: Kylen
mail: Kylen_Abedi@example.com
carLicense: EITVNIL
departmentNumber: 6680
employeeType: Employee
homePhone: +1 206 968-4087
initials: K. A.
mobile: +1 206 225-3502
pager: +1 303 928-6475
manager: cn=Norcal Monaco
secretary: cn=Joachim Ludwick
roomNumber: 7234

dn: cn=Par Chong, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Par Chong
sn: Chong
description: This is Par Chong's description
facsimileTelephoneNumber: +1 206 264-3645
l: Redwood Shores
ou: Accounting
postalAddress: example$Accounting$Dept # 730
telephoneNumber: +1 818 638-6083
title: Elite Accounting Dictator
userPassword: gnohCraP
uid: Par_Chong
givenName: Par
mail: Par_Chong@example.com
carLicense: B1BMTWO
departmentNumber: 4833
employeeType: Employee
homePhone: +1 303 325-8280
initials: P. C.
mobile: +1 213 437-3268
pager: +1 818 841-2460
manager: cn=Chiu Yun
secretary: cn=Shana Cuggy
roomNumber: 60

dn: cn=Roberta Liao, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Roberta Liao
sn: Liao
description: This is Roberta Liao's description
facsimileTelephoneNumber: +1 818 953-9645
l: Cupertino
ou: Payroll
postalAddress: example$Payroll$Dept # 317
telephoneNumber: +1 206 400-2865
title: Chief Payroll Developer
userPassword: oaiLatrebo
uid: Roberta_Liao
givenName: Roberta
mail: Roberta_Liao@example.com
carLicense: SRJNE9E
departmentNumber: 9613
employeeType: Temp
homePhone: +1 71 684-9809
initials: R. L.
mobile: +1 510 655-7108
pager: +1 818 788-8495
manager: cn=Canadian Bragg
secretary: cn=Drona Adjangba
roomNumber: 5633

dn: cn=Maycel El-Hawary, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Maycel El-Hawary
sn: El-Hawary
description: This is Maycel El-Hawary's description
facsimileTelephoneNumber: +1 303 399-7655
l: San Jose
ou: Payroll
postalAddress: example$Payroll$Dept # 531
telephoneNumber: +1 303 279-2842
title: Elite Payroll Visionary
userPassword: yrawaH-lEl
uid: Maycel_El-Hawary
givenName: Maycel
mail: Maycel_El-Hawary@example.com
carLicense: K77UEL4
departmentNumber: 6120
employeeType: Manager
homePhone: +1 71 196-9815
initials: M. E.
mobile: +1 206 441-2049
pager: +1 303 134-2807
manager: cn=Huan-yu Joffe
secretary: cn=Lyndia Baum
roomNumber: 9476

dn: cn=Jannel Chern, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jannel Chern
sn: Chern
description: This is Jannel Chern's description
facsimileTelephoneNumber: +1 818 740-6505
l: Redmond
ou: Peons
postalAddress: example$Peons$Dept # 986
telephoneNumber: +1 415 118-3889
title: Junior Peons Director
userPassword: nrehClenna
uid: Jannel_Chern
givenName: Jannel
mail: Jannel_Chern@example.com
carLicense: FBBLADO
departmentNumber: 3009
employeeType: Manager
homePhone: +1 206 822-6702
initials: J. C.
mobile: +1 818 789-4145
pager: +1 818 126-3562
manager: cn=Vrouwerff Revill
secretary: cn=Lyda Loudiadis
roomNumber: 8822

dn: cn=Corabelle Beckie, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Corabelle Beckie
sn: Beckie
description: This is Corabelle Beckie's description
facsimileTelephoneNumber: +1 510 124-8955
l: San Francisco
ou: Management
postalAddress: example$Management$Dept # 233
telephoneNumber: +1 510 412-2726
title: Supreme Management Stooge
userPassword: eikceBelle
uid: Corabelle_Beckie
givenName: Corabelle
mail: Corabelle_Beckie@example.com
carLicense: KLOQJP1
departmentNumber: 4890
employeeType: Contract
homePhone: +1 71 673-3892
initials: C. B.
mobile: +1 206 554-2122
pager: +1 415 130-5306
manager: cn=Hilda Lyon
secretary: cn=Sisile Behroozi
roomNumber: 3696

dn: cn=Jann Eansor, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jann Eansor
sn: Eansor
description: This is Jann Eansor's description
facsimileTelephoneNumber: +1 213 522-3563
l: Mountain View
ou: Janitorial
postalAddress: example$Janitorial$Dept # 40
telephoneNumber: +1 408 879-5681
title: Junior Janitorial Pinhead
userPassword: rosnaEnnaJ
uid: Jann_Eansor
givenName: Jann
mail: Jann_Eansor@example.com
carLicense: Q4P4YIZ
departmentNumber: 7675
employeeType: Employee
homePhone: +1 213 654-4780
initials: J. E.
mobile: +1 804 198-3764
pager: +1 213 666-1076
manager: cn=Garth Carevic
secretary: cn=Lillie Schnupp
roomNumber: 2879

dn: cn=Berrie Qu, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Berrie Qu
sn: Qu
description: This is Berrie Qu's description
facsimileTelephoneNumber: +1 213 166-4023
l: Cupertino
ou: Peons
postalAddress: example$Peons$Dept # 998
telephoneNumber: +1 510 192-3177
title: Junior Peons Developer
userPassword: uQeirreB
uid: Berrie_Qu
givenName: Berrie
mail: Berrie_Qu@example.com
carLicense: 3JNKHK6
departmentNumber: 4809
employeeType: Normal
homePhone: +1 818 289-4386
initials: B. Q.
mobile: +1 408 941-4537
pager: +1 804 660-5286
manager: cn=Nessy Kosowan
secretary: cn=Darleen Lalu
roomNumber: 7362

dn: cn=Martine Shabo, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Martine Shabo
sn: Shabo
description: This is Martine Shabo's description
facsimileTelephoneNumber: +1 804 672-9719
l: Alameda
ou: Product Testing
postalAddress: example$Product Testing$Dept # 446
telephoneNumber: +1 213 190-7848
title: Senior Product Testing President
userPassword: obahSenitr
uid: Martine_Shabo
givenName: Martine
mail: Martine_Shabo@example.com
carLicense: VGJPYSK
departmentNumber: 3001
employeeType: Normal
homePhone: +1 415 431-8974
initials: M. S.
mobile: +1 408 213-2304
pager: +1 804 606-5737
manager: cn=Drusie Padgett
secretary: cn=Viviyan Strock
roomNumber: 3728

dn: cn=Antonietta Nugent, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Antonietta Nugent
sn: Nugent
description: This is Antonietta Nugent's description
facsimileTelephoneNumber: +1 71 961-9430
l: Orem
ou: Janitorial
postalAddress: example$Janitorial$Dept # 419
telephoneNumber: +1 206 446-9235
title: Chief Janitorial Consultant
userPassword: tneguNatte
uid: Antonietta_Nugent
givenName: Antonietta
mail: Antonietta_Nugent@example.com
carLicense: R8UASV6
departmentNumber: 5142
employeeType: Employee
homePhone: +1 71 824-6004
initials: A. N.
mobile: +1 213 985-8253
pager: +1 415 224-2804
manager: cn=Violetta Darrimon
secretary: cn=Bernd Klapper
roomNumber: 5356

dn: cn=Jenn Driscoll, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jenn Driscoll
sn: Driscoll
description: This is Jenn Driscoll's description
facsimileTelephoneNumber: +1 818 910-6972
l: Alameda
ou: Planning
postalAddress: example$Planning$Dept # 390
telephoneNumber: +1 213 508-9453
title: Supreme Planning President
userPassword: llocsirDnn
uid: Jenn_Driscoll
givenName: Jenn
mail: Jenn_Driscoll@example.com
carLicense: H223TV7
departmentNumber: 9552
employeeType: Temp
homePhone: +1 510 990-8697
initials: J. D.
mobile: +1 510 228-3296
pager: +1 408 175-1221
manager: cn=Phelia Tadevich
secretary: cn=Latashia Fixsen
roomNumber: 9169

dn: cn=Franciska Dueck, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Franciska Dueck
sn: Dueck
description: This is Franciska Dueck's description
facsimileTelephoneNumber: +1 206 290-2456
l: Fremont
ou: Payroll
postalAddress: example$Payroll$Dept # 881
telephoneNumber: +1 408 913-9714
title: Elite Payroll Punk
userPassword: kceuDaksic
uid: Franciska_Dueck
givenName: Franciska
mail: Franciska_Dueck@example.com
carLicense: 683B79B
departmentNumber: 7906
employeeType: Temp
homePhone: +1 303 334-7097
initials: F. D.
mobile: +1 206 141-4537
pager: +1 71 899-7525
manager: cn=Linnea Conley
secretary: cn=Lorita Grills
roomNumber: 4590

dn: cn=Tape Coe, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tape Coe
sn: Coe
description: This is Tape Coe's description
facsimileTelephoneNumber: +1 804 928-7334
l: San Mateo
ou: Payroll
postalAddress: example$Payroll$Dept # 874
telephoneNumber: +1 206 177-6708
title: Junior Payroll Engineer
userPassword: eoCepaT
uid: Tape_Coe
givenName: Tape
mail: Tape_Coe@example.com
carLicense: M7J3ES5
departmentNumber: 1408
employeeType: Manager
homePhone: +1 408 777-2780
initials: T. C.
mobile: +1 303 182-5881
pager: +1 415 815-6331
manager: cn=Walley Shewchenko
secretary: cn=Stephan Boarder
roomNumber: 7866

dn: cn=Minnnie Gould, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Minnnie Gould
sn: Gould
description: This is Minnnie Gould's description
facsimileTelephoneNumber: +1 818 483-2575
l: Redwood Shores
ou: Administrative
postalAddress: example$Administrative$Dept # 138
telephoneNumber: +1 818 368-5111
title: Chief Administrative Accountant
userPassword: dluoGeinnn
uid: Minnnie_Gould
givenName: Minnnie
mail: Minnnie_Gould@example.com
carLicense: 2JDEEK3
departmentNumber: 3692
employeeType: Manager
homePhone: +1 206 275-8932
initials: M. G.
mobile: +1 213 233-7285
pager: +1 510 511-4242
manager: cn=Anya Moledina
secretary: cn=Hall Heybroek
roomNumber: 77

dn: cn=Tak Sherman, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tak Sherman
sn: Sherman
description: This is Tak Sherman's description
facsimileTelephoneNumber: +1 818 500-7283
l: Cambridge
ou: Administrative
postalAddress: example$Administrative$Dept # 888
telephoneNumber: +1 71 854-6383
title: Associate Administrative Visionary
userPassword: namrehSkaT
uid: Tak_Sherman
givenName: Tak
mail: Tak_Sherman@example.com
carLicense: W5C6733
departmentNumber: 7508
employeeType: Contract
homePhone: +1 804 679-3727
initials: T. S.
mobile: +1 818 218-6838
pager: +1 303 333-6467
manager: cn=Sukhendu Klingsporn
secretary: cn=Aruna Scotti
roomNumber: 3688

dn: cn=Lourdes Changes, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lourdes Changes
sn: Changes
description: This is Lourdes Changes's description
facsimileTelephoneNumber: +1 303 488-2704
l: Fremont
ou: Product Testing
postalAddress: example$Product Testing$Dept # 314
telephoneNumber: +1 415 986-5442
title: Elite Product Testing Architect
userPassword: segnahCsed
uid: Lourdes_Changes
givenName: Lourdes
mail: Lourdes_Changes@example.com
carLicense: 5ED3L3T
departmentNumber: 9178
employeeType: Temp
homePhone: +1 206 225-7834
initials: L. C.
mobile: +1 408 974-5773
pager: +1 71 650-2622
manager: cn=Gabey Ecroyd
secretary: cn=Eleanore Moghe
roomNumber: 9056

dn: cn=Anneliese Saward, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Anneliese Saward
sn: Saward
description: This is Anneliese Saward's description
facsimileTelephoneNumber: +1 303 368-5026
l: Orem
ou: Peons
postalAddress: example$Peons$Dept # 104
telephoneNumber: +1 818 334-9722
title: Associate Peons Engineer
userPassword: drawaSesei
uid: Anneliese_Saward
givenName: Anneliese
mail: Anneliese_Saward@example.com
carLicense: 4L1RFGG
departmentNumber: 601
employeeType: Employee
homePhone: +1 71 285-4245
initials: A. S.
mobile: +1 415 214-2467
pager: +1 804 210-1228
manager: cn=Albertine Stars
secretary: cn=Larysa Kreiger
roomNumber: 7429

dn: cn=Shahrokh Wennerstrom, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shahrokh Wennerstrom
sn: Wennerstrom
description: This is Shahrokh Wennerstrom's description
facsimileTelephoneNumber: +1 303 968-1645
l: Menlo Park
ou: Payroll
postalAddress: example$Payroll$Dept # 801
telephoneNumber: +1 303 344-6573
title: Chief Payroll Mascot
userPassword: mortsrenne
uid: Shahrokh_Wennerstrom
givenName: Shahrokh
mail: Shahrokh_Wennerstrom@example.com
carLicense: 60SLXN5
departmentNumber: 7525
employeeType: Normal
homePhone: +1 206 400-6719
initials: S. W.
mobile: +1 303 263-7783
pager: +1 415 177-8010
manager: cn=Ofella Drabek
secretary: cn=Muinck Ntprel
roomNumber: 7182

dn: cn=Adriaens Szaplonczay, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Adriaens Szaplonczay
sn: Szaplonczay
description: This is Adriaens Szaplonczay's description
facsimileTelephoneNumber: +1 804 690-9862
l: Redwood Shores
ou: Accounting
postalAddress: example$Accounting$Dept # 911
telephoneNumber: +1 818 908-7106
title: Elite Accounting Technician
userPassword: yazcnolpaz
uid: Adriaens_Szaplonczay
givenName: Adriaens
mail: Adriaens_Szaplonczay@example.com
carLicense: 50P52KS
departmentNumber: 9124
employeeType: Employee
homePhone: +1 408 996-8311
initials: A. S.
mobile: +1 71 908-6369
pager: +1 206 382-5829
manager: cn=Ester Dosenbach
secretary: cn=Clarence Patchett
roomNumber: 3335

dn: cn=Li Giekes, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Li Giekes
sn: Giekes
description: This is Li Giekes's description
facsimileTelephoneNumber: +1 804 744-9005
l: San Mateo
ou: Human Resources
postalAddress: example$Human Resources$Dept # 988
telephoneNumber: +1 303 898-4507
title: Elite Human Resources Madonna
userPassword: sekeiGiL
uid: Li_Giekes
givenName: Li
mail: Li_Giekes@example.com
carLicense: P8CFR4F
departmentNumber: 9012
employeeType: Employee
homePhone: +1 408 335-3756
initials: L. G.
mobile: +1 303 446-5296
pager: +1 71 389-6623
manager: cn=Roanne Japp
secretary: cn=Galen Khouderchan
roomNumber: 5981

dn: cn=Jeri Shemwell, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jeri Shemwell
sn: Shemwell
description: This is Jeri Shemwell's description
facsimileTelephoneNumber: +1 818 352-3008
l: San Mateo
ou: Administrative
postalAddress: example$Administrative$Dept # 781
telephoneNumber: +1 206 716-2647
title: Chief Administrative Vice President
userPassword: llewmehSir
uid: Jeri_Shemwell
givenName: Jeri
mail: Jeri_Shemwell@example.com
carLicense: IP0PUCY
departmentNumber: 100
employeeType: Manager
homePhone: +1 415 175-9006
initials: J. S.
mobile: +1 510 237-9592
pager: +1 213 682-7219
manager: cn=Dieter Tables
secretary: cn=Noellyn Dodds
roomNumber: 8663

dn: cn=Jennifer Croxford, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jennifer Croxford
sn: Croxford
description: This is Jennifer Croxford's description
facsimileTelephoneNumber: +1 415 621-2992
l: Orem
ou: Peons
postalAddress: example$Peons$Dept # 367
telephoneNumber: +1 71 594-9347
title: Junior Peons Admin
userPassword: drofxorCre
uid: Jennifer_Croxford
givenName: Jennifer
mail: Jennifer_Croxford@example.com
carLicense: UWO595X
departmentNumber: 7134
employeeType: Normal
homePhone: +1 303 654-1112
initials: J. C.
mobile: +1 408 180-5348
pager: +1 206 720-2029
manager: cn=Carlene Lande
secretary: cn=Rochette Ndububa
roomNumber: 9470

dn: cn=Dee dee Gockel, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dee dee Gockel
sn: Gockel
description: This is Dee dee Gockel's description
facsimileTelephoneNumber: +1 303 747-9489
l: Cambridge
ou: Peons
postalAddress: example$Peons$Dept # 798
telephoneNumber: +1 206 924-8771
title: Master Peons Technician
userPassword: lekcoGeede
uid: Dee dee_Gockel
givenName: Dee dee
mail: Dee dee_Gockel@example.com
carLicense: 7K3MQN8
departmentNumber: 8805
employeeType: Contract
homePhone: +1 71 346-4371
initials: D. G.
mobile: +1 303 661-4459
pager: +1 804 830-5591
manager: cn=Dacey Mims
secretary: cn=Ashly Taheri
roomNumber: 4604

dn: cn=Hoekstra Brightwell, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Hoekstra Brightwell
sn: Brightwell
description: This is Hoekstra Brightwell's description
facsimileTelephoneNumber: +1 71 612-4705
l: Palo Alto
ou: Management
postalAddress: example$Management$Dept # 120
telephoneNumber: +1 408 134-7123
title: Master Management Sales Rep
userPassword: llewthgirB
uid: Hoekstra_Brightwell
givenName: Hoekstra
mail: Hoekstra_Brightwell@example.com
carLicense: 93EY22B
departmentNumber: 8716
employeeType: Employee
homePhone: +1 804 531-8765
initials: H. B.
mobile: +1 213 442-4612
pager: +1 71 358-3421
manager: cn=France Verma
secretary: cn=Fulvia Konomis
roomNumber: 8922

dn: cn=Jany Beaudet, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jany Beaudet
sn: Beaudet
description: This is Jany Beaudet's description
facsimileTelephoneNumber: +1 206 814-4739
l: Mountain View
ou: Payroll
postalAddress: example$Payroll$Dept # 104
telephoneNumber: +1 415 968-6035
title: Chief Payroll Visionary
userPassword: teduaeByna
uid: Jany_Beaudet
givenName: Jany
mail: Jany_Beaudet@example.com
carLicense: OQPQ66Z
departmentNumber: 2168
employeeType: Contract
homePhone: +1 213 618-9981
initials: J. B.
mobile: +1 213 493-5690
pager: +1 818 385-2922
manager: cn=De-Anna Grafton
secretary: cn=Mohammed Pancholy
roomNumber: 9889

dn: cn=Laury Hundrieser, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Laury Hundrieser
sn: Hundrieser
description: This is Laury Hundrieser's description
facsimileTelephoneNumber: +1 818 519-4730
l: Redwood Shores
ou: Planning
postalAddress: example$Planning$Dept # 748
telephoneNumber: +1 804 710-4277
title: Chief Planning Vice President
userPassword: reseirdnuH
uid: Laury_Hundrieser
givenName: Laury
mail: Laury_Hundrieser@example.com
carLicense: 7VHPX3R
departmentNumber: 9176
employeeType: Temp
homePhone: +1 303 430-3192
initials: L. H.
mobile: +1 213 684-4143
pager: +1 408 367-8589
manager: cn=Fonnie Freeburn
secretary: cn=Jorrie Townson
roomNumber: 6925

dn: cn=Grey Wilby, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Grey Wilby
sn: Wilby
description: This is Grey Wilby's description
facsimileTelephoneNumber: +1 303 916-2498
l: Armonk
ou: Management
postalAddress: example$Management$Dept # 316
telephoneNumber: +1 213 671-2278
title: Associate Management Consultant
userPassword: ybliWyerG
uid: Grey_Wilby
givenName: Grey
mail: Grey_Wilby@example.com
carLicense: S1KFLPX
departmentNumber: 6452
employeeType: Employee
homePhone: +1 804 140-6181
initials: G. W.
mobile: +1 303 558-6863
pager: +1 510 337-8464
manager: cn=Ally Bruin
secretary: cn=Kerstin Howell
roomNumber: 8168

dn: cn=Elicia Mendolia, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Elicia Mendolia
sn: Mendolia
description: This is Elicia Mendolia's description
facsimileTelephoneNumber: +1 804 410-3925
l: Sunnyvale
ou: Janitorial
postalAddress: example$Janitorial$Dept # 447
telephoneNumber: +1 818 715-9162
title: Junior Janitorial Admin
userPassword: ailodneMai
uid: Elicia_Mendolia
givenName: Elicia
mail: Elicia_Mendolia@example.com
carLicense: B82QKWB
departmentNumber: 9572
employeeType: Temp
homePhone: +1 71 820-8129
initials: E. M.
mobile: +1 71 498-8907
pager: +1 71 705-3694
manager: cn=Jean-Paul Javallas-Ross
secretary: cn=Fil Toop
roomNumber: 3904

dn: cn=Malgosia Qainfo, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Malgosia Qainfo
sn: Qainfo
description: This is Malgosia Qainfo's description
facsimileTelephoneNumber: +1 415 636-7122
l: Cupertino
ou: Janitorial
postalAddress: example$Janitorial$Dept # 81
telephoneNumber: +1 303 618-2084
title: Elite Janitorial Admin
userPassword: ofniaQaiso
uid: Malgosia_Qainfo
givenName: Malgosia
mail: Malgosia_Qainfo@example.com
carLicense: RTLNZ9G
departmentNumber: 4425
employeeType: Employee
homePhone: +1 818 160-5331
initials: M. Q.
mobile: +1 510 358-5904
pager: +1 415 723-1233
manager: cn=Shahrokh Grossman
secretary: cn=Bin Culberson
roomNumber: 454

dn: cn=Vladimir Hsieh, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Vladimir Hsieh
sn: Hsieh
description: This is Vladimir Hsieh's description
facsimileTelephoneNumber: +1 415 114-1055
l: Palo Alto
ou: Peons
postalAddress: example$Peons$Dept # 736
telephoneNumber: +1 408 767-1357
title: Senior Peons Vice President
userPassword: heisHrimid
uid: Vladimir_Hsieh
givenName: Vladimir
mail: Vladimir_Hsieh@example.com
carLicense: MOBSD4C
departmentNumber: 4340
employeeType: Employee
homePhone: +1 206 702-2498
initials: V. H.
mobile: +1 804 246-5559
pager: +1 510 826-5717
manager: cn=Astra Bracewell
secretary: cn=Pradyumn Welten
roomNumber: 866

dn: cn=Patching Zattiero, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Patching Zattiero
sn: Zattiero
description: This is Patching Zattiero's description
facsimileTelephoneNumber: +1 303 369-5585
l: San Mateo
ou: Management
postalAddress: example$Management$Dept # 227
telephoneNumber: +1 804 545-4875
title: Elite Management Warrior
userPassword: oreittaZgn
uid: Patching_Zattiero
givenName: Patching
mail: Patching_Zattiero@example.com
carLicense: OGE6POK
departmentNumber: 1747
employeeType: Contract
homePhone: +1 818 645-7367
initials: P. Z.
mobile: +1 415 684-6807
pager: +1 408 693-8834
manager: cn=Lynnet Smerek
secretary: cn=Cathe Toyooka
roomNumber: 8368

dn: cn=Gretna Mansell, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gretna Mansell
sn: Mansell
description: This is Gretna Mansell's description
facsimileTelephoneNumber: +1 206 470-3366
l: Redmond
ou: Management
postalAddress: example$Management$Dept # 100
telephoneNumber: +1 206 974-8773
title: Associate Management Engineer
userPassword: llesnaMant
uid: Gretna_Mansell
givenName: Gretna
mail: Gretna_Mansell@example.com
carLicense: LI6HVH5
departmentNumber: 1320
employeeType: Employee
homePhone: +1 818 259-9827
initials: G. M.
mobile: +1 303 958-2838
pager: +1 818 383-2711
manager: cn=Jeannot Banville
secretary: cn=Fast Homa
roomNumber: 84

dn: cn=Grady Webster, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Grady Webster
sn: Webster
description: This is Grady Webster's description
facsimileTelephoneNumber: +1 804 390-7916
l: Redmond
ou: Administrative
postalAddress: example$Administrative$Dept # 817
telephoneNumber: +1 510 842-2039
title: Junior Administrative Mascot
userPassword: retsbeWyda
uid: Grady_Webster
givenName: Grady
mail: Grady_Webster@example.com
carLicense: DLG73CO
departmentNumber: 5407
employeeType: Contract
homePhone: +1 408 506-9869
initials: G. W.
mobile: +1 804 739-2305
pager: +1 804 961-7792
manager: cn=Sami Hovinga
secretary: cn=Chok Handschy
roomNumber: 6308

dn: cn=Codie Measures, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Codie Measures
sn: Measures
description: This is Codie Measures's description
facsimileTelephoneNumber: +1 206 615-7054
l: Armonk
ou: Administrative
postalAddress: example$Administrative$Dept # 28
telephoneNumber: +1 818 361-2279
title: Supreme Administrative Sales Rep
userPassword: serusaeMei
uid: Codie_Measures
givenName: Codie
mail: Codie_Measures@example.com
carLicense: MPDFFCN
departmentNumber: 2622
employeeType: Normal
homePhone: +1 818 910-8000
initials: C. M.
mobile: +1 818 431-4254
pager: +1 71 472-2635
manager: cn=Rosabel Devine
secretary: cn=Wanda Langett
roomNumber: 9489

dn: cn=Fekri Chanonat, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Fekri Chanonat
sn: Chanonat
description: This is Fekri Chanonat's description
facsimileTelephoneNumber: +1 818 418-6751
l: Emeryville
ou: Product Testing
postalAddress: example$Product Testing$Dept # 139
telephoneNumber: +1 206 857-9837
title: Elite Product Testing Czar
userPassword: tanonahCir
uid: Fekri_Chanonat
givenName: Fekri
mail: Fekri_Chanonat@example.com
carLicense: YFDP9YR
departmentNumber: 7476
employeeType: Normal
homePhone: +1 71 388-6010
initials: F. C.
mobile: +1 206 149-2290
pager: +1 71 405-8307
manager: cn=Dania Eastus
secretary: cn=Merle Rhew
roomNumber: 4801

dn: cn=Neysa Chawla, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Neysa Chawla
sn: Chawla
description: This is Neysa Chawla's description
facsimileTelephoneNumber: +1 213 434-3842
l: Redwood Shores
ou: Product Development
postalAddress: example$Product Development$Dept # 687
telephoneNumber: +1 408 707-1883
title: Master Product Development Evangelist
userPassword: alwahCasye
uid: Neysa_Chawla
givenName: Neysa
mail: Neysa_Chawla@example.com
carLicense: OJM0TQ2
departmentNumber: 9768
employeeType: Temp
homePhone: +1 303 148-6728
initials: N. C.
mobile: +1 804 405-3870
pager: +1 804 196-4354
manager: cn=Melford Circe
secretary: cn=Fidelia Mollerus
roomNumber: 5400

dn: cn=Stacey Kendi, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Stacey Kendi
sn: Kendi
description: This is Stacey Kendi's description
facsimileTelephoneNumber: +1 818 539-7566
l: Alameda
ou: Administrative
postalAddress: example$Administrative$Dept # 751
telephoneNumber: +1 213 137-8584
title: Master Administrative Stooge
userPassword: idneKyecat
uid: Stacey_Kendi
givenName: Stacey
mail: Stacey_Kendi@example.com
carLicense: 65O13C4
departmentNumber: 4015
employeeType: Normal
homePhone: +1 408 340-6594
initials: S. K.
mobile: +1 510 810-5629
pager: +1 408 320-2418
manager: cn=Gwyn Daoud
secretary: cn=Jillene Wendling
roomNumber: 6886

dn: cn=Jeroen Administration, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jeroen Administration
sn: Administration
description: This is Jeroen Administration's description
facsimileTelephoneNumber: +1 408 924-2365
l: Alameda
ou: Payroll
postalAddress: example$Payroll$Dept # 453
telephoneNumber: +1 71 296-1085
title: Senior Payroll Punk
userPassword: noitartsin
uid: Jeroen_Administration
givenName: Jeroen
mail: Jeroen_Administration@example.com
carLicense: N9SRZ2O
departmentNumber: 3186
employeeType: Manager
homePhone: +1 510 531-8751
initials: J. A.
mobile: +1 804 413-9269
pager: +1 804 713-2431
manager: cn=Kathe Plmcoop
secretary: cn=Annabell Chapman
roomNumber: 733

dn: cn=Ryoung Pusztai, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ryoung Pusztai
sn: Pusztai
description: This is Ryoung Pusztai's description
facsimileTelephoneNumber: +1 213 494-3334
l: Menlo Park
ou: Management
postalAddress: example$Management$Dept # 390
telephoneNumber: +1 818 549-1453
title: Master Management Engineer
userPassword: iatzsuPgnu
uid: Ryoung_Pusztai
givenName: Ryoung
mail: Ryoung_Pusztai@example.com
carLicense: 6HGRJKO
departmentNumber: 332
employeeType: Manager
homePhone: +1 206 555-6309
initials: R. P.
mobile: +1 415 376-2267
pager: +1 206 151-6274
manager: cn=Florine Mucklow
secretary: cn=Clarke Sy
roomNumber: 8171

dn: cn=Laine Niergarth, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Laine Niergarth
sn: Niergarth
description: This is Laine Niergarth's description
facsimileTelephoneNumber: +1 510 149-5644
l: Orem
ou: Management
postalAddress: example$Management$Dept # 438
telephoneNumber: +1 408 559-7902
title: Supreme Management Writer
userPassword: htragreiNe
uid: Laine_Niergarth
givenName: Laine
mail: Laine_Niergarth@example.com
carLicense: 7OO6DYL
departmentNumber: 8203
employeeType: Employee
homePhone: +1 415 923-1180
initials: L. N.
mobile: +1 818 505-1606
pager: +1 804 506-4204
manager: cn=Vinod Sotelo
secretary: cn=Leese Gahr
roomNumber: 636

dn: cn=Marit Pizzimenti, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marit Pizzimenti
sn: Pizzimenti
description: This is Marit Pizzimenti's description
facsimileTelephoneNumber: +1 804 381-5191
l: Santa Clara
ou: Accounting
postalAddress: example$Accounting$Dept # 357
telephoneNumber: +1 206 187-1217
title: Junior Accounting Consultant
userPassword: itnemizziP
uid: Marit_Pizzimenti
givenName: Marit
mail: Marit_Pizzimenti@example.com
carLicense: 7JX2WPV
departmentNumber: 8785
employeeType: Temp
homePhone: +1 71 982-4609
initials: M. P.
mobile: +1 415 993-7109
pager: +1 415 904-3987
manager: cn=Eric Madl
secretary: cn=Barton Hirshman
roomNumber: 2991

dn: cn=Mirabelle Okada, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mirabelle Okada
sn: Okada
description: This is Mirabelle Okada's description
facsimileTelephoneNumber: +1 510 943-1518
l: Cupertino
ou: Human Resources
postalAddress: example$Human Resources$Dept # 3
telephoneNumber: +1 71 344-7022
title: Junior Human Resources Accountant
userPassword: adakOelleb
uid: Mirabelle_Okada
givenName: Mirabelle
mail: Mirabelle_Okada@example.com
carLicense: 572GPA5
departmentNumber: 1003
employeeType: Contract
homePhone: +1 71 785-3573
initials: M. O.
mobile: +1 415 803-4450
pager: +1 303 543-3523
manager: cn=Cherilyn Gowan
secretary: cn=Robinet Biedermann
roomNumber: 3408

dn: cn=Randie Harman, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Randie Harman
sn: Harman
description: This is Randie Harman's description
facsimileTelephoneNumber: +1 818 909-7619
l: Palo Alto
ou: Product Development
postalAddress: example$Product Development$Dept # 167
telephoneNumber: +1 71 560-8030
title: Chief Product Development Grunt
userPassword: namraHeidn
uid: Randie_Harman
givenName: Randie
mail: Randie_Harman@example.com
carLicense: O0PYC9D
departmentNumber: 2640
employeeType: Manager
homePhone: +1 206 958-7794
initials: R. H.
mobile: +1 818 839-9054
pager: +1 213 616-5727
manager: cn=Vinod Wooley
secretary: cn=Byron Sugarbroad
roomNumber: 7615

dn: cn=Pinder Mahon, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Pinder Mahon
sn: Mahon
description: This is Pinder Mahon's description
facsimileTelephoneNumber: +1 71 544-3325
l: Sunnyvale
ou: Payroll
postalAddress: example$Payroll$Dept # 761
telephoneNumber: +1 206 571-9497
title: Senior Payroll Pinhead
userPassword: nohaMredni
uid: Pinder_Mahon
givenName: Pinder
mail: Pinder_Mahon@example.com
carLicense: APWDYIZ
departmentNumber: 2301
employeeType: Contract
homePhone: +1 510 577-5126
initials: P. M.
mobile: +1 213 345-1103
pager: +1 303 766-9029
manager: cn=Codee Ibach
secretary: cn=Siva Chiabaut
roomNumber: 348

dn: cn=Ryman Bernier, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ryman Bernier
sn: Bernier
description: This is Ryman Bernier's description
facsimileTelephoneNumber: +1 71 719-1927
l: Cupertino
ou: Management
postalAddress: example$Management$Dept # 242
telephoneNumber: +1 818 860-5074
title: Elite Management Manager
userPassword: reinreBnam
uid: Ryman_Bernier
givenName: Ryman
mail: Ryman_Bernier@example.com
carLicense: G4YGLY5
departmentNumber: 7920
employeeType: Manager
homePhone: +1 71 342-5109
initials: R. B.
mobile: +1 213 207-7528
pager: +1 213 739-4730
manager: cn=Layne Baines
secretary: cn=Margarethe Calis
roomNumber: 353

dn: cn=France Renaud, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: France Renaud
sn: Renaud
description: This is France Renaud's description
facsimileTelephoneNumber: +1 213 531-4309
l: Cambridge
ou: Planning
postalAddress: example$Planning$Dept # 743
telephoneNumber: +1 804 484-7194
title: Elite Planning Admin
userPassword: duaneRecna
uid: France_Renaud
givenName: France
mail: France_Renaud@example.com
carLicense: AKDBJLU
departmentNumber: 7453
employeeType: Temp
homePhone: +1 818 619-1611
initials: F. R.
mobile: +1 71 440-9894
pager: +1 408 610-7007
manager: cn=Anky Beshai
secretary: cn=Lucille Mcgehee
roomNumber: 2514

dn: cn=Hilliary Stough, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Hilliary Stough
sn: Stough
description: This is Hilliary Stough's description
facsimileTelephoneNumber: +1 510 795-3980
l: Redwood Shores
ou: Management
postalAddress: example$Management$Dept # 590
telephoneNumber: +1 818 481-8140
title: Senior Management President
userPassword: hguotSyrai
uid: Hilliary_Stough
givenName: Hilliary
mail: Hilliary_Stough@example.com
carLicense: 9LIY8JA
departmentNumber: 3431
employeeType: Temp
homePhone: +1 818 716-6533
initials: H. S.
mobile: +1 213 247-4780
pager: +1 206 653-7020
manager: cn=Mitch Sldisk
secretary: cn=Melamie Lemaire
roomNumber: 1679

dn: cn=Ardra Whaley, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ardra Whaley
sn: Whaley
description: This is Ardra Whaley's description
facsimileTelephoneNumber: +1 408 653-9993
l: San Jose
ou: Accounting
postalAddress: example$Accounting$Dept # 38
telephoneNumber: +1 818 565-9770
title: Elite Accounting Figurehead
userPassword: yelahWardr
uid: Ardra_Whaley
givenName: Ardra
mail: Ardra_Whaley@example.com
carLicense: 7W4STEZ
departmentNumber: 3225
employeeType: Employee
homePhone: +1 818 226-5315
initials: A. W.
mobile: +1 71 592-5625
pager: +1 303 420-3521
manager: cn=Una Boulay
secretary: cn=Carlee Bento
roomNumber: 4072

dn: cn=Tobe Vachon, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tobe Vachon
sn: Vachon
description: This is Tobe Vachon's description
facsimileTelephoneNumber: +1 206 238-2362
l: San Jose
ou: Peons
postalAddress: example$Peons$Dept # 136
telephoneNumber: +1 804 613-2148
title: Master Peons Assistant
userPassword: nohcaVeboT
uid: Tobe_Vachon
givenName: Tobe
mail: Tobe_Vachon@example.com
carLicense: 7IRTVU6
departmentNumber: 6022
employeeType: Normal
homePhone: +1 206 685-5685
initials: T. V.
mobile: +1 408 207-4050
pager: +1 213 858-5396
manager: cn=Sally Levasseur
secretary: cn=Abbe Shiu
roomNumber: 8565

dn: cn=Weiping Baldridge, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Weiping Baldridge
sn: Baldridge
description: This is Weiping Baldridge's description
facsimileTelephoneNumber: +1 213 826-7039
l: Cupertino
ou: Administrative
postalAddress: example$Administrative$Dept # 207
telephoneNumber: +1 206 872-7097
title: Elite Administrative Manager
userPassword: egdirdlaBg
uid: Weiping_Baldridge
givenName: Weiping
mail: Weiping_Baldridge@example.com
carLicense: MYBLA6E
departmentNumber: 7088
employeeType: Temp
homePhone: +1 303 421-8110
initials: W. B.
mobile: +1 71 859-5629
pager: +1 415 810-1490
manager: cn=Brenna Ostapiw
secretary: cn=Christian Cauthen
roomNumber: 9714

dn: cn=Joann Akbas, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Joann Akbas
sn: Akbas
description: This is Joann Akbas's description
facsimileTelephoneNumber: +1 818 754-8031
l: Cambridge
ou: Product Development
postalAddress: example$Product Development$Dept # 966
telephoneNumber: +1 71 938-4824
title: Chief Product Development Visionary
userPassword: sabkAnnaoJ
uid: Joann_Akbas
givenName: Joann
mail: Joann_Akbas@example.com
carLicense: RQ68PLE
departmentNumber: 2316
employeeType: Temp
homePhone: +1 303 127-1322
initials: J. A.
mobile: +1 303 478-1252
pager: +1 510 600-2815
manager: cn=Florencia Recycling
secretary: cn=Sibeal Esler
roomNumber: 133

dn: cn=Franc Stansby, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Franc Stansby
sn: Stansby
description: This is Franc Stansby's description
facsimileTelephoneNumber: +1 206 871-1717
l: Orem
ou: Administrative
postalAddress: example$Administrative$Dept # 18
telephoneNumber: +1 303 982-9680
title: Junior Administrative Sales Rep
userPassword: ybsnatScna
uid: Franc_Stansby
givenName: Franc
mail: Franc_Stansby@example.com
carLicense: TY61BMA
departmentNumber: 8591
employeeType: Employee
homePhone: +1 510 351-4722
initials: F. S.
mobile: +1 408 471-3809
pager: +1 415 220-4627
manager: cn=Frantisek Druzeta
secretary: cn=Kanu Communication
roomNumber: 6274

dn: cn=Gwynith Mealin, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gwynith Mealin
sn: Mealin
description: This is Gwynith Mealin's description
facsimileTelephoneNumber: +1 213 285-9267
l: Fremont
ou: Human Resources
postalAddress: example$Human Resources$Dept # 724
telephoneNumber: +1 818 147-6410
title: Chief Human Resources Pinhead
userPassword: nilaeMhtin
uid: Gwynith_Mealin
givenName: Gwynith
mail: Gwynith_Mealin@example.com
carLicense: HEJU11H
departmentNumber: 8464
employeeType: Normal
homePhone: +1 818 358-6634
initials: G. M.
mobile: +1 303 544-9949
pager: +1 510 417-9283
manager: cn=Wrennie Aryavong
secretary: cn=Giang Mansourati
roomNumber: 75

dn: cn=Demetri Behler, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Demetri Behler
sn: Behler
description: This is Demetri Behler's description
facsimileTelephoneNumber: +1 206 707-7983
l: Sunnyvale
ou: Planning
postalAddress: example$Planning$Dept # 371
telephoneNumber: +1 408 727-3643
title: Senior Planning Pinhead
userPassword: relheBirte
uid: Demetri_Behler
givenName: Demetri
mail: Demetri_Behler@example.com
carLicense: EL27ZTJ
departmentNumber: 7720
employeeType: Employee
homePhone: +1 415 403-5249
initials: D. B.
mobile: +1 303 297-5258
pager: +1 303 156-9503
manager: cn=Wallace Piel
secretary: cn=Doralynn Lebo
roomNumber: 4863

dn: cn=Saibal Naimpally, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Saibal Naimpally
sn: Naimpally
description: This is Saibal Naimpally's description
facsimileTelephoneNumber: +1 303 399-4748
l: Emeryville
ou: Peons
postalAddress: example$Peons$Dept # 945
telephoneNumber: +1 206 326-1975
title: Chief Peons Mascot
userPassword: yllapmiaNl
uid: Saibal_Naimpally
givenName: Saibal
mail: Saibal_Naimpally@example.com
carLicense: 1IF8IE8
departmentNumber: 4588
employeeType: Normal
homePhone: +1 71 675-2309
initials: S. N.
mobile: +1 804 155-1073
pager: +1 818 691-9926
manager: cn=Cristy Puelma
secretary: cn=Farrah Shane
roomNumber: 8812

dn: cn=Valenka Caceres, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Valenka Caceres
sn: Caceres
description: This is Valenka Caceres's description
facsimileTelephoneNumber: +1 206 491-1130
l: Orem
ou: Product Development
postalAddress: example$Product Development$Dept # 639
telephoneNumber: +1 510 541-1059
title: Senior Product Development Dictator
userPassword: serecaCakn
uid: Valenka_Caceres
givenName: Valenka
mail: Valenka_Caceres@example.com
carLicense: 7AHNYVT
departmentNumber: 5986
employeeType: Temp
homePhone: +1 213 520-3759
initials: V. C.
mobile: +1 510 691-3715
pager: +1 510 797-6306
manager: cn=Lidio Channa
secretary: cn=Caresa Cote
roomNumber: 8463

dn: cn=Vasu Watchmaker, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Vasu Watchmaker
sn: Watchmaker
description: This is Vasu Watchmaker's description
facsimileTelephoneNumber: +1 804 819-2355
l: Cupertino
ou: Product Testing
postalAddress: example$Product Testing$Dept # 802
telephoneNumber: +1 71 392-7758
title: Master Product Testing Accountant
userPassword: rekamhctaW
uid: Vasu_Watchmaker
givenName: Vasu
mail: Vasu_Watchmaker@example.com
carLicense: 4MFLKI3
departmentNumber: 9331
employeeType: Manager
homePhone: +1 206 400-5524
initials: V. W.
mobile: +1 408 546-6814
pager: +1 818 176-8576
manager: cn=Sibyl Coste
secretary: cn=Mickey Ange
roomNumber: 8931

dn: cn=Damien Lauruhn, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Damien Lauruhn
sn: Lauruhn
description: This is Damien Lauruhn's description
facsimileTelephoneNumber: +1 303 516-7904
l: San Francisco
ou: Product Testing
postalAddress: example$Product Testing$Dept # 793
telephoneNumber: +1 213 391-1756
title: Chief Product Testing Writer
userPassword: nhuruaLnei
uid: Damien_Lauruhn
givenName: Damien
mail: Damien_Lauruhn@example.com
carLicense: UFTSP3E
departmentNumber: 6053
employeeType: Manager
homePhone: +1 213 814-5375
initials: D. L.
mobile: +1 71 942-4781
pager: +1 510 909-5076
manager: cn=Marylinda Mabuchi
secretary: cn=Cherilynn Bcs
roomNumber: 9558

dn: cn=Udaya Sprules, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Udaya Sprules
sn: Sprules
description: This is Udaya Sprules's description
facsimileTelephoneNumber: +1 818 323-6035
l: Redmond
ou: Product Development
postalAddress: example$Product Development$Dept # 786
telephoneNumber: +1 213 139-8619
title: Supreme Product Development Manager
userPassword: selurpSaya
uid: Udaya_Sprules
givenName: Udaya
mail: Udaya_Sprules@example.com
carLicense: JADBISJ
departmentNumber: 988
employeeType: Contract
homePhone: +1 804 534-8042
initials: U. S.
mobile: +1 206 357-1336
pager: +1 206 295-7277
manager: cn=Tak Omura
secretary: cn=Lark Gattrell
roomNumber: 4673

dn: cn=Julietta Stasaski, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Julietta Stasaski
sn: Stasaski
description: This is Julietta Stasaski's description
facsimileTelephoneNumber: +1 408 519-1830
l: Fremont
ou: Janitorial
postalAddress: example$Janitorial$Dept # 642
telephoneNumber: +1 804 162-3064
title: Junior Janitorial Developer
userPassword: iksasatSat
uid: Julietta_Stasaski
givenName: Julietta
mail: Julietta_Stasaski@example.com
carLicense: EV5GC6N
departmentNumber: 8746
employeeType: Contract
homePhone: +1 206 536-5375
initials: J. S.
mobile: +1 213 692-7723
pager: +1 408 250-9081
manager: cn=Jurek Rhattigan
secretary: cn=Sharona Harris
roomNumber: 1904

dn: cn=Meghann Hashimoto, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Meghann Hashimoto
sn: Hashimoto
description: This is Meghann Hashimoto's description
facsimileTelephoneNumber: +1 408 434-4275
l: Milpitas
ou: Human Resources
postalAddress: example$Human Resources$Dept # 711
telephoneNumber: +1 206 758-7305
title: Master Human Resources President
userPassword: otomihsaHn
uid: Meghann_Hashimoto
givenName: Meghann
mail: Meghann_Hashimoto@example.com
carLicense: 0UFVWSC
departmentNumber: 3874
employeeType: Employee
homePhone: +1 206 678-6708
initials: M. H.
mobile: +1 303 871-3337
pager: +1 213 932-9972
manager: cn=Kartik Puddington
secretary: cn=Daune Updt
roomNumber: 3942

dn: cn=Augustina Evraire, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Augustina Evraire
sn: Evraire
description: This is Augustina Evraire's description
facsimileTelephoneNumber: +1 213 421-1985
l: Orem
ou: Administrative
postalAddress: example$Administrative$Dept # 21
telephoneNumber: +1 303 452-1137
title: Supreme Administrative Director
userPassword: eriarvEani
uid: Augustina_Evraire
givenName: Augustina
mail: Augustina_Evraire@example.com
carLicense: MBANHC1
departmentNumber: 6016
employeeType: Manager
homePhone: +1 71 807-6498
initials: A. E.
mobile: +1 303 484-4308
pager: +1 804 515-2585
manager: cn=Marcille Roussier
secretary: cn=Melinie Liskoff
roomNumber: 8639

dn: cn=Adorne Nguyen, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Adorne Nguyen
sn: Nguyen
description: This is Adorne Nguyen's description
facsimileTelephoneNumber: +1 804 485-4709
l: Orem
ou: Payroll
postalAddress: example$Payroll$Dept # 467
telephoneNumber: +1 415 325-3305
title: Chief Payroll Figurehead
userPassword: neyugNenro
uid: Adorne_Nguyen
givenName: Adorne
mail: Adorne_Nguyen@example.com
carLicense: L9J6Z6W
departmentNumber: 582
employeeType: Contract
homePhone: +1 213 536-2390
initials: A. N.
mobile: +1 510 374-4770
pager: +1 408 508-4686
manager: cn=Koray Tooyserkani
secretary: cn=Ermo Holvey
roomNumber: 3259

dn: cn=Manami SonHing, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Manami SonHing
sn: SonHing
description: This is Manami SonHing's description
facsimileTelephoneNumber: +1 303 715-6668
l: Cambridge
ou: Janitorial
postalAddress: example$Janitorial$Dept # 790
telephoneNumber: +1 408 490-3409
title: Senior Janitorial Dictator
userPassword: gniHnoSima
uid: Manami_SonHing
givenName: Manami
mail: Manami_SonHing@example.com
carLicense: COQE7XX
departmentNumber: 8655
employeeType: Normal
homePhone: +1 303 507-6643
initials: M. S.
mobile: +1 71 672-6745
pager: +1 804 185-9199
manager: cn=Cheng Kilner
secretary: cn=Marissa Traylor
roomNumber: 4389

dn: cn=Ree Cuffling, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ree Cuffling
sn: Cuffling
description: This is Ree Cuffling's description
facsimileTelephoneNumber: +1 206 323-8567
l: Redwood Shores
ou: Janitorial
postalAddress: example$Janitorial$Dept # 543
telephoneNumber: +1 213 434-1937
title: Supreme Janitorial Dictator
userPassword: gnilffuCee
uid: Ree_Cuffling
givenName: Ree
mail: Ree_Cuffling@example.com
carLicense: 1RI83W8
departmentNumber: 1414
employeeType: Employee
homePhone: +1 510 282-5019
initials: R. C.
mobile: +1 206 323-1264
pager: +1 303 134-7068
manager: cn=Asnat Falaki
secretary: cn=Alexander Miranda
roomNumber: 5983

dn: cn=Maurene Paylor, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Maurene Paylor
sn: Paylor
description: This is Maurene Paylor's description
facsimileTelephoneNumber: +1 415 120-9000
l: Palo Alto
ou: Payroll
postalAddress: example$Payroll$Dept # 856
telephoneNumber: +1 71 382-1296
title: Chief Payroll Sales Rep
userPassword: rolyaPener
uid: Maurene_Paylor
givenName: Maurene
mail: Maurene_Paylor@example.com
carLicense: 6RZ9N9S
departmentNumber: 2005
employeeType: Employee
homePhone: +1 303 438-8363
initials: M. P.
mobile: +1 415 691-7892
pager: +1 510 366-9452
manager: cn=Henk Lindt
secretary: cn=Smita Franco
roomNumber: 571

dn: cn=Wilmette Vidmer, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Wilmette Vidmer
sn: Vidmer
description: This is Wilmette Vidmer's description
facsimileTelephoneNumber: +1 415 934-2352
l: Sunnyvale
ou: Management
postalAddress: example$Management$Dept # 835
telephoneNumber: +1 510 702-2339
title: Chief Management Assistant
userPassword: remdiVette
uid: Wilmette_Vidmer
givenName: Wilmette
mail: Wilmette_Vidmer@example.com
carLicense: SGXD3ZW
departmentNumber: 179
employeeType: Manager
homePhone: +1 804 859-9312
initials: W. V.
mobile: +1 818 588-7656
pager: +1 804 551-3717
manager: cn=Marvette St.Denis
secretary: cn=Guy Lind
roomNumber: 5743

dn: cn=Jessalyn Schuster, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jessalyn Schuster
sn: Schuster
description: This is Jessalyn Schuster's description
facsimileTelephoneNumber: +1 206 957-7521
l: Cupertino
ou: Product Testing
postalAddress: example$Product Testing$Dept # 73
telephoneNumber: +1 415 681-3218
title: Supreme Product Testing Admin
userPassword: retsuhcSny
uid: Jessalyn_Schuster
givenName: Jessalyn
mail: Jessalyn_Schuster@example.com
carLicense: KEPK1IR
departmentNumber: 5271
employeeType: Employee
homePhone: +1 408 485-6916
initials: J. S.
mobile: +1 804 348-4734
pager: +1 415 356-6767
manager: cn=Ofella Fitzpatrick
secretary: cn=Patching Rambo
roomNumber: 9469

dn: cn=Willetta Cucchiaro, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Willetta Cucchiaro
sn: Cucchiaro
description: This is Willetta Cucchiaro's description
facsimileTelephoneNumber: +1 804 780-7795
l: San Jose
ou: Product Development
postalAddress: example$Product Development$Dept # 53
telephoneNumber: +1 303 573-5639
title: Senior Product Development Stooge
userPassword: oraihccuCa
uid: Willetta_Cucchiaro
givenName: Willetta
mail: Willetta_Cucchiaro@example.com
carLicense: QGGN9EW
departmentNumber: 9637
employeeType: Contract
homePhone: +1 408 210-6365
initials: W. C.
mobile: +1 206 290-6619
pager: +1 415 267-9465
manager: cn=Pamelina Schyving
secretary: cn=Pauline Bowser
roomNumber: 4092

dn: cn=Usa Gell, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Usa Gell
sn: Gell
description: This is Usa Gell's description
facsimileTelephoneNumber: +1 213 789-8081
l: Redmond
ou: Peons
postalAddress: example$Peons$Dept # 562
telephoneNumber: +1 510 487-5445
title: Associate Peons Visionary
userPassword: lleGasU
uid: Usa_Gell
givenName: Usa
mail: Usa_Gell@example.com
carLicense: K2KTGE7
departmentNumber: 4216
employeeType: Normal
homePhone: +1 510 808-8963
initials: U. G.
mobile: +1 408 439-1199
pager: +1 408 508-7793
manager: cn=Sukey Cicci
secretary: cn=Nara Planas
roomNumber: 2652

dn: cn=Benne Carli, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Benne Carli
sn: Carli
description: This is Benne Carli's description
facsimileTelephoneNumber: +1 303 473-5030
l: Mountain View
ou: Peons
postalAddress: example$Peons$Dept # 535
telephoneNumber: +1 818 671-2060
title: Associate Peons Director
userPassword: ilraCenneB
uid: Benne_Carli
givenName: Benne
mail: Benne_Carli@example.com
carLicense: NG7GZK1
departmentNumber: 8726
employeeType: Contract
homePhone: +1 303 685-4953
initials: B. C.
mobile: +1 408 293-1367
pager: +1 206 390-3291
manager: cn=Gilda Booker
secretary: cn=Karlene Savarimuthu
roomNumber: 7758

dn: cn=Ike Doublesin, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ike Doublesin
sn: Doublesin
description: This is Ike Doublesin's description
facsimileTelephoneNumber: +1 213 378-8634
l: Emeryville
ou: Peons
postalAddress: example$Peons$Dept # 706
telephoneNumber: +1 818 189-8843
title: Master Peons Grunt
userPassword: niselbuoDe
uid: Ike_Doublesin
givenName: Ike
mail: Ike_Doublesin@example.com
carLicense: FV44VG8
departmentNumber: 6560
employeeType: Normal
homePhone: +1 818 773-4158
initials: I. D.
mobile: +1 71 416-7937
pager: +1 804 356-6695
manager: cn=Dinesh Krause
secretary: cn=Michaela Plourde
roomNumber: 1621

dn: cn=Orlando Curtin, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Orlando Curtin
sn: Curtin
description: This is Orlando Curtin's description
facsimileTelephoneNumber: +1 408 911-4577
l: Redwood Shores
ou: Janitorial
postalAddress: example$Janitorial$Dept # 776
telephoneNumber: +1 206 245-1538
title: Senior Janitorial Vice President
userPassword: nitruCodna
uid: Orlando_Curtin
givenName: Orlando
mail: Orlando_Curtin@example.com
carLicense: UY88XNS
departmentNumber: 6202
employeeType: Employee
homePhone: +1 510 574-8432
initials: O. C.
mobile: +1 415 775-3773
pager: +1 408 739-9073
manager: cn=Mariele Bookings
secretary: cn=Matti Henthorne
roomNumber: 67

dn: cn=Ignatius Tesh, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ignatius Tesh
sn: Tesh
description: This is Ignatius Tesh's description
facsimileTelephoneNumber: +1 206 664-8920
l: San Mateo
ou: Product Development
postalAddress: example$Product Development$Dept # 633
telephoneNumber: +1 71 609-8325
title: Senior Product Development Consultant
userPassword: hseTsuitan
uid: Ignatius_Tesh
givenName: Ignatius
mail: Ignatius_Tesh@example.com
carLicense: APF5G3S
departmentNumber: 8252
employeeType: Contract
homePhone: +1 818 529-4841
initials: I. T.
mobile: +1 303 545-1219
pager: +1 408 305-8619
manager: cn=Bram Salehi
secretary: cn=Jean-Paul Courville
roomNumber: 1986

dn: cn=Giralda Fucito, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Giralda Fucito
sn: Fucito
description: This is Giralda Fucito's description
facsimileTelephoneNumber: +1 303 981-2313
l: Redwood Shores
ou: Planning
postalAddress: example$Planning$Dept # 931
telephoneNumber: +1 804 654-1897
title: Chief Planning Admin
userPassword: oticuFadla
uid: Giralda_Fucito
givenName: Giralda
mail: Giralda_Fucito@example.com
carLicense: DB2GI65
departmentNumber: 130
employeeType: Temp
homePhone: +1 415 904-4017
initials: G. F.
mobile: +1 415 329-1880
pager: +1 415 920-9291
manager: cn=Codee Montoya
secretary: cn=Darell Salehi
roomNumber: 7189

dn: cn=Mariejeanne Karia, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mariejeanne Karia
sn: Karia
description: This is Mariejeanne Karia's description
facsimileTelephoneNumber: +1 804 761-2143
l: San Jose
ou: Product Development
postalAddress: example$Product Development$Dept # 298
telephoneNumber: +1 71 397-7501
title: Master Product Development Czar
userPassword: airaKennae
uid: Mariejeanne_Karia
givenName: Mariejeanne
mail: Mariejeanne_Karia@example.com
carLicense: AHMSX2M
departmentNumber: 527
employeeType: Manager
homePhone: +1 213 845-9536
initials: M. K.
mobile: +1 408 469-7882
pager: +1 303 204-8795
manager: cn=Deirdre Abel
secretary: cn=Adaline Donkers
roomNumber: 5758

dn: cn=Johanne Jammu, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Johanne Jammu
sn: Jammu
description: This is Johanne Jammu's description
facsimileTelephoneNumber: +1 408 963-9729
l: Redmond
ou: Janitorial
postalAddress: example$Janitorial$Dept # 306
telephoneNumber: +1 71 194-2145
title: Master Janitorial Warrior
userPassword: ummaJennah
uid: Johanne_Jammu
givenName: Johanne
mail: Johanne_Jammu@example.com
carLicense: LEHOHNL
departmentNumber: 8623
employeeType: Contract
homePhone: +1 303 283-4228
initials: J. J.
mobile: +1 415 405-3473
pager: +1 71 355-2574
manager: cn=Buffy McGill
secretary: cn=Adriena Kwant
roomNumber: 6253

dn: cn=Rosemary Prunier, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rosemary Prunier
sn: Prunier
description: This is Rosemary Prunier's description
facsimileTelephoneNumber: +1 415 456-8748
l: Redwood Shores
ou: Janitorial
postalAddress: example$Janitorial$Dept # 182
telephoneNumber: +1 415 283-2606
title: Chief Janitorial Czar
userPassword: reinurPyra
uid: Rosemary_Prunier
givenName: Rosemary
mail: Rosemary_Prunier@example.com
carLicense: V71L9G0
departmentNumber: 6194
employeeType: Temp
homePhone: +1 510 635-4995
initials: R. P.
mobile: +1 213 429-6764
pager: +1 71 302-1672
manager: cn=Jasmina Barr
secretary: cn=Bebe Vawter
roomNumber: 8151

dn: cn=Gustie Carr, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gustie Carr
sn: Carr
description: This is Gustie Carr's description
facsimileTelephoneNumber: +1 213 392-3596
l: San Mateo
ou: Payroll
postalAddress: example$Payroll$Dept # 730
telephoneNumber: +1 206 202-4408
title: Associate Payroll Punk
userPassword: rraCeitsuG
uid: Gustie_Carr
givenName: Gustie
mail: Gustie_Carr@example.com
carLicense: TJCWFFR
departmentNumber: 2924
employeeType: Temp
homePhone: +1 818 507-2904
initials: G. C.
mobile: +1 213 159-9090
pager: +1 415 651-2876
manager: cn=Zonda Schrang
secretary: cn=Hedwig Maracle
roomNumber: 3116

dn: cn=Zulema Michailov, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zulema Michailov
sn: Michailov
description: This is Zulema Michailov's description
facsimileTelephoneNumber: +1 408 452-6849
l: San Mateo
ou: Administrative
postalAddress: example$Administrative$Dept # 1
telephoneNumber: +1 818 653-4055
title: Elite Administrative Admin
userPassword: voliahciMa
uid: Zulema_Michailov
givenName: Zulema
mail: Zulema_Michailov@example.com
carLicense: NDBVBLP
departmentNumber: 2524
employeeType: Employee
homePhone: +1 510 670-6873
initials: Z. M.
mobile: +1 804 222-9552
pager: +1 206 275-5874
manager: cn=Ernesta Ste-Marie
secretary: cn=Beverie Brissette
roomNumber: 4544

dn: cn=Karee Patry, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Karee Patry
sn: Patry
description: This is Karee Patry's description
facsimileTelephoneNumber: +1 303 814-9240
l: Cambridge
ou: Administrative
postalAddress: example$Administrative$Dept # 199
telephoneNumber: +1 408 977-3750
title: Senior Administrative Visionary
userPassword: yrtaPeeraK
uid: Karee_Patry
givenName: Karee
mail: Karee_Patry@example.com
carLicense: LP9CTZC
departmentNumber: 8345
employeeType: Contract
homePhone: +1 510 402-3673
initials: K. P.
mobile: +1 818 575-1415
pager: +1 408 117-4516
manager: cn=Sondra Vajentic
secretary: cn=Jenelle Whitlock
roomNumber: 6189

dn: cn=Follick Favreau, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Follick Favreau
sn: Favreau
description: This is Follick Favreau's description
facsimileTelephoneNumber: +1 213 100-7942
l: Sunnyvale
ou: Management
postalAddress: example$Management$Dept # 579
telephoneNumber: +1 415 998-6895
title: Senior Management Engineer
userPassword: uaervaFkci
uid: Follick_Favreau
givenName: Follick
mail: Follick_Favreau@example.com
carLicense: K0EE7GN
departmentNumber: 1329
employeeType: Temp
homePhone: +1 510 803-9770
initials: F. F.
mobile: +1 408 431-6943
pager: +1 804 995-6485
manager: cn=Pratibha Sandness
secretary: cn=Arabela Tarasewicz
roomNumber: 8504

dn: cn=Amrish McClintock, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Amrish McClintock
sn: McClintock
description: This is Amrish McClintock's description
facsimileTelephoneNumber: +1 510 952-3166
l: Sunnyvale
ou: Peons
postalAddress: example$Peons$Dept # 535
telephoneNumber: +1 71 695-1035
title: Chief Peons Pinhead
userPassword: kcotnilCcM
uid: Amrish_McClintock
givenName: Amrish
mail: Amrish_McClintock@example.com
carLicense: TFHS4FO
departmentNumber: 234
employeeType: Normal
homePhone: +1 303 246-6567
initials: A. M.
mobile: +1 415 144-9286
pager: +1 804 926-6727
manager: cn=Larissa Nabors
secretary: cn=Angeles Horianopoulos
roomNumber: 3099

dn: cn=Diego Kluke, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Diego Kluke
sn: Kluke
description: This is Diego Kluke's description
facsimileTelephoneNumber: +1 206 586-6864
l: Cupertino
ou: Product Testing
postalAddress: example$Product Testing$Dept # 20
telephoneNumber: +1 818 836-2797
title: Supreme Product Testing Czar
userPassword: ekulKogeiD
uid: Diego_Kluke
givenName: Diego
mail: Diego_Kluke@example.com
carLicense: SP138NT
departmentNumber: 7942
employeeType: Employee
homePhone: +1 510 601-4047
initials: D. K.
mobile: +1 415 525-7727
pager: +1 206 521-4177
manager: cn=Loralyn Rosche
secretary: cn=Edi Scapin
roomNumber: 1925

dn: cn=Soyong Weinkauf, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Soyong Weinkauf
sn: Weinkauf
description: This is Soyong Weinkauf's description
facsimileTelephoneNumber: +1 71 357-3311
l: Alameda
ou: Management
postalAddress: example$Management$Dept # 483
telephoneNumber: +1 71 536-3463
title: Elite Management Sales Rep
userPassword: fuaknieWgn
uid: Soyong_Weinkauf
givenName: Soyong
mail: Soyong_Weinkauf@example.com
carLicense: KNZKJ7M
departmentNumber: 9953
employeeType: Employee
homePhone: +1 415 252-9968
initials: S. W.
mobile: +1 818 902-3454
pager: +1 415 119-8006
manager: cn=Ermo Libov
secretary: cn=Merunix Curmon
roomNumber: 2825

dn: cn=Demetre Bajpeyi, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Demetre Bajpeyi
sn: Bajpeyi
description: This is Demetre Bajpeyi's description
facsimileTelephoneNumber: +1 71 320-8594
l: Armonk
ou: Janitorial
postalAddress: example$Janitorial$Dept # 402
telephoneNumber: +1 213 852-6001
title: Senior Janitorial Fellow
userPassword: iyepjaBert
uid: Demetre_Bajpeyi
givenName: Demetre
mail: Demetre_Bajpeyi@example.com
carLicense: 24RPQC7
departmentNumber: 3714
employeeType: Temp
homePhone: +1 303 814-9549
initials: D. B.
mobile: +1 303 639-2352
pager: +1 510 926-2787
manager: cn=Siu-Man Sawchuk
secretary: cn=Nikolia Mihara
roomNumber: 2065

dn: cn=Mathilde Maguire, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mathilde Maguire
sn: Maguire
description: This is Mathilde Maguire's description
facsimileTelephoneNumber: +1 213 446-2001
l: Emeryville
ou: Peons
postalAddress: example$Peons$Dept # 50
telephoneNumber: +1 71 378-1781
title: Elite Peons Developer
userPassword: eriugaMedl
uid: Mathilde_Maguire
givenName: Mathilde
mail: Mathilde_Maguire@example.com
carLicense: 1SL2FYB
departmentNumber: 8475
employeeType: Employee
homePhone: +1 804 593-1563
initials: M. M.
mobile: +1 71 457-3623
pager: +1 206 603-5930
manager: cn=Christyna Toplis
secretary: cn=Doc Gillon
roomNumber: 985

dn: cn=Karlon Ludwick, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Karlon Ludwick
sn: Ludwick
description: This is Karlon Ludwick's description
facsimileTelephoneNumber: +1 303 861-9193
l: Armonk
ou: Janitorial
postalAddress: example$Janitorial$Dept # 291
telephoneNumber: +1 818 528-2579
title: Chief Janitorial Warrior
userPassword: kciwduLnol
uid: Karlon_Ludwick
givenName: Karlon
mail: Karlon_Ludwick@example.com
carLicense: FAEIUBW
departmentNumber: 4948
employeeType: Employee
homePhone: +1 206 917-3938
initials: K. L.
mobile: +1 818 195-7948
pager: +1 818 322-4006
manager: cn=Logntp Bonneville
secretary: cn=Flossy Stds
roomNumber: 6342

dn: cn=Onette Kolenda, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Onette Kolenda
sn: Kolenda
description: This is Onette Kolenda's description
facsimileTelephoneNumber: +1 818 722-9149
l: San Mateo
ou: Accounting
postalAddress: example$Accounting$Dept # 231
telephoneNumber: +1 303 262-8838
title: Associate Accounting Technician
userPassword: adneloKett
uid: Onette_Kolenda
givenName: Onette
mail: Onette_Kolenda@example.com
carLicense: K37F6F8
departmentNumber: 1461
employeeType: Employee
homePhone: +1 71 969-8949
initials: O. K.
mobile: +1 303 580-8045
pager: +1 71 995-7476
manager: cn=Janifer Leiker
secretary: cn=Trudey Klasky
roomNumber: 3977

dn: cn=Gregg Likert, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gregg Likert
sn: Likert
description: This is Gregg Likert's description
facsimileTelephoneNumber: +1 71 440-5890
l: San Jose
ou: Accounting
postalAddress: example$Accounting$Dept # 315
telephoneNumber: +1 510 254-4513
title: Master Accounting Evangelist
userPassword: trekiLgger
uid: Gregg_Likert
givenName: Gregg
mail: Gregg_Likert@example.com
carLicense: I575QHB
departmentNumber: 8770
employeeType: Manager
homePhone: +1 818 919-8598
initials: G. L.
mobile: +1 408 138-7196
pager: +1 206 642-8807
manager: cn=Lucina Hyjek
secretary: cn=Mauro Tischler
roomNumber: 2908

dn: cn=Maye Kolb, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Maye Kolb
sn: Kolb
description: This is Maye Kolb's description
facsimileTelephoneNumber: +1 206 233-2759
l: Santa Clara
ou: Human Resources
postalAddress: example$Human Resources$Dept # 145
telephoneNumber: +1 510 465-9537
title: Junior Human Resources Janitor
userPassword: bloKeyaM
uid: Maye_Kolb
givenName: Maye
mail: Maye_Kolb@example.com
carLicense: PUQ2P5H
departmentNumber: 8212
employeeType: Employee
homePhone: +1 510 392-3267
initials: M. K.
mobile: +1 206 195-4931
pager: +1 213 337-6735
manager: cn=Alida Sunderland
secretary: cn=Sheela Bolgos
roomNumber: 2443

dn: cn=Chelsie Gabbai, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Chelsie Gabbai
sn: Gabbai
description: This is Chelsie Gabbai's description
facsimileTelephoneNumber: +1 206 378-2872
l: Emeryville
ou: Peons
postalAddress: example$Peons$Dept # 701
telephoneNumber: +1 408 604-6116
title: Elite Peons Janitor
userPassword: iabbaGeisl
uid: Chelsie_Gabbai
givenName: Chelsie
mail: Chelsie_Gabbai@example.com
carLicense: 7MWMK7L
departmentNumber: 6975
employeeType: Manager
homePhone: +1 213 757-6017
initials: C. G.
mobile: +1 804 458-4095
pager: +1 804 733-8875
manager: cn=Tammara Ledu
secretary: cn=Lynde Quelch
roomNumber: 2041

dn: cn=Rosemarie Zarate, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rosemarie Zarate
sn: Zarate
description: This is Rosemarie Zarate's description
facsimileTelephoneNumber: +1 71 351-4764
l: Fremont
ou: Payroll
postalAddress: example$Payroll$Dept # 174
telephoneNumber: +1 818 310-1578
title: Senior Payroll Grunt
userPassword: etaraZeira
uid: Rosemarie_Zarate
givenName: Rosemarie
mail: Rosemarie_Zarate@example.com
carLicense: WJ3IMXU
departmentNumber: 6610
employeeType: Temp
homePhone: +1 408 168-7149
initials: R. Z.
mobile: +1 303 865-3015
pager: +1 71 987-4480
manager: cn=Raychel Lassiter
secretary: cn=Collete Anker
roomNumber: 9719

dn: cn=Sacto McGilly, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sacto McGilly
sn: McGilly
description: This is Sacto McGilly's description
facsimileTelephoneNumber: +1 510 487-4649
l: Redwood Shores
ou: Administrative
postalAddress: example$Administrative$Dept # 960
telephoneNumber: +1 818 863-3301
title: Elite Administrative Pinhead
userPassword: ylliGcMotc
uid: Sacto_McGilly
givenName: Sacto
mail: Sacto_McGilly@example.com
carLicense: GP46RFT
departmentNumber: 6087
employeeType: Manager
homePhone: +1 510 642-2917
initials: S. M.
mobile: +1 818 137-5043
pager: +1 408 524-4610
manager: cn=Mico Degenova
secretary: cn=Ri Sides
roomNumber: 9001

dn: cn=Sybille Beaucaire, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sybille Beaucaire
sn: Beaucaire
description: This is Sybille Beaucaire's description
facsimileTelephoneNumber: +1 303 458-6926
l: Cupertino
ou: Administrative
postalAddress: example$Administrative$Dept # 505
telephoneNumber: +1 206 139-6275
title: Chief Administrative Technician
userPassword: eriacuaeBe
uid: Sybille_Beaucaire
givenName: Sybille
mail: Sybille_Beaucaire@example.com
carLicense: EBS2ZVQ
departmentNumber: 5628
employeeType: Employee
homePhone: +1 818 327-3720
initials: S. B.
mobile: +1 415 717-3718
pager: +1 206 109-7889
manager: cn=Dorelia Abrahim
secretary: cn=Melinda Nagy
roomNumber: 382

dn: cn=Melvin Instal, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Melvin Instal
sn: Instal
description: This is Melvin Instal's description
facsimileTelephoneNumber: +1 303 567-2740
l: Palo Alto
ou: Janitorial
postalAddress: example$Janitorial$Dept # 674
telephoneNumber: +1 71 925-3834
title: Chief Janitorial Janitor
userPassword: latsnInivl
uid: Melvin_Instal
givenName: Melvin
mail: Melvin_Instal@example.com
carLicense: 3UM0HAK
departmentNumber: 9649
employeeType: Manager
homePhone: +1 804 262-1158
initials: M. I.
mobile: +1 415 429-9014
pager: +1 206 544-1033
manager: cn=Mike Larkin
secretary: cn=Radoslav McLenaghan
roomNumber: 1928

dn: cn=Winne Clayton, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Winne Clayton
sn: Clayton
description: This is Winne Clayton's description
facsimileTelephoneNumber: +1 510 176-2473
l: Alameda
ou: Janitorial
postalAddress: example$Janitorial$Dept # 817
telephoneNumber: +1 818 436-1270
title: Junior Janitorial Developer
userPassword: notyalCenn
uid: Winne_Clayton
givenName: Winne
mail: Winne_Clayton@example.com
carLicense: MX6QWM2
departmentNumber: 5119
employeeType: Contract
homePhone: +1 408 529-5795
initials: W. C.
mobile: +1 408 820-2248
pager: +1 303 868-3000
manager: cn=Mary-Ellen Fastpack
secretary: cn=Marcela Wetherbee
roomNumber: 1395

dn: cn=Claire Prevatt, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Claire Prevatt
sn: Prevatt
description: This is Claire Prevatt's description
facsimileTelephoneNumber: +1 213 841-7855
l: Cambridge
ou: Product Development
postalAddress: example$Product Development$Dept # 451
telephoneNumber: +1 213 221-1032
title: Chief Product Development Developer
userPassword: ttaverPeri
uid: Claire_Prevatt
givenName: Claire
mail: Claire_Prevatt@example.com
carLicense: VGBHLEP
departmentNumber: 635
employeeType: Employee
homePhone: +1 804 474-7424
initials: C. P.
mobile: +1 206 485-6464
pager: +1 213 812-5024
manager: cn=Briney Honkakangas
secretary: cn=Lotta Sotelo
roomNumber: 7435

dn: cn=Harvey Bycenko, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Harvey Bycenko
sn: Bycenko
description: This is Harvey Bycenko's description
facsimileTelephoneNumber: +1 206 998-2600
l: Cupertino
ou: Management
postalAddress: example$Management$Dept # 82
telephoneNumber: +1 206 100-4821
title: Master Management Writer
userPassword: oknecyByev
uid: Harvey_Bycenko
givenName: Harvey
mail: Harvey_Bycenko@example.com
carLicense: 11NK484
departmentNumber: 8041
employeeType: Temp
homePhone: +1 71 506-2067
initials: H. B.
mobile: +1 510 577-7693
pager: +1 408 209-5780
manager: cn=Patch Keseris
secretary: cn=Silvestro Raynor
roomNumber: 7479

dn: cn=Gihan Gendron, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gihan Gendron
sn: Gendron
description: This is Gihan Gendron's description
facsimileTelephoneNumber: +1 818 526-5445
l: Emeryville
ou: Product Testing
postalAddress: example$Product Testing$Dept # 479
telephoneNumber: +1 213 841-5077
title: Chief Product Testing Pinhead
userPassword: nordneGnah
uid: Gihan_Gendron
givenName: Gihan
mail: Gihan_Gendron@example.com
carLicense: 8CBKWNX
departmentNumber: 1825
employeeType: Normal
homePhone: +1 206 253-7786
initials: G. G.
mobile: +1 804 812-5201
pager: +1 510 535-5711
manager: cn=Tonye Terzian
secretary: cn=Wendeline Loadbuild
roomNumber: 404

dn: cn=Harm Ruban, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Harm Ruban
sn: Ruban
description: This is Harm Ruban's description
facsimileTelephoneNumber: +1 818 231-4324
l: San Jose
ou: Product Development
postalAddress: example$Product Development$Dept # 36
telephoneNumber: +1 206 587-1627
title: Junior Product Development Pinhead
userPassword: nabuRmraH
uid: Harm_Ruban
givenName: Harm
mail: Harm_Ruban@example.com
carLicense: LA1TY56
departmentNumber: 1814
employeeType: Employee
homePhone: +1 415 659-9442
initials: H. R.
mobile: +1 71 432-8390
pager: +1 804 613-5224
manager: cn=Cathyleen Beznowski
secretary: cn=Nicholle Jeanes
roomNumber: 6940

dn: cn=Maryse Nagarur, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Maryse Nagarur
sn: Nagarur
description: This is Maryse Nagarur's description
facsimileTelephoneNumber: +1 213 272-9533
l: Redmond
ou: Management
postalAddress: example$Management$Dept # 576
telephoneNumber: +1 408 263-1293
title: Senior Management Grunt
userPassword: ruragaNesy
uid: Maryse_Nagarur
givenName: Maryse
mail: Maryse_Nagarur@example.com
carLicense: P5TEMJC
departmentNumber: 3563
employeeType: Temp
homePhone: +1 818 604-7330
initials: M. N.
mobile: +1 213 234-1237
pager: +1 415 854-6281
manager: cn=Carlen Dost
secretary: cn=Arabella Snider
roomNumber: 1980

dn: cn=Javed Swann, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Javed Swann
sn: Swann
description: This is Javed Swann's description
facsimileTelephoneNumber: +1 510 104-7877
l: Redwood Shores
ou: Planning
postalAddress: example$Planning$Dept # 74
telephoneNumber: +1 804 563-8433
title: Junior Planning Figurehead
userPassword: nnawSdevaJ
uid: Javed_Swann
givenName: Javed
mail: Javed_Swann@example.com
carLicense: GVQ3AFB
departmentNumber: 1888
employeeType: Manager
homePhone: +1 818 771-7074
initials: J. S.
mobile: +1 71 475-2671
pager: +1 303 136-6099
manager: cn=Rungroj Riley
secretary: cn=Jerald Reznechek
roomNumber: 7966

dn: cn=Ruchi Ficken, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ruchi Ficken
sn: Ficken
description: This is Ruchi Ficken's description
facsimileTelephoneNumber: +1 804 689-4774
l: Orem
ou: Accounting
postalAddress: example$Accounting$Dept # 90
telephoneNumber: +1 804 692-5041
title: Junior Accounting Writer
userPassword: nekciFihcu
uid: Ruchi_Ficken
givenName: Ruchi
mail: Ruchi_Ficken@example.com
carLicense: C38WSX3
departmentNumber: 5446
employeeType: Employee
homePhone: +1 206 557-3940
initials: R. F.
mobile: +1 408 171-1977
pager: +1 206 141-1921
manager: cn=Yoko Nichols
secretary: cn=Harmi Reinboth
roomNumber: 9620

dn: cn=Yousef Simcox, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Yousef Simcox
sn: Simcox
description: This is Yousef Simcox's description
facsimileTelephoneNumber: +1 213 130-1178
l: Redwood Shores
ou: Janitorial
postalAddress: example$Janitorial$Dept # 611
telephoneNumber: +1 303 449-6208
title: Junior Janitorial Punk
userPassword: xocmiSfesu
uid: Yousef_Simcox
givenName: Yousef
mail: Yousef_Simcox@example.com
carLicense: YJWIKVR
departmentNumber: 590
employeeType: Manager
homePhone: +1 206 950-1477
initials: Y. S.
mobile: +1 206 523-8158
pager: +1 510 949-5277
manager: cn=Domenick Kay
secretary: cn=Yolanthe Sells
roomNumber: 2200

dn: cn=Gwenda Risdal, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gwenda Risdal
sn: Risdal
description: This is Gwenda Risdal's description
facsimileTelephoneNumber: +1 510 914-1810
l: San Francisco
ou: Payroll
postalAddress: example$Payroll$Dept # 288
telephoneNumber: +1 408 669-1410
title: Associate Payroll Warrior
userPassword: ladsiRadne
uid: Gwenda_Risdal
givenName: Gwenda
mail: Gwenda_Risdal@example.com
carLicense: WMKFV52
departmentNumber: 4777
employeeType: Employee
homePhone: +1 804 236-1042
initials: G. R.
mobile: +1 415 766-7366
pager: +1 206 491-2947
manager: cn=Yatish Cregan
secretary: cn=Du-Tuan Nash
roomNumber: 3893

dn: cn=Giselle Gorberg, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Giselle Gorberg
sn: Gorberg
description: This is Giselle Gorberg's description
facsimileTelephoneNumber: +1 804 291-5293
l: Orem
ou: Human Resources
postalAddress: example$Human Resources$Dept # 635
telephoneNumber: +1 408 782-3902
title: Elite Human Resources Accountant
userPassword: grebroGell
uid: Giselle_Gorberg
givenName: Giselle
mail: Giselle_Gorberg@example.com
carLicense: EKN00S9
departmentNumber: 6640
employeeType: Manager
homePhone: +1 408 243-4677
initials: G. G.
mobile: +1 213 196-2187
pager: +1 213 229-1238
manager: cn=Fei-Yin Jolicoeur
secretary: cn=Cthrine DMS
roomNumber: 5898

dn: cn=Faizal Guthrie, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Faizal Guthrie
sn: Guthrie
description: This is Faizal Guthrie's description
facsimileTelephoneNumber: +1 818 647-1904
l: Orem
ou: Administrative
postalAddress: example$Administrative$Dept # 962
telephoneNumber: +1 71 212-5263
title: Master Administrative Madonna
userPassword: eirhtuGlaz
uid: Faizal_Guthrie
givenName: Faizal
mail: Faizal_Guthrie@example.com
carLicense: GILI2G6
departmentNumber: 3890
employeeType: Employee
homePhone: +1 818 319-1349
initials: F. G.
mobile: +1 818 935-3145
pager: +1 818 898-9167
manager: cn=Thia Glancey
secretary: cn=Ora Zahn
roomNumber: 7586

dn: cn=Dannie Cholet, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dannie Cholet
sn: Cholet
description: This is Dannie Cholet's description
facsimileTelephoneNumber: +1 510 141-1644
l: Sunnyvale
ou: Management
postalAddress: example$Management$Dept # 674
telephoneNumber: +1 213 254-8079
title: Associate Management Accountant
userPassword: telohCeinn
uid: Dannie_Cholet
givenName: Dannie
mail: Dannie_Cholet@example.com
carLicense: AEP8BN8
departmentNumber: 4132
employeeType: Manager
homePhone: +1 71 879-3210
initials: D. C.
mobile: +1 818 991-3354
pager: +1 206 865-9846
manager: cn=Layla Kirouac
secretary: cn=Xuong Giuntini
roomNumber: 2927

dn: cn=Joanna Vandervelde, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Joanna Vandervelde
sn: Vandervelde
description: This is Joanna Vandervelde's description
facsimileTelephoneNumber: +1 415 327-1503
l: San Mateo
ou: Accounting
postalAddress: example$Accounting$Dept # 587
telephoneNumber: +1 71 636-6576
title: Master Accounting Accountant
userPassword: edlevredna
uid: Joanna_Vandervelde
givenName: Joanna
mail: Joanna_Vandervelde@example.com
carLicense: C7ZA600
departmentNumber: 4480
employeeType: Manager
homePhone: +1 71 628-9440
initials: J. V.
mobile: +1 71 527-9619
pager: +1 415 613-6242
manager: cn=Cookie Vetil
secretary: cn=Olia Vettese
roomNumber: 1935

dn: cn=Hanco Knowles, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Hanco Knowles
sn: Knowles
description: This is Hanco Knowles's description
facsimileTelephoneNumber: +1 213 844-3315
l: Milpitas
ou: Peons
postalAddress: example$Peons$Dept # 52
telephoneNumber: +1 71 398-1392
title: Supreme Peons Vice President
userPassword: selwonKocn
uid: Hanco_Knowles
givenName: Hanco
mail: Hanco_Knowles@example.com
carLicense: TKIR393
departmentNumber: 3066
employeeType: Contract
homePhone: +1 408 226-5036
initials: H. K.
mobile: +1 818 561-6679
pager: +1 818 480-3156
manager: cn=Mehdi Barel
secretary: cn=Becki Eisnor
roomNumber: 2059

dn: cn=Ranee Boswick, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ranee Boswick
sn: Boswick
description: This is Ranee Boswick's description
facsimileTelephoneNumber: +1 213 754-4670
l: Mountain View
ou: Product Development
postalAddress: example$Product Development$Dept # 544
telephoneNumber: +1 408 123-3015
title: Junior Product Development Developer
userPassword: kciwsoBeen
uid: Ranee_Boswick
givenName: Ranee
mail: Ranee_Boswick@example.com
carLicense: M30KC7O
departmentNumber: 3497
employeeType: Employee
homePhone: +1 804 494-8119
initials: R. B.
mobile: +1 804 480-8588
pager: +1 804 898-8895
manager: cn=Carline Arnold
secretary: cn=Abu Van Eyk
roomNumber: 1572

dn: cn=Harrison Locken, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Harrison Locken
sn: Locken
description: This is Harrison Locken's description
facsimileTelephoneNumber: +1 804 911-9610
l: Alameda
ou: Janitorial
postalAddress: example$Janitorial$Dept # 537
telephoneNumber: +1 804 265-5384
title: Chief Janitorial Yahoo
userPassword: nekcoLnosi
uid: Harrison_Locken
givenName: Harrison
mail: Harrison_Locken@example.com
carLicense: 1O43SOX
departmentNumber: 5333
employeeType: Normal
homePhone: +1 415 667-4762
initials: H. L.
mobile: +1 71 276-4165
pager: +1 206 618-2207
manager: cn=Pryor Bouroncle
secretary: cn=Rilla Van Kessel
roomNumber: 7155

dn: cn=Petre Albers, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Petre Albers
sn: Albers
description: This is Petre Albers's description
facsimileTelephoneNumber: +1 71 779-7085
l: Redmond
ou: Payroll
postalAddress: example$Payroll$Dept # 388
telephoneNumber: +1 818 731-9347
title: Junior Payroll Dictator
userPassword: sreblAerte
uid: Petre_Albers
givenName: Petre
mail: Petre_Albers@example.com
carLicense: 7KXGT6Y
departmentNumber: 5870
employeeType: Manager
homePhone: +1 510 118-2864
initials: P. A.
mobile: +1 71 595-7803
pager: +1 408 388-8342
manager: cn=Leo Capindale
secretary: cn=Jon Bhatti
roomNumber: 2522

dn: cn=Chantalle Paetsch, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Chantalle Paetsch
sn: Paetsch
description: This is Chantalle Paetsch's description
facsimileTelephoneNumber: +1 206 893-6338
l: Mountain View
ou: Administrative
postalAddress: example$Administrative$Dept # 593
telephoneNumber: +1 818 668-3736
title: Chief Administrative Dictator
userPassword: hcsteaPell
uid: Chantalle_Paetsch
givenName: Chantalle
mail: Chantalle_Paetsch@example.com
carLicense: ZRYK67G
departmentNumber: 9292
employeeType: Contract
homePhone: +1 71 649-9067
initials: C. P.
mobile: +1 415 468-3564
pager: +1 804 441-9916
manager: cn=Bahadir Granger
secretary: cn=Sheela Ratnayake
roomNumber: 6721

dn: cn=Ginette Paperno, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ginette Paperno
sn: Paperno
description: This is Ginette Paperno's description
facsimileTelephoneNumber: +1 206 513-5433
l: Sunnyvale
ou: Product Testing
postalAddress: example$Product Testing$Dept # 894
telephoneNumber: +1 213 133-4449
title: Supreme Product Testing Developer
userPassword: onrepaPett
uid: Ginette_Paperno
givenName: Ginette
mail: Ginette_Paperno@example.com
carLicense: TJG4BB1
departmentNumber: 7981
employeeType: Employee
homePhone: +1 510 176-9118
initials: G. P.
mobile: +1 213 683-6781
pager: +1 804 917-3048
manager: cn=Jacque Krieg
secretary: cn=Renata Raxter
roomNumber: 5244

dn: cn=Raina Barker, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Raina Barker
sn: Barker
description: This is Raina Barker's description
facsimileTelephoneNumber: +1 415 336-8569
l: Cupertino
ou: Administrative
postalAddress: example$Administrative$Dept # 877
telephoneNumber: +1 408 352-5301
title: Senior Administrative Manager
userPassword: rekraBania
uid: Raina_Barker
givenName: Raina
mail: Raina_Barker@example.com
carLicense: IUX1LID
departmentNumber: 1351
employeeType: Temp
homePhone: +1 804 858-1479
initials: R. B.
mobile: +1 818 308-5863
pager: +1 213 420-4581
manager: cn=Karita Misczak
secretary: cn=Tyler Poma
roomNumber: 2632

dn: cn=Tresrch Marzella, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tresrch Marzella
sn: Marzella
description: This is Tresrch Marzella's description
facsimileTelephoneNumber: +1 213 620-8363
l: Sunnyvale
ou: Human Resources
postalAddress: example$Human Resources$Dept # 457
telephoneNumber: +1 415 511-5141
title: Supreme Human Resources Vice President
userPassword: allezraMhc
uid: Tresrch_Marzella
givenName: Tresrch
mail: Tresrch_Marzella@example.com
carLicense: DK1YY7W
departmentNumber: 6168
employeeType: Temp
homePhone: +1 804 468-7675
initials: T. M.
mobile: +1 206 232-3501
pager: +1 804 810-3755
manager: cn=Tomi Twolan
secretary: cn=Ernie Pasvar
roomNumber: 5947

dn: cn=Henrie Bonnar, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Henrie Bonnar
sn: Bonnar
description: This is Henrie Bonnar's description
facsimileTelephoneNumber: +1 415 523-3146
l: Orem
ou: Human Resources
postalAddress: example$Human Resources$Dept # 758
telephoneNumber: +1 206 495-8414
title: Senior Human Resources Janitor
userPassword: rannoBeirn
uid: Henrie_Bonnar
givenName: Henrie
mail: Henrie_Bonnar@example.com
carLicense: 4GZJV62
departmentNumber: 3090
employeeType: Employee
homePhone: +1 206 490-8915
initials: H. B.
mobile: +1 408 224-9073
pager: +1 213 989-4621
manager: cn=Anthiathia Zwierzchowski
secretary: cn=Doc McNulty
roomNumber: 419

dn: cn=Kat Gaudet-Montsion, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kat Gaudet-Montsion
sn: Gaudet-Montsion
description: This is Kat Gaudet-Montsion's description
facsimileTelephoneNumber: +1 510 698-3658
l: Orem
ou: Product Testing
postalAddress: example$Product Testing$Dept # 220
telephoneNumber: +1 408 507-3832
title: Senior Product Testing Artist
userPassword: noistnoM-t
uid: Kat_Gaudet-Montsion
givenName: Kat
mail: Kat_Gaudet-Montsion@example.com
carLicense: EGS97EN
departmentNumber: 383
employeeType: Manager
homePhone: +1 213 457-7658
initials: K. G.
mobile: +1 303 461-9680
pager: +1 415 147-3541
manager: cn=Carmella Dacal
secretary: cn=Julietta McNally
roomNumber: 2958

dn: cn=Pension Oshiro, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Pension Oshiro
sn: Oshiro
description: This is Pension Oshiro's description
facsimileTelephoneNumber: +1 818 965-6467
l: Redwood Shores
ou: Planning
postalAddress: example$Planning$Dept # 723
telephoneNumber: +1 510 645-9344
title: Master Planning Visionary
userPassword: orihsOnois
uid: Pension_Oshiro
givenName: Pension
mail: Pension_Oshiro@example.com
carLicense: 74G2D2L
departmentNumber: 6239
employeeType: Temp
homePhone: +1 804 119-5081
initials: P. O.
mobile: +1 71 923-1480
pager: +1 804 744-2328
manager: cn=Symen Atrc
secretary: cn=Michaelina Stiglitz
roomNumber: 8991

dn: cn=Fraser Kaid, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Fraser Kaid
sn: Kaid
description: This is Fraser Kaid's description
facsimileTelephoneNumber: +1 71 559-4722
l: Cambridge
ou: Product Development
postalAddress: example$Product Development$Dept # 28
telephoneNumber: +1 71 669-1797
title: Senior Product Development Developer
userPassword: diaKresarF
uid: Fraser_Kaid
givenName: Fraser
mail: Fraser_Kaid@example.com
carLicense: OOGFDHH
departmentNumber: 8184
employeeType: Employee
homePhone: +1 804 182-1705
initials: F. K.
mobile: +1 71 729-1487
pager: +1 804 368-3374
manager: cn=Lujanka Bredeck
secretary: cn=Nikoletta Pirkey
roomNumber: 9618

dn: cn=Tyke Kolb, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tyke Kolb
sn: Kolb
description: This is Tyke Kolb's description
facsimileTelephoneNumber: +1 415 472-7326
l: Menlo Park
ou: Product Development
postalAddress: example$Product Development$Dept # 144
telephoneNumber: +1 71 610-5927
title: Associate Product Development Dictator
userPassword: bloKekyT
uid: Tyke_Kolb
givenName: Tyke
mail: Tyke_Kolb@example.com
carLicense: K9EJ2PH
departmentNumber: 3833
employeeType: Temp
homePhone: +1 213 465-6338
initials: T. K.
mobile: +1 510 372-4172
pager: +1 206 155-8964
manager: cn=Lorrie Pegler
secretary: cn=Ainsley Mocock
roomNumber: 9951

dn: cn=Samantha Funston, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Samantha Funston
sn: Funston
description: This is Samantha Funston's description
facsimileTelephoneNumber: +1 818 408-8781
l: Mountain View
ou: Product Development
postalAddress: example$Product Development$Dept # 704
telephoneNumber: +1 510 441-1713
title: Senior Product Development Architect
userPassword: notsnuFaht
uid: Samantha_Funston
givenName: Samantha
mail: Samantha_Funston@example.com
carLicense: OGR5NC0
departmentNumber: 2283
employeeType: Normal
homePhone: +1 71 257-5561
initials: S. F.
mobile: +1 818 203-5965
pager: +1 213 394-3029
manager: cn=Aloise Auth
secretary: cn=Reggie Coupal
roomNumber: 9673

dn: cn=Bunni Rabatich, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bunni Rabatich
sn: Rabatich
description: This is Bunni Rabatich's description
facsimileTelephoneNumber: +1 213 317-1293
l: Alameda
ou: Planning
postalAddress: example$Planning$Dept # 247
telephoneNumber: +1 206 210-4486
title: Associate Planning Grunt
userPassword: hcitabaRin
uid: Bunni_Rabatich
givenName: Bunni
mail: Bunni_Rabatich@example.com
carLicense: F5X9Z13
departmentNumber: 3091
employeeType: Contract
homePhone: +1 206 132-2276
initials: B. R.
mobile: +1 818 598-6564
pager: +1 804 727-1911
manager: cn=Clotilda US
secretary: cn=Judy Dutil
roomNumber: 9364

dn: cn=Kristopher Gach, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kristopher Gach
sn: Gach
description: This is Kristopher Gach's description
facsimileTelephoneNumber: +1 206 537-1265
l: Armonk
ou: Peons
postalAddress: example$Peons$Dept # 168
telephoneNumber: +1 71 891-3814
title: Master Peons Writer
userPassword: hcaGrehpot
uid: Kristopher_Gach
givenName: Kristopher
mail: Kristopher_Gach@example.com
carLicense: CZFQ8BU
departmentNumber: 9591
employeeType: Manager
homePhone: +1 818 440-4469
initials: K. G.
mobile: +1 804 306-7534
pager: +1 206 569-8283
manager: cn=Cecilla Brockhouse
secretary: cn=Giselle Mathew
roomNumber: 4242

dn: cn=Zyg Chawla, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zyg Chawla
sn: Chawla
description: This is Zyg Chawla's description
facsimileTelephoneNumber: +1 71 476-7592
l: Palo Alto
ou: Accounting
postalAddress: example$Accounting$Dept # 634
telephoneNumber: +1 71 917-5784
title: Supreme Accounting Janitor
userPassword: alwahCgyZ
uid: Zyg_Chawla
givenName: Zyg
mail: Zyg_Chawla@example.com
carLicense: JWVPXTE
departmentNumber: 2557
employeeType: Normal
homePhone: +1 213 402-1477
initials: Z. C.
mobile: +1 206 900-8420
pager: +1 408 871-2112
manager: cn=Malethia Van Alphen
secretary: cn=Tilda Jarmon
roomNumber: 1768

dn: cn=Peach Whited, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Peach Whited
sn: Whited
description: This is Peach Whited's description
facsimileTelephoneNumber: +1 71 488-8463
l: Fremont
ou: Management
postalAddress: example$Management$Dept # 453
telephoneNumber: +1 206 407-4377
title: Associate Management Director
userPassword: detihWhcae
uid: Peach_Whited
givenName: Peach
mail: Peach_Whited@example.com
carLicense: D5T79NF
departmentNumber: 4942
employeeType: Employee
homePhone: +1 818 529-1218
initials: P. W.
mobile: +1 408 603-1099
pager: +1 804 273-8788
manager: cn=Kora Gargul
secretary: cn=Rosalinde Cotuna
roomNumber: 9895

dn: cn=Abigail Filer, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Abigail Filer
sn: Filer
description: This is Abigail Filer's description
facsimileTelephoneNumber: +1 213 889-2130
l: Emeryville
ou: Product Testing
postalAddress: example$Product Testing$Dept # 804
telephoneNumber: +1 510 120-7068
title: Master Product Testing Admin
userPassword: reliFliagi
uid: Abigail_Filer
givenName: Abigail
mail: Abigail_Filer@example.com
carLicense: H8GGDRB
departmentNumber: 1626
employeeType: Manager
homePhone: +1 206 776-9048
initials: A. F.
mobile: +1 804 106-4595
pager: +1 71 606-4128
manager: cn=Koral Javallas
secretary: cn=Career Sauls
roomNumber: 1114

dn: cn=Goldia Beton, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Goldia Beton
sn: Beton
description: This is Goldia Beton's description
facsimileTelephoneNumber: +1 206 199-1710
l: San Francisco
ou: Payroll
postalAddress: example$Payroll$Dept # 521
telephoneNumber: +1 71 379-1352
title: Master Payroll Technician
userPassword: noteBaidlo
uid: Goldia_Beton
givenName: Goldia
mail: Goldia_Beton@example.com
carLicense: 8EZLKDZ
departmentNumber: 2103
employeeType: Contract
homePhone: +1 71 194-7458
initials: G. B.
mobile: +1 804 180-8843
pager: +1 510 831-4931
manager: cn=Maxie Kadlecik
secretary: cn=Gateway Drewes
roomNumber: 3898

dn: cn=Reginald Nessman, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Reginald Nessman
sn: Nessman
description: This is Reginald Nessman's description
facsimileTelephoneNumber: +1 804 429-5489
l: Alameda
ou: Janitorial
postalAddress: example$Janitorial$Dept # 155
telephoneNumber: +1 206 763-8825
title: Junior Janitorial Sales Rep
userPassword: namsseNdla
uid: Reginald_Nessman
givenName: Reginald
mail: Reginald_Nessman@example.com
carLicense: BDNVUZR
departmentNumber: 3570
employeeType: Contract
homePhone: +1 408 412-7967
initials: R. N.
mobile: +1 303 833-9104
pager: +1 408 356-6145
manager: cn=Joey Hagenbuck
secretary: cn=Tessa Friton
roomNumber: 9969

dn: cn=Astra Murris, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Astra Murris
sn: Murris
description: This is Astra Murris's description
facsimileTelephoneNumber: +1 804 933-8836
l: San Jose
ou: Management
postalAddress: example$Management$Dept # 914
telephoneNumber: +1 71 788-5412
title: Supreme Management Assistant
userPassword: sirruMarts
uid: Astra_Murris
givenName: Astra
mail: Astra_Murris@example.com
carLicense: 2XIZ7KY
departmentNumber: 2063
employeeType: Normal
homePhone: +1 206 383-8712
initials: A. M.
mobile: +1 415 508-2174
pager: +1 510 963-1069
manager: cn=Rea Streng
secretary: cn=Bernhard Pham
roomNumber: 4108

dn: cn=Bobby Bomstein, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bobby Bomstein
sn: Bomstein
description: This is Bobby Bomstein's description
facsimileTelephoneNumber: +1 510 561-1880
l: Milpitas
ou: Peons
postalAddress: example$Peons$Dept # 192
telephoneNumber: +1 303 660-3962
title: Senior Peons Visionary
userPassword: nietsmoByb
uid: Bobby_Bomstein
givenName: Bobby
mail: Bobby_Bomstein@example.com
carLicense: TE5KGBZ
departmentNumber: 2175
employeeType: Manager
homePhone: +1 415 765-4342
initials: B. B.
mobile: +1 818 728-5181
pager: +1 510 176-2561
manager: cn=Bernardo Worsley
secretary: cn=Nonie Noronha
roomNumber: 371

dn: cn=Kelli Dacre, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kelli Dacre
sn: Dacre
description: This is Kelli Dacre's description
facsimileTelephoneNumber: +1 510 235-3372
l: Redwood Shores
ou: Janitorial
postalAddress: example$Janitorial$Dept # 478
telephoneNumber: +1 303 331-7585
title: Junior Janitorial Artist
userPassword: ercaDilleK
uid: Kelli_Dacre
givenName: Kelli
mail: Kelli_Dacre@example.com
carLicense: OI9MRO9
departmentNumber: 7747
employeeType: Contract
homePhone: +1 303 682-6371
initials: K. D.
mobile: +1 303 950-8800
pager: +1 818 474-2753
manager: cn=Sabina Traxler
secretary: cn=Carol Wessels
roomNumber: 5932

dn: cn=Brittney Matton, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Brittney Matton
sn: Matton
description: This is Brittney Matton's description
facsimileTelephoneNumber: +1 408 809-7442
l: Sunnyvale
ou: Peons
postalAddress: example$Peons$Dept # 698
telephoneNumber: +1 213 641-2094
title: Associate Peons Director
userPassword: nottaMyent
uid: Brittney_Matton
givenName: Brittney
mail: Brittney_Matton@example.com
carLicense: 9ZWN92D
departmentNumber: 6035
employeeType: Contract
homePhone: +1 213 963-1877
initials: B. M.
mobile: +1 303 413-7502
pager: +1 415 588-4764
manager: cn=Cleve Brombal
secretary: cn=Sibella Vanderwel
roomNumber: 1059

dn: cn=Konstance Vandervelde, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Konstance Vandervelde
sn: Vandervelde
description: This is Konstance Vandervelde's description
facsimileTelephoneNumber: +1 71 999-5812
l: Alameda
ou: Janitorial
postalAddress: example$Janitorial$Dept # 593
telephoneNumber: +1 303 384-8754
title: Associate Janitorial Warrior
userPassword: edlevredna
uid: Konstance_Vandervelde
givenName: Konstance
mail: Konstance_Vandervelde@example.com
carLicense: CYZFEGE
departmentNumber: 8861
employeeType: Manager
homePhone: +1 415 626-6615
initials: K. V.
mobile: +1 303 588-8016
pager: +1 510 553-5762
manager: cn=Fenelia Westcott
secretary: cn=Amil Lukic
roomNumber: 6124

dn: cn=Elva Ciskowski, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Elva Ciskowski
sn: Ciskowski
description: This is Elva Ciskowski's description
facsimileTelephoneNumber: +1 510 149-9386
l: Orem
ou: Human Resources
postalAddress: example$Human Resources$Dept # 614
telephoneNumber: +1 303 853-8966
title: Master Human Resources Accountant
userPassword: ikswoksiCa
uid: Elva_Ciskowski
givenName: Elva
mail: Elva_Ciskowski@example.com
carLicense: RDDAVGO
departmentNumber: 1977
employeeType: Contract
homePhone: +1 213 452-8366
initials: E. C.
mobile: +1 415 835-6597
pager: +1 213 997-4726
manager: cn=Almeria Rowell
secretary: cn=Bep Hasen
roomNumber: 70

dn: cn=Bertha Schrader, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bertha Schrader
sn: Schrader
description: This is Bertha Schrader's description
facsimileTelephoneNumber: +1 408 540-4086
l: San Francisco
ou: Management
postalAddress: example$Management$Dept # 416
telephoneNumber: +1 415 934-3510
title: Master Management Artist
userPassword: redarhcSah
uid: Bertha_Schrader
givenName: Bertha
mail: Bertha_Schrader@example.com
carLicense: ABLEY7X
departmentNumber: 4403
employeeType: Contract
homePhone: +1 408 398-5632
initials: B. S.
mobile: +1 408 420-3289
pager: +1 408 822-4828
manager: cn=Hiroshi Forrest
secretary: cn=Meridith Chaintreuil
roomNumber: 1615

dn: cn=Kim-Tram Kennaday, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kim-Tram Kennaday
sn: Kennaday
description: This is Kim-Tram Kennaday's description
facsimileTelephoneNumber: +1 818 227-4636
l: Santa Clara
ou: Human Resources
postalAddress: example$Human Resources$Dept # 826
telephoneNumber: +1 415 680-2987
title: Supreme Human Resources Developer
userPassword: yadanneKma
uid: Kim-Tram_Kennaday
givenName: Kim-Tram
mail: Kim-Tram_Kennaday@example.com
carLicense: FBLSUNG
departmentNumber: 8972
employeeType: Normal
homePhone: +1 408 162-4428
initials: K. K.
mobile: +1 510 443-7611
pager: +1 303 585-5491
manager: cn=Ron Risto
secretary: cn=Shandee Reich
roomNumber: 2683

dn: cn=Isl Majmudar, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Isl Majmudar
sn: Majmudar
description: This is Isl Majmudar's description
facsimileTelephoneNumber: +1 510 441-3073
l: Cupertino
ou: Human Resources
postalAddress: example$Human Resources$Dept # 440
telephoneNumber: +1 213 591-4036
title: Supreme Human Resources Assistant
userPassword: radumjaMls
uid: Isl_Majmudar
givenName: Isl
mail: Isl_Majmudar@example.com
carLicense: FQBB9VE
departmentNumber: 7134
employeeType: Normal
homePhone: +1 206 965-8556
initials: I. M.
mobile: +1 510 334-6507
pager: +1 415 344-7708
manager: cn=Natalie Narayanan
secretary: cn=Pepita Hoadley
roomNumber: 9783

dn: cn=Ragu Moyce, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ragu Moyce
sn: Moyce
description: This is Ragu Moyce's description
facsimileTelephoneNumber: +1 71 533-1173
l: Armonk
ou: Planning
postalAddress: example$Planning$Dept # 296
telephoneNumber: +1 510 839-6879
title: Senior Planning Dictator
userPassword: ecyoMugaR
uid: Ragu_Moyce
givenName: Ragu
mail: Ragu_Moyce@example.com
carLicense: E6K0IUP
departmentNumber: 4746
employeeType: Normal
homePhone: +1 818 354-8223
initials: R. M.
mobile: +1 408 463-9780
pager: +1 303 743-5872
manager: cn=Wing-Ki Cherrier
secretary: cn=Lanita Amalu
roomNumber: 6173

dn: cn=Marie-ann Molochko, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marie-ann Molochko
sn: Molochko
description: This is Marie-ann Molochko's description
facsimileTelephoneNumber: +1 415 520-8403
l: Palo Alto
ou: Management
postalAddress: example$Management$Dept # 58
telephoneNumber: +1 303 251-4803
title: Chief Management Architect
userPassword: okhcoloMnn
uid: Marie-ann_Molochko
givenName: Marie-ann
mail: Marie-ann_Molochko@example.com
carLicense: K3V0NES
departmentNumber: 7819
employeeType: Normal
homePhone: +1 818 760-5020
initials: M. M.
mobile: +1 408 810-6275
pager: +1 804 848-4694
manager: cn=Daron Polashock
secretary: cn=Alica Lamers
roomNumber: 3504

dn: cn=Babbie Van Sickle, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Babbie Van Sickle
sn: Van Sickle
description: This is Babbie Van Sickle's description
facsimileTelephoneNumber: +1 415 976-5668
l: Cambridge
ou: Management
postalAddress: example$Management$Dept # 396
telephoneNumber: +1 303 746-3888
title: Elite Management Warrior
userPassword: elkciSnaVe
uid: Babbie_Van Sickle
givenName: Babbie
mail: Babbie_Van Sickle@example.com
carLicense: UPRJEKE
departmentNumber: 8121
employeeType: Temp
homePhone: +1 213 463-7970
initials: B. V.
mobile: +1 415 118-2179
pager: +1 408 937-2074
manager: cn=Karrah Kingdon
secretary: cn=Hang-Tong Timpson
roomNumber: 8504

dn: cn=Justinn Rockley, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Justinn Rockley
sn: Rockley
description: This is Justinn Rockley's description
facsimileTelephoneNumber: +1 71 712-7820
l: Cambridge
ou: Payroll
postalAddress: example$Payroll$Dept # 726
telephoneNumber: +1 510 858-7589
title: Chief Payroll Consultant
userPassword: yelkcoRnni
uid: Justinn_Rockley
givenName: Justinn
mail: Justinn_Rockley@example.com
carLicense: 6XXFOXU
departmentNumber: 1007
employeeType: Manager
homePhone: +1 71 971-2135
initials: J. R.
mobile: +1 213 601-3610
pager: +1 818 132-2519
manager: cn=Tessa Hurteau
secretary: cn=Caine Scholman
roomNumber: 8447

dn: cn=Azhar Sprules, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Azhar Sprules
sn: Sprules
description: This is Azhar Sprules's description
facsimileTelephoneNumber: +1 408 892-8515
l: Santa Clara
ou: Product Testing
postalAddress: example$Product Testing$Dept # 238
telephoneNumber: +1 415 888-3252
title: Elite Product Testing Director
userPassword: selurpSrah
uid: Azhar_Sprules
givenName: Azhar
mail: Azhar_Sprules@example.com
carLicense: F7UQXK9
departmentNumber: 9433
employeeType: Temp
homePhone: +1 818 208-5102
initials: A. S.
mobile: +1 818 605-1373
pager: +1 206 915-2931
manager: cn=Tuan Awadalla
secretary: cn=Kessley Kimbrough
roomNumber: 1754

dn: cn=Kimihiko Nielson, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kimihiko Nielson
sn: Nielson
description: This is Kimihiko Nielson's description
facsimileTelephoneNumber: +1 804 217-6916
l: San Francisco
ou: Management
postalAddress: example$Management$Dept # 574
telephoneNumber: +1 206 167-1806
title: Junior Management Assistant
userPassword: nosleiNoki
uid: Kimihiko_Nielson
givenName: Kimihiko
mail: Kimihiko_Nielson@example.com
carLicense: FF9BMUF
departmentNumber: 2273
employeeType: Contract
homePhone: +1 71 845-9774
initials: K. N.
mobile: +1 408 468-2881
pager: +1 303 978-7876
manager: cn=Susil Gumbley
secretary: cn=Previn Puent
roomNumber: 429

dn: cn=Shaib Breedlove, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shaib Breedlove
sn: Breedlove
description: This is Shaib Breedlove's description
facsimileTelephoneNumber: +1 804 607-2464
l: Fremont
ou: Planning
postalAddress: example$Planning$Dept # 710
telephoneNumber: +1 408 194-6015
title: Master Planning Yahoo
userPassword: evoldeerBb
uid: Shaib_Breedlove
givenName: Shaib
mail: Shaib_Breedlove@example.com
carLicense: REG24HO
departmentNumber: 8091
employeeType: Temp
homePhone: +1 408 602-8188
initials: S. B.
mobile: +1 206 518-6401
pager: +1 303 220-3328
manager: cn=Naresh Abou-Ezze
secretary: cn=Osama Islam
roomNumber: 6427

dn: cn=Vonni Stegman, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Vonni Stegman
sn: Stegman
description: This is Vonni Stegman's description
facsimileTelephoneNumber: +1 818 913-2880
l: San Mateo
ou: Product Testing
postalAddress: example$Product Testing$Dept # 759
telephoneNumber: +1 303 203-7946
title: Associate Product Testing Manager
userPassword: namgetSinn
uid: Vonni_Stegman
givenName: Vonni
mail: Vonni_Stegman@example.com
carLicense: JM3FB9Q
departmentNumber: 3062
employeeType: Temp
homePhone: +1 408 948-2295
initials: V. S.
mobile: +1 408 993-5048
pager: +1 303 442-3232
manager: cn=Allsun Pownall
secretary: cn=Blair Aston
roomNumber: 5732

dn: cn=Delly Kuczynski, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Delly Kuczynski
sn: Kuczynski
description: This is Delly Kuczynski's description
facsimileTelephoneNumber: +1 415 188-2065
l: Armonk
ou: Administrative
postalAddress: example$Administrative$Dept # 66
telephoneNumber: +1 303 287-6262
title: Associate Administrative Janitor
userPassword: iksnyzcuKy
uid: Delly_Kuczynski
givenName: Delly
mail: Delly_Kuczynski@example.com
carLicense: P3AXFUQ
departmentNumber: 9130
employeeType: Contract
homePhone: +1 303 225-2735
initials: D. K.
mobile: +1 415 544-7901
pager: +1 408 123-5545
manager: cn=Curtis Allison
secretary: cn=Mahmut Verrenneau
roomNumber: 3654

dn: cn=Rajinderpal Packard, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rajinderpal Packard
sn: Packard
description: This is Rajinderpal Packard's description
facsimileTelephoneNumber: +1 303 372-6050
l: Redwood Shores
ou: Peons
postalAddress: example$Peons$Dept # 506
telephoneNumber: +1 71 682-5254
title: Elite Peons Fellow
userPassword: drakcaPlap
uid: Rajinderpal_Packard
givenName: Rajinderpal
mail: Rajinderpal_Packard@example.com
carLicense: 1T5UUB0
departmentNumber: 7073
employeeType: Manager
homePhone: +1 206 505-2004
initials: R. P.
mobile: +1 303 552-9669
pager: +1 206 375-9250
manager: cn=Cheryl Silgardo
secretary: cn=Siobhan St-Pierre
roomNumber: 6068

dn: cn=Tarte Koren, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tarte Koren
sn: Koren
description: This is Tarte Koren's description
facsimileTelephoneNumber: +1 408 235-6422
l: San Francisco
ou: Payroll
postalAddress: example$Payroll$Dept # 807
telephoneNumber: +1 804 293-2694
title: Supreme Payroll Warrior
userPassword: neroKetraT
uid: Tarte_Koren
givenName: Tarte
mail: Tarte_Koren@example.com
carLicense: P0ZB3YN
departmentNumber: 8246
employeeType: Temp
homePhone: +1 510 277-4993
initials: T. K.
mobile: +1 206 675-5109
pager: +1 408 224-2113
manager: cn=Sophi Borum
secretary: cn=Kylen Ryan
roomNumber: 2346

dn: cn=Bevvy Roden, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Bevvy Roden
sn: Roden
description: This is Bevvy Roden's description
facsimileTelephoneNumber: +1 213 863-6260
l: Cupertino
ou: Product Development
postalAddress: example$Product Development$Dept # 290
telephoneNumber: +1 408 300-1187
title: Elite Product Development Accountant
userPassword: nedoRyvveB
uid: Bevvy_Roden
givenName: Bevvy
mail: Bevvy_Roden@example.com
carLicense: MLXY26G
departmentNumber: 5513
employeeType: Manager
homePhone: +1 510 207-1740
initials: B. R.
mobile: +1 818 720-2123
pager: +1 408 440-2650
manager: cn=Rafi Eastick
secretary: cn=Massoud Mapile
roomNumber: 897

dn: cn=Rozett Mathis, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rozett Mathis
sn: Mathis
description: This is Rozett Mathis's description
facsimileTelephoneNumber: +1 303 833-5751
l: Mountain View
ou: Product Testing
postalAddress: example$Product Testing$Dept # 449
telephoneNumber: +1 213 683-5687
title: Associate Product Testing Consultant
userPassword: sihtaMttez
uid: Rozett_Mathis
givenName: Rozett
mail: Rozett_Mathis@example.com
carLicense: BVHD1AO
departmentNumber: 6038
employeeType: Normal
homePhone: +1 818 339-4344
initials: R. M.
mobile: +1 303 642-3684
pager: +1 510 443-5679
manager: cn=Yutaka Drummond
secretary: cn=Mandie Results
roomNumber: 1372

dn: cn=Sunshine Glew, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sunshine Glew
sn: Glew
description: This is Sunshine Glew's description
facsimileTelephoneNumber: +1 408 100-1156
l: Alameda
ou: Planning
postalAddress: example$Planning$Dept # 113
telephoneNumber: +1 213 289-2825
title: Junior Planning Consultant
userPassword: welGenihsn
uid: Sunshine_Glew
givenName: Sunshine
mail: Sunshine_Glew@example.com
carLicense: MZY889U
departmentNumber: 1089
employeeType: Temp
homePhone: +1 206 148-4758
initials: S. G.
mobile: +1 415 865-3351
pager: +1 510 522-5831
manager: cn=King-Haut Townsend
secretary: cn=Hillary Mcellistrem
roomNumber: 2021

dn: cn=Sherie Dba, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sherie Dba
sn: Dba
description: This is Sherie Dba's description
facsimileTelephoneNumber: +1 818 389-8239
l: Redmond
ou: Product Testing
postalAddress: example$Product Testing$Dept # 124
telephoneNumber: +1 818 187-1777
title: Junior Product Testing Director
userPassword: abDeirehS
uid: Sherie_Dba
givenName: Sherie
mail: Sherie_Dba@example.com
carLicense: 71U7RQE
departmentNumber: 5749
employeeType: Temp
homePhone: +1 206 576-1081
initials: S. D.
mobile: +1 303 598-7803
pager: +1 415 538-5682
manager: cn=Fernando Sonier
secretary: cn=Vivianne Faley
roomNumber: 6609

dn: cn=Cherianne Batchelder, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Cherianne Batchelder
sn: Batchelder
description: This is Cherianne Batchelder's description
facsimileTelephoneNumber: +1 818 665-9755
l: Menlo Park
ou: Planning
postalAddress: example$Planning$Dept # 948
telephoneNumber: +1 71 629-8969
title: Supreme Planning Pinhead
userPassword: redlehctaB
uid: Cherianne_Batchelder
givenName: Cherianne
mail: Cherianne_Batchelder@example.com
carLicense: 9GDOZHG
departmentNumber: 6133
employeeType: Normal
homePhone: +1 510 592-1483
initials: C. B.
mobile: +1 303 305-8621
pager: +1 71 483-4716
manager: cn=Jada Kolodziej
secretary: cn=Marcela Lahlum
roomNumber: 7346

dn: cn=Yuksel Deatrick, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Yuksel Deatrick
sn: Deatrick
description: This is Yuksel Deatrick's description
facsimileTelephoneNumber: +1 804 459-3158
l: Palo Alto
ou: Product Testing
postalAddress: example$Product Testing$Dept # 55
telephoneNumber: +1 213 917-9451
title: Junior Product Testing President
userPassword: kcirtaeDle
uid: Yuksel_Deatrick
givenName: Yuksel
mail: Yuksel_Deatrick@example.com
carLicense: AKH9YO6
departmentNumber: 3460
employeeType: Contract
homePhone: +1 415 918-8804
initials: Y. D.
mobile: +1 206 411-8574
pager: +1 303 859-6762
manager: cn=Lorelle Ormsby
secretary: cn=Megen Merrill
roomNumber: 2895

dn: cn=Kip Zaloker, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kip Zaloker
sn: Zaloker
description: This is Kip Zaloker's description
facsimileTelephoneNumber: +1 510 203-6273
l: Mountain View
ou: Janitorial
postalAddress: example$Janitorial$Dept # 231
telephoneNumber: +1 71 858-5108
title: Senior Janitorial Dictator
userPassword: rekolaZpiK
uid: Kip_Zaloker
givenName: Kip
mail: Kip_Zaloker@example.com
carLicense: O6T7LCS
departmentNumber: 2501
employeeType: Normal
homePhone: +1 303 294-5589
initials: K. Z.
mobile: +1 818 939-5645
pager: +1 415 449-3063
manager: cn=Zitella Projects
secretary: cn=Eden Sherif
roomNumber: 1344

dn: cn=Andaree Brady, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Andaree Brady
sn: Brady
description: This is Andaree Brady's description
facsimileTelephoneNumber: +1 818 701-7411
l: Cupertino
ou: Product Development
postalAddress: example$Product Development$Dept # 239
telephoneNumber: +1 804 185-1426
title: Supreme Product Development Manager
userPassword: ydarBeerad
uid: Andaree_Brady
givenName: Andaree
mail: Andaree_Brady@example.com
carLicense: SQK8WWU
departmentNumber: 4226
employeeType: Temp
homePhone: +1 818 663-4622
initials: A. B.
mobile: +1 415 868-5365
pager: +1 510 317-1116
manager: cn=Kelli Paone
secretary: cn=Mil Lum
roomNumber: 5382

dn: cn=Seth Ladet, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Seth Ladet
sn: Ladet
description: This is Seth Ladet's description
facsimileTelephoneNumber: +1 206 122-6688
l: San Jose
ou: Payroll
postalAddress: example$Payroll$Dept # 545
telephoneNumber: +1 206 972-3584
title: Chief Payroll Pinhead
userPassword: tedaLhteS
uid: Seth_Ladet
givenName: Seth
mail: Seth_Ladet@example.com
carLicense: T7WAH8Z
departmentNumber: 3616
employeeType: Temp
homePhone: +1 510 149-3139
initials: S. L.
mobile: +1 415 192-2299
pager: +1 303 745-8757
manager: cn=Annemarie Hennessy
secretary: cn=Elio Laine
roomNumber: 7672

dn: cn=Pierre-Henri Weidinger, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Pierre-Henri Weidinger
sn: Weidinger
description: This is Pierre-Henri Weidinger's description
facsimileTelephoneNumber: +1 213 447-5128
l: Armonk
ou: Administrative
postalAddress: example$Administrative$Dept # 514
telephoneNumber: +1 71 533-4703
title: Chief Administrative Technician
userPassword: regnidieWi
uid: Pierre-Henri_Weidinger
givenName: Pierre-Henri
mail: Pierre-Henri_Weidinger@example.com
carLicense: 0190X9E
departmentNumber: 729
employeeType: Temp
homePhone: +1 303 749-1223
initials: P. W.
mobile: +1 415 191-5113
pager: +1 213 813-1837
manager: cn=Manny Pitcavage
secretary: cn=Harper Decapua
roomNumber: 7150

dn: cn=Daveen Dann, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Daveen Dann
sn: Dann
description: This is Daveen Dann's description
facsimileTelephoneNumber: +1 206 874-6373
l: Alameda
ou: Planning
postalAddress: example$Planning$Dept # 238
telephoneNumber: +1 510 544-4091
title: Junior Planning Manager
userPassword: nnaDneevaD
uid: Daveen_Dann
givenName: Daveen
mail: Daveen_Dann@example.com
carLicense: JI065N7
departmentNumber: 5939
employeeType: Normal
homePhone: +1 206 882-5957
initials: D. D.
mobile: +1 818 340-4499
pager: +1 303 881-4225
manager: cn=Ashely Goulet
secretary: cn=Margalit Kahan
roomNumber: 3461

dn: cn=Wallis Cochran, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Wallis Cochran
sn: Cochran
description: This is Wallis Cochran's description
facsimileTelephoneNumber: +1 415 370-9855
l: Emeryville
ou: Accounting
postalAddress: example$Accounting$Dept # 921
telephoneNumber: +1 818 552-8988
title: Master Accounting Accountant
userPassword: narhcoCsil
uid: Wallis_Cochran
givenName: Wallis
mail: Wallis_Cochran@example.com
carLicense: 0KJN84D
departmentNumber: 1461
employeeType: Normal
homePhone: +1 510 729-2049
initials: W. C.
mobile: +1 510 579-4811
pager: +1 510 366-4052
manager: cn=Gerard Malkani
secretary: cn=Matti Marette
roomNumber: 4108

dn: cn=Katalin Lommen, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Katalin Lommen
sn: Lommen
description: This is Katalin Lommen's description
facsimileTelephoneNumber: +1 818 906-8383
l: Orem
ou: Planning
postalAddress: example$Planning$Dept # 604
telephoneNumber: +1 303 883-1889
title: Elite Planning Vice President
userPassword: nemmoLnila
uid: Katalin_Lommen
givenName: Katalin
mail: Katalin_Lommen@example.com
carLicense: 0VNJP50
departmentNumber: 2059
employeeType: Normal
homePhone: +1 71 992-2035
initials: K. L.
mobile: +1 303 820-6903
pager: +1 408 379-5944
manager: cn=Ailey Wingfield
secretary: cn=Corey Marcelissen
roomNumber: 3676

dn: cn=Kathi Kunecke, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kathi Kunecke
sn: Kunecke
description: This is Kathi Kunecke's description
facsimileTelephoneNumber: +1 415 932-3783
l: Cupertino
ou: Management
postalAddress: example$Management$Dept # 354
telephoneNumber: +1 206 792-2906
title: Chief Management President
userPassword: ekcenuKiht
uid: Kathi_Kunecke
givenName: Kathi
mail: Kathi_Kunecke@example.com
carLicense: 1AV73X4
departmentNumber: 2037
employeeType: Contract
homePhone: +1 213 250-9680
initials: K. K.
mobile: +1 804 752-2203
pager: +1 408 445-1685
manager: cn=Hari Womble
secretary: cn=Deana Patchcor
roomNumber: 7759

dn: cn=Jun Showers, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jun Showers
sn: Showers
description: This is Jun Showers's description
facsimileTelephoneNumber: +1 213 952-8464
l: Menlo Park
ou: Accounting
postalAddress: example$Accounting$Dept # 219
telephoneNumber: +1 415 138-4745
title: Elite Accounting Engineer
userPassword: srewohSnuJ
uid: Jun_Showers
givenName: Jun
mail: Jun_Showers@example.com
carLicense: BXM4XCL
departmentNumber: 4228
employeeType: Temp
homePhone: +1 213 761-8326
initials: J. S.
mobile: +1 818 784-8400
pager: +1 303 877-4036
manager: cn=Janos Pkg
secretary: cn=Mercy Smothers
roomNumber: 6408

dn: cn=Karisa Blethen, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Karisa Blethen
sn: Blethen
description: This is Karisa Blethen's description
facsimileTelephoneNumber: +1 206 378-4638
l: San Jose
ou: Product Development
postalAddress: example$Product Development$Dept # 192
telephoneNumber: +1 818 617-8160
title: Senior Product Development Figurehead
userPassword: nehtelBasi
uid: Karisa_Blethen
givenName: Karisa
mail: Karisa_Blethen@example.com
carLicense: 2IOFJ9P
departmentNumber: 4573
employeeType: Employee
homePhone: +1 303 934-6410
initials: K. B.
mobile: +1 818 353-7906
pager: +1 818 892-1447
manager: cn=Anallese Silverthorn
secretary: cn=Ginnie Kiel
roomNumber: 5237

dn: cn=Arda Njo, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Arda Njo
sn: Njo
description: This is Arda Njo's description
facsimileTelephoneNumber: +1 804 412-7464
l: Armonk
ou: Planning
postalAddress: example$Planning$Dept # 938
telephoneNumber: +1 510 593-9460
title: Senior Planning Developer
userPassword: ojNadrA
uid: Arda_Njo
givenName: Arda
mail: Arda_Njo@example.com
carLicense: XL3XF3M
departmentNumber: 9245
employeeType: Contract
homePhone: +1 510 965-6740
initials: A. N.
mobile: +1 213 207-6189
pager: +1 206 341-6816
manager: cn=Berni Hahn
secretary: cn=Heda Gattrell
roomNumber: 4815

dn: cn=GeorgeAnn Paschall, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: GeorgeAnn Paschall
sn: Paschall
description: This is GeorgeAnn Paschall's description
facsimileTelephoneNumber: +1 415 131-1692
l: Mountain View
ou: Management
postalAddress: example$Management$Dept # 381
telephoneNumber: +1 818 585-8895
title: Elite Management Mascot
userPassword: llahcsaPnn
uid: GeorgeAnn_Paschall
givenName: GeorgeAnn
mail: GeorgeAnn_Paschall@example.com
carLicense: MQSVK2P
departmentNumber: 5077
employeeType: Temp
homePhone: +1 408 318-7319
initials: G. P.
mobile: +1 206 141-9887
pager: +1 408 765-2390
manager: cn=Mid Cheal
secretary: cn=Lacie Knittel
roomNumber: 7796

dn: cn=Manimozhi Nambride, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Manimozhi Nambride
sn: Nambride
description: This is Manimozhi Nambride's description
facsimileTelephoneNumber: +1 415 950-3359
l: Santa Clara
ou: Janitorial
postalAddress: example$Janitorial$Dept # 934
telephoneNumber: +1 303 100-6826
title: Junior Janitorial Consultant
userPassword: edirbmaNih
uid: Manimozhi_Nambride
givenName: Manimozhi
mail: Manimozhi_Nambride@example.com
carLicense: KE3BXSB
departmentNumber: 6895
employeeType: Manager
homePhone: +1 804 913-9282
initials: M. N.
mobile: +1 71 457-3978
pager: +1 818 649-3056
manager: cn=Karisa Wasitova
secretary: cn=Edwina Dempster
roomNumber: 7328

dn: cn=Cheng Herling, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Cheng Herling
sn: Herling
description: This is Cheng Herling's description
facsimileTelephoneNumber: +1 510 799-7267
l: Santa Clara
ou: Peons
postalAddress: example$Peons$Dept # 643
telephoneNumber: +1 415 732-9185
title: Elite Peons Stooge
userPassword: gnilreHgne
uid: Cheng_Herling
givenName: Cheng
mail: Cheng_Herling@example.com
carLicense: Y2KQCUM
departmentNumber: 6119
employeeType: Normal
homePhone: +1 804 921-3267
initials: C. H.
mobile: +1 415 485-2002
pager: +1 303 622-8565
manager: cn=Neely Nickells
secretary: cn=Darya Lucas
roomNumber: 7426

dn: cn=Rigoberto Cohea, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rigoberto Cohea
sn: Cohea
description: This is Rigoberto Cohea's description
facsimileTelephoneNumber: +1 804 916-3581
l: Emeryville
ou: Product Development
postalAddress: example$Product Development$Dept # 641
telephoneNumber: +1 804 174-4795
title: Supreme Product Development Sales Rep
userPassword: aehoCotreb
uid: Rigoberto_Cohea
givenName: Rigoberto
mail: Rigoberto_Cohea@example.com
carLicense: CBKSEYP
departmentNumber: 8439
employeeType: Contract
homePhone: +1 206 331-7116
initials: R. C.
mobile: +1 71 832-6927
pager: +1 213 846-6202
manager: cn=Delphine Astalos
secretary: cn=Paul Asgharzadeh
roomNumber: 3490

dn: cn=Cindelyn Pedigo, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Cindelyn Pedigo
sn: Pedigo
description: This is Cindelyn Pedigo's description
facsimileTelephoneNumber: +1 818 113-3409
l: Palo Alto
ou: Product Development
postalAddress: example$Product Development$Dept # 468
telephoneNumber: +1 818 371-1388
title: Elite Product Development Developer
userPassword: ogidePnyle
uid: Cindelyn_Pedigo
givenName: Cindelyn
mail: Cindelyn_Pedigo@example.com
carLicense: CIDGKGE
departmentNumber: 452
employeeType: Manager
homePhone: +1 415 280-1529
initials: C. P.
mobile: +1 510 104-9328
pager: +1 415 231-2322
manager: cn=Uday McCaughey
secretary: cn=Richardson Chapen
roomNumber: 227

dn: cn=Loni Navarro, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Loni Navarro
sn: Navarro
description: This is Loni Navarro's description
facsimileTelephoneNumber: +1 206 824-5495
l: Fremont
ou: Human Resources
postalAddress: example$Human Resources$Dept # 664
telephoneNumber: +1 510 166-7452
title: Chief Human Resources Artist
userPassword: orravaNino
uid: Loni_Navarro
givenName: Loni
mail: Loni_Navarro@example.com
carLicense: HB9XT3Q
departmentNumber: 2653
employeeType: Contract
homePhone: +1 804 155-8530
initials: L. N.
mobile: +1 213 197-5641
pager: +1 415 400-9972
manager: cn=Htd Wasserman
secretary: cn=Dotty Spindler
roomNumber: 7180

dn: cn=Della Sabol, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Della Sabol
sn: Sabol
description: This is Della Sabol's description
facsimileTelephoneNumber: +1 510 483-9525
l: Emeryville
ou: Product Testing
postalAddress: example$Product Testing$Dept # 696
telephoneNumber: +1 804 220-6286
title: Junior Product Testing Dictator
userPassword: lobaSalleD
uid: Della_Sabol
givenName: Della
mail: Della_Sabol@example.com
carLicense: YEYFCSF
departmentNumber: 9542
employeeType: Normal
homePhone: +1 408 435-7591
initials: D. S.
mobile: +1 213 955-8935
pager: +1 213 769-5940
manager: cn=Kirby Coste
secretary: cn=Peggi Rabjohn
roomNumber: 8388

dn: cn=Veleta Ganguly, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Veleta Ganguly
sn: Ganguly
description: This is Veleta Ganguly's description
facsimileTelephoneNumber: +1 804 461-4636
l: Redwood Shores
ou: Human Resources
postalAddress: example$Human Resources$Dept # 326
telephoneNumber: +1 303 548-4933
title: Master Human Resources Sales Rep
userPassword: ylugnaGate
uid: Veleta_Ganguly
givenName: Veleta
mail: Veleta_Ganguly@example.com
carLicense: FCKFIIA
departmentNumber: 8350
employeeType: Manager
homePhone: +1 213 308-6469
initials: V. G.
mobile: +1 415 244-5336
pager: +1 213 180-3400
manager: cn=Quoc-Vu Coupal
secretary: cn=Dru Bredeck
roomNumber: 644

dn: cn=Susann Appleyard, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Susann Appleyard
sn: Appleyard
description: This is Susann Appleyard's description
facsimileTelephoneNumber: +1 303 867-5090
l: Sunnyvale
ou: Planning
postalAddress: example$Planning$Dept # 233
telephoneNumber: +1 303 728-7897
title: Junior Planning Czar
userPassword: drayelppAn
uid: Susann_Appleyard
givenName: Susann
mail: Susann_Appleyard@example.com
carLicense: S8PYXWW
departmentNumber: 7783
employeeType: Normal
homePhone: +1 415 651-8434
initials: S. A.
mobile: +1 408 689-6582
pager: +1 818 885-2995
manager: cn=Minnaminnie Pieron
secretary: cn=Nenad Bagetakos
roomNumber: 3714

dn: cn=Hermien Purchasing, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Hermien Purchasing
sn: Purchasing
description: This is Hermien Purchasing's description
facsimileTelephoneNumber: +1 408 121-9607
l: Milpitas
ou: Accounting
postalAddress: example$Accounting$Dept # 545
telephoneNumber: +1 206 385-9588
title: Associate Accounting Evangelist
userPassword: gnisahcruP
uid: Hermien_Purchasing
givenName: Hermien
mail: Hermien_Purchasing@example.com
carLicense: VO7L7V2
departmentNumber: 8034
employeeType: Normal
homePhone: +1 213 374-3103
initials: H. P.
mobile: +1 71 261-3443
pager: +1 818 791-3370
manager: cn=Marianne Crippen
secretary: cn=Kassi Prichard
roomNumber: 572

dn: cn=Nance Hawken, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Nance Hawken
sn: Hawken
description: This is Nance Hawken's description
facsimileTelephoneNumber: +1 71 546-4303
l: San Francisco
ou: Management
postalAddress: example$Management$Dept # 51
telephoneNumber: +1 303 154-4057
title: Associate Management Dictator
userPassword: nekwaHecna
uid: Nance_Hawken
givenName: Nance
mail: Nance_Hawken@example.com
carLicense: I0FGJBC
departmentNumber: 7420
employeeType: Normal
homePhone: +1 415 245-8639
initials: N. H.
mobile: +1 303 575-8328
pager: +1 408 636-3502
manager: cn=Narida Krieg
secretary: cn=Kana Azizuddin
roomNumber: 8582

dn: cn=Leisha Damena, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Leisha Damena
sn: Damena
description: This is Leisha Damena's description
facsimileTelephoneNumber: +1 303 780-2129
l: Redmond
ou: Administrative
postalAddress: example$Administrative$Dept # 950
telephoneNumber: +1 206 435-6314
title: Elite Administrative Developer
userPassword: anemaDahsi
uid: Leisha_Damena
givenName: Leisha
mail: Leisha_Damena@example.com
carLicense: 36W71OF
departmentNumber: 974
employeeType: Manager
homePhone: +1 71 375-3905
initials: L. D.
mobile: +1 415 577-2948
pager: +1 408 242-1941
manager: cn=Nico Nix
secretary: cn=Ernie Dunham
roomNumber: 75

dn: cn=Randy Hardcastle, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Randy Hardcastle
sn: Hardcastle
description: This is Randy Hardcastle's description
facsimileTelephoneNumber: +1 206 976-4416
l: Palo Alto
ou: Accounting
postalAddress: example$Accounting$Dept # 558
telephoneNumber: +1 303 554-1526
title: Senior Accounting Writer
userPassword: eltsacdraH
uid: Randy_Hardcastle
givenName: Randy
mail: Randy_Hardcastle@example.com
carLicense: 48ADIBF
departmentNumber: 9450
employeeType: Employee
homePhone: +1 818 651-2394
initials: R. H.
mobile: +1 818 777-2263
pager: +1 206 705-4779
manager: cn=Ilan Tombul
secretary: cn=Celia Szkarlat
roomNumber: 8487

dn: cn=Charmane Zeggil, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Charmane Zeggil
sn: Zeggil
description: This is Charmane Zeggil's description
facsimileTelephoneNumber: +1 510 455-4632
l: Cambridge
ou: Human Resources
postalAddress: example$Human Resources$Dept # 482
telephoneNumber: +1 206 175-7998
title: Supreme Human Resources Visionary
userPassword: liggeZenam
uid: Charmane_Zeggil
givenName: Charmane
mail: Charmane_Zeggil@example.com
carLicense: 0MAVAUF
departmentNumber: 9354
employeeType: Employee
homePhone: +1 408 126-3636
initials: C. Z.
mobile: +1 303 117-6915
pager: +1 510 270-9083
manager: cn=Manda Suwanawongse
secretary: cn=Ylaine Luettchau
roomNumber: 6672

dn: cn=Ansley Corner, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ansley Corner
sn: Corner
description: This is Ansley Corner's description
facsimileTelephoneNumber: +1 804 958-6413
l: Menlo Park
ou: Product Testing
postalAddress: example$Product Testing$Dept # 28
telephoneNumber: +1 213 482-9036
title: Master Product Testing Technician
userPassword: renroCyels
uid: Ansley_Corner
givenName: Ansley
mail: Ansley_Corner@example.com
carLicense: FE4A53S
departmentNumber: 940
employeeType: Manager
homePhone: +1 71 287-7267
initials: A. C.
mobile: +1 71 188-6134
pager: +1 415 927-6612
manager: cn=Pension Kotler
secretary: cn=Christabella Zumhagen
roomNumber: 7935

dn: cn=Colette Longo, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Colette Longo
sn: Longo
description: This is Colette Longo's description
facsimileTelephoneNumber: +1 206 502-4699
l: Mountain View
ou: Peons
postalAddress: example$Peons$Dept # 84
telephoneNumber: +1 415 940-4230
title: Senior Peons Pinhead
userPassword: ognoLettel
uid: Colette_Longo
givenName: Colette
mail: Colette_Longo@example.com
carLicense: H8VXVTT
departmentNumber: 1994
employeeType: Temp
homePhone: +1 303 871-6476
initials: C. L.
mobile: +1 510 620-6333
pager: +1 206 331-7790
manager: cn=Micro Herlihy
secretary: cn=Fahim Averett
roomNumber: 4931

dn: cn=Loretta Clason, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Loretta Clason
sn: Clason
description: This is Loretta Clason's description
facsimileTelephoneNumber: +1 303 261-3535
l: Orem
ou: Payroll
postalAddress: example$Payroll$Dept # 428
telephoneNumber: +1 510 389-8623
title: Chief Payroll Figurehead
userPassword: nosalCatte
uid: Loretta_Clason
givenName: Loretta
mail: Loretta_Clason@example.com
carLicense: BRE3ORB
departmentNumber: 7915
employeeType: Employee
homePhone: +1 206 716-6752
initials: L. C.
mobile: +1 415 234-1691
pager: +1 71 636-3923
manager: cn=Gihan Slade
secretary: cn=Caresse Hui
roomNumber: 1958

dn: cn=Eloise Haggarty, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Eloise Haggarty
sn: Haggarty
description: This is Eloise Haggarty's description
facsimileTelephoneNumber: +1 71 852-9666
l: Milpitas
ou: Management
postalAddress: example$Management$Dept # 223
telephoneNumber: +1 213 552-9784
title: Elite Management Engineer
userPassword: ytraggaHes
uid: Eloise_Haggarty
givenName: Eloise
mail: Eloise_Haggarty@example.com
carLicense: 9EIE7SA
departmentNumber: 217
employeeType: Contract
homePhone: +1 804 871-6041
initials: E. H.
mobile: +1 303 695-9076
pager: +1 408 276-3706
manager: cn=Opalina DeNoon
secretary: cn=Wilford Leiding
roomNumber: 2135

dn: cn=Allix Regier, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Allix Regier
sn: Regier
description: This is Allix Regier's description
facsimileTelephoneNumber: +1 408 654-3843
l: Orem
ou: Product Testing
postalAddress: example$Product Testing$Dept # 24
telephoneNumber: +1 510 353-4168
title: Chief Product Testing Developer
userPassword: reigeRxill
uid: Allix_Regier
givenName: Allix
mail: Allix_Regier@example.com
carLicense: KPZCB05
departmentNumber: 4046
employeeType: Employee
homePhone: +1 818 521-3665
initials: A. R.
mobile: +1 510 893-1785
pager: +1 510 670-5202
manager: cn=Madelaine Rahrer
secretary: cn=Valli Townley
roomNumber: 2766

dn: cn=Brittany McConnell, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Brittany McConnell
sn: McConnell
description: This is Brittany McConnell's description
facsimileTelephoneNumber: +1 303 841-2461
l: Cambridge
ou: Planning
postalAddress: example$Planning$Dept # 949
telephoneNumber: +1 818 187-1717
title: Junior Planning Czar
userPassword: llennoCcMy
uid: Brittany_McConnell
givenName: Brittany
mail: Brittany_McConnell@example.com
carLicense: RPJRY69
departmentNumber: 3424
employeeType: Manager
homePhone: +1 303 704-2226
initials: B. M.
mobile: +1 213 214-3692
pager: +1 804 757-4328
manager: cn=Surinder Sridhar
secretary: cn=Discover Beasley
roomNumber: 4794

dn: cn=Blythe Lathangue, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Blythe Lathangue
sn: Lathangue
description: This is Blythe Lathangue's description
facsimileTelephoneNumber: +1 206 202-9249
l: San Jose
ou: Administrative
postalAddress: example$Administrative$Dept # 63
telephoneNumber: +1 408 614-1741
title: Master Administrative Admin
userPassword: eugnahtaLe
uid: Blythe_Lathangue
givenName: Blythe
mail: Blythe_Lathangue@example.com
carLicense: NHQ5KNV
departmentNumber: 7548
employeeType: Temp
homePhone: +1 408 821-1540
initials: B. L.
mobile: +1 213 728-5608
pager: +1 303 265-1876
manager: cn=Caritta Visentin
secretary: cn=Gayl Fastfeat
roomNumber: 8766

dn: cn=Arts Sahli, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Arts Sahli
sn: Sahli
description: This is Arts Sahli's description
facsimileTelephoneNumber: +1 213 325-2328
l: Sunnyvale
ou: Product Development
postalAddress: example$Product Development$Dept # 702
telephoneNumber: +1 303 722-7085
title: Senior Product Development Admin
userPassword: ilhaSstrA
uid: Arts_Sahli
givenName: Arts
mail: Arts_Sahli@example.com
carLicense: 32X4B7R
departmentNumber: 1729
employeeType: Contract
homePhone: +1 510 156-9430
initials: A. S.
mobile: +1 303 830-5331
pager: +1 415 948-7378
manager: cn=Ardyth Veloz
secretary: cn=Canadian Wyble
roomNumber: 1973

dn: cn=Ryszard Finak, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ryszard Finak
sn: Finak
description: This is Ryszard Finak's description
facsimileTelephoneNumber: +1 510 364-3425
l: Mountain View
ou: Human Resources
postalAddress: example$Human Resources$Dept # 803
telephoneNumber: +1 213 164-9389
title: Elite Human Resources Technician
userPassword: kaniFdrazs
uid: Ryszard_Finak
givenName: Ryszard
mail: Ryszard_Finak@example.com
carLicense: C2TVT5L
departmentNumber: 684
employeeType: Employee
homePhone: +1 510 536-4963
initials: R. F.
mobile: +1 303 425-7763
pager: +1 510 214-3756
manager: cn=Yasmin Langer
secretary: cn=Elsy Porterfield
roomNumber: 5128

dn: cn=Marga Heroux, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marga Heroux
sn: Heroux
description: This is Marga Heroux's description
facsimileTelephoneNumber: +1 415 999-9233
l: Emeryville
ou: Janitorial
postalAddress: example$Janitorial$Dept # 88
telephoneNumber: +1 415 469-4223
title: Master Janitorial Grunt
userPassword: xuoreHagra
uid: Marga_Heroux
givenName: Marga
mail: Marga_Heroux@example.com
carLicense: X98UNCY
departmentNumber: 3188
employeeType: Normal
homePhone: +1 415 294-9705
initials: M. H.
mobile: +1 71 414-5756
pager: +1 818 491-3954
manager: cn=Edee Ludwig
secretary: cn=Claudetta Khorami
roomNumber: 6829

dn: cn=Warren Lingafelter, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Warren Lingafelter
sn: Lingafelter
description: This is Warren Lingafelter's description
facsimileTelephoneNumber: +1 213 184-3150
l: Cambridge
ou: Management
postalAddress: example$Management$Dept # 800
telephoneNumber: +1 303 895-2893
title: Supreme Management Architect
userPassword: retlefagni
uid: Warren_Lingafelter
givenName: Warren
mail: Warren_Lingafelter@example.com
carLicense: SWA3JR6
departmentNumber: 6906
employeeType: Employee
homePhone: +1 71 372-3359
initials: W. L.
mobile: +1 206 920-7077
pager: +1 415 676-6356
manager: cn=Ly-Khanh Biersach
secretary: cn=Peggi Bartley
roomNumber: 4949

dn: cn=Divine Pascas, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Divine Pascas
sn: Pascas
description: This is Divine Pascas's description
facsimileTelephoneNumber: +1 408 906-1374
l: Santa Clara
ou: Human Resources
postalAddress: example$Human Resources$Dept # 246
telephoneNumber: +1 206 486-4052
title: Elite Human Resources Director
userPassword: sacsaPeniv
uid: Divine_Pascas
givenName: Divine
mail: Divine_Pascas@example.com
carLicense: S2ZW1JU
departmentNumber: 7609
employeeType: Employee
homePhone: +1 510 533-4634
initials: D. P.
mobile: +1 818 823-8494
pager: +1 213 152-4421
manager: cn=Zena Taki
secretary: cn=Aurora DiSisto
roomNumber: 9317

dn: cn=Gavin Belcher, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gavin Belcher
sn: Belcher
description: This is Gavin Belcher's description
facsimileTelephoneNumber: +1 303 583-6380
l: Milpitas
ou: Product Development
postalAddress: example$Product Development$Dept # 839
telephoneNumber: +1 213 673-3041
title: Master Product Development Director
userPassword: rehcleBniv
uid: Gavin_Belcher
givenName: Gavin
mail: Gavin_Belcher@example.com
carLicense: FSNYGVG
departmentNumber: 8596
employeeType: Contract
homePhone: +1 213 683-9693
initials: G. B.
mobile: +1 818 250-9957
pager: +1 804 713-6338
manager: cn=Kennon Sabety
secretary: cn=Kristen Hage
roomNumber: 1730

dn: cn=Peng-David Nash, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Peng-David Nash
sn: Nash
description: This is Peng-David Nash's description
facsimileTelephoneNumber: +1 71 605-1245
l: Palo Alto
ou: Janitorial
postalAddress: example$Janitorial$Dept # 997
telephoneNumber: +1 415 446-9898
title: Junior Janitorial Mascot
userPassword: hsaNdivaD-
uid: Peng-David_Nash
givenName: Peng-David
mail: Peng-David_Nash@example.com
carLicense: GZ8V5YY
departmentNumber: 1320
employeeType: Employee
homePhone: +1 415 480-9066
initials: P. N.
mobile: +1 71 404-8977
pager: +1 818 532-1657
manager: cn=Luigi Schnackenberg
secretary: cn=Emlynn Aghili
roomNumber: 9389

dn: cn=Delancey Schiegl, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Delancey Schiegl
sn: Schiegl
description: This is Delancey Schiegl's description
facsimileTelephoneNumber: +1 415 707-2946
l: Mountain View
ou: Management
postalAddress: example$Management$Dept # 700
telephoneNumber: +1 818 509-7560
title: Elite Management Janitor
userPassword: lgeihcSyec
uid: Delancey_Schiegl
givenName: Delancey
mail: Delancey_Schiegl@example.com
carLicense: K5MOA6C
departmentNumber: 2402
employeeType: Contract
homePhone: +1 303 384-2162
initials: D. S.
mobile: +1 408 163-6633
pager: +1 818 183-8147
manager: cn=Chatri Xu
secretary: cn=Rheta O'Grady
roomNumber: 919

dn: cn=Jeremy Voduc, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jeremy Voduc
sn: Voduc
description: This is Jeremy Voduc's description
facsimileTelephoneNumber: +1 303 304-2896
l: Sunnyvale
ou: Accounting
postalAddress: example$Accounting$Dept # 720
telephoneNumber: +1 804 560-6668
title: Master Accounting Vice President
userPassword: cudoVymere
uid: Jeremy_Voduc
givenName: Jeremy
mail: Jeremy_Voduc@example.com
carLicense: O2N8VGG
departmentNumber: 1401
employeeType: Normal
homePhone: +1 818 421-6153
initials: J. V.
mobile: +1 804 379-5643
pager: +1 804 954-9926
manager: cn=Katsumi Saucerman
secretary: cn=Agnola McTiernan
roomNumber: 9998

dn: cn=Zita Kelleher, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Zita Kelleher
sn: Kelleher
description: This is Zita Kelleher's description
facsimileTelephoneNumber: +1 206 173-5719
l: Palo Alto
ou: Janitorial
postalAddress: example$Janitorial$Dept # 435
telephoneNumber: +1 510 401-5342
title: Elite Janitorial Technician
userPassword: rehelleKat
uid: Zita_Kelleher
givenName: Zita
mail: Zita_Kelleher@example.com
carLicense: 8P4XOUC
departmentNumber: 8096
employeeType: Temp
homePhone: +1 510 778-2566
initials: Z. K.
mobile: +1 804 403-4644
pager: +1 303 842-1400
manager: cn=Ania Sidor
secretary: cn=JR Kendrick
roomNumber: 4638

dn: cn=Melva Feutlinske, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Melva Feutlinske
sn: Feutlinske
description: This is Melva Feutlinske's description
facsimileTelephoneNumber: +1 303 816-7419
l: Redmond
ou: Payroll
postalAddress: example$Payroll$Dept # 569
telephoneNumber: +1 206 254-9854
title: Supreme Payroll Developer
userPassword: eksniltueF
uid: Melva_Feutlinske
givenName: Melva
mail: Melva_Feutlinske@example.com
carLicense: GTWW23E
departmentNumber: 7474
employeeType: Contract
homePhone: +1 818 295-5341
initials: M. F.
mobile: +1 71 688-4474
pager: +1 408 815-2399
manager: cn=Tdr Chalker
secretary: cn=Nalani Hibler
roomNumber: 1557

dn: cn=Pey-Kee Tharby, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Pey-Kee Tharby
sn: Tharby
description: This is Pey-Kee Tharby's description
facsimileTelephoneNumber: +1 804 782-7676
l: San Mateo
ou: Planning
postalAddress: example$Planning$Dept # 350
telephoneNumber: +1 213 542-1425
title: Supreme Planning Technician
userPassword: ybrahTeeK-
uid: Pey-Kee_Tharby
givenName: Pey-Kee
mail: Pey-Kee_Tharby@example.com
carLicense: X4AIZ8B
departmentNumber: 9292
employeeType: Contract
homePhone: +1 71 183-2147
initials: P. T.
mobile: +1 510 109-8298
pager: +1 213 557-3084
manager: cn=Nader Noguchi
secretary: cn=Jessica Drayton
roomNumber: 2936

dn: cn=Harli Zagrodney, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Harli Zagrodney
sn: Zagrodney
description: This is Harli Zagrodney's description
facsimileTelephoneNumber: +1 415 711-4965
l: Alameda
ou: Human Resources
postalAddress: example$Human Resources$Dept # 227
telephoneNumber: +1 206 546-5529
title: Master Human Resources Architect
userPassword: yendorgaZi
uid: Harli_Zagrodney
givenName: Harli
mail: Harli_Zagrodney@example.com
carLicense: JV2I693
departmentNumber: 7952
employeeType: Contract
homePhone: +1 71 191-6885
initials: H. Z.
mobile: +1 71 223-5022
pager: +1 303 760-1890
manager: cn=Seamus Bonneau
secretary: cn=Willamina Drakage
roomNumber: 2895

dn: cn=Desmond MacLennan, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Desmond MacLennan
sn: MacLennan
description: This is Desmond MacLennan's description
facsimileTelephoneNumber: +1 415 637-2954
l: Orem
ou: Administrative
postalAddress: example$Administrative$Dept # 822
telephoneNumber: +1 415 168-8626
title: Chief Administrative Writer
userPassword: nanneLcaMd
uid: Desmond_MacLennan
givenName: Desmond
mail: Desmond_MacLennan@example.com
carLicense: EBVD8VQ
departmentNumber: 9259
employeeType: Temp
homePhone: +1 818 950-6537
initials: D. M.
mobile: +1 408 946-9432
pager: +1 804 975-4570
manager: cn=Mer Testsds
secretary: cn=Ranique Chapin
roomNumber: 944

dn: cn=Jaquith Chesterfield, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jaquith Chesterfield
sn: Chesterfield
description: This is Jaquith Chesterfield's description
facsimileTelephoneNumber: +1 213 680-7831
l: Alameda
ou: Accounting
postalAddress: example$Accounting$Dept # 690
telephoneNumber: +1 804 157-8232
title: Junior Accounting Mascot
userPassword: dleifretse
uid: Jaquith_Chesterfield
givenName: Jaquith
mail: Jaquith_Chesterfield@example.com
carLicense: VE9DC2B
departmentNumber: 9425
employeeType: Contract
homePhone: +1 206 552-1958
initials: J. C.
mobile: +1 510 965-7707
pager: +1 213 517-1647
manager: cn=Hpone Irccar
secretary: cn=Ginette Benchimol
roomNumber: 7035

dn: cn=Shara Okon, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shara Okon
sn: Okon
description: This is Shara Okon's description
facsimileTelephoneNumber: +1 415 655-2039
l: Santa Clara
ou: Human Resources
postalAddress: example$Human Resources$Dept # 828
telephoneNumber: +1 415 390-4298
title: Associate Human Resources Director
userPassword: nokOarahS
uid: Shara_Okon
givenName: Shara
mail: Shara_Okon@example.com
carLicense: 6WQE2UM
departmentNumber: 927
employeeType: Temp
homePhone: +1 804 178-4483
initials: S. O.
mobile: +1 818 381-8543
pager: +1 408 449-7208
manager: cn=Erina Cleary
secretary: cn=Truman Lonsdale
roomNumber: 7763

dn: cn=Lynnelle Kausche, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lynnelle Kausche
sn: Kausche
description: This is Lynnelle Kausche's description
facsimileTelephoneNumber: +1 818 434-9046
l: Redwood Shores
ou: Management
postalAddress: example$Management$Dept # 442
telephoneNumber: +1 71 157-4161
title: Supreme Management Accountant
userPassword: ehcsuaKell
uid: Lynnelle_Kausche
givenName: Lynnelle
mail: Lynnelle_Kausche@example.com
carLicense: E4QWHXO
departmentNumber: 6363
employeeType: Employee
homePhone: +1 408 494-9482
initials: L. K.
mobile: +1 206 959-1722
pager: +1 818 545-9988
manager: cn=Mariquilla Semler
secretary: cn=Hodge Marinos
roomNumber: 1976

dn: cn=Mirjam Lenox, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Mirjam Lenox
sn: Lenox
description: This is Mirjam Lenox's description
facsimileTelephoneNumber: +1 206 835-1106
l: San Mateo
ou: Planning
postalAddress: example$Planning$Dept # 979
telephoneNumber: +1 408 690-1570
title: Master Planning Assistant
userPassword: xoneLmajri
uid: Mirjam_Lenox
givenName: Mirjam
mail: Mirjam_Lenox@example.com
carLicense: DL2TQCH
departmentNumber: 5969
employeeType: Manager
homePhone: +1 408 678-9627
initials: M. L.
mobile: +1 303 556-4017
pager: +1 206 268-6154
manager: cn=Rennie Lightfoot
secretary: cn=Karon Zeidler
roomNumber: 3581

dn: cn=Nayneshkumar Rylott, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Nayneshkumar Rylott
sn: Rylott
description: This is Nayneshkumar Rylott's description
facsimileTelephoneNumber: +1 804 224-3420
l: Alameda
ou: Planning
postalAddress: example$Planning$Dept # 839
telephoneNumber: +1 818 795-4425
title: Junior Planning Technician
userPassword: ttolyRramu
uid: Nayneshkumar_Rylott
givenName: Nayneshkumar
mail: Nayneshkumar_Rylott@example.com
carLicense: CC6TJTD
departmentNumber: 7263
employeeType: Employee
homePhone: +1 510 845-9141
initials: N. R.
mobile: +1 71 388-7333
pager: +1 804 403-9908
manager: cn=Lucilia Sliter
secretary: cn=Mario Foley
roomNumber: 5827

dn: cn=Wai-Hung Hikita, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Wai-Hung Hikita
sn: Hikita
description: This is Wai-Hung Hikita's description
facsimileTelephoneNumber: +1 303 692-1566
l: San Mateo
ou: Peons
postalAddress: example$Peons$Dept # 297
telephoneNumber: +1 71 340-8208
title: Elite Peons Figurehead
userPassword: atikiHgnuH
uid: Wai-Hung_Hikita
givenName: Wai-Hung
mail: Wai-Hung_Hikita@example.com
carLicense: 6ZUD5PW
departmentNumber: 2205
employeeType: Contract
homePhone: +1 408 216-8094
initials: W. H.
mobile: +1 71 190-3801
pager: +1 213 825-8550
manager: cn=Belen Baskaran
secretary: cn=Amabel Sizto
roomNumber: 200

dn: cn=Marguerite St.Laurent, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marguerite St.Laurent
sn: St.Laurent
description: This is Marguerite St.Laurent's description
facsimileTelephoneNumber: +1 71 262-8043
l: Cambridge
ou: Management
postalAddress: example$Management$Dept # 181
telephoneNumber: +1 206 747-5983
title: Master Management Yahoo
userPassword: tneruaL.tS
uid: Marguerite_St.Laurent
givenName: Marguerite
mail: Marguerite_St.Laurent@example.com
carLicense: XS00HHV
departmentNumber: 6589
employeeType: Temp
homePhone: +1 213 904-4673
initials: M. S.
mobile: +1 804 381-1347
pager: +1 408 183-8578
manager: cn=Armine Klapper
secretary: cn=Josine Rowe
roomNumber: 985

dn: cn=Tessty Purson, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tessty Purson
sn: Purson
description: This is Tessty Purson's description
facsimileTelephoneNumber: +1 415 685-7459
l: Redmond
ou: Janitorial
postalAddress: example$Janitorial$Dept # 373
telephoneNumber: +1 415 280-6139
title: Elite Janitorial Vice President
userPassword: nosruPytss
uid: Tessty_Purson
givenName: Tessty
mail: Tessty_Purson@example.com
carLicense: G5XNA9M
departmentNumber: 254
employeeType: Employee
homePhone: +1 510 591-2471
initials: T. P.
mobile: +1 303 327-4095
pager: +1 206 461-7570
manager: cn=Adeline Harrison
secretary: cn=Elfrida Netlink
roomNumber: 489

dn: cn=Shaylah Riddall, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shaylah Riddall
sn: Riddall
description: This is Shaylah Riddall's description
facsimileTelephoneNumber: +1 818 542-3574
l: Armonk
ou: Accounting
postalAddress: example$Accounting$Dept # 23
telephoneNumber: +1 213 978-1302
title: Senior Accounting Mascot
userPassword: lladdiRhal
uid: Shaylah_Riddall
givenName: Shaylah
mail: Shaylah_Riddall@example.com
carLicense: 5L1I6GB
departmentNumber: 2947
employeeType: Employee
homePhone: +1 303 501-9242
initials: S. R.
mobile: +1 415 325-6243
pager: +1 415 869-9605
manager: cn=Aloisia Freno
secretary: cn=Gin Mejury
roomNumber: 2149

dn: cn=Dorothy Drabek, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dorothy Drabek
sn: Drabek
description: This is Dorothy Drabek's description
facsimileTelephoneNumber: +1 206 358-7831
l: Mountain View
ou: Planning
postalAddress: example$Planning$Dept # 200
telephoneNumber: +1 408 889-5119
title: Elite Planning Architect
userPassword: kebarDyhto
uid: Dorothy_Drabek
givenName: Dorothy
mail: Dorothy_Drabek@example.com
carLicense: AUJ0A94
departmentNumber: 8158
employeeType: Manager
homePhone: +1 408 579-1212
initials: D. D.
mobile: +1 206 258-1611
pager: +1 408 912-1506
manager: cn=Jemie Varia
secretary: cn=Cornelle Saito
roomNumber: 8163

dn: cn=Karlie Puchala, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Karlie Puchala
sn: Puchala
description: This is Karlie Puchala's description
facsimileTelephoneNumber: +1 818 255-2304
l: Cupertino
ou: Management
postalAddress: example$Management$Dept # 293
telephoneNumber: +1 71 137-3502
title: Senior Management President
userPassword: alahcuPeil
uid: Karlie_Puchala
givenName: Karlie
mail: Karlie_Puchala@example.com
carLicense: FKN3WCH
departmentNumber: 6877
employeeType: Temp
homePhone: +1 408 191-5389
initials: K. P.
mobile: +1 804 796-2391
pager: +1 804 330-4034
manager: cn=Darell Yeo
secretary: cn=Twiggy Lumsden
roomNumber: 6693

dn: cn=Shir Steinhart, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shir Steinhart
sn: Steinhart
description: This is Shir Steinhart's description
facsimileTelephoneNumber: +1 415 441-3047
l: Menlo Park
ou: Administrative
postalAddress: example$Administrative$Dept # 348
telephoneNumber: +1 408 169-5447
title: Master Administrative Accountant
userPassword: trahnietSr
uid: Shir_Steinhart
givenName: Shir
mail: Shir_Steinhart@example.com
carLicense: A3YI0EV
departmentNumber: 5362
employeeType: Manager
homePhone: +1 804 374-7051
initials: S. S.
mobile: +1 71 598-8269
pager: +1 415 965-2419
manager: cn=Swact Palczuk
secretary: cn=Athena Gronwall
roomNumber: 4292

dn: cn=Gabrila Moyce, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Gabrila Moyce
sn: Moyce
description: This is Gabrila Moyce's description
facsimileTelephoneNumber: +1 408 645-6645
l: San Francisco
ou: Accounting
postalAddress: example$Accounting$Dept # 903
telephoneNumber: +1 206 167-8766
title: Master Accounting Writer
userPassword: ecyoMalirb
uid: Gabrila_Moyce
givenName: Gabrila
mail: Gabrila_Moyce@example.com
carLicense: CFEYKKJ
departmentNumber: 9923
employeeType: Normal
homePhone: +1 213 986-3628
initials: G. M.
mobile: +1 213 235-8211
pager: +1 408 611-7139
manager: cn=Priscella Barnhill
secretary: cn=Joshi Harsham
roomNumber: 2020

dn: cn=Kyle Horak, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kyle Horak
sn: Horak
description: This is Kyle Horak's description
facsimileTelephoneNumber: +1 510 730-8450
l: San Mateo
ou: Peons
postalAddress: example$Peons$Dept # 520
telephoneNumber: +1 804 647-7251
title: Senior Peons Madonna
userPassword: karoHelyK
uid: Kyle_Horak
givenName: Kyle
mail: Kyle_Horak@example.com
carLicense: MF80EPP
departmentNumber: 3616
employeeType: Employee
homePhone: +1 408 533-1915
initials: K. H.
mobile: +1 303 734-7002
pager: +1 818 491-2294
manager: cn=Jean-Pierre Glasa
secretary: cn=Cory Sherow
roomNumber: 5023

dn: cn=Diena Sasore, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Diena Sasore
sn: Sasore
description: This is Diena Sasore's description
facsimileTelephoneNumber: +1 804 861-8442
l: Armonk
ou: Accounting
postalAddress: example$Accounting$Dept # 476
telephoneNumber: +1 71 280-8828
title: Junior Accounting President
userPassword: erosaSanei
uid: Diena_Sasore
givenName: Diena
mail: Diena_Sasore@example.com
carLicense: 9KCS9VP
departmentNumber: 4238
employeeType: Normal
homePhone: +1 303 915-7053
initials: D. S.
mobile: +1 71 564-4545
pager: +1 71 232-8994
manager: cn=Alia Ladouceur
secretary: cn=Suk-Yin Emmons
roomNumber: 1667

dn: cn=Nadine Terranova, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Nadine Terranova
sn: Terranova
description: This is Nadine Terranova's description
facsimileTelephoneNumber: +1 408 560-2327
l: Alameda
ou: Human Resources
postalAddress: example$Human Resources$Dept # 375
telephoneNumber: +1 818 211-5277
title: Senior Human Resources Accountant
userPassword: avonarreTe
uid: Nadine_Terranova
givenName: Nadine
mail: Nadine_Terranova@example.com
carLicense: FQGXMUQ
departmentNumber: 997
employeeType: Manager
homePhone: +1 213 967-8604
initials: N. T.
mobile: +1 415 685-3299
pager: +1 408 833-5430
manager: cn=Gerti Lavigne
secretary: cn=Lois Senten
roomNumber: 9709

dn: cn=Shayna Samac, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Shayna Samac
sn: Samac
description: This is Shayna Samac's description
facsimileTelephoneNumber: +1 804 259-8033
l: Fremont
ou: Janitorial
postalAddress: example$Janitorial$Dept # 443
telephoneNumber: +1 206 690-7507
title: Supreme Janitorial Architect
userPassword: camaSanyah
uid: Shayna_Samac
givenName: Shayna
mail: Shayna_Samac@example.com
carLicense: MIEQWHF
departmentNumber: 2494
employeeType: Normal
homePhone: +1 818 520-9204
initials: S. S.
mobile: +1 71 253-6155
pager: +1 213 387-5474
manager: cn=Gwynne Alzofon
secretary: cn=Rozalia Shew
roomNumber: 8940

dn: cn=Lisa Cohea, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Lisa Cohea
sn: Cohea
description: This is Lisa Cohea's description
facsimileTelephoneNumber: +1 804 966-8051
l: Fremont
ou: Planning
postalAddress: example$Planning$Dept # 83
telephoneNumber: +1 415 603-3466
title: Chief Planning Janitor
userPassword: aehoCasiL
uid: Lisa_Cohea
givenName: Lisa
mail: Lisa_Cohea@example.com
carLicense: E801YYY
departmentNumber: 4875
employeeType: Normal
homePhone: +1 818 585-3387
initials: L. C.
mobile: +1 415 260-5195
pager: +1 510 498-2437
manager: cn=Huib Harris
secretary: cn=Fei Petrescu
roomNumber: 3870

dn: cn=Isadora Iyer, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Isadora Iyer
sn: Iyer
description: This is Isadora Iyer's description
facsimileTelephoneNumber: +1 804 256-3308
l: San Francisco
ou: Janitorial
postalAddress: example$Janitorial$Dept # 103
telephoneNumber: +1 415 974-7113
title: Junior Janitorial Madonna
userPassword: reyIarodas
uid: Isadora_Iyer
givenName: Isadora
mail: Isadora_Iyer@example.com
carLicense: EC8KKXC
departmentNumber: 5124
employeeType: Normal
homePhone: +1 415 739-9864
initials: I. I.
mobile: +1 213 322-6643
pager: +1 510 234-5627
manager: cn=Coop Cuu
secretary: cn=Esmaria Scarffe
roomNumber: 2485

dn: cn=Daveen Gawargy, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Daveen Gawargy
sn: Gawargy
description: This is Daveen Gawargy's description
facsimileTelephoneNumber: +1 510 471-7023
l: San Jose
ou: Accounting
postalAddress: example$Accounting$Dept # 898
telephoneNumber: +1 71 574-8452
title: Senior Accounting Sales Rep
userPassword: ygrawaGnee
uid: Daveen_Gawargy
givenName: Daveen
mail: Daveen_Gawargy@example.com
carLicense: MKM7FFR
departmentNumber: 9162
employeeType: Contract
homePhone: +1 71 769-4150
initials: D. G.
mobile: +1 804 137-1538
pager: +1 303 280-2981
manager: cn=Carine Shukster
secretary: cn=Alexina Olinyk
roomNumber: 847

dn: cn=Joellen Goodman, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Joellen Goodman
sn: Goodman
description: This is Joellen Goodman's description
facsimileTelephoneNumber: +1 206 231-2887
l: Menlo Park
ou: Management
postalAddress: example$Management$Dept # 938
telephoneNumber: +1 71 651-4278
title: Chief Management Grunt
userPassword: namdooGnel
uid: Joellen_Goodman
givenName: Joellen
mail: Joellen_Goodman@example.com
carLicense: 86WSOEC
departmentNumber: 9611
employeeType: Contract
homePhone: +1 818 874-3167
initials: J. G.
mobile: +1 206 211-5457
pager: +1 71 602-6109
manager: cn=Terrijo Willmott
secretary: cn=Zdenek Bobar
roomNumber: 1124

dn: cn=Dareen Gittins, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dareen Gittins
sn: Gittins
description: This is Dareen Gittins's description
facsimileTelephoneNumber: +1 818 280-8973
l: San Jose
ou: Product Development
postalAddress: example$Product Development$Dept # 360
telephoneNumber: +1 71 395-6697
title: Senior Product Development Vice President
userPassword: snittiGnee
uid: Dareen_Gittins
givenName: Dareen
mail: Dareen_Gittins@example.com
carLicense: DO3CWLS
departmentNumber: 9326
employeeType: Manager
homePhone: +1 408 665-8088
initials: D. G.
mobile: +1 303 874-2234
pager: +1 818 703-1757
manager: cn=Ailyn Joshi
secretary: cn=Clement Sollee
roomNumber: 3397

dn: cn=Hermine Cronk, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Hermine Cronk
sn: Cronk
description: This is Hermine Cronk's description
facsimileTelephoneNumber: +1 510 895-1959
l: Redwood Shores
ou: Peons
postalAddress: example$Peons$Dept # 658
telephoneNumber: +1 213 962-7007
title: Chief Peons Evangelist
userPassword: knorCenimr
uid: Hermine_Cronk
givenName: Hermine
mail: Hermine_Cronk@example.com
carLicense: PPVI97E
departmentNumber: 8517
employeeType: Contract
homePhone: +1 415 150-9419
initials: H. C.
mobile: +1 415 788-7483
pager: +1 213 170-4509
manager: cn=Juan Vosu
secretary: cn=Mary-Ellen Donaghue
roomNumber: 4344

dn: cn=Thor Jasny, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Thor Jasny
sn: Jasny
description: This is Thor Jasny's description
facsimileTelephoneNumber: +1 213 354-4590
l: Palo Alto
ou: Product Development
postalAddress: example$Product Development$Dept # 965
telephoneNumber: +1 510 279-9807
title: Chief Product Development Grunt
userPassword: ynsaJrohT
uid: Thor_Jasny
givenName: Thor
mail: Thor_Jasny@example.com
carLicense: O2TJ41Q
departmentNumber: 4045
employeeType: Manager
homePhone: +1 818 371-7625
initials: T. J.
mobile: +1 213 589-7902
pager: +1 818 264-9432
manager: cn=Minnnie Kinos
secretary: cn=Hensley Yong
roomNumber: 5606

dn: cn=Carlie Jaworsky, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Carlie Jaworsky
sn: Jaworsky
description: This is Carlie Jaworsky's description
facsimileTelephoneNumber: +1 303 728-9050
l: Fremont
ou: Administrative
postalAddress: example$Administrative$Dept # 757
telephoneNumber: +1 206 453-8543
title: Junior Administrative Engineer
userPassword: yksrowaJei
uid: Carlie_Jaworsky
givenName: Carlie
mail: Carlie_Jaworsky@example.com
carLicense: 741F73C
departmentNumber: 2795
employeeType: Manager
homePhone: +1 206 931-8088
initials: C. J.
mobile: +1 206 588-3333
pager: +1 71 780-9748
manager: cn=Mireielle Sayer
secretary: cn=Gursharan Limerick
roomNumber: 8714

dn: cn=Flory Galasso, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Flory Galasso
sn: Galasso
description: This is Flory Galasso's description
facsimileTelephoneNumber: +1 213 342-8878
l: Emeryville
ou: Payroll
postalAddress: example$Payroll$Dept # 750
telephoneNumber: +1 71 657-9872
title: Senior Payroll Pinhead
userPassword: ossalaGyro
uid: Flory_Galasso
givenName: Flory
mail: Flory_Galasso@example.com
carLicense: 8W6ALH6
departmentNumber: 3963
employeeType: Employee
homePhone: +1 510 384-6589
initials: F. G.
mobile: +1 408 503-9620
pager: +1 510 821-5844
manager: cn=Turus Helfrick
secretary: cn=Anetta Van Atta
roomNumber: 103

dn: cn=Eolanda Marcheck, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Eolanda Marcheck
sn: Marcheck
description: This is Eolanda Marcheck's description
facsimileTelephoneNumber: +1 213 981-4975
l: Mountain View
ou: Human Resources
postalAddress: example$Human Resources$Dept # 871
telephoneNumber: +1 408 313-6715
title: Senior Human Resources Evangelist
userPassword: kcehcraMad
uid: Eolanda_Marcheck
givenName: Eolanda
mail: Eolanda_Marcheck@example.com
carLicense: 2HYT26G
departmentNumber: 8600
employeeType: Employee
homePhone: +1 71 900-1527
initials: E. M.
mobile: +1 415 127-9226
pager: +1 303 584-4187
manager: cn=Jinann Tahir
secretary: cn=Graham Partin
roomNumber: 7980

dn: cn=Cindra Wyndham, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Cindra Wyndham
sn: Wyndham
description: This is Cindra Wyndham's description
facsimileTelephoneNumber: +1 415 630-4285
l: San Jose
ou: Product Testing
postalAddress: example$Product Testing$Dept # 398
telephoneNumber: +1 510 998-6945
title: Junior Product Testing Director
userPassword: mahdnyWard
uid: Cindra_Wyndham
givenName: Cindra
mail: Cindra_Wyndham@example.com
carLicense: G7DTZLX
departmentNumber: 4385
employeeType: Manager
homePhone: +1 71 109-9080
initials: C. W.
mobile: +1 510 529-7198
pager: +1 206 878-3972
manager: cn=Whitney Diersch
secretary: cn=How-Kee Zanga
roomNumber: 9338

dn: cn=Donald Zelenka, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Donald Zelenka
sn: Zelenka
description: This is Donald Zelenka's description
facsimileTelephoneNumber: +1 71 582-5331
l: Cupertino
ou: Janitorial
postalAddress: example$Janitorial$Dept # 645
telephoneNumber: +1 206 415-7851
title: Master Janitorial Dictator
userPassword: akneleZdla
uid: Donald_Zelenka
givenName: Donald
mail: Donald_Zelenka@example.com
carLicense: CYKBZQM
departmentNumber: 2861
employeeType: Temp
homePhone: +1 415 569-9913
initials: D. Z.
mobile: +1 206 517-2694
pager: +1 510 443-7549
manager: cn=Ali Wassel
secretary: cn=Meggy Closson
roomNumber: 4406

dn: cn=Tehchi Decker, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Tehchi Decker
sn: Decker
description: This is Tehchi Decker's description
facsimileTelephoneNumber: +1 415 757-6013
l: Palo Alto
ou: Product Testing
postalAddress: example$Product Testing$Dept # 804
telephoneNumber: +1 818 697-5325
title: Elite Product Testing Evangelist
userPassword: rekceDihch
uid: Tehchi_Decker
givenName: Tehchi
mail: Tehchi_Decker@example.com
carLicense: 7APFZ3G
departmentNumber: 8712
employeeType: Normal
homePhone: +1 71 779-4257
initials: T. D.
mobile: +1 303 239-7566
pager: +1 804 201-3465
manager: cn=Kalpit Gilmore
secretary: cn=Marcela Lopes
roomNumber: 6481

dn: cn=Harrison Niu, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Harrison Niu
sn: Niu
description: This is Harrison Niu's description
facsimileTelephoneNumber: +1 303 951-4953
l: Redmond
ou: Product Development
postalAddress: example$Product Development$Dept # 150
telephoneNumber: +1 408 237-4758
title: Master Product Development Fellow
userPassword: uiNnosirra
uid: Harrison_Niu
givenName: Harrison
mail: Harrison_Niu@example.com
carLicense: P10DFW6
departmentNumber: 9212
employeeType: Temp
homePhone: +1 804 549-6549
initials: H. N.
mobile: +1 206 162-9598
pager: +1 303 678-3059
manager: cn=Jordanna Pufpaff
secretary: cn=Seiko Juers
roomNumber: 9865

dn: cn=Donnie Brissette, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Donnie Brissette
sn: Brissette
description: This is Donnie Brissette's description
facsimileTelephoneNumber: +1 408 596-9307
l: Redwood Shores
ou: Human Resources
postalAddress: example$Human Resources$Dept # 689
telephoneNumber: +1 206 667-8532
title: Junior Human Resources Artist
userPassword: ettessirBe
uid: Donnie_Brissette
givenName: Donnie
mail: Donnie_Brissette@example.com
carLicense: XJC3KQV
departmentNumber: 1761
employeeType: Temp
homePhone: +1 415 452-6922
initials: D. B.
mobile: +1 71 958-4191
pager: +1 303 675-4208
manager: cn=Terrell Witkowski
secretary: cn=Elsey Michaelson
roomNumber: 4992

dn: cn=May Gaul, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: May Gaul
sn: Gaul
description: This is May Gaul's description
facsimileTelephoneNumber: +1 804 673-8690
l: Sunnyvale
ou: Accounting
postalAddress: example$Accounting$Dept # 930
telephoneNumber: +1 408 696-5756
title: Associate Accounting Developer
userPassword: luaGyaM
uid: May_Gaul
givenName: May
mail: May_Gaul@example.com
carLicense: 1BMCX31
departmentNumber: 5148
employeeType: Temp
homePhone: +1 415 245-8979
initials: M. G.
mobile: +1 818 141-8493
pager: +1 804 480-4264
manager: cn=Fqa McMannen
secretary: cn=Daniel Encomenderos
roomNumber: 5933

dn: cn=Kaylee Golaszewski, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kaylee Golaszewski
sn: Golaszewski
description: This is Kaylee Golaszewski's description
facsimileTelephoneNumber: +1 415 527-7345
l: Palo Alto
ou: Management
postalAddress: example$Management$Dept # 358
telephoneNumber: +1 71 161-2141
title: Junior Management Warrior
userPassword: ikswezsalo
uid: Kaylee_Golaszewski
givenName: Kaylee
mail: Kaylee_Golaszewski@example.com
carLicense: HLN6PNH
departmentNumber: 5533
employeeType: Temp
homePhone: +1 206 212-5534
initials: K. G.
mobile: +1 71 947-6940
pager: +1 408 685-6889
manager: cn=Dulcy Runkel
secretary: cn=Duane Debord
roomNumber: 9891

dn: cn=Teodora Bjornson, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Teodora Bjornson
sn: Bjornson
description: This is Teodora Bjornson's description
facsimileTelephoneNumber: +1 408 657-3540
l: Alameda
ou: Planning
postalAddress: example$Planning$Dept # 401
telephoneNumber: +1 415 533-9032
title: Elite Planning Manager
userPassword: nosnrojBar
uid: Teodora_Bjornson
givenName: Teodora
mail: Teodora_Bjornson@example.com
carLicense: PKTB45Y
departmentNumber: 7265
employeeType: Contract
homePhone: +1 71 620-6107
initials: T. B.
mobile: +1 804 609-9614
pager: +1 206 842-3624
manager: cn=Nagaraj Barwikowski
secretary: cn=Charo Markell
roomNumber: 1590

dn: cn=GeorgeAnn Stanulis, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: GeorgeAnn Stanulis
sn: Stanulis
description: This is GeorgeAnn Stanulis's description
facsimileTelephoneNumber: +1 415 772-7968
l: San Jose
ou: Accounting
postalAddress: example$Accounting$Dept # 365
telephoneNumber: +1 804 626-8319
title: Senior Accounting Dictator
userPassword: silunatSnn
uid: GeorgeAnn_Stanulis
givenName: GeorgeAnn
mail: GeorgeAnn_Stanulis@example.com
carLicense: SQ5C9MI
departmentNumber: 3229
employeeType: Contract
homePhone: +1 818 351-2650
initials: G. S.
mobile: +1 818 765-9210
pager: +1 415 373-6213
manager: cn=Jean-Robert Beconovich
secretary: cn=Rosabelle Strober
roomNumber: 1896

dn: cn=Yvette Ludwig, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Yvette Ludwig
sn: Ludwig
description: This is Yvette Ludwig's description
facsimileTelephoneNumber: +1 303 306-5445
l: San Mateo
ou: Planning
postalAddress: example$Planning$Dept # 481
telephoneNumber: +1 415 811-1657
title: Senior Planning Yahoo
userPassword: giwduLette
uid: Yvette_Ludwig
givenName: Yvette
mail: Yvette_Ludwig@example.com
carLicense: YLN7XSN
departmentNumber: 7920
employeeType: Normal
homePhone: +1 206 960-4637
initials: Y. L.
mobile: +1 510 634-4008
pager: +1 818 922-2134
manager: cn=Willa Dirilten
secretary: cn=Tresrch Veloz
roomNumber: 3845

dn: cn=Fahim Eimer, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Fahim Eimer
sn: Eimer
description: This is Fahim Eimer's description
facsimileTelephoneNumber: +1 206 871-3266
l: Sunnyvale
ou: Human Resources
postalAddress: example$Human Resources$Dept # 49
telephoneNumber: +1 206 250-7419
title: Associate Human Resources Czar
userPassword: remiEmihaF
uid: Fahim_Eimer
givenName: Fahim
mail: Fahim_Eimer@example.com
carLicense: QU9NIPK
departmentNumber: 9211
employeeType: Employee
homePhone: +1 510 923-5012
initials: F. E.
mobile: +1 206 206-1510
pager: +1 71 554-8588
manager: cn=Cindra Williamson
secretary: cn=Reza Rickborn
roomNumber: 1814

dn: cn=Spenser Purson, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Spenser Purson
sn: Purson
description: This is Spenser Purson's description
facsimileTelephoneNumber: +1 510 270-3830
l: Cupertino
ou: Human Resources
postalAddress: example$Human Resources$Dept # 827
telephoneNumber: +1 206 703-2247
title: Associate Human Resources President
userPassword: nosruPresn
uid: Spenser_Purson
givenName: Spenser
mail: Spenser_Purson@example.com
carLicense: NIGEIRV
departmentNumber: 7309
employeeType: Employee
homePhone: +1 71 442-4522
initials: S. P.
mobile: +1 804 535-5046
pager: +1 510 949-7721
manager: cn=Alison Burdett
secretary: cn=Nellie Jagernauth
roomNumber: 9391

dn: cn=Ertha Calva, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ertha Calva
sn: Calva
description: This is Ertha Calva's description
facsimileTelephoneNumber: +1 408 521-8377
l: Cupertino
ou: Administrative
postalAddress: example$Administrative$Dept # 28
telephoneNumber: +1 804 948-5631
title: Senior Administrative President
userPassword: avlaCahtrE
uid: Ertha_Calva
givenName: Ertha
mail: Ertha_Calva@example.com
carLicense: KFNBO9N
departmentNumber: 1757
employeeType: Employee
homePhone: +1 818 821-5081
initials: E. C.
mobile: +1 206 671-1674
pager: +1 303 461-2259
manager: cn=Cassie Wepf
secretary: cn=Lorette Sawchuk
roomNumber: 386

dn: cn=Rickrd Integration, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rickrd Integration
sn: Integration
description: This is Rickrd Integration's description
facsimileTelephoneNumber: +1 415 264-5076
l: Sunnyvale
ou: Management
postalAddress: example$Management$Dept # 753
telephoneNumber: +1 415 941-3374
title: Associate Management Fellow
userPassword: noitargetn
uid: Rickrd_Integration
givenName: Rickrd
mail: Rickrd_Integration@example.com
carLicense: R9T7FZC
departmentNumber: 3644
employeeType: Contract
homePhone: +1 303 751-6571
initials: R. I.
mobile: +1 303 201-8042
pager: +1 213 486-4874
manager: cn=Reine Lidster
secretary: cn=Elayne Sutter
roomNumber: 3015

dn: cn=Starlet Cervantes, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Starlet Cervantes
sn: Cervantes
description: This is Starlet Cervantes's description
facsimileTelephoneNumber: +1 303 691-3116
l: Fremont
ou: Accounting
postalAddress: example$Accounting$Dept # 187
telephoneNumber: +1 71 548-4911
title: Senior Accounting Dictator
userPassword: setnavreCt
uid: Starlet_Cervantes
givenName: Starlet
mail: Starlet_Cervantes@example.com
carLicense: C7YVC78
departmentNumber: 790
employeeType: Manager
homePhone: +1 71 898-3034
initials: S. C.
mobile: +1 213 121-2578
pager: +1 206 364-9296
manager: cn=Xiaojing Destech
secretary: cn=Inessa Brousseau
roomNumber: 2153

dn: cn=Blanca Perkinson, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Blanca Perkinson
sn: Perkinson
description: This is Blanca Perkinson's description
facsimileTelephoneNumber: +1 804 366-1482
l: San Mateo
ou: Payroll
postalAddress: example$Payroll$Dept # 811
telephoneNumber: +1 818 448-6362
title: Senior Payroll Technician
userPassword: nosnikrePa
uid: Blanca_Perkinson
givenName: Blanca
mail: Blanca_Perkinson@example.com
carLicense: P3C60RZ
departmentNumber: 4444
employeeType: Normal
homePhone: +1 408 517-8796
initials: B. P.
mobile: +1 213 603-6343
pager: +1 71 475-5851
manager: cn=Valerie Pearse
secretary: cn=Oliy Yvon
roomNumber: 9752

dn: cn=Heinz Ibarra, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Heinz Ibarra
sn: Ibarra
description: This is Heinz Ibarra's description
facsimileTelephoneNumber: +1 415 905-1594
l: Palo Alto
ou: Administrative
postalAddress: example$Administrative$Dept # 784
telephoneNumber: +1 206 459-4529
title: Senior Administrative Dictator
userPassword: arrabIznie
uid: Heinz_Ibarra
givenName: Heinz
mail: Heinz_Ibarra@example.com
carLicense: O08CDMU
departmentNumber: 2885
employeeType: Contract
homePhone: +1 303 816-5503
initials: H. I.
mobile: +1 510 901-7355
pager: +1 206 652-3929
manager: cn=Maressa Janseen
secretary: cn=Yihban Teran
roomNumber: 4463

dn: cn=Sieber Hilaire, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sieber Hilaire
sn: Hilaire
description: This is Sieber Hilaire's description
facsimileTelephoneNumber: +1 415 606-3740
l: Milpitas
ou: Product Testing
postalAddress: example$Product Testing$Dept # 769
telephoneNumber: +1 408 531-5819
title: Junior Product Testing Manager
userPassword: erialiHreb
uid: Sieber_Hilaire
givenName: Sieber
mail: Sieber_Hilaire@example.com
carLicense: MUZ3KRL
departmentNumber: 7832
employeeType: Temp
homePhone: +1 415 524-8424
initials: S. H.
mobile: +1 408 960-3501
pager: +1 213 125-6590
manager: cn=Norrie Bedi
secretary: cn=Louis-Philippe Artspssa
roomNumber: 1313

dn: cn=Alana Depelteau, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Alana Depelteau
sn: Depelteau
description: This is Alana Depelteau's description
facsimileTelephoneNumber: +1 303 767-5215
l: San Mateo
ou: Human Resources
postalAddress: example$Human Resources$Dept # 431
telephoneNumber: +1 206 516-5014
title: Junior Human Resources Architect
userPassword: uaetlepeDa
uid: Alana_Depelteau
givenName: Alana
mail: Alana_Depelteau@example.com
carLicense: UNQUU8E
departmentNumber: 9573
employeeType: Manager
homePhone: +1 510 171-5036
initials: A. D.
mobile: +1 303 108-1591
pager: +1 408 492-6874
manager: cn=Loren Drane
secretary: cn=Gee-Meng Pelz
roomNumber: 8281

dn: cn=Viole Wun, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Viole Wun
sn: Wun
description: This is Viole Wun's description
facsimileTelephoneNumber: +1 71 734-2397
l: Cupertino
ou: Planning
postalAddress: example$Planning$Dept # 462
telephoneNumber: +1 408 178-6082
title: Supreme Planning Fellow
userPassword: nuWeloiV
uid: Viole_Wun
givenName: Viole
mail: Viole_Wun@example.com
carLicense: ZI7BP5X
departmentNumber: 7758
employeeType: Normal
homePhone: +1 804 952-4514
initials: V. W.
mobile: +1 213 115-1234
pager: +1 818 630-4157
manager: cn=Naveen Gooley
secretary: cn=Seamus Kruziak
roomNumber: 2010

dn: cn=Melisande Chanchlani, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Melisande Chanchlani
sn: Chanchlani
description: This is Melisande Chanchlani's description
facsimileTelephoneNumber: +1 510 721-3725
l: Redwood Shores
ou: Accounting
postalAddress: example$Accounting$Dept # 302
telephoneNumber: +1 303 412-9372
title: Elite Accounting Punk
userPassword: inalhcnahC
uid: Melisande_Chanchlani
givenName: Melisande
mail: Melisande_Chanchlani@example.com
carLicense: FJELVY9
departmentNumber: 1063
employeeType: Temp
homePhone: +1 818 751-6755
initials: M. C.
mobile: +1 303 611-6743
pager: +1 818 606-2865
manager: cn=Bud Andros
secretary: cn=Ronna Rothwell
roomNumber: 9661

dn: cn=Nha Heyward, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Nha Heyward
sn: Heyward
description: This is Nha Heyward's description
facsimileTelephoneNumber: +1 71 578-9078
l: Emeryville
ou: Human Resources
postalAddress: example$Human Resources$Dept # 384
telephoneNumber: +1 510 875-8611
title: Junior Human Resources Yahoo
userPassword: drawyeHahN
uid: Nha_Heyward
givenName: Nha
mail: Nha_Heyward@example.com
carLicense: L56T36P
departmentNumber: 5119
employeeType: Manager
homePhone: +1 818 948-7798
initials: N. H.
mobile: +1 303 717-9325
pager: +1 415 649-6984
manager: cn=Bria O'Meara
secretary: cn=Ida Hamid
roomNumber: 3891

dn: cn=Ragu Dearaujo, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Ragu Dearaujo
sn: Dearaujo
description: This is Ragu Dearaujo's description
facsimileTelephoneNumber: +1 303 769-4641
l: Menlo Park
ou: Janitorial
postalAddress: example$Janitorial$Dept # 267
telephoneNumber: +1 804 809-1583
title: Master Janitorial Warrior
userPassword: ojuaraeDug
uid: Ragu_Dearaujo
givenName: Ragu
mail: Ragu_Dearaujo@example.com
carLicense: XXYHCRX
departmentNumber: 4647
employeeType: Contract
homePhone: +1 408 698-4340
initials: R. D.
mobile: +1 213 499-7835
pager: +1 206 382-1975
manager: cn=Ehab Bergeson
secretary: cn=Hot Lazarou
roomNumber: 3535

dn: cn=Rosemonde Eales, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Rosemonde Eales
sn: Eales
description: This is Rosemonde Eales's description
facsimileTelephoneNumber: +1 510 556-9306
l: Santa Clara
ou: Management
postalAddress: example$Management$Dept # 925
telephoneNumber: +1 408 841-9033
title: Elite Management Developer
userPassword: selaEednom
uid: Rosemonde_Eales
givenName: Rosemonde
mail: Rosemonde_Eales@example.com
carLicense: PI989TL
departmentNumber: 1473
employeeType: Normal
homePhone: +1 415 813-9480
initials: R. E.
mobile: +1 206 402-3725
pager: +1 510 160-2175
manager: cn=Courtenay Mansbridge
secretary: cn=Sallie Hagan
roomNumber: 3922

dn: cn=Dieuwertje Akita, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dieuwertje Akita
sn: Akita
description: This is Dieuwertje Akita's description
facsimileTelephoneNumber: +1 510 197-7938
l: Fremont
ou: Product Testing
postalAddress: example$Product Testing$Dept # 192
telephoneNumber: +1 71 432-2992
title: Associate Product Testing Mascot
userPassword: atikAejtre
uid: Dieuwertje_Akita
givenName: Dieuwertje
mail: Dieuwertje_Akita@example.com
carLicense: IE2ZCM2
departmentNumber: 4084
employeeType: Normal
homePhone: +1 213 733-4676
initials: D. A.
mobile: +1 408 425-8337
pager: +1 804 877-3834
manager: cn=Clevon Geyer
secretary: cn=Weitzel Bhoday
roomNumber: 8319

dn: cn=Kellyann Horton, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kellyann Horton
sn: Horton
description: This is Kellyann Horton's description
facsimileTelephoneNumber: +1 415 416-3406
l: San Mateo
ou: Human Resources
postalAddress: example$Human Resources$Dept # 670
telephoneNumber: +1 818 886-5953
title: Junior Human Resources Developer
userPassword: notroHnnay
uid: Kellyann_Horton
givenName: Kellyann
mail: Kellyann_Horton@example.com
carLicense: 6ENFXVA
departmentNumber: 7279
employeeType: Contract
homePhone: +1 213 531-8248
initials: K. H.
mobile: +1 510 744-5231
pager: +1 415 231-8672
manager: cn=Jacintha Crowe
secretary: cn=Melosa Kirley
roomNumber: 6502

dn: cn=Truman Nilson, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Truman Nilson
sn: Nilson
description: This is Truman Nilson's description
facsimileTelephoneNumber: +1 206 661-2468
l: Redmond
ou: Product Testing
postalAddress: example$Product Testing$Dept # 733
telephoneNumber: +1 71 896-8519
title: Associate Product Testing Pinhead
userPassword: nosliNnamu
uid: Truman_Nilson
givenName: Truman
mail: Truman_Nilson@example.com
carLicense: JC4C6XQ
departmentNumber: 1844
employeeType: Temp
homePhone: +1 71 146-6637
initials: T. N.
mobile: +1 206 966-6795
pager: +1 804 926-8955
manager: cn=Eliot Paye
secretary: cn=Elissa Steinhart
roomNumber: 4814

dn: cn=Blondie Kember, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Blondie Kember
sn: Kember
description: This is Blondie Kember's description
facsimileTelephoneNumber: +1 415 399-5093
l: Sunnyvale
ou: Janitorial
postalAddress: example$Janitorial$Dept # 54
telephoneNumber: +1 804 111-2699
title: Supreme Janitorial Manager
userPassword: rebmeKeidn
uid: Blondie_Kember
givenName: Blondie
mail: Blondie_Kember@example.com
carLicense: QE7XZMC
departmentNumber: 6003
employeeType: Temp
homePhone: +1 804 146-3460
initials: B. K.
mobile: +1 818 487-4717
pager: +1 213 296-9942
manager: cn=Anabelle Laniel
secretary: cn=Herbert Wakabayashi
roomNumber: 1107

dn: cn=Elladine Bose, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Elladine Bose
sn: Bose
description: This is Elladine Bose's description
facsimileTelephoneNumber: +1 71 289-5853
l: Sunnyvale
ou: Peons
postalAddress: example$Peons$Dept # 947
telephoneNumber: +1 303 971-6203
title: Junior Peons Madonna
userPassword: esoBenidal
uid: Elladine_Bose
givenName: Elladine
mail: Elladine_Bose@example.com
carLicense: QIHW6TY
departmentNumber: 9064
employeeType: Temp
homePhone: +1 206 764-4911
initials: E. B.
mobile: +1 206 848-4223
pager: +1 804 785-8358
manager: cn=Canute Ryder
secretary: cn=Hojjat Debassige
roomNumber: 9444

dn: cn=Demetria Deibert, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Demetria Deibert
sn: Deibert
description: This is Demetria Deibert's description
facsimileTelephoneNumber: +1 71 704-5894
l: San Mateo
ou: Product Development
postalAddress: example$Product Development$Dept # 767
telephoneNumber: +1 415 846-3651
title: Elite Product Development Manager
userPassword: trebieDair
uid: Demetria_Deibert
givenName: Demetria
mail: Demetria_Deibert@example.com
carLicense: L90IKCH
departmentNumber: 2866
employeeType: Contract
homePhone: +1 804 915-6346
initials: D. D.
mobile: +1 303 871-8249
pager: +1 818 791-1975
manager: cn=Meriel Dmuchalsky
secretary: cn=Inger Dba
roomNumber: 6308

dn: cn=Chie Sallee, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Chie Sallee
sn: Sallee
description: This is Chie Sallee's description
facsimileTelephoneNumber: +1 415 492-2678
l: Cambridge
ou: Payroll
postalAddress: example$Payroll$Dept # 370
telephoneNumber: +1 818 704-2633
title: Junior Payroll Director
userPassword: eellaSeihC
uid: Chie_Sallee
givenName: Chie
mail: Chie_Sallee@example.com
carLicense: G6MUNJZ
departmentNumber: 7095
employeeType: Normal
homePhone: +1 71 113-5855
initials: C. S.
mobile: +1 213 701-4279
pager: +1 818 166-2016
manager: cn=Lalitha Archambault
secretary: cn=Pasiedb Bazerghi
roomNumber: 4000

dn: cn=Scot Fralick, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Scot Fralick
sn: Fralick
description: This is Scot Fralick's description
facsimileTelephoneNumber: +1 303 585-1223
l: Santa Clara
ou: Management
postalAddress: example$Management$Dept # 530
telephoneNumber: +1 415 442-4717
title: Associate Management President
userPassword: kcilarFtoc
uid: Scot_Fralick
givenName: Scot
mail: Scot_Fralick@example.com
carLicense: YH6BO4M
departmentNumber: 6096
employeeType: Employee
homePhone: +1 303 598-4845
initials: S. F.
mobile: +1 818 165-3943
pager: +1 510 626-8553
manager: cn=Pde Fulk
secretary: cn=Lowell Tweddle
roomNumber: 6318

dn: cn=Monroe Christensen, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Monroe Christensen
sn: Christensen
description: This is Monroe Christensen's description
facsimileTelephoneNumber: +1 206 356-8279
l: Mountain View
ou: Payroll
postalAddress: example$Payroll$Dept # 148
telephoneNumber: +1 206 912-6866
title: Supreme Payroll Grunt
userPassword: nesnetsirh
uid: Monroe_Christensen
givenName: Monroe
mail: Monroe_Christensen@example.com
carLicense: R6O0NAQ
departmentNumber: 8407
employeeType: Contract
homePhone: +1 213 766-9533
initials: M. C.
mobile: +1 303 556-6768
pager: +1 510 286-2604
manager: cn=Michaeline Health-Safety
secretary: cn=Jobye Brombal
roomNumber: 5351

dn: cn=Kirsteni Finckler, ou=Janitorial, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Kirsteni Finckler
sn: Finckler
description: This is Kirsteni Finckler's description
facsimileTelephoneNumber: +1 510 868-2823
l: Orem
ou: Janitorial
postalAddress: example$Janitorial$Dept # 47
telephoneNumber: +1 818 629-5584
title: Elite Janitorial Janitor
userPassword: relkcniFin
uid: Kirsteni_Finckler
givenName: Kirsteni
mail: Kirsteni_Finckler@example.com
carLicense: V0YDBED
departmentNumber: 3159
employeeType: Contract
homePhone: +1 206 678-7419
initials: K. F.
mobile: +1 818 399-1514
pager: +1 804 253-1019
manager: cn=Jordan Rondeau
secretary: cn=Jasver Cherrier
roomNumber: 3602

dn: cn=Audre Berenbach, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Audre Berenbach
sn: Berenbach
description: This is Audre Berenbach's description
facsimileTelephoneNumber: +1 804 256-5474
l: Emeryville
ou: Product Development
postalAddress: example$Product Development$Dept # 871
telephoneNumber: +1 818 992-8711
title: Associate Product Development Fellow
userPassword: hcabnereBe
uid: Audre_Berenbach
givenName: Audre
mail: Audre_Berenbach@example.com
carLicense: 35TW1J9
departmentNumber: 2112
employeeType: Normal
homePhone: +1 510 334-5649
initials: A. B.
mobile: +1 415 565-1897
pager: +1 818 845-1036
manager: cn=Eolande Colpitts
secretary: cn=Garth Tse
roomNumber: 2863

dn: cn=Anabelle Ludwig, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Anabelle Ludwig
sn: Ludwig
description: This is Anabelle Ludwig's description
facsimileTelephoneNumber: +1 408 855-8095
l: Cupertino
ou: Human Resources
postalAddress: example$Human Resources$Dept # 26
telephoneNumber: +1 408 822-2622
title: Master Human Resources Fellow
userPassword: giwduLelle
uid: Anabelle_Ludwig
givenName: Anabelle
mail: Anabelle_Ludwig@example.com
carLicense: BMWFISB
departmentNumber: 8253
employeeType: Temp
homePhone: +1 71 117-2604
initials: A. L.
mobile: +1 408 915-1480
pager: +1 818 457-5763
manager: cn=Trev Zug
secretary: cn=Cen Subasinghe
roomNumber: 9477

dn: cn=Florri Vandenheede, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Florri Vandenheede
sn: Vandenheede
description: This is Florri Vandenheede's description
facsimileTelephoneNumber: +1 408 149-8781
l: Menlo Park
ou: Product Development
postalAddress: example$Product Development$Dept # 167
telephoneNumber: +1 408 952-1219
title: Chief Product Development Consultant
userPassword: edeehnedna
uid: Florri_Vandenheede
givenName: Florri
mail: Florri_Vandenheede@example.com
carLicense: L3JXPX6
departmentNumber: 9177
employeeType: Contract
homePhone: +1 804 503-2451
initials: F. V.
mobile: +1 804 866-9787
pager: +1 818 747-7775
manager: cn=Esmail Buckalew
secretary: cn=Snehal Jolicoeur
roomNumber: 2265

dn: cn=Sileas Gallo, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sileas Gallo
sn: Gallo
description: This is Sileas Gallo's description
facsimileTelephoneNumber: +1 408 574-5500
l: Mountain View
ou: Product Testing
postalAddress: example$Product Testing$Dept # 926
telephoneNumber: +1 71 907-6499
title: Elite Product Testing Grunt
userPassword: ollaGsaeli
uid: Sileas_Gallo
givenName: Sileas
mail: Sileas_Gallo@example.com
carLicense: HD1VVJI
departmentNumber: 4101
employeeType: Temp
homePhone: +1 818 814-1046
initials: S. G.
mobile: +1 303 440-5276
pager: +1 510 369-1491
manager: cn=Fausto Clipperton
secretary: cn=Sylvie Zieber
roomNumber: 2886

dn: cn=Dinker Vlad, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Dinker Vlad
sn: Vlad
description: This is Dinker Vlad's description
facsimileTelephoneNumber: +1 415 997-1200
l: Sunnyvale
ou: Accounting
postalAddress: example$Accounting$Dept # 383
telephoneNumber: +1 510 270-9293
title: Chief Accounting Grunt
userPassword: dalVrekniD
uid: Dinker_Vlad
givenName: Dinker
mail: Dinker_Vlad@example.com
carLicense: JIEK9OZ
departmentNumber: 3320
employeeType: Employee
homePhone: +1 213 295-5225
initials: D. V.
mobile: +1 213 636-6002
pager: +1 206 198-2064
manager: cn=Haroon Roldan
secretary: cn=Chand Jasrotia
roomNumber: 6160

dn: cn=Norvie Chomik, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Norvie Chomik
sn: Chomik
description: This is Norvie Chomik's description
facsimileTelephoneNumber: +1 213 408-4683
l: Fremont
ou: Human Resources
postalAddress: example$Human Resources$Dept # 514
telephoneNumber: +1 206 338-6639
title: Chief Human Resources Visionary
userPassword: kimohCeivr
uid: Norvie_Chomik
givenName: Norvie
mail: Norvie_Chomik@example.com
carLicense: 4M0417D
departmentNumber: 4617
employeeType: Employee
homePhone: +1 804 131-5356
initials: N. C.
mobile: +1 415 910-2510
pager: +1 213 936-9730
manager: cn=Herbie Andrassy
secretary: cn=Virgina Brungardt
roomNumber: 4408

dn: cn=Eran Preston-Thomas, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Eran Preston-Thomas
sn: Preston-Thomas
description: This is Eran Preston-Thomas's description
facsimileTelephoneNumber: +1 415 805-9017
l: Orem
ou: Management
postalAddress: example$Management$Dept # 70
telephoneNumber: +1 510 490-7704
title: Chief Management Engineer
userPassword: samohT-not
uid: Eran_Preston-Thomas
givenName: Eran
mail: Eran_Preston-Thomas@example.com
carLicense: 8PCVG00
departmentNumber: 4339
employeeType: Temp
homePhone: +1 303 834-3076
initials: E. P.
mobile: +1 804 585-1018
pager: +1 818 871-3707
manager: cn=Rafaela Macquistan
secretary: cn=Mufi Lesourd
roomNumber: 7079

dn: cn=Romulus Giese, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Romulus Giese
sn: Giese
description: This is Romulus Giese's description
facsimileTelephoneNumber: +1 206 196-4952
l: Redwood Shores
ou: Human Resources
postalAddress: example$Human Resources$Dept # 188
telephoneNumber: +1 415 114-5444
title: Junior Human Resources Sales Rep
userPassword: eseiGsulum
uid: Romulus_Giese
givenName: Romulus
mail: Romulus_Giese@example.com
carLicense: 7CMJBHX
departmentNumber: 8930
employeeType: Contract
homePhone: +1 510 460-2913
initials: R. G.
mobile: +1 415 107-8612
pager: +1 303 819-4889
manager: cn=Di Majumdar
secretary: cn=Gerry Overton
roomNumber: 6740

dn: cn=Randal Twarog, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Randal Twarog
sn: Twarog
description: This is Randal Twarog's description
facsimileTelephoneNumber: +1 804 754-9609
l: Alameda
ou: Administrative
postalAddress: example$Administrative$Dept # 952
telephoneNumber: +1 206 956-5103
title: Master Administrative Dictator
userPassword: gorawTladn
uid: Randal_Twarog
givenName: Randal
mail: Randal_Twarog@example.com
carLicense: RFRBHCJ
departmentNumber: 0
employeeType: Temp
homePhone: +1 213 486-7709
initials: R. T.
mobile: +1 213 567-5657
pager: +1 510 314-1948
manager: cn=Patti Goupil
secretary: cn=Sanchez Jachym
roomNumber: 2599

dn: cn=Fritz Claise, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Fritz Claise
sn: Claise
description: This is Fritz Claise's description
facsimileTelephoneNumber: +1 510 531-7295
l: San Mateo
ou: Peons
postalAddress: example$Peons$Dept # 298
telephoneNumber: +1 804 683-2606
title: Elite Peons Pinhead
userPassword: esialCztir
uid: Fritz_Claise
givenName: Fritz
mail: Fritz_Claise@example.com
carLicense: U1PMLLY
departmentNumber: 5506
employeeType: Temp
homePhone: +1 408 907-5910
initials: F. C.
mobile: +1 408 624-3588
pager: +1 71 914-9489
manager: cn=Pierrick Cline
secretary: cn=Herve Bonner
roomNumber: 3200

dn: cn=Djenana Dumouchelle, ou=Payroll, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Djenana Dumouchelle
sn: Dumouchelle
description: This is Djenana Dumouchelle's description
facsimileTelephoneNumber: +1 804 870-4350
l: Redmond
ou: Payroll
postalAddress: example$Payroll$Dept # 960
telephoneNumber: +1 408 482-7997
title: Supreme Payroll Vice President
userPassword: ellehcuomu
uid: Djenana_Dumouchelle
givenName: Djenana
mail: Djenana_Dumouchelle@example.com
carLicense: SCUQAST
departmentNumber: 9280
employeeType: Employee
homePhone: +1 415 992-1209
initials: D. D.
mobile: +1 213 117-4250
pager: +1 415 699-8888
manager: cn=Delores Eyers
secretary: cn=Delly Kok
roomNumber: 7452

dn: cn=La Valko, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: La Valko
sn: Valko
description: This is La Valko's description
facsimileTelephoneNumber: +1 818 542-1967
l: Redmond
ou: Peons
postalAddress: example$Peons$Dept # 132
telephoneNumber: +1 415 193-4715
title: Supreme Peons Sales Rep
userPassword: oklaVaL
uid: La_Valko
givenName: La
mail: La_Valko@example.com
carLicense: GQU0EB6
departmentNumber: 1170
employeeType: Contract
homePhone: +1 408 637-4126
initials: L. V.
mobile: +1 415 141-6794
pager: +1 415 266-6615
manager: cn=Wai-Man Arnauld
secretary: cn=Zola Karhuniemi
roomNumber: 922

dn: cn=Clary Mand, ou=Administrative, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Clary Mand
sn: Mand
description: This is Clary Mand's description
facsimileTelephoneNumber: +1 408 928-6670
l: Alameda
ou: Administrative
postalAddress: example$Administrative$Dept # 55
telephoneNumber: +1 804 111-7612
title: Senior Administrative Stooge
userPassword: dnaMyralC
uid: Clary_Mand
givenName: Clary
mail: Clary_Mand@example.com
carLicense: 2AVIDV3
departmentNumber: 2566
employeeType: Temp
homePhone: +1 213 446-5803
initials: C. M.
mobile: +1 71 687-3763
pager: +1 206 452-2059
manager: cn=Ainslee Hanlan
secretary: cn=Mallik MacArthur
roomNumber: 8363

dn: cn=Freida Gann, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Freida Gann
sn: Gann
description: This is Freida Gann's description
facsimileTelephoneNumber: +1 415 795-8798
l: San Jose
ou: Product Testing
postalAddress: example$Product Testing$Dept # 879
telephoneNumber: +1 213 622-1643
title: Master Product Testing Assistant
userPassword: nnaGadierF
uid: Freida_Gann
givenName: Freida
mail: Freida_Gann@example.com
carLicense: FGJFOA8
departmentNumber: 7796
employeeType: Contract
homePhone: +1 71 782-6115
initials: F. G.
mobile: +1 408 899-5110
pager: +1 804 707-1253
manager: cn=Nat Calkins
secretary: cn=Atmane Guenette
roomNumber: 4156

dn: cn=Sharyl Wefers, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Sharyl Wefers
sn: Wefers
description: This is Sharyl Wefers's description
facsimileTelephoneNumber: +1 408 943-3670
l: Milpitas
ou: Peons
postalAddress: example$Peons$Dept # 451
telephoneNumber: +1 415 853-8475
title: Supreme Peons Yahoo
userPassword: srefeWlyra
uid: Sharyl_Wefers
givenName: Sharyl
mail: Sharyl_Wefers@example.com
carLicense: BHQAXH8
departmentNumber: 2071
employeeType: Contract
homePhone: +1 303 509-4625
initials: S. W.
mobile: +1 71 854-7951
pager: +1 303 947-1628
manager: cn=Fernanda Pachek
secretary: cn=Katie Nicol
roomNumber: 7430

dn: cn=Riki Lahteenmaa, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Riki Lahteenmaa
sn: Lahteenmaa
description: This is Riki Lahteenmaa's description
facsimileTelephoneNumber: +1 303 683-7590
l: Fremont
ou: Product Testing
postalAddress: example$Product Testing$Dept # 419
telephoneNumber: +1 408 392-8634
title: Junior Product Testing Consultant
userPassword: aamneethaL
uid: Riki_Lahteenmaa
givenName: Riki
mail: Riki_Lahteenmaa@example.com
carLicense: LRUSPU7
departmentNumber: 2691
employeeType: Normal
homePhone: +1 303 890-6562
initials: R. L.
mobile: +1 206 112-6791
pager: +1 804 284-2760
manager: cn=Tawauna Masapati
secretary: cn=Leddy Majury
roomNumber: 4146

dn: cn=Damon Leeson, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Damon Leeson
sn: Leeson
description: This is Damon Leeson's description
facsimileTelephoneNumber: +1 818 446-8649
l: Milpitas
ou: Product Development
postalAddress: example$Product Development$Dept # 709
telephoneNumber: +1 303 822-1700
title: Chief Product Development Yahoo
userPassword: noseeLnoma
uid: Damon_Leeson
givenName: Damon
mail: Damon_Leeson@example.com
carLicense: NJN9C3A
departmentNumber: 6908
employeeType: Normal
homePhone: +1 415 206-2910
initials: D. L.
mobile: +1 206 460-9897
pager: +1 71 652-6585
manager: cn=Sydel Bhardwaj
secretary: cn=Nikki Kiel
roomNumber: 2880

dn: cn=Aziz Hawley, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Aziz Hawley
sn: Hawley
description: This is Aziz Hawley's description
facsimileTelephoneNumber: +1 71 936-5160
l: Menlo Park
ou: Human Resources
postalAddress: example$Human Resources$Dept # 238
telephoneNumber: +1 415 139-9638
title: Elite Human Resources Punk
userPassword: yelwaHzizA
uid: Aziz_Hawley
givenName: Aziz
mail: Aziz_Hawley@example.com
carLicense: JJB405S
departmentNumber: 1322
employeeType: Contract
homePhone: +1 206 165-8362
initials: A. H.
mobile: +1 804 985-9213
pager: +1 206 444-5129
manager: cn=Karyn Holloway
secretary: cn=Olav OPSPLNG
roomNumber: 4248

dn: cn=Fox Receiving, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Fox Receiving
sn: Receiving
description: This is Fox Receiving's description
facsimileTelephoneNumber: +1 303 372-8912
l: Sunnyvale
ou: Product Development
postalAddress: example$Product Development$Dept # 522
telephoneNumber: +1 415 455-5816
title: Associate Product Development Pinhead
userPassword: gnivieceRx
uid: Fox_Receiving
givenName: Fox
mail: Fox_Receiving@example.com
carLicense: 1IMTNIY
departmentNumber: 891
employeeType: Normal
homePhone: +1 510 485-3318
initials: F. R.
mobile: +1 303 719-1736
pager: +1 303 665-1600
manager: cn=Anstice Smithson
secretary: cn=Torrie Arnon
roomNumber: 5785

dn: cn=Elleke Wessel, ou=Peons, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Elleke Wessel
sn: Wessel
description: This is Elleke Wessel's description
facsimileTelephoneNumber: +1 213 394-3666
l: Alameda
ou: Peons
postalAddress: example$Peons$Dept # 944
telephoneNumber: +1 71 236-5017
title: Junior Peons Writer
userPassword: lesseWekel
uid: Elleke_Wessel
givenName: Elleke
mail: Elleke_Wessel@example.com
carLicense: 03F3UHJ
departmentNumber: 6509
employeeType: Manager
homePhone: +1 818 284-3358
initials: E. W.
mobile: +1 71 971-2866
pager: +1 415 430-2905
manager: cn=LeRoy Goodner
secretary: cn=Brenn Silieff
roomNumber: 944

dn: cn=Jody Caie, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Jody Caie
sn: Caie
description: This is Jody Caie's description
facsimileTelephoneNumber: +1 303 527-2692
l: Mountain View
ou: Product Testing
postalAddress: example$Product Testing$Dept # 114
telephoneNumber: +1 206 679-1888
title: Chief Product Testing Developer
userPassword: eiaCydoJ
uid: Jody_Caie
givenName: Jody
mail: Jody_Caie@example.com
carLicense: 9BJH2K7
departmentNumber: 5758
employeeType: Temp
homePhone: +1 818 498-6489
initials: J. C.
mobile: +1 71 326-6995
pager: +1 206 514-8361
manager: cn=Verena Misslitz
secretary: cn=Leandra Marschewaki
roomNumber: 1529

dn: cn=Hoa Dandurand, ou=Management, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Hoa Dandurand
sn: Dandurand
description: This is Hoa Dandurand's description
facsimileTelephoneNumber: +1 510 295-6636
l: Redwood Shores
ou: Management
postalAddress: example$Management$Dept # 628
telephoneNumber: +1 71 984-5150
title: Chief Management Pinhead
userPassword: dnarudnaDa
uid: Hoa_Dandurand
givenName: Hoa
mail: Hoa_Dandurand@example.com
carLicense: 9YJFYL1
departmentNumber: 499
employeeType: Manager
homePhone: +1 415 809-1590
initials: H. D.
mobile: +1 71 907-2817
pager: +1 510 862-3357
manager: cn=Sioux Pancholy
secretary: cn=Julietta Briel
roomNumber: 2015

dn: cn=Vlad Hord, ou=Planning, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Vlad Hord
sn: Hord
description: This is Vlad Hord's description
facsimileTelephoneNumber: +1 206 490-1246
l: Sunnyvale
ou: Planning
postalAddress: example$Planning$Dept # 447
telephoneNumber: +1 303 750-4975
title: Senior Planning Director
userPassword: droHdalV
uid: Vlad_Hord
givenName: Vlad
mail: Vlad_Hord@example.com
carLicense: 45D4ZMU
departmentNumber: 7057
employeeType: Manager
homePhone: +1 408 217-9879
initials: V. H.
mobile: +1 206 674-4523
pager: +1 818 315-5452
manager: cn=Irita Swanson
secretary: cn=Anker Mathewson
roomNumber: 9259

dn: cn=Logntp Yaung, ou=Accounting, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Logntp Yaung
sn: Yaung
description: This is Logntp Yaung's description
facsimileTelephoneNumber: +1 804 637-1515
l: Fremont
ou: Accounting
postalAddress: example$Accounting$Dept # 577
telephoneNumber: +1 818 846-6671
title: Chief Accounting Consultant
userPassword: gnuaYptngo
uid: Logntp_Yaung
givenName: Logntp
mail: Logntp_Yaung@example.com
carLicense: 8X0DAD7
departmentNumber: 9019
employeeType: Employee
homePhone: +1 71 243-2137
initials: L. Y.
mobile: +1 408 899-9752
pager: +1 408 186-4812
manager: cn=Aili Benton
secretary: cn=Lilias Settles
roomNumber: 1094

dn: cn=Habeeb Rosenblum, ou=Human Resources, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Habeeb Rosenblum
sn: Rosenblum
description: This is Habeeb Rosenblum's description
facsimileTelephoneNumber: +1 415 101-2600
l: San Francisco
ou: Human Resources
postalAddress: example$Human Resources$Dept # 304
telephoneNumber: +1 206 631-6108
title: Chief Human Resources Figurehead
userPassword: mulbnesoRb
uid: Habeeb_Rosenblum
givenName: Habeeb
mail: Habeeb_Rosenblum@example.com
carLicense: 24KN6IU
departmentNumber: 3001
employeeType: Employee
homePhone: +1 206 227-6825
initials: H. R.
mobile: +1 804 844-7445
pager: +1 510 896-4710
manager: cn=Marek Percy
secretary: cn=Colette Blanche
roomNumber: 4961

dn: cn=Enid Goridkov, ou=Product Development, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Enid Goridkov
sn: Goridkov
description: This is Enid Goridkov's description
facsimileTelephoneNumber: +1 303 636-7461
l: Cambridge
ou: Product Development
postalAddress: example$Product Development$Dept # 484
telephoneNumber: +1 510 658-9835
title: Junior Product Development Madonna
userPassword: vokdiroGdi
uid: Enid_Goridkov
givenName: Enid
mail: Enid_Goridkov@example.com
carLicense: 0FW2NN4
departmentNumber: 3347
employeeType: Normal
homePhone: +1 408 343-1126
initials: E. G.
mobile: +1 818 372-2038
pager: +1 71 423-5647
manager: cn=Mougy Mong
secretary: cn=Nicola Paone
roomNumber: 1860

dn: cn=Marice McCaugherty, ou=Product Testing, dc=example,dc=com
objectClass: top
objectClass: person
objectClass: organizationalPerson
objectClass: inetOrgPerson
cn: Marice McCaugherty
sn: McCaugherty
description: This is Marice McCaugherty's description
facsimileTelephoneNumber: +1 510 519-3382
l: Menlo Park
ou: Product Testing
postalAddress: example$Product Testing$Dept # 26
telephoneNumber: +1 213 496-8242
title: Elite Product Testing Stooge
userPassword: ytrehguaCc
uid: Marice_McCaugherty
givenName: Marice
mail: Marice_McCaugherty@example.com
carLicense: D3P8WWY
departmentNumber: 7100
employeeType: Manager
homePhone: +1 408 101-6964
initials: M. M.
mobile: +1 510 590-8719
pager: +1 303 638-5731
manager: cn=Sada McNeilly
secretary: cn=Antonia Fallows
roomNumber: 7891
EOF


echo $STARTSERVER
$STARTSERVER

echo "Waiting for slapd to complete its startup..."
sleep 10

echo "You can now perform a test search. During the search you can hit the"
echo "interrupt (ctrl-c) key to stop the search at any time."
pecho "Would you like to perform the test search now (y/n)? "
read t

if [ "$t" = "y" ]; then
	echo $TESTSEARCH
	eval $TESTSEARCH
else
	echo "You can run the test yourself any time by typing:"
	echo $TESTSEARCH
fi
